// -------------------------------------------------------------------------
//
//  PROJECT: PCI Express
//  COMPANY: Northwest Logic, Inc.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//                 Copyright 2010 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or
//  transmitted in any form or by any means, electronic or mechanical,
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -------------
// -- Defines --
// -------------

`define BFM_PATH                    tb_top.pcie_bfm
`define INC_ERRORS                  `BFM_PATH.inc_errors
`define DUT_PATH                    tb_top.dut



// -----------------------
// -- Module Definition --
// -----------------------

module master_bfm (

    clk,
    rst_n

);    



// ----------------
// -- Parameters --
// ----------------



// ----------------------
// -- Port Definitions --
// ----------------------

input                               clk;
input                               rst_n;



// ----------------
// -- Port Types --
// ----------------

wire                                clk;
wire                                rst_n;



// -------------------
// -- Local Signals --
// -------------------

wire                                mst_ready;
reg                                 mst_req;
reg     [6:0]                       mst_type;
reg     [31:0]                      mst_data;
reg     [3:0]                       mst_be;
reg     [63:0]                      mst_addr;
reg     [7:0]                       mst_msgcode;
wire    [31:0]                      mst_rd_data;
wire    [2:0]                       mst_status;
wire                                mst_done;

// Misc 
integer                             xfer_keys = 1;      // Number of simultaneous calls allowed to task "xfer"; only 1 is allowed
reg     [31:0]                      next_prior = 32'h0; // For managing access to task xfer: Next priority to assign
reg     [31:0]                      curr_prior = 32'h0; // For managing access to task xfer: Current priority number

reg                                 expect_status_001_error = 1'b0; // Set when Read Status == 0001 is expected to avoid error message being generated



// ---------------
// -- Equations --
// ---------------

assign `DUT_PATH.mst_req      = mst_req;
assign `DUT_PATH.mst_type     = mst_type;
assign `DUT_PATH.mst_data     = mst_data;
assign `DUT_PATH.mst_be       = mst_be;
assign `DUT_PATH.mst_addr     = mst_addr;
assign `DUT_PATH.mst_msgcode  = mst_msgcode;
assign mst_ready              = `DUT_PATH.mst_ready;
assign mst_done               = `DUT_PATH.mst_done;
assign mst_rd_data            = `DUT_PATH.mst_rd_data;
assign mst_status             = `DUT_PATH.mst_status;

// Initialize to idle state
initial
begin
    mst_req      = 0;
    mst_type     = 7'b0;
    mst_data     = 32'b0;
    mst_be       = 4'b0;
    mst_addr     = 64'b0;
    mst_msgcode  = 8'b0;
end

// Semaphore task used to ensure that only 1 xfer task call can 
//   use the core transmit interface at one time; get_xfer_key may
//   only be called by task xfer
task automatic get_xfer_key; 
begin 
    wait (xfer_keys > 0)
        xfer_keys = xfer_keys - 1;
end  
endtask 

// Semaphore task used to ensure that only 1 xfer task call can 
//   use the core transmit interface at one time; put_xfer_key may
//   only be called by task xfer
task automatic put_xfer_key; 
begin 
    xfer_keys = xfer_keys + 1; 
end 
endtask 

task automatic xfer;
    input [6:0]     ttype;
    input [63:0]    addr;
    input [3:0]     be;
    inout [31:0]    data;
    input [7:0]     msg_code;
    output [2:0]    status;

    reg     [31:0]  my_prior;
    reg     [2:0]   cpl_status;

    begin
        get_xfer_key;
        my_prior = next_prior;
        next_prior = next_prior + 1;
        #1;
        put_xfer_key;
        while (my_prior != curr_prior)
            @(posedge clk);
        xfer_action (ttype, addr, be, data, msg_code, status);
    end
endtask

task automatic xfer_action;
    input [6:0]     ttype;
    input [63:0]    addr;
    input [3:0]     be;
    inout [31:0]    data;
    input [7:0]     msg_code;
    output [2:0]    status;

    reg             valid;
    reg             has_payload;
    reg             has_mcode;
    reg             has_response;

    begin
        // wait for ready to be asserted (last operation complete)
        while (mst_ready == 1'b0) begin
            @(posedge clk);
            #1;
        end

        casex (ttype)
            7'b00_00000 : begin valid = 1; has_payload = 0; has_mcode = 0; has_response = 1; end    // 32-bit mem read
            7'b01_00000 : begin valid = 1; has_payload = 0; has_mcode = 0; has_response = 1; end    // 64-bit mem read
            7'b10_00000 : begin valid = 1; has_payload = 1; has_mcode = 0; has_response = 0; end    // 32-bit mem write
            7'b11_00000 : begin valid = 1; has_payload = 1; has_mcode = 0; has_response = 0; end    // 64-bit mem write
            7'b00_00010 : begin valid = 1; has_payload = 0; has_mcode = 0; has_response = 1; end    // io read
            7'b10_00010 : begin valid = 1; has_payload = 1; has_mcode = 0; has_response = 0; end    // io write
            7'b00_00100 : begin valid = 1; has_payload = 0; has_mcode = 0; has_response = 1; end    // config read type0
            7'b10_00100 : begin valid = 1; has_payload = 1; has_mcode = 0; has_response = 1; end    // config write type0
            7'b00_00101 : begin valid = 1; has_payload = 0; has_mcode = 0; has_response = 1; end    // config read type1
            7'b10_00101 : begin valid = 1; has_payload = 1; has_mcode = 0; has_response = 1; end    // config write type 1
            7'b01_10xxx : begin valid = 1; has_payload = 0; has_mcode = 1; has_response = 0; end    // message w/o data
            7'b11_10xxx : begin valid = 1; has_payload = 1; has_mcode = 1; has_response = 0; end    // message with data
            default     : begin valid = 0; has_payload = 0; has_mcode = 0; has_response = 0; end    // Unsupported
        endcase

        if (valid == 1) begin
            // Wait for last operation to clear
            while (mst_done == 1'b1) begin
                @(posedge clk);
                #1;
            end
            // Issue request
            mst_req = 1'b1;
            mst_type = ttype;
            mst_addr = addr;
            if (has_payload)
                mst_data = data;
            else
                mst_data = 32'b0;
            if (has_mcode) begin
                mst_msgcode = msg_code;
                mst_be      = 4'b0;
            end
            else begin
                mst_msgcode = 8'b0;
                mst_be      =  be;
            end
            @(posedge clk);
            #1;

            // Wait for accept
            while (mst_done == 1'b0) begin
                @(posedge clk);
                #1;
            end

            if (has_response) begin
                data = mst_rd_data;
                status = mst_status;
            end
            else begin
                status = 3'b0;
            end
            mst_req     = 0;
            mst_type    = 7'b0;
            mst_addr    = 64'b0;
            mst_be      = 4'b0;
            mst_data    = 32'b0;
            mst_msgcode = 8'b0;

            curr_prior = curr_prior + 1;
        end
        else begin
            // Wait for last operation to clear
            while (mst_done == 1'b1) begin
                @(posedge clk);
                #1;
            end
            // Issue invalid request
            mst_req     = 1'b1;
            mst_type    = ttype;
            mst_addr    = addr;
            mst_data    = 32'b0;
            mst_msgcode = 8'b0;
            mst_be      =  be;
            @(posedge clk);
            #1;
            // Wait for accept
            while (mst_done == 1'b0) begin
                @(posedge clk);
                #1;
            end
            mst_req     = 0;
            mst_type    = 7'b0;
            mst_addr    = 64'b0;
            mst_be      = 4'b0;
            mst_data    = 32'b0;
            mst_msgcode = 8'b0;

            curr_prior = curr_prior + 1;
        end
    end
endtask 

task automatic mem_read;
    input   [63:0]      addr;
    input   [3:0]       be;
    output  [31:0]      data;

    reg     [2:0]       status;
    reg     [31:0]      read_data;

    begin
        if (addr[63:32] == 32'b0)
            xfer (7'b00_00000, addr, be, read_data, 8'h00, status);
        else    
            xfer (7'b01_00000, addr, be, read_data, 8'h00, status);

        data = read_data;
        if (status != 3'b0) begin
            if ((status == 3'b001) & (expect_status_001_error == 1'b1))
                $display ("%m : EXPECTED : Unsuccessful Memory Read on master port (time %t)", $time);
            else
            begin
                $display ("%m : ERROR : Unsuccessful Memory Read on master port (time %t)", $time);
                `INC_ERRORS;
            end
        end
    end
endtask

task automatic mem_write;
    input   [63:0]      addr;
    input   [31:0]      data;
    input   [3:0]       be;

    reg     [2:0]       status;

    begin
        if (addr[63:32] == 32'b0) 
            xfer (7'b10_00000, addr, be, data, 8'h00, status);
        else    
            xfer (7'b11_00000, addr, be, data, 8'h00, status);
    end
endtask

task automatic io_read;
    input   [63:0]      addr;
    output  [31:0]      data;

    reg     [2:0]       status;

    begin
        xfer (7'b00_00010, addr, 4'hf, data, 8'h00, status);
        if (status != 3'b0) begin
            if ((status == 3'b001) & (expect_status_001_error == 1'b1))
                $display ("%m : EXPECTED : Unsuccessful I/O Read on master port (time %t)", $time);
            else
            begin
                $display ("%m : ERROR : Unsuccessful I/O Read on master port (time %t)", $time);
                `INC_ERRORS;
            end
        end
    end
endtask

task automatic io_write;
    input   [63:0]      addr;
    input   [31:0]      data;
    input   [3:0]       be;

    reg     [2:0]       status;

    begin
        xfer (7'b10_00010, addr, be, data, 8'h00, status);
        if (status != 3'b0) begin
            if ((status == 3'b001) & (expect_status_001_error == 1'b1))
                $display ("%m : EXPECTED : Unsuccessful I/O Write on master port (time %t)", $time);
            else
            begin
                $display ("%m : ERROR : Unsuccessful I/O Write on master port (time %t)", $time);
                `INC_ERRORS;
            end
        end
    end
endtask

task automatic config_read;
    input   [63:0]      addr;
    output  [31:0]      data;
    output  [2:0]       status;

    begin
        xfer (7'b00_00100, addr, 4'hf, data, 8'h00, status);
        if (status != 3'b0) begin
            if ((status == 3'b001) & (expect_status_001_error == 1'b1))
                $display ("%m : EXPECTED : Unsuccessful Config Read on master port (time %t)", $time);
            else
            begin
                $display ("%m : ERROR : Unsuccessful Config Read on master port (time %t)", $time);
                `INC_ERRORS;
            end
        end
    end
endtask

task automatic config_write;
    input   [63:0]      addr;
    input   [31:0]      data;
    input   [3:0]       be;
    output  [2:0]       status;

    begin
        xfer (7'b10_00100, addr, be, data, 8'h00, status);
        if (status != 3'b0) begin
            if ((status == 3'b001) & (expect_status_001_error == 1'b1))
                $display ("%m : EXPECTED : Unsuccessful Config Write on master port (time %t)", $time);
            else
            begin
                $display ("%m : ERROR : Unsuccessful Config Write on master port (time %t)", $time);
                `INC_ERRORS;
            end
        end
    end
endtask

task automatic config1_read;
    input   [63:0]      addr;
    output  [31:0]      data;
    output  [2:0]       status;

    begin
        xfer (7'b00_00101, addr, 4'hf, data, 8'h00, status);
        if (status != 3'b0) begin
            if ((status == 3'b001) & (expect_status_001_error == 1'b1))
                $display ("%m : EXPECTED : Unsuccessful Config Read on master port (time %t)", $time);
            else
            begin
                $display ("%m : ERROR : Unsuccessful Config Read on master port (time %t)", $time);
                `INC_ERRORS;
            end
        end
    end
endtask

task automatic config1_write;
    input   [63:0]      addr;
    input   [31:0]      data;
    input   [3:0]       be;
    output  [2:0]       status;

    begin
        xfer (7'b10_00101, addr, be, data, 8'h00, status);
        if (status != 3'b0) begin
            if ((status == 3'b001) & (expect_status_001_error == 1'b1))
                $display ("%m : EXPECTED : Unsuccessful Config Write on master port (time %t)", $time);
            else
            begin
                $display ("%m : ERROR : Unsuccessful Config Write on master port (time %t)", $time);
                `INC_ERRORS;
            end
        end
    end
endtask

task automatic message;
    input   [63:0]      addr;
    input   [7:0]       msg_code;
    input   [2:0]       routing;

    reg     [2:0]       status;
    reg     [31:0]      data;

    begin
        data = 32'b0;
    
        xfer ({4'b01_10, routing}, addr, 4'h0, data, msg_code, status);
    end
endtask

task automatic message_w_data;
    input   [63:0]      addr;
    input   [7:0]       msg_code;
    input   [2:0]       routing;
    input   [31:0]      data;

    reg     [2:0]       status;

    begin
        xfer ({4'b11_10, routing}, addr, 4'h0, data, msg_code, status);
    end
endtask



endmodule

