//  -------------------------------------------------------------------------
// Logic Core:          DMA Back End
//
// Module Name:         xil_pcie_wrapper
//
// MODULE DESCRIPTION:
// Handles the initialization of the SDRAM when the system
// leaves reset state. Also handles the periodic refresh
// operations to the SDRAM after initialization
//
// $Date: 2013-02-21 14:10:40 -0800 (Thu, 21 Feb 2013) $
// $Revision: 34665 $
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//              (c) Copyright 2011 by Northwest Logic, Inc.
//                        All rights reserved.
//
// Trade Secret of Northwest Logic, Inc.  Do not disclose.
//
// Use of this source code in any form or means is permitted only
// with a valid, written license agreement with Northwest Logic, Inc.
//
// Licensee shall keep all information contained herein confidential
// and shall protect same in whole or in part from disclosure and
// dissemination to all third parties.
//
//                        Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//
//                        Ph:  +1 503 533 5800
//                        Fax: +1 503 533 5900
//                      E-Mail: info@nwlogic.com
//                           www.nwlogic.com
//
//  -------------------------------------------------------------------------

// -------------------------------------------------------------------------
//
// This module is a top level reference design file containing:
//   * Xilinx V6 Hard PCI Express Core
//   * Wrapper logic to interface to NW Logic DMA Back-End
//
// -------------------------------------------------------------------------

`timescale 1ns / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module xil_pcie_wrapper # (

    parameter   PL_FAST_TRAIN           = "FALSE",  // "TRUE" Speed up training for simulation; "FALSE" for hardware implementation

    parameter   REF_CLK_FREQ            = 2,    // 0 - 100 MHz, 1 - 125 MHz, 2 - 250 MHz

    parameter   VENDOR_ID               = 16'h19AA,
    parameter   DEVICE_ID               = 16'hE004,
    parameter   SUBSYSTEM_VENDOR_ID     = 16'h19AA,
    parameter   SUBSYSTEM_ID            = 16'hE004,
    parameter   REVISION_ID             = 8'h05,

    parameter   BAR0                    = 32'hFFFF0000,
    parameter   BAR1                    = 32'hFFFFE000,
    parameter   BAR2                    = 32'hFFFFE000,
    parameter   BAR3                    = 32'h00000000,
    parameter   BAR4                    = 32'h00000000,
    parameter   BAR5                    = 32'h00000000,

    parameter   CLASS_CODE              = 24'h11_80_00,

    parameter   DEVICE_SN               = 64'h0,

    // Do not override parameters below this line
    parameter   NUM_LANES               = 8,            // Number of PCI Express Lanes

    parameter   XIL_DATA_WIDTH          = 128,          // RX/TX interface data width
    parameter   XIL_STRB_WIDTH          = 16,           // TSTRB width

    parameter  CORE_BE_WIDTH            = 16,
    parameter  CORE_DATA_WIDTH          = 128,
    parameter  CORE_REMAIN_WIDTH        = 4,    // 2^CORE_REMAIN_WIDTH represents the number of bytes in CORE_DATA_WIDTH

    parameter  NUM_C2S                  = 2,
    parameter  NUM_S2C                  = 2,

// AXI Master implements 1 interrupt vector per DMA Engine plus one for User Interrupt
    parameter   M_INT_VECTORS           = 5,


    parameter   USER_CONTROL_WIDTH      = 64,
    parameter   USER_STATUS_WIDTH       = 64,
    parameter   REG_ADDR_WIDTH          = 12, // Register BAR is 64KBytes

    parameter  DESC_STATUS_WIDTH       = 160,
    parameter  DESC_WIDTH              = 256

)
(

    input                                       perst_n,

    input                                       ref_clk_p,
    input                                       ref_clk_n,

    output  [NUM_LANES-1:0]                     tx_p,
    output  [NUM_LANES-1:0]                     tx_n,

    input   [NUM_LANES-1:0]                     rx_p,
    input   [NUM_LANES-1:0]                     rx_n,

    output                                      user_clk,
    output                                      user_rst_n,
    input                                       user_interrupt,

    input   [31:0]                              mgmt_interrupt,
    output  [31:0]                              mgmt_pcie_version,
    output  [31:0]                              mgmt_be_version,
    input   [31:0]                              mgmt_user_version,
    output  reg                                 mgmt_mst_en,
    output  reg                                 mgmt_msi_en,
    output  reg                                 mgmt_msix_en,

    output                                      mst_ready,
    input                                       mst_req,
    input   [6:0]                               mst_type,
    input   [31:0]                              mst_data,
    input   [3:0]                               mst_be,
    input   [63:0]                              mst_addr,
    input   [7:0]                               mst_msgcode,
    output  [31:0]                              mst_rd_data,
    output  [2:0]                               mst_status,
    output                                      mst_done,

    input                                       c2s0_disable_desc_updates,
    input                                       c2s0_desc_req,
    output                                      c2s0_desc_ready,
    input   [31:0]                              c2s0_desc_ptr,
    input   [DESC_WIDTH-1:0]                    c2s0_desc_data,
    input                                       c2s0_desc_abort,
    output                                      c2s0_desc_abort_ack,
    input                                       c2s0_desc_rst_n,

    output                                      c2s0_desc_done,
    output  [7:0]                               c2s0_desc_done_channel,
    output  [DESC_STATUS_WIDTH-1:0]             c2s0_desc_done_status,

    output                                      c2s0_cmd_rst_n,
    output                                      c2s0_cmd_req,
    input                                       c2s0_cmd_ready,
    output                                      c2s0_cmd_first_chain,
    output                                      c2s0_cmd_last_chain,
    output  [63:0]                              c2s0_cmd_addr,
    output  [31:0]                              c2s0_cmd_bcount,
    output  [63:0]                              c2s0_cmd_user_control,
    output                                      c2s0_cmd_abort,
    input                                       c2s0_cmd_abort_ack,

    output                                      c2s0_data_req,
    input                                       c2s0_data_ready,
    output  [CORE_REMAIN_WIDTH-1:0]             c2s0_data_req_remain,
    output                                      c2s0_data_req_last_desc,
    output  [63:0]                              c2s0_data_addr,
    output  [9:0]                               c2s0_data_bcount,
    input                                       c2s0_data_stop,
    input   [9:0]                               c2s0_data_stop_bcount,

    output                                      c2s0_data_en,
    output  [CORE_REMAIN_WIDTH-1:0]             c2s0_data_remain,
    output  [CORE_REMAIN_WIDTH:0]               c2s0_data_valid,
    output                                      c2s0_data_first_req,
    output                                      c2s0_data_last_req,
    output                                      c2s0_data_first_desc,
    output                                      c2s0_data_last_desc,
    output                                      c2s0_data_first_chain,
    output                                      c2s0_data_last_chain,
    input                                       c2s0_data_sop,
    input                                       c2s0_data_eop,
    input   [CORE_DATA_WIDTH-1:0]               c2s0_data_data,
    input   [63:0]                              c2s0_data_user_status,
    input                                       s2c0_disable_desc_updates,
    input                                       s2c0_desc_req,
    output                                      s2c0_desc_ready,
    input   [31:0]                              s2c0_desc_ptr,
    input   [255:0]                             s2c0_desc_data,
    input                                       s2c0_desc_abort,
    output                                      s2c0_desc_abort_ack,
    input                                       s2c0_desc_rst_n,

    output                                      s2c0_desc_done,
    output  [7:0]                               s2c0_desc_done_channel,
    output  [159:0]                             s2c0_desc_done_status,

    output                                      s2c0_cmd_rst_n,
    output                                      s2c0_cmd_req,
    input                                       s2c0_cmd_ready,
    output  [63:0]                              s2c0_cmd_addr,
    output  [9:0]                               s2c0_cmd_bcount,
    output  [63:0]                              s2c0_cmd_user_control,
    output                                      s2c0_cmd_abort,
    input                                       s2c0_cmd_abort_ack,
    input                                       s2c0_cmd_stop,
    input   [9:0]                               s2c0_cmd_stop_bcount,

    output                                      s2c0_data_req,
    input                                       s2c0_data_ready,
    output  [63:0]                              s2c0_data_addr,
    output  [9:0]                               s2c0_data_bcount,
    output                                      s2c0_data_en,
    output                                      s2c0_data_error,
    output  [CORE_REMAIN_WIDTH-1:0]             s2c0_data_remain,
    output  [CORE_REMAIN_WIDTH:0]               s2c0_data_valid,
    output                                      s2c0_data_first_req,
    output                                      s2c0_data_last_req,
    output                                      s2c0_data_first_desc,
    output                                      s2c0_data_last_desc,
    output                                      s2c0_data_first_chain,
    output                                      s2c0_data_last_chain,
    output  [CORE_DATA_WIDTH-1:0]               s2c0_data_data,
    output  [63:0]                              s2c0_data_user_control,
    output                                      targ_wr_req,
    output                                      targ_wr_core_ready,
    input                                       targ_wr_user_ready,
    output  [5:0]                               targ_wr_cs,
    output                                      targ_wr_start,
    output  [31:0]                              targ_wr_addr,
    output  [12:0]                              targ_wr_count,
    output                                      targ_wr_en,
    output  [CORE_DATA_WIDTH-1:0]               targ_wr_data,
    output  [CORE_BE_WIDTH-1:0]                 targ_wr_be,

    output                                      targ_rd_req,
    output                                      targ_rd_core_ready,
    input                                       targ_rd_user_ready,
    output  [5:0]                               targ_rd_cs,
    output                                      targ_rd_start,
    output  [31:0]                              targ_rd_addr,
    output  [CORE_BE_WIDTH-1:0]                 targ_rd_first_be,
    output  [CORE_BE_WIDTH-1:0]                 targ_rd_last_be,
    output  [12:0]                              targ_rd_count,
    output                                      targ_rd_en,
    input   [CORE_DATA_WIDTH-1:0]               targ_rd_data,

    output  [REG_ADDR_WIDTH-1:0]                reg_wr_addr,
    output                                      reg_wr_en,
    output  [CORE_BE_WIDTH-1:0]                 reg_wr_be,
    output  [CORE_DATA_WIDTH-1:0]               reg_wr_data,
    output  [REG_ADDR_WIDTH-1:0]                reg_rd_addr,
    output  [CORE_BE_WIDTH-1:0]                 reg_rd_be,
    input   [CORE_DATA_WIDTH-1:0]               reg_rd_data
);


// -------------------
// -- Local Signals --
// -------------------

wire                                v6_clk;

wire                                user_reset;
wire                                reset;
reg                                 user_rst;
reg                                 d_user_rst;
wire                                user_lnk_up;


// Xilinx Hard Core Instantiation
wire    [5:0]                       tx_buf_av;
wire                                tx_err_drop;
wire                                tx_cfg_req;
wire                                tx_cfg_gnt;
wire                                s_axis_tx_tready;
wire    [XIL_DATA_WIDTH-1:0]        s_axis_tx_tdata;
wire    [XIL_STRB_WIDTH-1:0]        s_axis_tx_tkeep;
wire    [3:0]                       s_axis_tx_tuser;
wire                                s_axis_tx_tlast;
wire                                s_axis_tx_tvalid;

wire    [XIL_DATA_WIDTH-1:0]        m_axis_rx_tdata;
wire    [XIL_STRB_WIDTH-1:0]        m_axis_rx_tkeep;
wire                                m_axis_rx_tlast;
wire                                m_axis_rx_tvalid;
wire                                m_axis_rx_tready;
wire    [21:0]                      m_axis_rx_tuser;
wire                                rx_np_ok;

wire    [11:0]                      fc_cpld;
wire    [7:0]                       fc_cplh;
wire    [11:0]                      fc_npd;
wire    [7:0]                       fc_nph;
wire    [11:0]                      fc_pd;
wire    [7:0]                       fc_ph;
wire    [2:0]                       fc_sel;
wire                                mgmt_ch_infinite;
wire                                mgmt_cd_infinite;

wire    [31:0]                      cfg_do;
wire                                cfg_rd_wr_done;
wire    [31:0]                      cfg_di;
wire    [3:0]                       cfg_byte_en;
wire    [9:0]                       cfg_dwaddr;
wire                                cfg_wr_en;
wire                                cfg_rd_en;

wire                                cfg_err_cor;
wire                                cfg_err_ur;
wire                                cfg_err_ecrc;
reg                                 cfg_err_cpl_timeout;
wire                                cfg_err_cpl_abort;
reg                                 cfg_err_cpl_unexpect;
reg                                 cfg_err_posted;
wire                                cfg_err_locked;
wire    [47:0]                      cfg_err_tlp_cpl_header;
wire                                cfg_err_cpl_rdy;

wire                                cfg_interrupt;
wire                                cfg_interrupt_rdy;
wire                                cfg_interrupt_assert;
wire    [7:0]                       cfg_interrupt_di;
wire    [7:0]                       cfg_interrupt_do;
wire    [2:0]                       cfg_interrupt_mmenable;
wire                                cfg_interrupt_msienable;
wire                                cfg_interrupt_msixenable;
wire                                cfg_interrupt_msixfm;

reg                                 cfg_turnoff_ok;
wire                                cfg_to_turnoff;
reg                                 cfg_trn_pending;
wire                                cfg_pm_wake;

wire    [7:0]                       cfg_bus_number;
wire    [4:0]                       cfg_device_number;
wire    [2:0]                       cfg_function_number;
wire    [15:0]                      cfg_status;
wire    [15:0]                      cfg_command;
wire    [15:0]                      cfg_dstatus;
wire    [15:0]                      cfg_dcommand;
wire    [15:0]                      cfg_lstatus;
wire    [15:0]                      cfg_lcommand;
wire    [15:0]                      cfg_dcommand2;
wire    [2:0]                       cfg_pcie_link_state;
wire    [63:0]                      cfg_dsn;

wire    [2:0]                       pl_initial_link_width;
wire    [1:0]                       pl_lane_reversal_mode;
wire                                pl_link_gen2_capable;
wire                                pl_link_partner_gen2_supported;
wire                                pl_link_upcfg_capable;
wire    [5:0]                       pl_ltssm_state;
wire                                pl_received_hot_rst;
wire                                pl_sel_link_rate;
wire    [1:0]                       pl_sel_link_width;
wire                                pl_directed_link_auton;
wire    [1:0]                       pl_directed_link_change;
wire                                pl_directed_link_speed;
wire    [1:0]                       pl_directed_link_width;
wire                                pl_upstream_prefer_deemph;

wire                                ref_clk;
wire                                sys_reset_n_c;

wire    [63:0]                      s2c0_cfg_constants;
wire    [63:0]                      c2s0_cfg_constants;

// Configuration Interrupts
reg                                 mgmt_msix_fm;

reg     [2:0]                       mgmt_max_payload_size;
reg     [2:0]                       mgmt_max_rd_req_size;
reg     [15:0]                      mgmt_cfg_id;

wire    [7:0]                       mgmt_ch_credits;
wire    [11:0]                      mgmt_cd_credits;

reg                                 mgmt_cpl_timeout_disable;
reg     [3:0]                       mgmt_cpl_timeout_value;
wire    [7:0]                       mgmt_clk_period_in_ns;

wire                                err_cpl_to_closed_tag;
wire                                err_cpl_timeout;
wire                                cpl_tag_active;

genvar                              i;


// ---------------
// -- Equations --
// ---------------

// ---------------
// Clock and Reset

IBUFDS_GTXE1 refclk_ibuf (.O(ref_clk), .ODIV2(), .I(ref_clk_p), .IB(ref_clk_n), .CEB(1'b0));

IBUF   sys_reset_n_ibuf (.O(sys_reset_n_c), .I(perst_n));


assign user_clk = v6_clk;



// Hold core in reset whenever hard core reset is asserted or link is down
assign reset = user_reset | ~user_lnk_up;

// Synchronize reset to user_clk
always @(posedge user_clk or posedge reset)
begin
    if (reset == 1'b1)
    begin
        d_user_rst  <= 1'b1;
        user_rst    <= 1'b1;
    end
    else
    begin
        d_user_rst  <= 1'b0;
        user_rst    <= d_user_rst;
    end
end

assign user_rst_n = ~user_rst;


// ------------------------------
// Xilinx Hard Core Instantiation

assign mgmt_pcie_version = 32'h10EE_0205;

v6_pcie_v2_5 #(

    .PL_FAST_TRAIN                  (PL_FAST_TRAIN                  ),

    .REF_CLK_FREQ                   (REF_CLK_FREQ                   ),

    .VENDOR_ID                      (VENDOR_ID                      ),
    .DEVICE_ID                      (DEVICE_ID                      ),
    .SUBSYSTEM_VENDOR_ID            (SUBSYSTEM_VENDOR_ID            ),
    .SUBSYSTEM_ID                   (SUBSYSTEM_ID                   ),
    .REVISION_ID                    (REVISION_ID                    ),

    .BAR0                           (BAR0                           ),
    .BAR1                           (BAR1                           ),
    .BAR2                           (BAR2                           ),
    .BAR3                           (BAR3                           ),
    .BAR4                           (BAR4                           ),
    .BAR5                           (BAR5                           ),

    .CLASS_CODE                     (CLASS_CODE                     )

) v6_pcie (

    // Tx
    .pci_exp_txp                    (tx_p                           ),
    .pci_exp_txn                    (tx_n                           ),

    // Rx
    .pci_exp_rxp                    (rx_p                           ),
    .pci_exp_rxn                    (rx_n                           ),

    // Common
    .user_clk_out                   (v6_clk                         ),
    .user_reset_out                 (user_reset                     ),
    .user_lnk_up                    (user_lnk_up                    ),

    // Tx
    .s_axis_tx_tready               (s_axis_tx_tready               ),
    .s_axis_tx_tdata                (s_axis_tx_tdata                ),
    .s_axis_tx_tkeep                (s_axis_tx_tkeep                ),
    .s_axis_tx_tuser                (s_axis_tx_tuser                ),
    .s_axis_tx_tlast                (s_axis_tx_tlast                ),
    .s_axis_tx_tvalid               (s_axis_tx_tvalid               ),

    .tx_buf_av                      (tx_buf_av                      ),
    .tx_err_drop                    (tx_err_drop                    ),

    // Rx
    .m_axis_rx_tdata                (m_axis_rx_tdata                ),
    .m_axis_rx_tkeep                (m_axis_rx_tkeep                ),
    .m_axis_rx_tlast                (m_axis_rx_tlast                ),
    .m_axis_rx_tvalid               (m_axis_rx_tvalid               ),
    .m_axis_rx_tready               (m_axis_rx_tready               ),
    .m_axis_rx_tuser                (m_axis_rx_tuser                ),
    .rx_np_ok                       (rx_np_ok                       ),

    // Flow Control
    .fc_cpld                        (fc_cpld                        ),
    .fc_cplh                        (fc_cplh                        ),
    .fc_npd                         (fc_npd                         ),
    .fc_nph                         (fc_nph                         ),
    .fc_pd                          (fc_pd                          ),
    .fc_ph                          (fc_ph                          ),
    .fc_sel                         (fc_sel                         ),

    // Configuration (CFG) Interface
    .cfg_do                         (cfg_do                         ),
    .cfg_rd_wr_done                 (cfg_rd_wr_done                 ),
    .cfg_di                         (cfg_di                         ),
    .cfg_byte_en                    (cfg_byte_en                    ),
    .cfg_dwaddr                     (cfg_dwaddr                     ),
    .cfg_wr_en                      (cfg_wr_en                      ),
    .cfg_rd_en                      (cfg_rd_en                      ),

    .cfg_err_cor                    (cfg_err_cor                    ),
    .cfg_err_ur                     (cfg_err_ur                     ),
    .cfg_err_ecrc                   (cfg_err_ecrc                   ),
    .cfg_err_cpl_timeout            (cfg_err_cpl_timeout            ),
    .cfg_err_cpl_abort              (cfg_err_cpl_abort              ),
    .cfg_err_cpl_unexpect           (cfg_err_cpl_unexpect           ),
    .cfg_err_posted                 (cfg_err_posted                 ),
    .cfg_err_locked                 (cfg_err_locked                 ),
    .cfg_err_tlp_cpl_header         (cfg_err_tlp_cpl_header         ),
    .cfg_err_cpl_rdy                (cfg_err_cpl_rdy                ),

    .cfg_interrupt                  (cfg_interrupt                  ),
    .cfg_interrupt_rdy              (cfg_interrupt_rdy              ),
    .cfg_interrupt_assert           (cfg_interrupt_assert           ),
    .cfg_interrupt_di               (cfg_interrupt_di               ),
    .cfg_interrupt_do               (cfg_interrupt_do               ),
    .cfg_interrupt_mmenable         (cfg_interrupt_mmenable         ),
    .cfg_interrupt_msienable        (cfg_interrupt_msienable        ),
    .cfg_interrupt_msixenable       (cfg_interrupt_msixenable       ),
    .cfg_interrupt_msixfm           (cfg_interrupt_msixfm           ),

    .cfg_turnoff_ok                 (cfg_turnoff_ok                 ),
    .cfg_to_turnoff                 (cfg_to_turnoff                 ),
    .cfg_trn_pending                (cfg_trn_pending                ),
    .cfg_pm_wake                    (cfg_pm_wake                    ),

    .cfg_bus_number                 (cfg_bus_number                 ),
    .cfg_device_number              (cfg_device_number              ),
    .cfg_function_number            (cfg_function_number            ),
    .cfg_status                     (cfg_status                     ),
    .cfg_command                    (cfg_command                    ),
    .cfg_dstatus                    (cfg_dstatus                    ),
    .cfg_dcommand                   (cfg_dcommand                   ),
    .cfg_lstatus                    (cfg_lstatus                    ),
    .cfg_lcommand                   (cfg_lcommand                   ),
    .cfg_dcommand2                  (cfg_dcommand2                  ),
    .cfg_pcie_link_state            (cfg_pcie_link_state            ),
    .cfg_dsn                        (cfg_dsn                        ),
    .cfg_pmcsr_pme_en               (                               ),
    .cfg_pmcsr_pme_status           (                               ),
    .cfg_pmcsr_powerstate           (                               ),

    // Physical Layer Control and Status (PL) Interface
    .pl_initial_link_width          (pl_initial_link_width          ),
    .pl_lane_reversal_mode          (pl_lane_reversal_mode          ),
    .pl_link_gen2_capable           (pl_link_gen2_capable           ),
    .pl_link_partner_gen2_supported (pl_link_partner_gen2_supported ),
    .pl_link_upcfg_capable          (pl_link_upcfg_capable          ),
    .pl_ltssm_state                 (pl_ltssm_state                 ),
    .pl_received_hot_rst            (pl_received_hot_rst            ),
    .pl_sel_link_rate               (pl_sel_link_rate               ),
    .pl_sel_link_width              (pl_sel_link_width              ),
    .pl_directed_link_auton         (pl_directed_link_auton         ),
    .pl_directed_link_change        (pl_directed_link_change        ),
    .pl_directed_link_speed         (pl_directed_link_speed         ),
    .pl_directed_link_width         (pl_directed_link_width         ),
    .pl_upstream_prefer_deemph      (pl_upstream_prefer_deemph      ),

    // System  (SYS) Interface
    .sys_clk                        (ref_clk                        ),
    .sys_reset                      (!sys_reset_n_c                 )

);

assign tx_cfg_gnt = 1'b1;

// ------------
// Flow Control

// Set to report Receive Buffer Available Space
assign fc_sel = 3'b000;

assign mgmt_ch_infinite = 1'b0;
assign mgmt_cd_infinite = 1'b0;

assign mgmt_ch_credits  = fc_cplh;
assign mgmt_cd_credits  = fc_cpld;


// ------------------------
// Configuration Read/Write

// Ports not used
assign cfg_di                       = 32'h0;
assign cfg_byte_en                  = 4'h0;
assign cfg_dwaddr                   = 10'h0;
assign cfg_wr_en                    = 1'b0;
assign cfg_rd_en                    = 1'b0;


// --------------------
// Configuration Errors

always @(posedge v6_clk or negedge user_rst_n)
begin
    if (user_rst_n == 1'b0)
    begin
        cfg_err_cpl_timeout         <= 1'b0;
        cfg_err_cpl_unexpect        <= 1'b0;
        cfg_err_posted              <= 1'b0;
    end
    else
    begin
        // Assert cfg_err_posted so message response rather than completion response is used
        cfg_err_cpl_timeout         <=                          err_cpl_timeout;
        cfg_err_cpl_unexpect        <=  err_cpl_to_closed_tag;
        cfg_err_posted              <= (err_cpl_to_closed_tag | err_cpl_timeout);
    end
end

assign cfg_err_cor                  = 1'b0;
assign cfg_err_ur                   = 1'b0;
assign cfg_err_ecrc                 = 1'b0;
assign cfg_err_cpl_abort            = 1'b0;
assign cfg_err_locked               = 1'b0;

// Header for completion error response (unused)
//   If used, the following info should be put here from
//   taken from the error non-posted TLP
//   [47:41] Lower Address
//   [40:29] Byte Count
//   [28:26] TC
//   [25:24] Attr
//   [23:8] Requester ID
//   [7:0] Tag
assign cfg_err_tlp_cpl_header       = 48'h0;


    // Configure C2S DMA Engine(s)
    assign c2s0_cfg_constants[    0] = 1'b0;        // Reserved; was use sequence/continue functionality
    assign c2s0_cfg_constants[    1] = 1'b1;        // 1 == Support 32/64-bit system addresses; 0 == 32-bit address support only
    assign c2s0_cfg_constants[    2] = 1'b1;        // 1 == Support 32/64-bit descriptor pointer system addresses; 0 == 32-bit address support only (Block DMA Only)
    assign c2s0_cfg_constants[    3] = 1'b0;        // 1 == Enable increased command overlapping (Block DMA Only); 0 == Don't overlap
    assign c2s0_cfg_constants[ 7: 4] = 4'h0;        // Reserved
    assign c2s0_cfg_constants[14: 8] = 7'd32;       // Address space implemented on the card for this engine == 2^DMA_DEST_ADDR_WIDTH (2^32 = 4GB)
    assign c2s0_cfg_constants[   15] = 1'b0;        // Reserved
    assign c2s0_cfg_constants[21:16] = 6'h0;        // Implemented byte count width; 0 selects maximum supported DMA engine value
    assign c2s0_cfg_constants[23:22] = 2'h0;        // Reserved
    assign c2s0_cfg_constants[27:24] = 4'h0;        // Implemented channel width; 0 == only 1 channel
    assign c2s0_cfg_constants[31:28] = 4'h0;        // Reserved
    assign c2s0_cfg_constants[38:32] = 7'd64;       // Implemented user status width; 64 == max value
    assign c2s0_cfg_constants[   39] = 1'b0;        // Reserved
    assign c2s0_cfg_constants[46:40] = 7'h0;        // Implemented user control width; 0 == not used; not supported for C2S Engines
    assign c2s0_cfg_constants[   47] = c2s0_disable_desc_updates;        // Set to disable descriptor update writes
    assign c2s0_cfg_constants[63:48] = 16'h0;       // Reserved
    // Configure S2C DMA Engine(s)
    assign s2c0_cfg_constants[    0] = 1'b0;        // Reserved; was use sequence/continue functionality
    assign s2c0_cfg_constants[    1] = 1'b1;        // 1 == Support 32/64-bit system addresses; 0 == 32-bit address support only
    assign s2c0_cfg_constants[    2] = 1'b1;        // 1 == Support 32/64-bit descriptor pointer system addresses; 0 == 32-bit address support only (Block DMA Only)
    assign s2c0_cfg_constants[    3] = 1'b0;        // 1 == Enable increased command overlapping (Block DMA Only); 0 == Don't overlap
    assign s2c0_cfg_constants[ 7: 4] = 4'h0;        // Reserved
    assign s2c0_cfg_constants[14: 8] = 7'd32;       // Address space implemented on the card for this engine == 2^DMA_DEST_ADDR_WIDTH (2^32 = 4GB)
    assign s2c0_cfg_constants[   15] = 1'b0;        // Reserved
    assign s2c0_cfg_constants[21:16] = 6'h0;        // Implemented byte count width; 0 selects maximum supported DMA engine value
    assign s2c0_cfg_constants[23:22] = 2'h0;        // Reserved
    assign s2c0_cfg_constants[27:24] = 4'h0;        // Implemented channel width; 0 == only 1 channel
    assign s2c0_cfg_constants[31:28] = 4'h0;        // Reserved
    assign s2c0_cfg_constants[38:32] = 7'h0;        // Implemented user status width; 0 == not used
    assign s2c0_cfg_constants[   39] = 1'b0;        // Reserved
    assign s2c0_cfg_constants[46:40] = 7'd64;       // Implemented user control width; 64 == maximum width
    assign s2c0_cfg_constants[   47] = s2c0_disable_desc_updates;        // Set to disable descriptor update writes
    assign s2c0_cfg_constants[63:48] = 16'h0;       // Reserved
// --------------------
// Configuration Status

always @(posedge user_clk or negedge user_rst_n)
begin
    if (user_rst_n == 1'b0)
    begin
        cfg_turnoff_ok              <= 1'b0;
        cfg_trn_pending             <= 1'b0;

        mgmt_mst_en                 <= 1'b0;
        mgmt_msi_en                 <= 1'b0;
        mgmt_msix_en                <= 1'b0;
        mgmt_msix_fm                <= 1'b0;
        mgmt_max_payload_size       <= 3'b000;
        mgmt_max_rd_req_size        <= 3'b000;
        mgmt_cfg_id                 <= 16'h0;

        mgmt_cpl_timeout_disable    <= 1'b0;
        mgmt_cpl_timeout_value      <= 4'h0;
    end
    else
    begin
        if (cfg_to_turnoff & ~cfg_trn_pending) // Turn off request and no completions pending
            cfg_turnoff_ok <= 1'b1;
        else
            cfg_turnoff_ok <= 1'b0;

        cfg_trn_pending             <= cpl_tag_active;

        mgmt_mst_en                 <= cfg_command[2];
        mgmt_msi_en                 <= cfg_interrupt_msienable;
        mgmt_msix_en                <= cfg_interrupt_msixenable;
        mgmt_msix_fm                <= cfg_interrupt_msixfm;
        mgmt_max_payload_size       <= cfg_dcommand[ 7: 5];
        mgmt_max_rd_req_size        <= cfg_dcommand[14:12];
        mgmt_cfg_id                 <= {cfg_bus_number, cfg_device_number, cfg_function_number};

        mgmt_cpl_timeout_disable    <= cfg_dcommand2[4];
        mgmt_cpl_timeout_value      <= cfg_dcommand2[3:0];
    end
end

assign mgmt_clk_period_in_ns = 8'd4;  // Back-End is 250MHz for 5G and 2.5G

assign cfg_pm_wake                  = 1'b0;



// ---------------------------------
// Physical Layer Control and Status

assign pl_directed_link_change      = 2'h0;
assign pl_directed_link_width       = 2'h0;
assign pl_directed_link_speed       = 1'b0;
assign pl_directed_link_auton       = 1'b0;
assign pl_upstream_prefer_deemph    = 1'b1;



// -------------------------------
// Device Serial Number Capability

assign cfg_dsn                      = DEVICE_SN;



// ------------
// DMA Back End

dma_back_end_pkt #(

    .REG_ADDR_WIDTH                 (REG_ADDR_WIDTH                 )

) dma_back_end_pkt (

    .rst_n                          (user_rst_n                     ),
    .clk                            (v6_clk                         ),
    .testmode                       (1'b0                           ),

    .tx_buf_av                      (tx_buf_av                      ),
    .tx_err_drop                    (tx_err_drop                    ),
    .s_axis_tx_tready               (s_axis_tx_tready               ),
    .s_axis_tx_tdata                (s_axis_tx_tdata                ),
    .s_axis_tx_tkeep                (s_axis_tx_tkeep                ),
    .s_axis_tx_tuser                (s_axis_tx_tuser                ),
    .s_axis_tx_tlast                (s_axis_tx_tlast                ),
    .s_axis_tx_tvalid               (s_axis_tx_tvalid               ),

    .m_axis_rx_tdata                (m_axis_rx_tdata                ),
    .m_axis_rx_tkeep                (m_axis_rx_tkeep                ),
    .m_axis_rx_tlast                (m_axis_rx_tlast                ),
    .m_axis_rx_tvalid               (m_axis_rx_tvalid               ),
    .m_axis_rx_tready               (m_axis_rx_tready               ),
    .m_axis_rx_tuser                (m_axis_rx_tuser                ),
    .rx_np_ok                       (rx_np_ok                       ),

    .mgmt_mst_en                    (mgmt_mst_en                    ),
    .mgmt_msi_en                    (mgmt_msi_en                    ), // MSI En
    .mgmt_msix_en                   (mgmt_msix_en                   ), // MSI-X En
    .mgmt_msix_table_offset         (32'h6000                       ),
    .mgmt_msix_pba_offset           (32'h7000                       ),
    .mgmt_msix_function_mask        (mgmt_msix_fm                   ),
    .mgmt_max_payload_size          (mgmt_max_payload_size          ),
    .mgmt_max_rd_req_size           (mgmt_max_rd_req_size           ),
    .mgmt_clk_period_in_ns          (mgmt_clk_period_in_ns          ),
    .mgmt_version                   (mgmt_be_version                ),
    .mgmt_pcie_version              (mgmt_pcie_version              ),
    .mgmt_user_version              (mgmt_user_version              ),
    .mgmt_cfg_id                    (mgmt_cfg_id                    ),
    .mgmt_interrupt                 (mgmt_interrupt                 ),
    .user_interrupt                 (user_interrupt                 ),

    .cfg_interrupt_rdy              (cfg_interrupt_rdy              ),
    .cfg_interrupt_assert           (cfg_interrupt_assert           ),
    .cfg_interrupt                  (cfg_interrupt                  ),
    .cfg_interrupt_di               (cfg_interrupt_di               ),
    .cfg_interrupt_do               (cfg_interrupt_do               ),


    .mgmt_ch_infinite               (mgmt_ch_infinite               ),
    .mgmt_cd_infinite               (mgmt_cd_infinite               ),
    .mgmt_ch_credits                (mgmt_ch_credits                ),
    .mgmt_cd_credits                (mgmt_cd_credits                ),

    .mgmt_adv_cpl_timeout_disable   (                               ),
    .mgmt_adv_cpl_timeout_value     (                               ),
    .mgmt_cpl_timeout_disable       (mgmt_cpl_timeout_disable       ),
    .mgmt_cpl_timeout_value         (mgmt_cpl_timeout_value         ),

    .err_cpl_to_closed_tag          (err_cpl_to_closed_tag          ),
    .err_cpl_timeout                (err_cpl_timeout                ),
    .cpl_tag_active                 (cpl_tag_active                 ),

    .err_pkt_poison                 (                               ),
    .err_pkt_header                 (                               ),

    .mst_ready                      (mst_ready                      ),
    .mst_rd_data                    (mst_rd_data                    ),
    .mst_status                     (mst_status                     ),
    .mst_done                       (mst_done                       ),
    .mst_req                        (mst_req                        ),
    .mst_type                       (mst_type                       ),
    .mst_data                       (mst_data                       ),
    .mst_be                         (mst_be                         ),
    .mst_addr                       (mst_addr                       ),
    .mst_msgcode                    (mst_msgcode                    ),

    .c2s0_cfg_constants             (c2s0_cfg_constants             ),

    .c2s0_desc_req                  (c2s0_desc_req                  ),
    .c2s0_desc_ready                (c2s0_desc_ready                ),
    .c2s0_desc_ptr                  (c2s0_desc_ptr                  ),
    .c2s0_desc_data                 (c2s0_desc_data                 ),
    .c2s0_desc_abort                (c2s0_desc_abort                ),
    .c2s0_desc_abort_ack            (c2s0_desc_abort_ack            ),
    .c2s0_desc_rst_n                (c2s0_desc_rst_n                ),

    .c2s0_desc_done                 (c2s0_desc_done                 ),
    .c2s0_desc_done_channel         (c2s0_desc_done_channel         ),
    .c2s0_desc_done_status          (c2s0_desc_done_status          ),

    .c2s0_cmd_rst_n                 (c2s0_cmd_rst_n                 ),
    .c2s0_cmd_req                   (c2s0_cmd_req                   ),
    .c2s0_cmd_ready                 (c2s0_cmd_ready                 ),
    .c2s0_cmd_first_chain           (c2s0_cmd_first_chain           ),
    .c2s0_cmd_last_chain            (c2s0_cmd_last_chain            ),
    .c2s0_cmd_addr                  (c2s0_cmd_addr                  ),
    .c2s0_cmd_bcount                (c2s0_cmd_bcount                ),
    .c2s0_cmd_user_control          (c2s0_cmd_user_control          ),
    .c2s0_cmd_abort                 (c2s0_cmd_abort                 ),
    .c2s0_cmd_abort_ack             (c2s0_cmd_abort_ack             ),

    .c2s0_data_req                  (c2s0_data_req                  ),
    .c2s0_data_ready                (c2s0_data_ready                ),
    .c2s0_data_req_remain           (c2s0_data_req_remain           ),
    .c2s0_data_req_last_desc        (c2s0_data_req_last_desc        ),
    .c2s0_data_addr                 (c2s0_data_addr                 ),
    .c2s0_data_bcount               (c2s0_data_bcount               ),
    .c2s0_data_stop                 (c2s0_data_stop                 ),
    .c2s0_data_stop_bcount          (c2s0_data_stop_bcount          ),

    .c2s0_data_en                   (c2s0_data_en                   ),
    .c2s0_data_remain               (c2s0_data_remain               ),
    .c2s0_data_valid                (c2s0_data_valid                ),
    .c2s0_data_first_req            (c2s0_data_first_req            ),
    .c2s0_data_last_req             (c2s0_data_last_req             ),
    .c2s0_data_first_desc           (c2s0_data_first_desc           ),
    .c2s0_data_last_desc            (c2s0_data_last_desc            ),
    .c2s0_data_first_chain          (c2s0_data_first_chain          ),
    .c2s0_data_last_chain           (c2s0_data_last_chain           ),
    .c2s0_data_sop                  (c2s0_data_sop                  ),
    .c2s0_data_eop                  (c2s0_data_eop                  ),
    .c2s0_data_data                 (c2s0_data_data                 ),
    .c2s0_data_user_status          (c2s0_data_user_status          ),
    .s2c0_cfg_constants             (s2c0_cfg_constants             ),

    .s2c0_desc_req                  (s2c0_desc_req                  ),
    .s2c0_desc_ready                (s2c0_desc_ready                ),
    .s2c0_desc_ptr                  (s2c0_desc_ptr                  ),
    .s2c0_desc_data                 (s2c0_desc_data                 ),
    .s2c0_desc_abort                (s2c0_desc_abort                ),
    .s2c0_desc_abort_ack            (s2c0_desc_abort_ack            ),
    .s2c0_desc_rst_n                (s2c0_desc_rst_n                ),

    .s2c0_desc_done                 (s2c0_desc_done                 ),
    .s2c0_desc_done_channel         (s2c0_desc_done_channel         ),
    .s2c0_desc_done_status          (s2c0_desc_done_status          ),

    .s2c0_cmd_rst_n                 (s2c0_cmd_rst_n                 ),
    .s2c0_cmd_req                   (s2c0_cmd_req                   ),
    .s2c0_cmd_ready                 (s2c0_cmd_ready                 ),
    .s2c0_cmd_addr                  (s2c0_cmd_addr                  ),
    .s2c0_cmd_bcount                (s2c0_cmd_bcount                ),
    .s2c0_cmd_user_control          (s2c0_cmd_user_control          ),
    .s2c0_cmd_abort                 (s2c0_cmd_abort                 ),
    .s2c0_cmd_abort_ack             (s2c0_cmd_abort_ack             ),
    .s2c0_cmd_stop                  (s2c0_cmd_stop                  ),
    .s2c0_cmd_stop_bcount           (s2c0_cmd_stop_bcount           ),

    .s2c0_data_req                  (s2c0_data_req                  ),
    .s2c0_data_ready                (s2c0_data_ready                ),
    .s2c0_data_addr                 (s2c0_data_addr                 ),
    .s2c0_data_bcount               (s2c0_data_bcount               ),
    .s2c0_data_en                   (s2c0_data_en                   ),
    .s2c0_data_error                (s2c0_data_error                ),
    .s2c0_data_remain               (s2c0_data_remain               ),
    .s2c0_data_valid                (s2c0_data_valid                ),
    .s2c0_data_first_req            (s2c0_data_first_req            ),
    .s2c0_data_last_req             (s2c0_data_last_req             ),
    .s2c0_data_first_desc           (s2c0_data_first_desc           ),
    .s2c0_data_last_desc            (s2c0_data_last_desc            ),
    .s2c0_data_first_chain          (s2c0_data_first_chain          ),
    .s2c0_data_last_chain           (s2c0_data_last_chain           ),
    .s2c0_data_data                 (s2c0_data_data                 ),
    .s2c0_data_user_control         (s2c0_data_user_control         ),
    .targ_wr_req                    (targ_wr_req                    ),
    .targ_wr_core_ready             (targ_wr_core_ready             ),
    .targ_wr_user_ready             (targ_wr_user_ready             ),
    .targ_wr_cs                     (targ_wr_cs                     ),
    .targ_wr_start                  (targ_wr_start                  ),
    .targ_wr_addr                   (targ_wr_addr                   ),
    .targ_wr_count                  (targ_wr_count                  ),
    .targ_wr_en                     (targ_wr_en                     ),
    .targ_wr_data                   (targ_wr_data                   ),
    .targ_wr_be                     (targ_wr_be                     ),

    .targ_rd_req                    (targ_rd_req                    ),
    .targ_rd_core_ready             (targ_rd_core_ready             ),
    .targ_rd_user_ready             (targ_rd_user_ready             ),
    .targ_rd_cs                     (targ_rd_cs                     ),
    .targ_rd_start                  (targ_rd_start                  ),
    .targ_rd_addr                   (targ_rd_addr                   ),
    .targ_rd_first_be               (targ_rd_first_be               ),
    .targ_rd_last_be                (targ_rd_last_be                ),
    .targ_rd_count                  (targ_rd_count                  ),
    .targ_rd_en                     (targ_rd_en                     ),
    .targ_rd_data                   (targ_rd_data                   ),

    .reg_wr_addr                    (reg_wr_addr                    ),
    .reg_wr_en                      (reg_wr_en                      ),
    .reg_wr_be                      (reg_wr_be                      ),
    .reg_wr_data                    (reg_wr_data                    ),
    .reg_rd_addr                    (reg_rd_addr                    ),
    .reg_rd_be                      (reg_rd_be                      ),
    .reg_rd_data                    (reg_rd_data                    )

);


endmodule
