// -------------------------------------------------------------------------
//
//  PROJECT: PCI Express Core
//  COMPANY: Northwest Logic, Inc.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//                 Copyright 2006 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or
//  transmitted in any form or by any means, electronic or mechanical,
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

// -------------------------------------------------------------------------
//
// This module implements example logic to illustrate use of the DMA Back-End
//   Target Write and Target Read Interfaces; all target access from the DMA
//   Back-End are terminated by this module; this module implements only
//   one target memory and does not use the targ_wr_cs and targ_rd_cs chip
//   seclects; if multiple BARs are implemented, they will all map to the
//   same RAM in this module
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module target_example (

    rst_n,
    clk,

    // Target Write Interface
    targ_wr_req,
    targ_wr_core_ready,
    targ_wr_user_ready,
    targ_wr_cs,
    targ_wr_start,
    targ_wr_addr,
    targ_wr_count,
    targ_wr_en,
    targ_wr_data,
    targ_wr_be,

    // Target Read Interface
    targ_rd_req,
    targ_rd_core_ready,
    targ_rd_user_ready,
    targ_rd_cs,
    targ_rd_start,
    targ_rd_addr,
    targ_rd_count,
    targ_rd_en,
    targ_rd_data

);



// ----------------
// -- Parameters --
// ----------------

parameter   SRAM_ADDR_WIDTH         = 9;


// None of the following local parameters are intended to be modified
localparam  TARG_CS_WIDTH           = 5;
localparam  TARG_ADDR_WIDTH         = 32;
localparam  TARG_COUNT_WIDTH        = 13;

localparam  CORE_DATA_WIDTH         = 128;
localparam  CORE_BE_WIDTH           = 16;
localparam  CORE_ADDR_SHIFT         = 4;



// ----------------------
// -- Port Definitions --
// ----------------------

input                               rst_n;
input                               clk;

input                               targ_wr_req;
input                               targ_wr_core_ready;
output                              targ_wr_user_ready;
input   [TARG_CS_WIDTH-1:0]         targ_wr_cs;
input                               targ_wr_start;
input   [TARG_ADDR_WIDTH-1:0]       targ_wr_addr;
input   [TARG_COUNT_WIDTH-1:0]      targ_wr_count;
input                               targ_wr_en;
input   [CORE_DATA_WIDTH-1:0]       targ_wr_data;
input   [CORE_BE_WIDTH-1:0]         targ_wr_be;

input                               targ_rd_req;
input                               targ_rd_core_ready;
output                              targ_rd_user_ready;
input   [TARG_CS_WIDTH-1:0]         targ_rd_cs;
input                               targ_rd_start;
input   [TARG_ADDR_WIDTH-1:0]       targ_rd_addr;
input   [TARG_COUNT_WIDTH-1:0]      targ_rd_count;
input                               targ_rd_en;
output  [CORE_DATA_WIDTH-1:0]       targ_rd_data;



// ----------------
// -- Port Types --
// ----------------

wire                                rst_n;
wire                                clk;

wire                                targ_wr_req;
wire                                targ_wr_core_ready;
wire                                targ_wr_user_ready;
wire    [TARG_CS_WIDTH-1:0]         targ_wr_cs;
wire                                targ_wr_start;
wire    [TARG_ADDR_WIDTH-1:0]       targ_wr_addr;
wire    [TARG_COUNT_WIDTH-1:0]      targ_wr_count;
wire                                targ_wr_en;
wire    [CORE_DATA_WIDTH-1:0]       targ_wr_data;
wire    [CORE_BE_WIDTH-1:0]         targ_wr_be;

wire                                targ_rd_req;
wire                                targ_rd_core_ready;
wire                                targ_rd_user_ready;
wire    [TARG_CS_WIDTH-1:0]         targ_rd_cs;
wire                                targ_rd_start;
wire    [TARG_ADDR_WIDTH-1:0]       targ_rd_addr;
wire    [TARG_COUNT_WIDTH-1:0]      targ_rd_count;
wire                                targ_rd_en;
reg     [CORE_DATA_WIDTH-1:0]       targ_rd_data;



// -------------------
// -- Local Signals --
// -------------------

// Instantiate SRAM Target Memory
wire    [CORE_DATA_WIDTH-1:0]       sram_rd_data;

// Target Reads
wire                                targ_rd_req_accept;

reg     [SRAM_ADDR_WIDTH-1:0]       sram_rd_addr;
wire    [SRAM_ADDR_WIDTH-1:0]       next_sram_rd_addr;
wire    [SRAM_ADDR_WIDTH-1:0]       c_sram_rd_addr;

// Target Writes
reg                                 sram_wr_en;
reg     [SRAM_ADDR_WIDTH-1:0]       sram_wr_addr;
reg     [CORE_BE_WIDTH-1:0]         sram_wr_be;
reg     [CORE_DATA_WIDTH-1:0]       sram_wr_data;
reg     [TARG_CS_WIDTH-1:0]         hold_wr_cs;



// ---------------
// -- Equations --
// ---------------

// ------------------------------
// Instantiate SRAM Target Memory

// Generate 1 RAM for each byte so that we can use byte enables
genvar i;
generate for (i=0; i<CORE_BE_WIDTH; i=i+1)
    begin : gen_sram
        ref_inferred_block_ram #(

            .ADDR_WIDTH (SRAM_ADDR_WIDTH                    ),
            .DATA_WIDTH (8                                  )

        ) sram (

            .wr_clk     (clk                                ),
            .wr_addr    (sram_wr_addr                       ),
            .wr_en      (sram_wr_en & sram_wr_be[i]         ),
            .wr_data    (sram_wr_data[(((i+1)*8)-1):(i*8)]  ),

            .rd_clk     (clk                                ),
            .rd_addr    (c_sram_rd_addr                     ),
            .rd_data    (sram_rd_data[(((i+1)*8)-1):(i*8)]  )

        );
    end
endgenerate



// ------------
// Target Reads

// RAM is always ready for reads
assign targ_rd_user_ready = 1'b1;

// Signal that indicates that targ_rd_req will be accepted; use to pipeline read
assign targ_rd_req_accept = targ_rd_req & targ_rd_core_ready & targ_rd_user_ready;

always @(posedge clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
        sram_rd_addr <= {SRAM_ADDR_WIDTH{1'b0}};
    else
        sram_rd_addr <= c_sram_rd_addr;
end

assign next_sram_rd_addr = sram_rd_addr + {{(SRAM_ADDR_WIDTH-1){1'b0}}, 1'b1};

// Use look ahead read to reduce latency of RAM
assign c_sram_rd_addr = targ_rd_req_accept ? targ_rd_addr[SRAM_ADDR_WIDTH-1:0] : (targ_rd_en ? next_sram_rd_addr : sram_rd_addr);

always @(posedge clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
        targ_rd_data <= {CORE_DATA_WIDTH{1'b0}};
    else
        targ_rd_data <= sram_rd_data;
end



// -------------
// Target Writes

// RAM is always ready for writes
assign targ_wr_user_ready = 1'b1;

always @(posedge clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        sram_wr_en   <= 1'b0;
        sram_wr_addr <= {SRAM_ADDR_WIDTH{1'b0}};
        sram_wr_be   <= {CORE_BE_WIDTH{1'b0}};
        sram_wr_data <= {CORE_DATA_WIDTH{1'b0}};
        hold_wr_cs   <= {TARG_CS_WIDTH{1'b0}};
    end
    else
    begin
        if (targ_wr_req)
            hold_wr_cs <= targ_wr_cs;
        sram_wr_en   <= targ_wr_en & !hold_wr_cs[TARG_CS_WIDTH-1];      // Disable writes to Expansion ROM
        sram_wr_addr <= targ_wr_addr[SRAM_ADDR_WIDTH-1:0];
        sram_wr_be   <= targ_wr_be;
        sram_wr_data <= targ_wr_data;
    end
end



endmodule
