// -------------------------------------------------------------------------
//
//  PROJECT: PCI Express Core
//  COMPANY: Northwest Logic, Inc.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//                 Copyright 2012 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or
//  transmitted in any form or by any means, electronic or mechanical,
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//  $Date: 2013-10-14 17:26:49 -0700 (Mon, 14 Oct 2013) $
//  $Revision: 40726 $
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module sdram_dma_ref_design_pkt (

    pcie_srst_n,
    pcie_clk,

    sdram_srst_n,
    sdram_clk,

    testmode,
    // Target Write Interface
    targ_wr_req,
    targ_wr_core_ready,
    targ_wr_user_ready,
    targ_wr_cs,
    targ_wr_start,
    targ_wr_addr,
    targ_wr_count,
    targ_wr_en,
    targ_wr_data,
    targ_wr_be,

    // Target Read Interface
    targ_rd_req,
    targ_rd_core_ready,
    targ_rd_user_ready,
    targ_rd_cs,
    targ_rd_start,
    targ_rd_addr,
    targ_rd_count,
    targ_rd_en,
    targ_rd_data,

    // Register Interface
    reg_wr_addr,
    reg_wr_en,
    reg_wr_be,
    reg_wr_data,
    reg_rd_addr,
    reg_rd_data,


    // DMA Card to System Engine #0 User Interface
    c2s0_desc_req,
    c2s0_desc_ready,
    c2s0_desc_ptr,
    c2s0_desc_data,
    c2s0_desc_abort,
    c2s0_desc_abort_ack,
    c2s0_desc_rst_n,

    c2s0_desc_done,
    c2s0_desc_done_channel,
    c2s0_desc_done_status,

    c2s0_cmd_rst_n,
    c2s0_cmd_req,
    c2s0_cmd_ready,
    c2s0_cmd_first_chain,
    c2s0_cmd_last_chain,
    c2s0_cmd_addr,
    c2s0_cmd_bcount,
    c2s0_cmd_user_control,
    c2s0_cmd_abort,
    c2s0_cmd_abort_ack,

    c2s0_data_req,
    c2s0_data_ready,
    c2s0_data_req_remain,
    c2s0_data_req_last_desc,
    c2s0_data_addr,
    c2s0_data_bcount,
    c2s0_data_stop,
    c2s0_data_stop_bcount,

    c2s0_data_en,
    c2s0_data_remain,
    c2s0_data_valid,
    c2s0_data_first_req,
    c2s0_data_last_req,
    c2s0_data_first_desc,
    c2s0_data_last_desc,
    c2s0_data_first_chain,
    c2s0_data_last_chain,
    c2s0_data_sop,
    c2s0_data_eop,
    c2s0_data_data,
    c2s0_data_user_status,
    // DMA System to Card Engine #0 User Interface
    s2c0_desc_req,
    s2c0_desc_ready,
    s2c0_desc_ptr,
    s2c0_desc_data,
    s2c0_desc_abort,
    s2c0_desc_abort_ack,
    s2c0_desc_rst_n,

    s2c0_desc_done,
    s2c0_desc_done_channel,
    s2c0_desc_done_status,

    s2c0_cmd_rst_n,
    s2c0_cmd_req,
    s2c0_cmd_ready,
    s2c0_cmd_addr,
    s2c0_cmd_bcount,
    s2c0_cmd_user_control,
    s2c0_cmd_abort,
    s2c0_cmd_abort_ack,
    s2c0_cmd_stop,
    s2c0_cmd_stop_bcount,

    s2c0_data_req,
    s2c0_data_ready,
    s2c0_data_addr,
    s2c0_data_bcount,
    s2c0_data_en,
    s2c0_data_error,
    s2c0_data_remain,
    s2c0_data_valid,
    s2c0_data_first_req,
    s2c0_data_last_req,
    s2c0_data_first_desc,
    s2c0_data_last_desc,
    s2c0_data_first_chain,
    s2c0_data_last_chain,
    s2c0_data_data,
    s2c0_data_user_control,
 `ifdef INSTANTIATE_MPFE
    // Card to System Card DMA Engine #0 Multi-Port Interface
    c2s0_l_r_req,
    c2s0_l_w_req,
    c2s0_l_req_pri,
    c2s0_l_addr,
    c2s0_l_b_size,
    c2s0_l_auto_pch,
    c2s0_l_busy,
    c2s0_l_r_valid,
    c2s0_l_r_valid_last,
    c2s0_l_d_req,
    c2s0_l_d_req_last,
    c2s0_l_datain,
    c2s0_l_dm_in,
    c2s0_l_dataout,
 `endif
 `ifdef INSTANTIATE_MPFE
    // System to Card DMA Engine #0 Multi-Port Interface
    s2c0_l_r_req,
    s2c0_l_w_req,
    s2c0_l_req_pri,
    s2c0_l_addr,
    s2c0_l_b_size,
    s2c0_l_auto_pch,
    s2c0_l_busy,
    s2c0_l_r_valid,
    s2c0_l_r_valid_last,
    s2c0_l_d_req,
    s2c0_l_d_req_last,
    s2c0_l_datain,
    s2c0_l_dm_in,
    s2c0_l_dataout,
 `endif
    mgmt_core_ph,
    mgmt_core_pd,
    mgmt_core_nh,
    mgmt_core_nd,
    mgmt_core_ch,
    mgmt_core_cd,

    mgmt_chipset_ph,
    mgmt_chipset_pd,
    mgmt_chipset_nh,
    mgmt_chipset_nd,
    mgmt_chipset_ch,
    mgmt_chipset_cd,

    mgmt_user_version,
    mgmt_core_version,
    mgmt_be_version,
    mgmt_msi_en,
    mgmt_msix_en,
    mgmt_interrupt

);



// ----------------
// -- Parameters --
// ----------------

localparam  CORE_DATA_WIDTH             = 128;
localparam  CORE_BE_WIDTH               = 16;
localparam  CORE_REMAIN_WIDTH           = 4;

parameter   DMA_DEST_ADDR_WIDTH         = 7'd28;    // Size of DMA destination memory in bytes == 2^DMA_DEST_ADDR_WIDTH; set to 0 for FIFO and stream applications

parameter   FIFO_CADDR_WIDTH            =  4;                           // Address width of reference design command FIFOs; generally best left at 4
parameter   FIFO_DADDR_WIDTH            =  7 + (4 - CORE_REMAIN_WIDTH); // Address width of reference design data FIFOs; Want 2 KBytes minimum == 4 * Max Supported Payload Size

parameter   SRAM_ADDR_WIDTH             =  9 + (4 - CORE_REMAIN_WIDTH); // BAR1 and BAR2 example SRAM BAR size in bytes == 2^SRAM_ADDR_WIDTH; default is 8 KBytes; needs to be default of 8 KBytes for reference design example software to function correctly

localparam  REG_ADDR_WIDTH              = 12 + (4 - CORE_REMAIN_WIDTH); // BAR0 Register BAR size in bytes == 2^REG_ADDR_WIDTH; needs to be default of 64 KBytes for reference design example software to function correctly

// The PCI Express Complete Core implements an MSI-X arbiter, Table, and PBA
//   in BAR0 register space using the Register Interface; msix_table_bir and
//   msix_pba_bir must be 000 to indicate BAR0.  The MSI-X Table and PBA may
//   be moved by changing the default values below (mgmt_msix_table_offset &
//   mgmt_msix_pba_offset are outputs of this module to the MSI-X arbiter).  If
//   selecting non-default values, care must be taken to place the Table and
//   PBA in their own 4 KByte (8 KB prefferred) page of memory.  The Tables
//   must also be placed on addresses that are aligned to the Table Size.
// Each MSI-X vector requires 16 bytes of MSI-X Table Space.  The default Table
//   and PBA locations below allow for a maximum of 256 vectors.  The default
//   table size in this module is 32 vectors.  A MSI-X table supporting all
//   2048 possible MSI-X vectors requires 32 KBytes.  There is no unoccupied
//   space in the default BAR0 register layout to supports a MSI-X Table larger
//   than 256 vectors.  When more than 256 vectors are selected, the MSI-X Tables
//   are moved to offset +64KB and +96 KB respectively which can support the
//   maximum of 2048 vectors.  Since BAR0 is normally only 64 KBytes, BAR0 size
//   is expanded to 128 Bytes in this case.

localparam  DESC_STATUS_WIDTH           = 160;

localparam  USER_CONTROL_WIDTH          = 64;
localparam  USER_STATUS_WIDTH           = 64;

localparam  DESC_WIDTH                  = 256;

localparam  DSIZE                       = 128;      // MPFE: Local side data width
parameter   SDRAM_ADDR_WIDTH            = 33;       // MPFE: Local address width
parameter   BURST_SIZE_WIDTH            = 8;        // MPFE: Width of l_b_size parameter
parameter   BYTE_ADDRESSING             = 0;        // Use Byte Addressing on SDRAM Interface when 1

// If implemented, packet generators are located at BAR0 offset of DMA Engine registers + 16'hA000 bytes
localparam  PKT_GEN_REG_BASE_ADDR_C2S0  = 32'h200 + 32'hA00;
localparam  PKT_GEN_REG_BASE_ADDR_C2S1  = 32'h210 + 32'hA00;
localparam  PKT_GEN_REG_BASE_ADDR_C2S2  = 32'h220 + 32'hA00;
localparam  PKT_GEN_REG_BASE_ADDR_C2S3  = 32'h230 + 32'hA00;
localparam  PKT_GEN_REG_BASE_ADDR_C2S4  = 32'h240 + 32'hA00;
localparam  PKT_GEN_REG_BASE_ADDR_C2S5  = 32'h250 + 32'hA00;

// If implemented, packet checkers are located at BAR0 offset of DMA Engine registers + 16'hA000 bytes
localparam  PKT_CHK_REG_BASE_ADDR_S2C0  = 32'h000 + 32'hA00;
localparam  PKT_CHK_REG_BASE_ADDR_S2C1  = 32'h010 + 32'hA00;
localparam  PKT_CHK_REG_BASE_ADDR_S2C2  = 32'h020 + 32'hA00;
localparam  PKT_CHK_REG_BASE_ADDR_S2C3  = 32'h030 + 32'hA00;



// ----------------------
// -- Port Definitions --
// ----------------------

input                               pcie_srst_n;
input                               pcie_clk;

input                               sdram_srst_n;
input                               sdram_clk;
input                               testmode;

input                               targ_wr_req;
input                               targ_wr_core_ready;
output                              targ_wr_user_ready;
input   [4:0]                       targ_wr_cs;
input                               targ_wr_start;
input   [31:0]                      targ_wr_addr;
input   [12:0]                      targ_wr_count;
input                               targ_wr_en;
input   [CORE_DATA_WIDTH-1:0]       targ_wr_data;
input   [CORE_BE_WIDTH-1:0]         targ_wr_be;

input                               targ_rd_req;
input                               targ_rd_core_ready;
output                              targ_rd_user_ready;
input   [4:0]                       targ_rd_cs;
input                               targ_rd_start;
input   [31:0]                      targ_rd_addr;
input   [12:0]                      targ_rd_count;
input                               targ_rd_en;
output  [CORE_DATA_WIDTH-1:0]       targ_rd_data;

input   [REG_ADDR_WIDTH-1:0]        reg_wr_addr;
input                               reg_wr_en;
input   [CORE_BE_WIDTH-1:0]         reg_wr_be;
input   [CORE_DATA_WIDTH-1:0]       reg_wr_data;
input   [REG_ADDR_WIDTH-1:0]        reg_rd_addr;
output  [CORE_DATA_WIDTH-1:0]       reg_rd_data;

output                              c2s0_desc_req;
input                               c2s0_desc_ready;
output  [31:0]                      c2s0_desc_ptr;
output  [DESC_WIDTH-1:0]            c2s0_desc_data;
output                              c2s0_desc_abort;
input                               c2s0_desc_abort_ack;
output                              c2s0_desc_rst_n;

input                               c2s0_desc_done;
input   [7:0]                       c2s0_desc_done_channel;
input   [DESC_STATUS_WIDTH-1:0]     c2s0_desc_done_status;

input                               c2s0_cmd_rst_n;
input                               c2s0_cmd_req;
output                              c2s0_cmd_ready;
input                               c2s0_cmd_first_chain;
input                               c2s0_cmd_last_chain;
input   [63:0]                      c2s0_cmd_addr;
input   [31:0]                      c2s0_cmd_bcount;
input   [USER_CONTROL_WIDTH-1:0]    c2s0_cmd_user_control;
input                               c2s0_cmd_abort;
output                              c2s0_cmd_abort_ack;

input                               c2s0_data_req;
output                              c2s0_data_ready;
input   [CORE_REMAIN_WIDTH-1:0]     c2s0_data_req_remain;
input                               c2s0_data_req_last_desc;
input   [63:0]                      c2s0_data_addr;
input   [9:0]                       c2s0_data_bcount;
output                              c2s0_data_stop;
output  [9:0]                       c2s0_data_stop_bcount;

input                               c2s0_data_en;
input   [CORE_REMAIN_WIDTH-1:0]     c2s0_data_remain;
input   [CORE_REMAIN_WIDTH:0]       c2s0_data_valid;
input                               c2s0_data_first_req;
input                               c2s0_data_last_req;
input                               c2s0_data_first_desc;
input                               c2s0_data_last_desc;
input                               c2s0_data_first_chain;
input                               c2s0_data_last_chain;
output                              c2s0_data_sop;
output                              c2s0_data_eop;
output  [CORE_DATA_WIDTH-1:0]       c2s0_data_data;
output  [63:0]                      c2s0_data_user_status;
output                              s2c0_desc_req;
input                               s2c0_desc_ready;
output  [31:0]                      s2c0_desc_ptr;
output  [255:0]                     s2c0_desc_data;
output                              s2c0_desc_abort;
input                               s2c0_desc_abort_ack;
output                              s2c0_desc_rst_n;

input                               s2c0_desc_done;
input   [7:0]                       s2c0_desc_done_channel;
input   [159:0]                     s2c0_desc_done_status;

input                               s2c0_cmd_rst_n;
input                               s2c0_cmd_req;
output                              s2c0_cmd_ready;
input   [63:0]                      s2c0_cmd_addr;
input   [9:0]                       s2c0_cmd_bcount;
input   [63:0]                      s2c0_cmd_user_control;
input                               s2c0_cmd_abort;
output                              s2c0_cmd_abort_ack;
output                              s2c0_cmd_stop;
output  [9:0]                       s2c0_cmd_stop_bcount;

input                               s2c0_data_req;
output                              s2c0_data_ready;
input   [63:0]                      s2c0_data_addr;
input   [9:0]                       s2c0_data_bcount;
input                               s2c0_data_en;
input                               s2c0_data_error;
input   [CORE_REMAIN_WIDTH-1:0]     s2c0_data_remain;
input   [CORE_REMAIN_WIDTH:0]       s2c0_data_valid;
input                               s2c0_data_first_req;
input                               s2c0_data_last_req;
input                               s2c0_data_first_desc;
input                               s2c0_data_last_desc;
input                               s2c0_data_first_chain;
input                               s2c0_data_last_chain;
input   [CORE_DATA_WIDTH-1:0]       s2c0_data_data;
input   [63:0]                      s2c0_data_user_control;
 `ifdef INSTANTIATE_MPFE
output                              c2s0_l_r_req;
output                              c2s0_l_w_req;
output  [2:0]                       c2s0_l_req_pri;
output  [SDRAM_ADDR_WIDTH-1:0]      c2s0_l_addr;
output  [BURST_SIZE_WIDTH-1:0]      c2s0_l_b_size;
output                              c2s0_l_auto_pch;
input                               c2s0_l_busy;
input                               c2s0_l_r_valid;
input                               c2s0_l_r_valid_last;
input                               c2s0_l_d_req;
input                               c2s0_l_d_req_last;
output  [DSIZE-1:0]                 c2s0_l_datain;
output  [DSIZE/8-1:0]               c2s0_l_dm_in;
input   [DSIZE-1:0]                 c2s0_l_dataout;
 `endif
 `ifdef INSTANTIATE_MPFE
output                              s2c0_l_r_req;
output                              s2c0_l_w_req;
output  [2:0]                       s2c0_l_req_pri;
output  [SDRAM_ADDR_WIDTH-1:0]      s2c0_l_addr;
output  [BURST_SIZE_WIDTH-1:0]      s2c0_l_b_size;
output                              s2c0_l_auto_pch;
input                               s2c0_l_busy;
input                               s2c0_l_r_valid;
input                               s2c0_l_r_valid_last;
input                               s2c0_l_d_req;
input                               s2c0_l_d_req_last;
output  [DSIZE-1:0]                 s2c0_l_datain;
output  [DSIZE/8-1:0]               s2c0_l_dm_in;
input   [DSIZE-1:0]                 s2c0_l_dataout;
 `endif
input   [7:0]                       mgmt_core_ph;
input   [11:0]                      mgmt_core_pd;
input   [7:0]                       mgmt_core_nh;
input   [11:0]                      mgmt_core_nd;
input   [7:0]                       mgmt_core_ch;
input   [11:0]                      mgmt_core_cd;

input   [7:0]                       mgmt_chipset_ph;
input   [11:0]                      mgmt_chipset_pd;
input   [7:0]                       mgmt_chipset_nh;
input   [11:0]                      mgmt_chipset_nd;
input   [7:0]                       mgmt_chipset_ch;
input   [11:0]                      mgmt_chipset_cd;

output  [31:0]                      mgmt_user_version;
input   [31:0]                      mgmt_core_version;
input   [31:0]                      mgmt_be_version;
input                               mgmt_msi_en;
input                               mgmt_msix_en;
output  [31:0]                      mgmt_interrupt;



// ----------------
// -- Port Types --
// ----------------

wire                                pcie_srst_n;
wire                                pcie_clk;

wire                                sdram_srst_n;
wire                                sdram_clk;
wire                                testmode;

wire    [7:0]                       mgmt_core_ph;
wire    [11:0]                      mgmt_core_pd;
wire    [7:0]                       mgmt_core_nh;
wire    [11:0]                      mgmt_core_nd;
wire    [7:0]                       mgmt_core_ch;
wire    [11:0]                      mgmt_core_cd;

wire    [7:0]                       mgmt_chipset_ph;
wire    [11:0]                      mgmt_chipset_pd;
wire    [7:0]                       mgmt_chipset_nh;
wire    [11:0]                      mgmt_chipset_nd;
wire    [7:0]                       mgmt_chipset_ch;
wire    [11:0]                      mgmt_chipset_cd;

wire    [31:0]                      mgmt_user_version;
wire    [31:0]                      mgmt_core_version;
wire    [31:0]                      mgmt_be_version;
wire                                mgmt_msi_en;
wire                                mgmt_msix_en;
wire    [31:0]                      mgmt_interrupt;

wire                                targ_wr_req;
wire                                targ_wr_core_ready;
wire                                targ_wr_user_ready;
wire    [4:0]                       targ_wr_cs;
wire                                targ_wr_start;
wire    [31:0]                      targ_wr_addr;
wire    [12:0]                      targ_wr_count;
wire                                targ_wr_en;
wire    [CORE_DATA_WIDTH-1:0]       targ_wr_data;
wire    [CORE_BE_WIDTH-1:0]         targ_wr_be;

wire                                targ_rd_req;
wire                                targ_rd_core_ready;
wire                                targ_rd_user_ready;
wire    [4:0]                       targ_rd_cs;
wire                                targ_rd_start;
wire    [31:0]                      targ_rd_addr;
wire    [12:0]                      targ_rd_count;
wire                                targ_rd_en;
wire    [CORE_DATA_WIDTH-1:0]       targ_rd_data;

wire    [REG_ADDR_WIDTH-1:0]        reg_wr_addr;
wire                                reg_wr_en;
wire    [CORE_BE_WIDTH-1:0]         reg_wr_be;
wire    [CORE_DATA_WIDTH-1:0]       reg_wr_data;
wire    [REG_ADDR_WIDTH-1:0]        reg_rd_addr;
wire    [CORE_DATA_WIDTH-1:0]       reg_rd_data;

wire                                c2s0_desc_req;
wire                                c2s0_desc_ready;
wire    [31:0]                      c2s0_desc_ptr;
wire    [DESC_WIDTH-1:0]            c2s0_desc_data;
wire                                c2s0_desc_abort;
wire                                c2s0_desc_abort_ack;
wire                                c2s0_desc_rst_n;

wire                                c2s0_desc_done;
wire    [7:0]                       c2s0_desc_done_channel;
wire    [DESC_STATUS_WIDTH-1:0]     c2s0_desc_done_status;

wire                                c2s0_cmd_rst_n;
wire                                c2s0_cmd_req;
wire                                c2s0_cmd_ready;
wire                                c2s0_cmd_first_chain;
wire                                c2s0_cmd_last_chain;
wire    [63:0]                      c2s0_cmd_addr;
wire    [31:0]                      c2s0_cmd_bcount;
wire    [USER_CONTROL_WIDTH-1:0]    c2s0_cmd_user_control;
wire                                c2s0_cmd_abort;
wire                                c2s0_cmd_abort_ack;

wire                                c2s0_data_req;
wire                                c2s0_data_ready;
wire    [CORE_REMAIN_WIDTH-1:0]     c2s0_data_req_remain;
wire                                c2s0_data_req_last_desc;
wire    [63:0]                      c2s0_data_addr;
wire    [9:0]                       c2s0_data_bcount;
wire                                c2s0_data_stop;
wire    [9:0]                       c2s0_data_stop_bcount;

wire                                c2s0_data_en;
wire    [CORE_REMAIN_WIDTH-1:0]     c2s0_data_remain;
wire    [CORE_REMAIN_WIDTH:0]       c2s0_data_valid;
wire                                c2s0_data_first_req;
wire                                c2s0_data_last_req;
wire                                c2s0_data_first_desc;
wire                                c2s0_data_last_desc;
wire                                c2s0_data_first_chain;
wire                                c2s0_data_last_chain;
wire                                c2s0_data_sop;
wire                                c2s0_data_eop;
wire    [CORE_DATA_WIDTH-1:0]       c2s0_data_data;
wire    [63:0]                      c2s0_data_user_status;

wire    [63:0]                      c2s0_user_status;
wire                                c2s0_sop;
wire                                c2s0_eop;
wire    [CORE_DATA_WIDTH-1:0]       c2s0_data;
wire    [CORE_REMAIN_WIDTH-1:0]     c2s0_valid;
wire                                c2s0_src_rdy;
wire                                c2s0_dst_rdy;
wire                                c2s0_abort;
wire                                c2s0_abort_ack;
wire                                c2s0_user_rst_n;
wire                                c2s0_apkt_req;
wire                                c2s0_apkt_ready;
wire    [63:0]                      c2s0_apkt_addr;
wire    [31:0]                      c2s0_apkt_bcount;
wire                                c2s0_apkt_eop;
wire                                s2c0_desc_req;
wire                                s2c0_desc_ready;
wire    [31:0]                      s2c0_desc_ptr;
wire    [255:0]                     s2c0_desc_data;
wire                                s2c0_desc_abort;
wire                                s2c0_desc_abort_ack;
wire                                s2c0_desc_rst_n;

wire                                s2c0_desc_done;
wire    [7:0]                       s2c0_desc_done_channel;
wire    [159:0]                     s2c0_desc_done_status;

wire                                s2c0_cmd_rst_n;
wire                                s2c0_cmd_req;
wire                                s2c0_cmd_ready;
wire    [63:0]                      s2c0_cmd_addr;
wire    [9:0]                       s2c0_cmd_bcount;
wire    [63:0]                      s2c0_cmd_user_control;
wire                                s2c0_cmd_abort;
wire                                s2c0_cmd_abort_ack;
wire                                s2c0_cmd_stop;
wire    [9:0]                       s2c0_cmd_stop_bcount;

wire                                s2c0_data_req;
wire                                s2c0_data_ready;
wire    [63:0]                      s2c0_data_addr;
wire    [9:0]                       s2c0_data_bcount;
wire                                s2c0_data_en;
wire                                s2c0_data_error;
wire    [CORE_REMAIN_WIDTH-1:0]     s2c0_data_remain;
wire    [CORE_REMAIN_WIDTH:0]       s2c0_data_valid;
wire                                s2c0_data_first_req;
wire                                s2c0_data_last_req;
wire                                s2c0_data_first_desc;
wire                                s2c0_data_last_desc;
wire                                s2c0_data_first_chain;
wire                                s2c0_data_last_chain;
wire    [CORE_DATA_WIDTH-1:0]       s2c0_data_data;
wire    [63:0]                      s2c0_data_user_control;

wire    [USER_STATUS_WIDTH-1:0]     s2c0_lpbk_user_status;
wire                                s2c0_lpbk_sop;
wire                                s2c0_lpbk_eop;
wire    [CORE_DATA_WIDTH-1:0]       s2c0_lpbk_data;
wire    [CORE_REMAIN_WIDTH-1:0]     s2c0_lpbk_valid;
wire                                s2c0_lpbk_src_rdy;
wire                                s2c0_lpbk_dst_rdy;

wire                                s2c0_error_sop;
wire                                s2c0_error_eop;
wire                                s2c0_error_cpl;
wire                                s2c0_error_data;
wire                                s2c0_error_valid;
wire                                s2c0_error_user_control;

wire    [USER_CONTROL_WIDTH-1:0]    s2c0_user_control;
wire                                s2c0_sop;
wire                                s2c0_eop;
wire    [CORE_DATA_WIDTH-1:0]       s2c0_data;
wire    [CORE_REMAIN_WIDTH-1:0]     s2c0_valid;
wire                                s2c0_err;
wire                                s2c0_src_rdy;
wire                                s2c0_dst_rdy;
wire                                s2c0_abort;
wire                                s2c0_abort_ack;
wire                                s2c0_user_rst_n;
wire                                s2c0_apkt_req;
wire                                s2c0_apkt_ready;
wire   [63:0]                       s2c0_apkt_addr;
wire   [9:0]                        s2c0_apkt_bcount;
 `ifdef INSTANTIATE_MPFE
wire                                c2s0_l_r_req;
wire                                c2s0_l_w_req;
wire    [2:0]                       c2s0_l_req_pri;
wire    [SDRAM_ADDR_WIDTH-1:0]      c2s0_l_addr;
wire    [BURST_SIZE_WIDTH-1:0]      c2s0_l_b_size;
wire                                c2s0_l_auto_pch;
wire                                c2s0_l_busy;
wire                                c2s0_l_r_valid;
wire                                c2s0_l_r_valid_last;
wire                                c2s0_l_d_req;
wire                                c2s0_l_d_req_last;
wire    [DSIZE-1:0]                 c2s0_l_datain;
wire    [DSIZE/8-1:0]               c2s0_l_dm_in;
wire    [DSIZE-1:0]                 c2s0_l_dataout;
 `endif
 `ifdef INSTANTIATE_MPFE
wire                                s2c0_l_r_req;
wire                                s2c0_l_w_req;
wire    [2:0]                       s2c0_l_req_pri;
wire    [SDRAM_ADDR_WIDTH-1:0]      s2c0_l_addr;
wire    [BURST_SIZE_WIDTH-1:0]      s2c0_l_b_size;
wire                                s2c0_l_auto_pch;
wire                                s2c0_l_busy;
wire                                s2c0_l_r_valid;
wire                                s2c0_l_r_valid_last;
wire                                s2c0_l_d_req;
wire                                s2c0_l_d_req_last;
wire    [DSIZE-1:0]                 s2c0_l_datain;
wire    [DSIZE/8-1:0]               s2c0_l_dm_in;
wire    [DSIZE-1:0]                 s2c0_l_dataout;
wire                                s2c0_l_error_unused;
 `endif
// -------------------
// -- Local Signals --
// -------------------

// Convert base address parameters to wires so that bits can be picked off
wire    [31:0]                      pg_c2s0 = PKT_GEN_REG_BASE_ADDR_C2S0;
wire    [31:0]                      pg_c2s1 = PKT_GEN_REG_BASE_ADDR_C2S1;
wire    [31:0]                      pg_c2s2 = PKT_GEN_REG_BASE_ADDR_C2S2;
wire    [31:0]                      pg_c2s3 = PKT_GEN_REG_BASE_ADDR_C2S3;
wire    [31:0]                      pg_c2s4 = PKT_GEN_REG_BASE_ADDR_C2S4;
wire    [31:0]                      pg_c2s5 = PKT_GEN_REG_BASE_ADDR_C2S5;

wire    [31:0]                      pc_s2c0 = PKT_CHK_REG_BASE_ADDR_S2C0;
wire    [31:0]                      pc_s2c1 = PKT_CHK_REG_BASE_ADDR_S2C1;
wire    [31:0]                      pc_s2c2 = PKT_CHK_REG_BASE_ADDR_S2C2;
wire    [31:0]                      pc_s2c3 = PKT_CHK_REG_BASE_ADDR_S2C3;


// Register Interface Example
wire    [CORE_DATA_WIDTH-1:0]       example_reg_rd_data;

wire    [CORE_DATA_WIDTH-1:0]       c2s0_reg_rd_data;
wire    [CORE_DATA_WIDTH-1:0]       c2s1_reg_rd_data;
wire    [CORE_DATA_WIDTH-1:0]       c2s2_reg_rd_data;
wire    [CORE_DATA_WIDTH-1:0]       c2s3_reg_rd_data;
wire    [CORE_DATA_WIDTH-1:0]       c2s4_reg_rd_data;
wire    [CORE_DATA_WIDTH-1:0]       c2s5_reg_rd_data;

wire    [CORE_DATA_WIDTH-1:0]       s2c0_reg_rd_data;
wire    [CORE_DATA_WIDTH-1:0]       s2c1_reg_rd_data;
wire    [CORE_DATA_WIDTH-1:0]       s2c2_reg_rd_data;
wire    [CORE_DATA_WIDTH-1:0]       s2c3_reg_rd_data;

wire                                c2s0_pkt_sel_ram_pkt_n;
wire    [63:0]                      c2s0_pkt_gen_user_status;
wire                                c2s0_pkt_gen_src_rdy;
wire                                c2s0_pkt_gen_dst_rdy;
wire    [CORE_REMAIN_WIDTH-1:0]     c2s0_pkt_gen_valid;
wire    [CORE_DATA_WIDTH-1:0]       c2s0_pkt_gen_data;
wire                                c2s0_pkt_gen_sop;
wire                                c2s0_pkt_gen_eop;
wire                                c2s0_pkt_gen_abort;
wire                                c2s0_pkt_gen_abort_ack;

wire                                c2s0_ap_cmd_rst_n;
wire                                c2s0_ap_cmd_req;
wire                                c2s0_ap_cmd_ready;
wire                                c2s0_ap_cmd_first_chain;
wire                                c2s0_ap_cmd_last_chain;
wire    [63:0]                      c2s0_ap_cmd_addr;
wire    [31:0]                      c2s0_ap_cmd_bcount;
wire    [USER_CONTROL_WIDTH-1:0]    c2s0_ap_cmd_user_control;
wire                                c2s0_ap_cmd_abort;
wire                                c2s0_ap_cmd_abort_ack;

wire                                c2s0_ap_data_req;
wire                                c2s0_ap_data_ready;
wire    [CORE_REMAIN_WIDTH-1:0]     c2s0_ap_data_req_remain;
wire                                c2s0_ap_data_req_last_desc;
wire    [63:0]                      c2s0_ap_data_addr;
wire    [9:0]                       c2s0_ap_data_bcount;
wire                                c2s0_ap_data_stop;
wire    [9:0]                       c2s0_ap_data_stop_bcount;

wire                                c2s0_ap_data_en;
wire    [CORE_REMAIN_WIDTH-1:0]     c2s0_ap_data_remain;
wire    [CORE_REMAIN_WIDTH:0]       c2s0_ap_data_valid;
wire                                c2s0_ap_data_first_req;
wire                                c2s0_ap_data_last_req;
wire                                c2s0_ap_data_first_desc;
wire                                c2s0_ap_data_last_desc;
wire                                c2s0_ap_data_first_chain;
wire                                c2s0_ap_data_last_chain;
wire                                c2s0_ap_data_sop;
wire                                c2s0_ap_data_eop;
wire    [CORE_DATA_WIDTH-1:0]       c2s0_ap_data_data;
wire    [63:0]                      c2s0_ap_data_user_status;


wire                                c2s0_sf_cmd_rst_n;
wire                                c2s0_sf_cmd_req;
wire                                c2s0_sf_cmd_ready;
wire                                c2s0_sf_cmd_first_chain;
wire                                c2s0_sf_cmd_last_chain;
wire    [63:0]                      c2s0_sf_cmd_addr;
wire    [31:0]                      c2s0_sf_cmd_bcount;
wire    [USER_CONTROL_WIDTH-1:0]    c2s0_sf_cmd_user_control;
wire                                c2s0_sf_cmd_abort;
wire                                c2s0_sf_cmd_abort_ack;

wire                                c2s0_sf_data_req;
wire                                c2s0_sf_data_ready;
wire    [CORE_REMAIN_WIDTH-1:0]     c2s0_sf_data_req_remain;
wire                                c2s0_sf_data_req_last_desc;
wire    [63:0]                      c2s0_sf_data_addr;
wire    [9:0]                       c2s0_sf_data_bcount;
wire                                c2s0_sf_data_stop;
wire    [9:0]                       c2s0_sf_data_stop_bcount;

wire                                c2s0_sf_data_en;
wire    [CORE_REMAIN_WIDTH-1:0]     c2s0_sf_data_remain;
wire    [CORE_REMAIN_WIDTH:0]       c2s0_sf_data_valid;
wire                                c2s0_sf_data_first_req;
wire                                c2s0_sf_data_last_req;
wire                                c2s0_sf_data_first_desc;
wire                                c2s0_sf_data_last_desc;
wire                                c2s0_sf_data_first_chain;
wire                                c2s0_sf_data_last_chain;
wire                                c2s0_sf_data_sop;
wire                                c2s0_sf_data_eop;
wire    [CORE_DATA_WIDTH-1:0]       c2s0_sf_data_data;
wire    [63:0]                      c2s0_sf_data_user_status;
wire                                s2c0_pkt_chk_dst_rdy;
wire                                s2c0_pkt_chk_src_rdy;
wire                                s2c0_pkt_chk_abort;
wire                                s2c0_pkt_chk_abort_ack;
wire                                s2c0_pkt_sel_ram_pkt_n;

  `ifdef INSTANTIATE_MPFE
wire                                s2c0_ap_cmd_req;
wire                                s2c0_ap_cmd_ready;
wire    [63:0]                      s2c0_ap_cmd_addr;
wire    [9:0]                       s2c0_ap_cmd_bcount;
wire    [63:0]                      s2c0_ap_cmd_user_control;
wire                                s2c0_ap_cmd_abort;
wire                                s2c0_ap_cmd_abort_ack;
wire                                s2c0_ap_cmd_stop;
wire    [9:0]                       s2c0_ap_cmd_stop_bcount;

wire                                s2c0_ap_data_req;
wire                                s2c0_ap_data_ready;
wire    [63:0]                      s2c0_ap_data_addr;
wire    [9:0]                       s2c0_ap_data_bcount;
wire                                s2c0_ap_data_en;
wire                                s2c0_ap_data_error;
wire    [CORE_REMAIN_WIDTH-1:0]     s2c0_ap_data_remain;
wire    [CORE_REMAIN_WIDTH:0]       s2c0_ap_data_valid;
wire                                s2c0_ap_data_first_req;
wire                                s2c0_ap_data_last_req;
wire                                s2c0_ap_data_first_desc;
wire                                s2c0_ap_data_last_desc;
wire                                s2c0_ap_data_first_chain;
wire                                s2c0_ap_data_last_chain;
wire    [CORE_DATA_WIDTH-1:0]       s2c0_ap_data_data;
wire    [63:0]                      s2c0_ap_data_user_control;

  `endif
wire                                s2c0_sf_cmd_req;
wire                                s2c0_sf_cmd_ready;
wire    [63:0]                      s2c0_sf_cmd_addr;
wire    [9:0]                       s2c0_sf_cmd_bcount;
wire    [63:0]                      s2c0_sf_cmd_user_control;
wire                                s2c0_sf_cmd_abort;
wire                                s2c0_sf_cmd_abort_ack;
wire                                s2c0_sf_cmd_stop;
wire    [9:0]                       s2c0_sf_cmd_stop_bcount;

wire                                s2c0_sf_data_req;
wire                                s2c0_sf_data_ready;
wire    [63:0]                      s2c0_sf_data_addr;
wire    [9:0]                       s2c0_sf_data_bcount;
wire                                s2c0_sf_data_en;
wire                                s2c0_sf_data_error;
wire    [CORE_REMAIN_WIDTH-1:0]     s2c0_sf_data_remain;
wire    [CORE_REMAIN_WIDTH:0]       s2c0_sf_data_valid;
wire                                s2c0_sf_data_first_req;
wire                                s2c0_sf_data_last_req;
wire                                s2c0_sf_data_first_desc;
wire                                s2c0_sf_data_last_desc;
wire                                s2c0_sf_data_first_chain;
wire                                s2c0_sf_data_last_chain;
wire    [CORE_DATA_WIDTH-1:0]       s2c0_sf_data_data;
wire    [63:0]                      s2c0_sf_data_user_control;

// ---------------
// -- Equations --
// ---------------

// ---------------------
// Core Revision History

// Version Information
//   Unused[31:24]
//   Increment on Major Release/Feature Add[23:16]
//   Increment on Major Bug Fix[15:8]
//   Increment on Minor Bug Fix[7:0]

assign mgmt_user_version = 32'h00_01_01_00;

// Version 00.01.00.01 (03/05/09)
//   Introduced mgmt_user_version to track the version of the reference design
//
// Version 00.01.01.00 (05/11/09)
//   Added support for Packet DMA



// ------------------------
// Target Interface Example

target_example #(

    .SRAM_ADDR_WIDTH                (SRAM_ADDR_WIDTH            )

) target_example (

    .rst_n                          (pcie_srst_n                ),
    .clk                            (pcie_clk                   ),

    .targ_wr_req                    (targ_wr_req                ),
    .targ_wr_core_ready             (targ_wr_core_ready         ),
    .targ_wr_user_ready             (targ_wr_user_ready         ),
    .targ_wr_cs                     (targ_wr_cs                 ),
    .targ_wr_start                  (targ_wr_start              ),
    .targ_wr_addr                   (targ_wr_addr               ),
    .targ_wr_count                  (targ_wr_count              ),
    .targ_wr_en                     (targ_wr_en                 ),
    .targ_wr_data                   (targ_wr_data               ),
    .targ_wr_be                     (targ_wr_be                 ),

    .targ_rd_req                    (targ_rd_req                ),
    .targ_rd_core_ready             (targ_rd_core_ready         ),
    .targ_rd_user_ready             (targ_rd_user_ready         ),
    .targ_rd_cs                     (targ_rd_cs                 ),
    .targ_rd_start                  (targ_rd_start              ),
    .targ_rd_addr                   (targ_rd_addr               ),
    .targ_rd_count                  (targ_rd_count              ),
    .targ_rd_en                     (targ_rd_en                 ),
    .targ_rd_data                   (targ_rd_data               )

);

// --------------------------
// Register Interface Example

register_example #(.REG_ADDR_WIDTH (REG_ADDR_WIDTH)) register_example (

    .rst_n                          (pcie_srst_n                ),
    .clk                            (pcie_clk                   ),

    .mgmt_core_ph                   ({8'h0, mgmt_core_ph}       ),
    .mgmt_core_pd                   ({4'h0, mgmt_core_pd}       ),
    .mgmt_core_nh                   ({8'h0, mgmt_core_nh}       ),
    .mgmt_core_nd                   ({4'h0, mgmt_core_nd}       ),
    .mgmt_core_ch                   ({8'h0, mgmt_core_ch}       ),
    .mgmt_core_cd                   ({4'h0, mgmt_core_cd}       ),

    .mgmt_chipset_ph                ({8'h0, mgmt_chipset_ph}    ),
    .mgmt_chipset_pd                ({4'h0, mgmt_chipset_pd}    ),
    .mgmt_chipset_nh                ({8'h0, mgmt_chipset_nh}    ),
    .mgmt_chipset_nd                ({4'h0, mgmt_chipset_nd}    ),
    .mgmt_chipset_ch                ({8'h0, mgmt_chipset_ch}    ),
    .mgmt_chipset_cd                ({4'h0, mgmt_chipset_cd}    ),

    .mgmt_msi_en                    (mgmt_msi_en                ),
    .mgmt_msix_en                   (mgmt_msix_en               ),
    .mgmt_interrupt                 (mgmt_interrupt             ),

    .reg_wr_addr                    (reg_wr_addr                ),
    .reg_wr_en                      (reg_wr_en                  ),
    .reg_wr_be                      (reg_wr_be                  ),
    .reg_wr_data                    (reg_wr_data                ),
    .reg_rd_addr                    (reg_rd_addr                ),
    .reg_rd_data                    (example_reg_rd_data        )

);

assign reg_rd_data = example_reg_rd_data |
                     c2s0_reg_rd_data    |
                     c2s1_reg_rd_data    |
                     c2s2_reg_rd_data    |
                     c2s3_reg_rd_data    |
                     c2s4_reg_rd_data    |
                     c2s5_reg_rd_data    |
                     s2c0_reg_rd_data    |
                     s2c1_reg_rd_data    |
                     s2c2_reg_rd_data    |
                     s2c3_reg_rd_data;



// --------------
// Card to System

// DMA Direct Control Port is unused
assign c2s0_desc_req             = 1'b0;
assign c2s0_desc_ptr             = 32'h0;
assign c2s0_desc_data            = {DESC_WIDTH{1'b0}};
assign c2s0_desc_abort           = 1'b0;
assign c2s0_desc_rst_n           = 1'b1;

packet_gen #(.REG_ADDR_WIDTH (REG_ADDR_WIDTH)) c2s0_packet_gen (

    .rst_n                  (c2s0_user_rst_n            ),
    .clk                    (pcie_clk                   ),

    .reg_base_addr          (pg_c2s0[REG_ADDR_WIDTH-1:0]),

    .reg_wr_addr            (reg_wr_addr                ),
    .reg_wr_en              (reg_wr_en                  ),
    .reg_wr_be              (reg_wr_be                  ),
    .reg_wr_data            (reg_wr_data                ),
    .reg_rd_addr            (reg_rd_addr                ),
    .reg_rd_data            (c2s0_reg_rd_data           ),

    .lpbk_user_status       (s2c0_lpbk_user_status      ),
    .lpbk_sop               (s2c0_lpbk_sop              ),
    .lpbk_eop               (s2c0_lpbk_eop              ),
    .lpbk_data              (s2c0_lpbk_data             ),
    .lpbk_valid             (s2c0_lpbk_valid            ),
    .lpbk_src_rdy           (s2c0_lpbk_src_rdy          ),
    .lpbk_dst_rdy           (s2c0_lpbk_dst_rdy          ),

    .user_status            (c2s0_pkt_gen_user_status   ),
    .sop                    (c2s0_pkt_gen_sop           ),
    .eop                    (c2s0_pkt_gen_eop           ),
    .data                   (c2s0_pkt_gen_data          ),
    .valid                  (c2s0_pkt_gen_valid         ),
    .src_rdy                (c2s0_pkt_gen_src_rdy       ),
    .dst_rdy                (c2s0_pkt_gen_dst_rdy       ),
    .abort                  (c2s0_pkt_gen_abort         ),
    .abort_ack              (c2s0_pkt_gen_abort_ack     ),

    .pkt_sel_ram_pkt_n      (c2s0_pkt_sel_ram_pkt_n     )
);

  `ifdef INSTANTIATE_MPFE
assign c2s0_user_status = c2s0_pkt_sel_ram_pkt_n ? 64'h0 : c2s0_pkt_gen_user_status;
assign c2s0_src_rdy     = c2s0_pkt_gen_src_rdy;
assign c2s0_valid       = c2s0_pkt_gen_valid;
assign c2s0_data        = c2s0_pkt_gen_data;
assign c2s0_sop         = c2s0_pkt_gen_sop;
assign c2s0_eop         = c2s0_pkt_gen_eop;
assign c2s0_abort_ack   = c2s0_pkt_gen_abort_ack;
assign c2s0_pkt_gen_dst_rdy = c2s0_dst_rdy;
assign c2s0_pkt_gen_abort   = c2s0_abort    & ~c2s0_pkt_sel_ram_pkt_n;

assign c2s0_sf_cmd_req            = c2s0_pkt_sel_ram_pkt_n ?  1'b0 : c2s0_cmd_req;
assign c2s0_sf_cmd_abort          = c2s0_pkt_sel_ram_pkt_n ?  1'b0 : c2s0_cmd_abort;
assign c2s0_sf_cmd_first_chain    = c2s0_cmd_first_chain;
assign c2s0_sf_cmd_last_chain     = c2s0_cmd_last_chain;
assign c2s0_sf_cmd_bcount         = c2s0_cmd_bcount;
assign c2s0_sf_cmd_addr           = c2s0_cmd_addr;
assign c2s0_sf_cmd_user_control   = c2s0_cmd_user_control;

assign c2s0_sf_data_req           = c2s0_pkt_sel_ram_pkt_n ?  1'b0 : c2s0_data_req;
assign c2s0_sf_data_req_remain    = c2s0_data_req_remain;
assign c2s0_sf_data_req_last_desc = c2s0_data_req_last_desc;
assign c2s0_sf_data_addr          = c2s0_data_addr;
assign c2s0_sf_data_bcount        = c2s0_data_bcount;
assign c2s0_sf_data_en            = c2s0_pkt_sel_ram_pkt_n ?  1'b0 : c2s0_data_en;
assign c2s0_sf_data_remain        = c2s0_data_remain;
assign c2s0_sf_data_valid         = c2s0_data_valid;
assign c2s0_sf_data_first_req     = c2s0_data_first_req;
assign c2s0_sf_data_last_req      = c2s0_data_last_req;
assign c2s0_sf_data_first_desc    = c2s0_data_first_desc;
assign c2s0_sf_data_last_desc     = c2s0_data_last_desc;
assign c2s0_sf_data_first_chain   = c2s0_data_first_chain;
assign c2s0_sf_data_last_chain    = c2s0_data_last_chain;


assign c2s0_ap_cmd_req            = c2s0_pkt_sel_ram_pkt_n ? c2s0_cmd_req    : 1'b0;
assign c2s0_ap_cmd_abort          = c2s0_pkt_sel_ram_pkt_n ? c2s0_cmd_abort  : 1'b0;
assign c2s0_ap_cmd_first_chain    = c2s0_cmd_first_chain;
assign c2s0_ap_cmd_last_chain     = c2s0_cmd_last_chain;
assign c2s0_ap_cmd_bcount         = c2s0_cmd_bcount;
assign c2s0_ap_cmd_addr           = c2s0_cmd_addr;
assign c2s0_ap_cmd_user_control   = c2s0_cmd_user_control;

assign c2s0_ap_data_req           = c2s0_pkt_sel_ram_pkt_n ? c2s0_data_req   : 1'b0;
assign c2s0_ap_data_req_remain    = c2s0_data_req_remain;
assign c2s0_ap_data_req_last_desc = c2s0_data_req_last_desc;
assign c2s0_ap_data_addr          = c2s0_data_addr;
assign c2s0_ap_data_bcount        = c2s0_data_bcount;
assign c2s0_ap_data_en            = c2s0_pkt_sel_ram_pkt_n ? c2s0_data_en    : 1'b0;
assign c2s0_ap_data_remain        = c2s0_data_remain;
assign c2s0_ap_data_valid         = c2s0_data_valid;
assign c2s0_ap_data_first_req     = c2s0_data_first_req;
assign c2s0_ap_data_last_req      = c2s0_data_last_req;
assign c2s0_ap_data_first_desc    = c2s0_data_first_desc;
assign c2s0_ap_data_last_desc     = c2s0_data_last_desc;
assign c2s0_ap_data_first_chain   = c2s0_data_first_chain;
assign c2s0_ap_data_last_chain    = c2s0_data_last_chain;

assign c2s0_cmd_ready        = c2s0_pkt_sel_ram_pkt_n ? c2s0_ap_cmd_ready         : c2s0_sf_cmd_ready;
assign c2s0_cmd_abort_ack    = c2s0_pkt_sel_ram_pkt_n ? c2s0_ap_cmd_abort_ack     : c2s0_sf_cmd_abort_ack;
assign c2s0_data_stop        = c2s0_pkt_sel_ram_pkt_n ? c2s0_ap_data_stop         : c2s0_sf_data_stop;
assign c2s0_data_stop_bcount = c2s0_pkt_sel_ram_pkt_n ? c2s0_ap_data_stop_bcount  : c2s0_sf_data_stop_bcount;
assign c2s0_data_ready       = c2s0_pkt_sel_ram_pkt_n ? c2s0_ap_data_ready        : c2s0_sf_data_ready;
assign c2s0_data_data        = c2s0_pkt_sel_ram_pkt_n ? c2s0_ap_data_data         : c2s0_sf_data_data;
assign c2s0_data_sop         = c2s0_pkt_sel_ram_pkt_n ? c2s0_ap_data_sop          : c2s0_sf_data_sop;
assign c2s0_data_eop         = c2s0_pkt_sel_ram_pkt_n ? c2s0_ap_data_eop          : c2s0_sf_data_eop;
assign c2s0_data_user_status = c2s0_pkt_sel_ram_pkt_n ? c2s0_ap_data_user_status  : c2s0_sf_data_user_status;


// Instantiate Card to System FIFO to Interface to MPFE
c2s_adr_pkt  #(

    .SDRAM_ADDR_WIDTH       (SDRAM_ADDR_WIDTH           ),
    .SDRAM_BSIZE_WIDTH      (BURST_SIZE_WIDTH           ),
    .BYTE_ADDRESSING        (BYTE_ADDRESSING            ),
    .DMA_DEST_ADDR_WIDTH    (DMA_DEST_ADDR_WIDTH        ),
    .FIFO_CADDR_WIDTH       (FIFO_CADDR_WIDTH           ),
    .FIFO_DADDR_WIDTH       (FIFO_DADDR_WIDTH           )

) c2s0_adr_pkt (

    .pcie_rst_n             (c2s0_cmd_rst_n             ),
    .pcie_clk               (pcie_clk                   ),

    .sdram_rst_n            (sdram_srst_n               ),
    .sdram_clk              (sdram_clk                  ),

    .cmd_req                (c2s0_ap_cmd_req            ),
    .cmd_ready              (c2s0_ap_cmd_ready          ),
    .cmd_first_chain        (c2s0_ap_cmd_first_chain    ),
    .cmd_last_chain         (c2s0_ap_cmd_last_chain     ),
    .cmd_addr               (c2s0_ap_cmd_addr           ),
    .cmd_bcount             (c2s0_ap_cmd_bcount         ),
    .cmd_user_control       (c2s0_ap_cmd_user_control   ),
    .cmd_abort              (c2s0_ap_cmd_abort          ),
    .cmd_abort_ack          (c2s0_ap_cmd_abort_ack      ),

    .data_req               (c2s0_ap_data_req           ),
    .data_ready             (c2s0_ap_data_ready         ),
    .data_req_remain        (c2s0_ap_data_req_remain    ),
    .data_req_last_desc     (c2s0_ap_data_req_last_desc ),
    .data_addr              (c2s0_ap_data_addr          ),
    .data_bcount            (c2s0_ap_data_bcount        ),
    .data_stop              (c2s0_ap_data_stop          ),
    .data_stop_bcount       (c2s0_ap_data_stop_bcount   ),

    .data_en                (c2s0_ap_data_en            ),
    .data_remain            (c2s0_ap_data_remain        ),
    .data_valid             (c2s0_ap_data_valid         ),
    .data_first_req         (c2s0_ap_data_first_req     ),
    .data_last_req          (c2s0_ap_data_last_req      ),
    .data_first_desc        (c2s0_ap_data_first_desc    ),
    .data_last_desc         (c2s0_ap_data_last_desc     ),
    .data_first_chain       (c2s0_ap_data_first_chain   ),
    .data_last_chain        (c2s0_ap_data_last_chain    ),
    .data_sop               (c2s0_ap_data_sop           ),
    .data_eop               (c2s0_ap_data_eop           ),
    .data_data              (c2s0_ap_data_data          ),
    .data_user_status       (c2s0_ap_data_user_status   ),

    .l_r_req                (c2s0_l_r_req               ),
    .l_w_req                (c2s0_l_w_req               ),
    .l_req_pri              (c2s0_l_req_pri             ),
    .l_addr                 (c2s0_l_addr                ),
    .l_b_size               (c2s0_l_b_size              ),
    .l_auto_pch             (c2s0_l_auto_pch            ),
    .l_busy                 (c2s0_l_busy                ),
    .l_r_valid              (c2s0_l_r_valid             ),
    .l_r_valid_last         (c2s0_l_r_valid_last        ),
    .l_d_req                (c2s0_l_d_req               ),
    .l_d_req_last           (c2s0_l_d_req_last          ),
    .l_datain               (c2s0_l_datain              ),
    .l_dm_in                (c2s0_l_dm_in               ),
    .l_dataout              (c2s0_l_dataout             )

);
  `else
assign c2s0_sf_cmd_req            = c2s0_cmd_req;
assign c2s0_sf_cmd_abort          = c2s0_cmd_abort;
assign c2s0_sf_cmd_first_chain    = c2s0_cmd_first_chain;
assign c2s0_sf_cmd_last_chain     = c2s0_cmd_last_chain;
assign c2s0_sf_cmd_bcount         = c2s0_cmd_bcount;
assign c2s0_sf_cmd_addr           = c2s0_cmd_addr;
assign c2s0_sf_cmd_user_control   = c2s0_cmd_user_control;

assign c2s0_sf_data_req           = c2s0_data_req;
assign c2s0_sf_data_req_remain    = c2s0_data_req_remain;
assign c2s0_sf_data_req_last_desc = c2s0_data_req_last_desc;
assign c2s0_sf_data_addr          = c2s0_data_addr;
assign c2s0_sf_data_bcount        = c2s0_data_bcount;
assign c2s0_sf_data_en            = c2s0_data_en;
assign c2s0_sf_data_remain        = c2s0_data_remain;
assign c2s0_sf_data_valid         = c2s0_data_valid;
assign c2s0_sf_data_first_req     = c2s0_data_first_req;
assign c2s0_sf_data_last_req      = c2s0_data_last_req;
assign c2s0_sf_data_first_desc    = c2s0_data_first_desc;
assign c2s0_sf_data_last_desc     = c2s0_data_last_desc;
assign c2s0_sf_data_first_chain   = c2s0_data_first_chain;
assign c2s0_sf_data_last_chain    = c2s0_data_last_chain;

assign c2s0_user_status     = c2s0_pkt_gen_user_status;
assign c2s0_src_rdy         = c2s0_pkt_gen_src_rdy;
assign c2s0_valid           = c2s0_pkt_gen_valid;
assign c2s0_data            = c2s0_pkt_gen_data;
assign c2s0_sop             = c2s0_pkt_gen_sop;
assign c2s0_eop             = c2s0_pkt_gen_eop;
assign c2s0_abort_ack       = c2s0_pkt_gen_abort_ack;
assign c2s0_pkt_gen_dst_rdy = c2s0_dst_rdy;
assign c2s0_pkt_gen_abort   = c2s0_abort;

assign c2s0_cmd_ready        = c2s0_sf_cmd_ready;
assign c2s0_cmd_abort_ack    = c2s0_sf_cmd_abort_ack;
assign c2s0_data_stop        = c2s0_sf_data_stop;
assign c2s0_data_stop_bcount = c2s0_sf_data_stop_bcount;
assign c2s0_data_ready       = c2s0_sf_data_ready;
assign c2s0_data_data        = c2s0_sf_data_data;
assign c2s0_data_sop         = c2s0_sf_data_sop;
assign c2s0_data_eop         = c2s0_sf_data_eop;
assign c2s0_data_user_status = c2s0_sf_data_user_status;

  `endif
c2s_pkt_streaming_fifo #(

    .FIFO_ADDR_WIDTH        (FIFO_DADDR_WIDTH           )

) c2s0_c2s_pkt_streaming_fifo (

    .rst_n                  (c2s0_cmd_rst_n             ),
    .clk                    (pcie_clk                   ),

    .cmd_req                (c2s0_sf_cmd_req            ),
    .cmd_ready              (c2s0_sf_cmd_ready          ),
    .cmd_first_chain        (c2s0_sf_cmd_first_chain    ),
    .cmd_last_chain         (c2s0_sf_cmd_last_chain     ),
    .cmd_addr               (c2s0_sf_cmd_addr           ),
    .cmd_bcount             (c2s0_sf_cmd_bcount         ),
    .cmd_user_control       (c2s0_sf_cmd_user_control   ),
    .cmd_abort              (c2s0_sf_cmd_abort          ),
    .cmd_abort_ack          (c2s0_sf_cmd_abort_ack      ),

    .data_req               (c2s0_sf_data_req           ),
    .data_ready             (c2s0_sf_data_ready         ),
    .data_req_remain        (c2s0_sf_data_req_remain    ),
    .data_req_last_desc     (c2s0_sf_data_req_last_desc ),
    .data_addr              (c2s0_sf_data_addr          ),
    .data_bcount            (c2s0_sf_data_bcount        ),
    .data_stop              (c2s0_sf_data_stop          ),
    .data_stop_bcount       (c2s0_sf_data_stop_bcount   ),

    .data_en                (c2s0_sf_data_en            ),
    .data_remain            (c2s0_sf_data_remain        ),
    .data_valid             (c2s0_sf_data_valid         ),
    .data_first_req         (c2s0_sf_data_first_req     ),
    .data_last_req          (c2s0_sf_data_last_req      ),
    .data_first_desc        (c2s0_sf_data_first_desc    ),
    .data_last_desc         (c2s0_sf_data_last_desc     ),
    .data_first_chain       (c2s0_sf_data_first_chain   ),
    .data_last_chain        (c2s0_sf_data_last_chain    ),
    .data_sop               (c2s0_sf_data_sop           ),
    .data_eop               (c2s0_sf_data_eop           ),
    .data_data              (c2s0_sf_data_data          ),
    .data_user_status       (c2s0_sf_data_user_status   ),

    .user_status            (c2s0_user_status           ),
    .sop                    (c2s0_sop                   ),
    .eop                    (c2s0_eop                   ),
    .data                   (c2s0_data                  ),
    .valid                  (c2s0_valid                 ),
    .src_rdy                (c2s0_src_rdy               ),
    .dst_rdy                (c2s0_dst_rdy               ),
    .abort                  (c2s0_abort                 ),
    .abort_ack              (c2s0_abort_ack             ),
    .user_rst_n             (c2s0_user_rst_n            )
);
assign c2s1_reg_rd_data = {CORE_DATA_WIDTH{1'b0}};
assign c2s2_reg_rd_data = {CORE_DATA_WIDTH{1'b0}};
assign c2s3_reg_rd_data = {CORE_DATA_WIDTH{1'b0}};
assign c2s4_reg_rd_data = {CORE_DATA_WIDTH{1'b0}};
assign c2s5_reg_rd_data = {CORE_DATA_WIDTH{1'b0}};



// --------------
// System to Card

// DMA Direct Control Port is unused
assign s2c0_desc_req             = 1'b0;
assign s2c0_desc_ptr             = 32'h0;
assign s2c0_desc_data            = {DESC_WIDTH{1'b0}};
assign s2c0_desc_abort           = 1'b0;
assign s2c0_desc_rst_n           = 1'b1;

packet_check #(.REG_ADDR_WIDTH (REG_ADDR_WIDTH)) s2c0_packet_check (

    .rst_n                  (s2c0_user_rst_n            ),
    .clk                    (pcie_clk                   ),

    .check_control          (1'b1                       ),

    .reg_base_addr          (pc_s2c0[REG_ADDR_WIDTH-1:0]),

    .reg_wr_addr            (reg_wr_addr                ),
    .reg_wr_en              (reg_wr_en                  ),
    .reg_wr_be              (reg_wr_be                  ),
    .reg_wr_data            (reg_wr_data                ),
    .reg_rd_addr            (reg_rd_addr                ),
    .reg_rd_data            (s2c0_reg_rd_data           ),

    .lpbk_user_status       (s2c0_lpbk_user_status      ),
    .lpbk_sop               (s2c0_lpbk_sop              ),
    .lpbk_eop               (s2c0_lpbk_eop              ),
    .lpbk_data              (s2c0_lpbk_data             ),
    .lpbk_valid             (s2c0_lpbk_valid            ),
    .lpbk_src_rdy           (s2c0_lpbk_src_rdy          ),
    .lpbk_dst_rdy           (s2c0_lpbk_dst_rdy          ),

    .user_control           (s2c0_user_control          ),
    .sop                    (s2c0_sop                   ),
    .eop                    (s2c0_eop                   ),
    .err                    (s2c0_err                   ),
    .data                   (s2c0_data                  ),
    .valid                  (s2c0_valid                 ),
    .src_rdy                (s2c0_pkt_chk_src_rdy       ),
    .dst_rdy                (s2c0_pkt_chk_dst_rdy       ),
    .abort                  (s2c0_pkt_chk_abort         ),
    .abort_ack              (s2c0_pkt_chk_abort_ack     ),

    .error_sop              (s2c0_error_sop             ),
    .error_eop              (s2c0_error_eop             ),
    .error_cpl              (s2c0_error_cpl             ),
    .error_data             (s2c0_error_data            ),
    .error_valid            (s2c0_error_valid           ),
    .error_user_control     (s2c0_error_user_control    ),

    .pkt_sel_ram_pkt_n      (s2c0_pkt_sel_ram_pkt_n     )

);

  `ifdef INSTANTIATE_MPFE
assign s2c0_dst_rdy         = s2c0_pkt_chk_dst_rdy;
assign s2c0_abort_ack       = s2c0_pkt_chk_abort_ack;
assign s2c0_pkt_chk_src_rdy = s2c0_src_rdy  & ~s2c0_pkt_sel_ram_pkt_n;
assign s2c0_pkt_chk_abort   = s2c0_abort    & ~s2c0_pkt_sel_ram_pkt_n;


assign s2c0_sf_cmd_req           = s2c0_pkt_sel_ram_pkt_n ?  1'b0 : s2c0_cmd_req;
assign s2c0_sf_cmd_abort         = s2c0_pkt_sel_ram_pkt_n ?  1'b0 : s2c0_cmd_abort;
assign s2c0_sf_cmd_addr          = s2c0_cmd_addr;
assign s2c0_sf_cmd_bcount        = s2c0_cmd_bcount;
assign s2c0_sf_cmd_user_control  = s2c0_cmd_user_control;

assign s2c0_sf_data_req          = s2c0_pkt_sel_ram_pkt_n ?  1'b0 : s2c0_data_req;
assign s2c0_sf_data_addr         = s2c0_data_addr;
assign s2c0_sf_data_bcount       = s2c0_data_bcount;
assign s2c0_sf_data_en           = s2c0_pkt_sel_ram_pkt_n ?  1'b0 : s2c0_data_en;
assign s2c0_sf_data_error        = s2c0_pkt_sel_ram_pkt_n ?  1'b0 : s2c0_data_error;
assign s2c0_sf_data_remain       = s2c0_data_remain;
assign s2c0_sf_data_valid        = s2c0_data_valid;
assign s2c0_sf_data_first_req    = s2c0_data_first_req;
assign s2c0_sf_data_last_req     = s2c0_data_last_req;
assign s2c0_sf_data_first_desc   = s2c0_data_first_desc;
assign s2c0_sf_data_last_desc    = s2c0_data_last_desc;
assign s2c0_sf_data_first_chain  = s2c0_data_first_chain;
assign s2c0_sf_data_last_chain   = s2c0_data_last_chain;
assign s2c0_sf_data_data         = s2c0_data_data;
assign s2c0_sf_data_user_control = s2c0_data_user_control;


assign s2c0_ap_cmd_req           = s2c0_pkt_sel_ram_pkt_n ? s2c0_cmd_req    : 1'b0;
assign s2c0_ap_cmd_abort         = s2c0_pkt_sel_ram_pkt_n ? s2c0_cmd_abort  : 1'b0;
assign s2c0_ap_cmd_addr          = s2c0_cmd_addr;
assign s2c0_ap_cmd_bcount        = s2c0_cmd_bcount;
assign s2c0_ap_cmd_user_control  = s2c0_cmd_user_control;

assign s2c0_ap_data_req          = s2c0_pkt_sel_ram_pkt_n ? s2c0_data_req   : 1'b0;
assign s2c0_ap_data_addr         = s2c0_data_addr;
assign s2c0_ap_data_bcount       = s2c0_data_bcount;
assign s2c0_ap_data_en           = s2c0_pkt_sel_ram_pkt_n ? s2c0_data_en    : 1'b0;
assign s2c0_ap_data_error        = s2c0_pkt_sel_ram_pkt_n ? s2c0_data_error : 1'b0;
assign s2c0_ap_data_remain       = s2c0_data_remain;
assign s2c0_ap_data_valid        = s2c0_data_valid;
assign s2c0_ap_data_first_req    = s2c0_data_first_req;
assign s2c0_ap_data_last_req     = s2c0_data_last_req;
assign s2c0_ap_data_first_desc   = s2c0_data_first_desc;
assign s2c0_ap_data_last_desc    = s2c0_data_last_desc;
assign s2c0_ap_data_first_chain  = s2c0_data_first_chain;
assign s2c0_ap_data_last_chain   = s2c0_data_last_chain;
assign s2c0_ap_data_data         = s2c0_data_data;
assign s2c0_ap_data_user_control = s2c0_data_user_control;

assign s2c0_cmd_ready       = s2c0_pkt_sel_ram_pkt_n ? s2c0_ap_cmd_ready       : s2c0_sf_cmd_ready;
assign s2c0_cmd_abort_ack   = s2c0_pkt_sel_ram_pkt_n ? s2c0_ap_cmd_abort_ack   : s2c0_sf_cmd_abort_ack;
assign s2c0_cmd_stop        = s2c0_pkt_sel_ram_pkt_n ? s2c0_ap_cmd_stop        : s2c0_sf_cmd_stop;
assign s2c0_cmd_stop_bcount = s2c0_pkt_sel_ram_pkt_n ? s2c0_ap_cmd_stop_bcount : s2c0_sf_cmd_stop_bcount;
assign s2c0_data_ready      = s2c0_pkt_sel_ram_pkt_n ? s2c0_ap_data_ready      : s2c0_sf_data_ready;

s2c_adr_pkt #(

    .SDRAM_ADDR_WIDTH       (SDRAM_ADDR_WIDTH           ),
    .SDRAM_BSIZE_WIDTH      (BURST_SIZE_WIDTH           ),
    .BYTE_ADDRESSING        (BYTE_ADDRESSING            ),
    .DMA_DEST_ADDR_WIDTH    (DMA_DEST_ADDR_WIDTH        ),
    .FIFO_CADDR_WIDTH       (FIFO_CADDR_WIDTH           ),
    .FIFO_DADDR_WIDTH       (FIFO_DADDR_WIDTH           )

) s2c0_adr_pkt (

    .pcie_rst_n             (s2c0_cmd_rst_n             ),
    .pcie_clk               (pcie_clk                   ),

    .sdram_rst_n            (sdram_srst_n               ),
    .sdram_clk              (sdram_clk                  ),

    .testmode               (testmode                   ),

    .cmd_req                (s2c0_ap_cmd_req            ),
    .cmd_ready              (s2c0_ap_cmd_ready          ),
    .cmd_addr               (s2c0_ap_cmd_addr           ),
    .cmd_bcount             (s2c0_ap_cmd_bcount         ),
    .cmd_user_control       (s2c0_ap_cmd_user_control   ),
    .cmd_abort              (s2c0_ap_cmd_abort          ),
    .cmd_abort_ack          (s2c0_ap_cmd_abort_ack      ),
    .cmd_stop               (s2c0_ap_cmd_stop           ),
    .cmd_stop_bcount        (s2c0_ap_cmd_stop_bcount    ),

    .data_req               (s2c0_ap_data_req           ),
    .data_ready             (s2c0_ap_data_ready         ),
    .data_addr              (s2c0_ap_data_addr          ),
    .data_bcount            (s2c0_ap_data_bcount        ),
    .data_en                (s2c0_ap_data_en            ),
    .data_error             (s2c0_ap_data_error         ),
    .data_remain            (s2c0_ap_data_remain        ),
    .data_valid             (s2c0_ap_data_valid         ),
    .data_first_req         (s2c0_ap_data_first_req     ),
    .data_last_req          (s2c0_ap_data_last_req      ),
    .data_first_desc        (s2c0_ap_data_first_desc    ),
    .data_last_desc         (s2c0_ap_data_last_desc     ),
    .data_first_chain       (s2c0_ap_data_first_chain   ),
    .data_last_chain        (s2c0_ap_data_last_chain    ),
    .data_data              (s2c0_ap_data_data          ),
    .data_user_control      (s2c0_ap_data_user_control  ),

    .l_r_req                (s2c0_l_r_req               ),
    .l_w_req                (s2c0_l_w_req               ),
    .l_req_pri              (s2c0_l_req_pri             ),
    .l_addr                 (s2c0_l_addr                ),
    .l_b_size               (s2c0_l_b_size              ),
    .l_auto_pch             (s2c0_l_auto_pch            ),
    .l_busy                 (s2c0_l_busy                ),
    .l_r_valid              (s2c0_l_r_valid             ),
    .l_r_valid_last         (s2c0_l_r_valid_last        ),
    .l_d_req                (s2c0_l_d_req               ),
    .l_d_req_last           (s2c0_l_d_req_last          ),
    .l_datain               (s2c0_l_datain              ),
    .l_dm_in                (s2c0_l_dm_in               ),
    .l_dataout              (s2c0_l_dataout             ),
    .l_error                (s2c0_l_error_unused        )
);
  `else
assign s2c0_dst_rdy         = s2c0_pkt_chk_dst_rdy;
assign s2c0_abort_ack       = s2c0_pkt_chk_abort_ack;
assign s2c0_pkt_chk_src_rdy = s2c0_src_rdy;
assign s2c0_pkt_chk_abort   = s2c0_abort;

assign s2c0_sf_cmd_req           = s2c0_cmd_req;
assign s2c0_sf_cmd_abort         = s2c0_cmd_abort;
assign s2c0_sf_cmd_addr          = s2c0_cmd_addr;
assign s2c0_sf_cmd_bcount        = s2c0_cmd_bcount;
assign s2c0_sf_cmd_user_control  = s2c0_cmd_user_control;

assign s2c0_sf_data_req          = s2c0_data_req;
assign s2c0_sf_data_addr         = s2c0_data_addr;
assign s2c0_sf_data_bcount       = s2c0_data_bcount;
assign s2c0_sf_data_en           = s2c0_data_en;
assign s2c0_sf_data_error        = s2c0_data_error;
assign s2c0_sf_data_remain       = s2c0_data_remain;
assign s2c0_sf_data_valid        = s2c0_data_valid;
assign s2c0_sf_data_first_req    = s2c0_data_first_req;
assign s2c0_sf_data_last_req     = s2c0_data_last_req;
assign s2c0_sf_data_first_desc   = s2c0_data_first_desc;
assign s2c0_sf_data_last_desc    = s2c0_data_last_desc;
assign s2c0_sf_data_first_chain  = s2c0_data_first_chain;
assign s2c0_sf_data_last_chain   = s2c0_data_last_chain;
assign s2c0_sf_data_data         = s2c0_data_data;
assign s2c0_sf_data_user_control = s2c0_data_user_control;

assign s2c0_cmd_ready            = s2c0_sf_cmd_ready;
assign s2c0_cmd_abort_ack        = s2c0_sf_cmd_abort_ack;
assign s2c0_cmd_stop             = s2c0_sf_cmd_stop;
assign s2c0_cmd_stop_bcount      = s2c0_sf_cmd_stop_bcount;
assign s2c0_data_ready           = s2c0_sf_data_ready;
  `endif
s2c_pkt_streaming_fifo #(

    .FIFO_ADDR_WIDTH        (FIFO_DADDR_WIDTH           )

) s2c0_s2c_pkt_streaming_fifo (

    .rst_n                  (s2c0_cmd_rst_n             ),
    .clk                    (pcie_clk                   ),

    .cmd_req                (s2c0_sf_cmd_req            ),
    .cmd_ready              (s2c0_sf_cmd_ready          ),
    .cmd_addr               (s2c0_sf_cmd_addr           ),
    .cmd_bcount             (s2c0_sf_cmd_bcount         ),
    .cmd_user_control       (s2c0_sf_cmd_user_control   ),
    .cmd_abort              (s2c0_sf_cmd_abort          ),
    .cmd_abort_ack          (s2c0_sf_cmd_abort_ack      ),
    .cmd_stop               (s2c0_sf_cmd_stop           ),
    .cmd_stop_bcount        (s2c0_sf_cmd_stop_bcount    ),

    .data_req               (s2c0_sf_data_req           ),
    .data_ready             (s2c0_sf_data_ready         ),
    .data_addr              (s2c0_sf_data_addr          ),
    .data_bcount            (s2c0_sf_data_bcount        ),
    .data_en                (s2c0_sf_data_en            ),
    .data_error             (s2c0_sf_data_error         ),
    .data_remain            (s2c0_sf_data_remain        ),
    .data_valid             (s2c0_sf_data_valid         ),
    .data_first_req         (s2c0_sf_data_first_req     ),
    .data_last_req          (s2c0_sf_data_last_req      ),
    .data_first_desc        (s2c0_sf_data_first_desc    ),
    .data_last_desc         (s2c0_sf_data_last_desc     ),
    .data_first_chain       (s2c0_sf_data_first_chain   ),
    .data_last_chain        (s2c0_sf_data_last_chain    ),
    .data_data              (s2c0_sf_data_data          ),
    .data_user_control      (s2c0_sf_data_user_control  ),

    .user_control           (s2c0_user_control          ),
    .sop                    (s2c0_sop                   ),
    .eop                    (s2c0_eop                   ),
    .err                    (s2c0_err                   ),
    .data                   (s2c0_data                  ),
    .valid                  (s2c0_valid                 ),
    .src_rdy                (s2c0_src_rdy               ),
    .dst_rdy                (s2c0_dst_rdy               ),
    .abort                  (s2c0_abort                 ),
    .abort_ack              (s2c0_abort_ack             ),
    .user_rst_n             (s2c0_user_rst_n            )
);
assign s2c1_reg_rd_data = {CORE_DATA_WIDTH{1'b0}};
assign s2c2_reg_rd_data = {CORE_DATA_WIDTH{1'b0}};
assign s2c3_reg_rd_data = {CORE_DATA_WIDTH{1'b0}};



endmodule
