// -------------------------------------------------------------------------
//
// LOGIC CORE:          Expresso
// MODULE NAME:         dma_ref_design
//
//
// MODULE DESCRIPTION:
//
// This module is a top level reference design file containing:
//   * PCI Express Core
//   * DMA Back-End Core
//   * Reference Design
//
// This reference design is intended both for simulation to illustrate
//   use of the above cores as well as implemenation in hardware to
//   illustrate core capabilities.
//
// $Revision: 41252 $
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//              Copyright (c) 2013 by Northwest Logic, Inc.
//                       All rights reserved.
//
// Trade Secret of Northwest Logic, Inc.  Do not disclose.
//
// Use of this source code in any form or means is permitted only
// with a valid, written license agreement with Northwest Logic, Inc.
//
// Licensee shall keep all information contained herein confidential
// and shall protect same in whole or in part from disclosure and
// dissemination to all third parties.
//
//
//                        Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//
//                        Ph:  +1 503 533 5800
//                        Fax: +1 503 533 5900
//                      E-Mail: info@nwlogic.com
//                           www.nwlogic.com
//
// -------------------------------------------------------------------------



`timescale 1ps / 1ps

// -----------------------
// -- Module Definition --
// -----------------------

module dma_ref_design (

    perst_n,            // PCI Express slot PERST# reset signal

    pcie_clk_p,         // PCIe Reference Clock Input
    pcie_clk_n,         // PCIe Reference Clock Input


    tx_p,               // PCIe differential transmit output
    tx_n,               // PCIe differential transmit output

    rx_p,               // PCIe differential receive output
    rx_n,               // PCIe differential receive output

    led                 // Diagnostic LEDs


);



// ----------------
// -- Parameters --
// ----------------

localparam  NUM_LANES               = 8;

localparam  CORE_DATA_WIDTH         = 128;
localparam  CORE_BE_WIDTH           = 16;
localparam  CORE_REMAIN_WIDTH       = 4;
localparam  CORE_STS_WIDTH          = (CORE_BE_WIDTH * 3);
localparam  FDBK_BITS               = 8;   // Number of bits provided by PHY when reporting equalization quality

// 32 interrupt vectors are implemented by the DMA Back-End
//   and reference design logic; an MSI-X interrupt expansion
//   port is provided for users to support additional vectors;
//   If additional vectors are supported, then the MSI-X Table
//   and MSI-X PBA offset locations may need to be adjusted.
//   See msix_table_offset and msix_pba_offset below
localparam  INTERRUPT_VECTOR_BITS   = 5;

// BAR0 must be doubled in size when more than 256 MSI-X Vectors are used to make room for the correspondingly large MSI-X Table
localparam  REG_ADDR_WIDTH          = (INTERRUPT_VECTOR_BITS > 8) ? 13 : 12; // Register BAR is 128KBytes : 64KBytes

localparam  DESC_STATUS_WIDTH       = 160;  // Packet DMA Engine Descriptor Status port width

localparam  DESC_WIDTH              = 256;
`ifdef INSTANTIATE_MPFE

// Sets the size of SRAM DMA Destination Memory in words
//   *   1 MByte allows for doing larger DMA operations for behavioral testing;
//   * 128 KByte is a reasonable size for a design that is targeted to hardware;
// DMAs which exceed the amount of implemented memory will alias
 `ifdef SIMULATION
localparam  DMA_DEST_ADDR_WIDTH      = 16;                      // 2^16 16 byte words == 1 MByte
localparam  DMA_DEST_BADDR_WIDTH     = DMA_DEST_ADDR_WIDTH + 4; // Express above in bytes
 `else
localparam  DMA_DEST_ADDR_WIDTH      = 13;                      // 2^13 16 byte words == 128 KBytes
localparam  DMA_DEST_BADDR_WIDTH     = DMA_DEST_ADDR_WIDTH + 4; // Express above in bytes
 `endif

// Multi-port Front End Parameters
localparam  DSIZE                   = 128; // Local side data width
localparam  ADDR_WIDTH              = 33;  // Local address width
localparam  BURST_SIZE_WIDTH        = 8;   // Width of l_b_size parameter
`else
localparam  DMA_DEST_ADDR_WIDTH      = 0;  // FIFO implementation; no card address space
localparam  DMA_DEST_BADDR_WIDTH     = 28; // FIFO implementation; no card address space
`endif // ifdef INSTANTIATE_MPFE

localparam  LED_CTR_WIDTH           = 26;  // Sets period of LED flashing (~twice per Second @ 8nS clock rate, ~once per Second @ 16nS clock period)

// Configure size and indexes of led[LED_HI:LED_LO] port
localparam  LED_HI                  = 2;    // LED port high bit
localparam  LED_LO                  = 0;    // LED port low bit

// ----------------------
// -- Port Definitions --
// ----------------------

input                               perst_n;

input                               pcie_clk_p;
input                               pcie_clk_n;

output  [NUM_LANES-1:0]             tx_p;
output  [NUM_LANES-1:0]             tx_n;

input   [NUM_LANES-1:0]             rx_p;
input   [NUM_LANES-1:0]             rx_n;

output  [LED_HI:LED_LO]             led;


// ----------------
// -- Port Types --
// ----------------

wire                                perst_n;

wire                                pcie_clk_p;
wire                                pcie_clk_n;

wire    [NUM_LANES-1:0]             tx_p;
wire    [NUM_LANES-1:0]             tx_n;

wire    [NUM_LANES-1:0]             rx_p;
wire    [NUM_LANES-1:0]             rx_n;
wire    [LED_HI:LED_LO]             led;


// -------------------
// -- Local Signals --
// -------------------

// PCI Express PHY Instantiation
wire                                mgmt_dl_link_up_i;
wire                                link_up_i;

// Instantiate PCIe Complete Core
wire                                core_rst_n;
wire                                core_clk;
wire    [5:0]                       core_clk_period_in_ns;
wire                                testmode;

wire                                phy_tx_swing;
wire    [2:0]                       phy_tx_margin;
wire    [NUM_LANES-1:0]             phy_tx_deemph;

wire    [NUM_LANES-1:0]             phy_tx_detect_rx_loopback;
wire    [(NUM_LANES*2)-1:0]         phy_power_down;
wire    [NUM_LANES-1:0]             phy_rate;
wire    [NUM_LANES-1:0]             phy_phy_status;
wire    [NUM_LANES-1:0]             phy_rx_polarity;

wire    [CORE_DATA_WIDTH-1:0]       phy_tx_data;
wire    [CORE_BE_WIDTH-1:0]         phy_tx_is_k;
wire    [CORE_BE_WIDTH-1:0]         phy_tx_fneg;
wire    [NUM_LANES-1:0]             phy_tx_elec_idle;

wire    [CORE_DATA_WIDTH-1:0]       phy_rx_data;
wire    [CORE_BE_WIDTH-1:0]         phy_rx_is_k;
wire    [NUM_LANES-1:0]             phy_rx_data_valid;
wire    [NUM_LANES-1:0]             phy_rx_valid;
wire    [CORE_STS_WIDTH-1:0]        phy_rx_status;
wire    [NUM_LANES-1:0]             phy_rx_elec_idle;
`ifdef SIMULATION
reg     [NUM_LANES-1:0]             phy_rx_elec_idle_override;
`else
wire    [NUM_LANES-1:0]             phy_rx_elec_idle_override = {NUM_LANES{1'b0}};
`endif

// Instantiate Reference Design
wire                                user_clk;
wire                                user_rst_n;
wire                                mst_req;
wire                                mst_ready;
wire    [6:0]                       mst_type;
wire    [31:0]                      mst_data;
wire    [3:0]                       mst_be;
wire    [63:0]                      mst_addr;
wire    [7:0]                       mst_msgcode;
wire    [31:0]                      mst_rd_data;
wire    [2:0]                       mst_status;
wire                                mst_done;

wire                                targ_wr_req;
wire                                targ_wr_core_ready;
wire                                targ_wr_user_ready;
wire    [5:0]                       targ_wr_cs;
wire                                targ_wr_start;
wire    [31:0]                      targ_wr_addr;
wire    [12:0]                      targ_wr_count;
wire                                targ_wr_en;
wire    [CORE_DATA_WIDTH-1:0]       targ_wr_data;
wire    [CORE_BE_WIDTH-1:0]         targ_wr_be;

wire                                targ_rd_req;
wire                                targ_rd_core_ready;
wire                                targ_rd_user_ready;
wire    [5:0]                       targ_rd_cs;
wire                                targ_rd_start;
wire    [31:0]                      targ_rd_addr;
wire    [CORE_BE_WIDTH-1:0]         targ_rd_first_be;
wire    [CORE_BE_WIDTH-1:0]         targ_rd_last_be;
wire    [12:0]                      targ_rd_count;
wire                                targ_rd_en;
wire    [CORE_DATA_WIDTH-1:0]       targ_rd_data;

wire    [4:0]                       ref_design_wr_cs;
wire    [4:0]                       ref_design_rd_cs;

wire    [REG_ADDR_WIDTH-1:0]        reg_wr_addr;
wire                                reg_wr_en;
wire    [CORE_BE_WIDTH-1:0]         reg_wr_be;
wire    [CORE_DATA_WIDTH-1:0]       reg_wr_data;
wire    [REG_ADDR_WIDTH-1:0]        reg_rd_addr;
wire    [CORE_BE_WIDTH-1:0]         reg_rd_be;
wire    [CORE_DATA_WIDTH-1:0]       reg_rd_data;

wire                                c2s0_desc_req;
wire                                c2s0_desc_ready;
wire    [31:0]                      c2s0_desc_ptr;
wire    [DESC_WIDTH-1:0]            c2s0_desc_data;
wire                                c2s0_desc_abort;
wire                                c2s0_desc_abort_ack;
wire                                c2s0_desc_rst_n;

wire                                c2s0_desc_done;
wire    [7:0]                       c2s0_desc_done_channel;
wire    [DESC_STATUS_WIDTH-1:0]     c2s0_desc_done_status;

wire                                c2s0_cmd_rst_n;
wire                                c2s0_cmd_req;
wire                                c2s0_cmd_ready;
wire                                c2s0_cmd_first_chain;
wire                                c2s0_cmd_last_chain;
wire    [63:0]                      c2s0_cmd_addr;
wire    [31:0]                      c2s0_cmd_bcount;
wire    [63:0]                      c2s0_cmd_user_control;
wire                                c2s0_cmd_abort;
wire                                c2s0_cmd_abort_ack;

wire                                c2s0_data_req;
wire                                c2s0_data_ready;
wire    [CORE_REMAIN_WIDTH-1:0]     c2s0_data_req_remain;
wire                                c2s0_data_req_last_desc;
wire    [63:0]                      c2s0_data_addr;
wire    [9:0]                       c2s0_data_bcount;
wire                                c2s0_data_stop;
wire    [9:0]                       c2s0_data_stop_bcount;

wire                                c2s0_data_en;
wire    [CORE_REMAIN_WIDTH-1:0]     c2s0_data_remain;
wire    [CORE_REMAIN_WIDTH:0]       c2s0_data_valid;
wire                                c2s0_data_first_req;
wire                                c2s0_data_last_req;
wire                                c2s0_data_first_desc;
wire                                c2s0_data_last_desc;
wire                                c2s0_data_first_chain;
wire                                c2s0_data_last_chain;
wire                                c2s0_data_sop;
wire                                c2s0_data_eop;
wire    [CORE_DATA_WIDTH-1:0]       c2s0_data_data;
wire    [63:0]                      c2s0_data_user_status;
wire                                s2c0_desc_req;
wire                                s2c0_desc_ready;
wire    [31:0]                      s2c0_desc_ptr;
wire    [255:0]                     s2c0_desc_data;
wire                                s2c0_desc_abort;
wire                                s2c0_desc_abort_ack;
wire                                s2c0_desc_rst_n;

wire                                s2c0_desc_done;
wire    [7:0]                       s2c0_desc_done_channel;
wire    [159:0]                     s2c0_desc_done_status;

wire                                s2c0_cmd_rst_n;
wire                                s2c0_cmd_req;
wire                                s2c0_cmd_ready;
wire    [63:0]                      s2c0_cmd_addr;
wire    [9:0]                       s2c0_cmd_bcount;
wire    [63:0]                      s2c0_cmd_user_control;
wire                                s2c0_cmd_abort;
wire                                s2c0_cmd_abort_ack;
wire                                s2c0_cmd_stop;
wire    [9:0]                       s2c0_cmd_stop_bcount;

wire                                s2c0_data_req;
wire                                s2c0_data_ready;
wire    [63:0]                      s2c0_data_addr;
wire    [9:0]                       s2c0_data_bcount;
wire                                s2c0_data_en;
wire                                s2c0_data_error;
wire    [CORE_REMAIN_WIDTH-1:0]     s2c0_data_remain;
wire    [CORE_REMAIN_WIDTH:0]       s2c0_data_valid;
wire                                s2c0_data_first_req;
wire                                s2c0_data_last_req;
wire                                s2c0_data_first_desc;
wire                                s2c0_data_last_desc;
wire                                s2c0_data_first_chain;
wire                                s2c0_data_last_chain;
wire    [CORE_DATA_WIDTH-1:0]       s2c0_data_data;
wire    [63:0]                      s2c0_data_user_control;
wire    [31:0]                      mgmt_user_version;
wire    [31:0]                      mgmt_core_version;
wire    [31:0]                      mgmt_be_version;
wire    [15:0]                      mgmt_cfg_id;

wire                                mgmt_mst_en;
wire                                mgmt_msix_en;
wire                                mgmt_msi_en;
wire    [31:0]                      mgmt_interrupt;

 `ifdef SIMULATION
reg                                 user_interrupt;
 `else
wire                                user_interrupt;
 `endif
wire    [7:0]                       mgmt_core_ph;
wire    [11:0]                      mgmt_core_pd;
wire    [7:0]                       mgmt_core_nh;
wire    [11:0]                      mgmt_core_nd;
wire    [7:0]                       mgmt_core_ch;
wire    [11:0]                      mgmt_core_cd;

wire    [7:0]                       mgmt_chipset_ph;
wire    [11:0]                      mgmt_chipset_pd;
wire    [7:0]                       mgmt_chipset_nh;
wire    [11:0]                      mgmt_chipset_nd;
wire    [7:0]                       mgmt_chipset_ch;
wire    [11:0]                      mgmt_chipset_cd;
`ifdef INSTANTIATE_MPFE

// Internal SDRAM DMA Destination Memory
wire                                l_r_req_p0;
wire                                l_w_req_p0;
wire    [2:0]                       l_req_pri_p0;
wire    [ADDR_WIDTH-1:0]            l_addr_p0;
wire    [BURST_SIZE_WIDTH-1:0]      l_b_size_p0;
wire                                l_auto_pch_p0;
wire                                l_busy_p0;
wire                                l_r_valid_p0;
wire                                l_r_valid_last_p0;
wire                                l_d_req_p0;
wire                                l_d_req_last_p0;
wire    [DSIZE-1:0]                 l_datain_p0;
wire    [DSIZE/8-1:0]               l_dm_in_p0;

wire                                l_r_req_p1;
wire                                l_w_req_p1;
wire    [2:0]                       l_req_pri_p1;
wire    [ADDR_WIDTH-1:0]            l_addr_p1;
wire    [BURST_SIZE_WIDTH-1:0]      l_b_size_p1;
wire                                l_auto_pch_p1;
wire                                l_busy_p1;
wire                                l_r_valid_p1;
wire                                l_r_valid_last_p1;
wire                                l_d_req_p1;
wire                                l_d_req_last_p1;
wire    [DSIZE-1:0]                 l_datain_p1;
wire    [DSIZE/8-1:0]               l_dm_in_p1;

wire                                l_r_req_p2;
wire                                l_w_req_p2;
wire    [2:0]                       l_req_pri_p2;
wire    [ADDR_WIDTH-1:0]            l_addr_p2;
wire    [BURST_SIZE_WIDTH-1:0]      l_b_size_p2;
wire                                l_auto_pch_p2;
wire                                l_busy_p2;
wire                                l_r_valid_p2;
wire                                l_r_valid_last_p2;
wire                                l_d_req_p2;
wire                                l_d_req_last_p2;
wire    [DSIZE-1:0]                 l_datain_p2;
wire    [DSIZE/8-1:0]               l_dm_in_p2;

wire                                l_r_req_p3;
wire                                l_w_req_p3;
wire    [2:0]                       l_req_pri_p3;
wire    [ADDR_WIDTH-1:0]            l_addr_p3;
wire    [BURST_SIZE_WIDTH-1:0]      l_b_size_p3;
wire                                l_auto_pch_p3;
wire                                l_busy_p3;
wire                                l_r_valid_p3;
wire                                l_r_valid_last_p3;
wire                                l_d_req_p3;
wire                                l_d_req_last_p3;
wire    [DSIZE-1:0]                 l_datain_p3;
wire    [DSIZE/8-1:0]               l_dm_in_p3;

wire                                l_r_req_p4;
wire                                l_w_req_p4;
wire    [2:0]                       l_req_pri_p4;
wire    [ADDR_WIDTH-1:0]            l_addr_p4;
wire    [BURST_SIZE_WIDTH-1:0]      l_b_size_p4;
wire                                l_auto_pch_p4;
wire                                l_busy_p4;
wire                                l_r_valid_p4;
wire                                l_r_valid_last_p4;
wire                                l_d_req_p4;
wire                                l_d_req_last_p4;
wire    [DSIZE-1:0]                 l_datain_p4;
wire    [DSIZE/8-1:0]               l_dm_in_p4;

wire                                l_r_req_p5;
wire                                l_w_req_p5;
wire    [2:0]                       l_req_pri_p5;
wire    [ADDR_WIDTH-1:0]            l_addr_p5;
wire    [BURST_SIZE_WIDTH-1:0]      l_b_size_p5;
wire                                l_auto_pch_p5;
wire                                l_busy_p5;
wire                                l_r_valid_p5;
wire                                l_r_valid_last_p5;
wire                                l_d_req_p5;
wire                                l_d_req_last_p5;
wire    [DSIZE-1:0]                 l_datain_p5;
wire    [DSIZE/8-1:0]               l_dm_in_p5;

wire                                l_r_req_p6;
wire                                l_w_req_p6;
wire    [2:0]                       l_req_pri_p6;
wire    [ADDR_WIDTH-1:0]            l_addr_p6;
wire    [BURST_SIZE_WIDTH-1:0]      l_b_size_p6;
wire                                l_auto_pch_p6;
wire                                l_busy_p6;
wire                                l_r_valid_p6;
wire                                l_r_valid_last_p6;
wire                                l_d_req_p6;
wire                                l_d_req_last_p6;
wire    [DSIZE-1:0]                 l_datain_p6;
wire    [DSIZE/8-1:0]               l_dm_in_p6;

wire                                l_r_req_p7;
wire                                l_w_req_p7;
wire    [2:0]                       l_req_pri_p7;
wire    [ADDR_WIDTH-1:0]            l_addr_p7;
wire    [BURST_SIZE_WIDTH-1:0]      l_b_size_p7;
wire                                l_auto_pch_p7;
wire                                l_busy_p7;
wire                                l_r_valid_p7;
wire                                l_r_valid_last_p7;
wire                                l_d_req_p7;
wire                                l_d_req_last_p7;
wire    [DSIZE-1:0]                 l_datain_p7;
wire    [DSIZE/8-1:0]               l_dm_in_p7;

wire    [DSIZE-1:0]                 l_dataout;
`endif // ifdef INSTANTIATE_MPFE

// LEDs
reg     [LED_CTR_WIDTH-1:0]         led_ctr;
// ---------------
// -- Equations --
// ---------------

// -----------------------------
// PCI Express PHY Instantiation

assign link_up_i = mgmt_dl_link_up_i;



// ------------------------------
// Instantiate PCIe Complete Core

 `ifdef SIMULATION
initial user_interrupt = 1'b0;
 `else
assign user_interrupt = 1'b0;
 `endif

xil_pcie_wrapper #
    (

  `ifdef SIMULATION
    .PL_FAST_TRAIN                  ("TRUE"                     ),
  `else
    .PL_FAST_TRAIN                  ("FALSE"                    ),
  `endif

    .REF_CLK_FREQ                   (2                          ), // 0 - 100 MHz, 1 - 125 MHz, 2 - 250 MHz

    .VENDOR_ID                      (16'h19AA                   ), // 19AA is Northwest Logic's PCI SIG device ID
    .DEVICE_ID                      (16'hE004                   ),
    .SUBSYSTEM_VENDOR_ID            (16'h19AA                   ),
    .SUBSYSTEM_ID                   (16'hE004                   ),
    .REVISION_ID                    (8'h05                      ),

    .BAR0                           (32'hFFFF0000               ), // bar0; 64 KByte registers
    .BAR1                           (32'hFFFFE000               ), // bar1; 8K Byte internal SRAM
    .BAR2                           (32'hFFFFE000               ), // bar2; 8K Byte internal SRAM (same as bar1)
    .BAR3                           (32'h00000000               ),
    .BAR4                           (32'h00000000               ),
    .BAR5                           (32'h00000000               ),

    .CLASS_CODE                     (24'h11_80_00               ), // class_code == "Other data acquisition/signal processing controllers"

    .DEVICE_SN                      (64'h0                      )  // device serial number

)
xil_pcie_wrapper
    (
    .perst_n                        (perst_n                    ),

    .ref_clk_p                      (pcie_clk_p                 ),
    .ref_clk_n                      (pcie_clk_n                 ),

    .tx_p                           (tx_p                       ),
    .tx_n                           (tx_n                       ),

    .rx_p                           (rx_p                       ),
    .rx_n                           (rx_n                       ),

    .user_clk                       (core_clk                   ),
    .user_rst_n                     (mgmt_dl_link_up_i          ),
    .user_interrupt                 (user_interrupt             ),
    .mgmt_interrupt                 (mgmt_interrupt             ),

    .mgmt_pcie_version              (mgmt_core_version          ),
    .mgmt_be_version                (mgmt_be_version            ),
    .mgmt_user_version              (mgmt_user_version          ),
    .mgmt_mst_en                    (mgmt_mst_en                ),
    .mgmt_msi_en                    (mgmt_msi_en                ),
    .mgmt_msix_en                   (mgmt_msix_en               ),

    .mst_ready                      (mst_ready                  ),
    .mst_req                        (mst_req                    ),
    .mst_type                       (mst_type                   ),
    .mst_data                       (mst_data                   ),
    .mst_be                         (mst_be                     ),
    .mst_addr                       (mst_addr                   ),
    .mst_msgcode                    (mst_msgcode                ),
    .mst_rd_data                    (mst_rd_data                ),
    .mst_status                     (mst_status                 ),
    .mst_done                       (mst_done                   ),


    .c2s0_disable_desc_updates      (1'b0                       ),
    .c2s0_desc_req                  (c2s0_desc_req              ),
    .c2s0_desc_ready                (c2s0_desc_ready            ),
    .c2s0_desc_ptr                  (c2s0_desc_ptr              ),
    .c2s0_desc_data                 (c2s0_desc_data             ),
    .c2s0_desc_abort                (c2s0_desc_abort            ),
    .c2s0_desc_abort_ack            (c2s0_desc_abort_ack        ),
    .c2s0_desc_rst_n                (c2s0_desc_rst_n            ),

    .c2s0_desc_done                 (c2s0_desc_done             ),
    .c2s0_desc_done_channel         (c2s0_desc_done_channel     ),
    .c2s0_desc_done_status          (c2s0_desc_done_status      ),

    .c2s0_cmd_rst_n                 (c2s0_cmd_rst_n             ),
    .c2s0_cmd_req                   (c2s0_cmd_req               ),
    .c2s0_cmd_ready                 (c2s0_cmd_ready             ),
    .c2s0_cmd_first_chain           (c2s0_cmd_first_chain       ),
    .c2s0_cmd_last_chain            (c2s0_cmd_last_chain        ),
    .c2s0_cmd_addr                  (c2s0_cmd_addr              ),
    .c2s0_cmd_bcount                (c2s0_cmd_bcount            ),
    .c2s0_cmd_user_control          (c2s0_cmd_user_control      ),
    .c2s0_cmd_abort                 (c2s0_cmd_abort             ),
    .c2s0_cmd_abort_ack             (c2s0_cmd_abort_ack         ),

    .c2s0_data_req                  (c2s0_data_req              ),
    .c2s0_data_ready                (c2s0_data_ready            ),
    .c2s0_data_req_remain           (c2s0_data_req_remain       ),
    .c2s0_data_req_last_desc        (c2s0_data_req_last_desc    ),
    .c2s0_data_addr                 (c2s0_data_addr             ),
    .c2s0_data_bcount               (c2s0_data_bcount           ),
    .c2s0_data_stop                 (c2s0_data_stop             ),
    .c2s0_data_stop_bcount          (c2s0_data_stop_bcount      ),

    .c2s0_data_en                   (c2s0_data_en               ),
    .c2s0_data_remain               (c2s0_data_remain           ),
    .c2s0_data_valid                (c2s0_data_valid            ),
    .c2s0_data_first_req            (c2s0_data_first_req        ),
    .c2s0_data_last_req             (c2s0_data_last_req         ),
    .c2s0_data_first_desc           (c2s0_data_first_desc       ),
    .c2s0_data_last_desc            (c2s0_data_last_desc        ),
    .c2s0_data_first_chain          (c2s0_data_first_chain      ),
    .c2s0_data_last_chain           (c2s0_data_last_chain       ),
    .c2s0_data_sop                  (c2s0_data_sop              ),
    .c2s0_data_eop                  (c2s0_data_eop              ),
    .c2s0_data_data                 (c2s0_data_data             ),
    .c2s0_data_user_status          (c2s0_data_user_status      ),
    .s2c0_disable_desc_updates      (1'b0                       ),
    .s2c0_desc_req                  (s2c0_desc_req              ),
    .s2c0_desc_ready                (s2c0_desc_ready            ),
    .s2c0_desc_ptr                  (s2c0_desc_ptr              ),
    .s2c0_desc_data                 (s2c0_desc_data             ),
    .s2c0_desc_abort                (s2c0_desc_abort            ),
    .s2c0_desc_abort_ack            (s2c0_desc_abort_ack        ),
    .s2c0_desc_rst_n                (s2c0_desc_rst_n            ),

    .s2c0_desc_done                 (s2c0_desc_done             ),
    .s2c0_desc_done_channel         (s2c0_desc_done_channel     ),
    .s2c0_desc_done_status          (s2c0_desc_done_status      ),

    .s2c0_cmd_rst_n                 (s2c0_cmd_rst_n             ),
    .s2c0_cmd_req                   (s2c0_cmd_req               ),
    .s2c0_cmd_ready                 (s2c0_cmd_ready             ),
    .s2c0_cmd_addr                  (s2c0_cmd_addr              ),
    .s2c0_cmd_bcount                (s2c0_cmd_bcount            ),
    .s2c0_cmd_user_control          (s2c0_cmd_user_control      ),
    .s2c0_cmd_abort                 (s2c0_cmd_abort             ),
    .s2c0_cmd_abort_ack             (s2c0_cmd_abort_ack         ),
    .s2c0_cmd_stop                  (s2c0_cmd_stop              ),
    .s2c0_cmd_stop_bcount           (s2c0_cmd_stop_bcount       ),

    .s2c0_data_req                  (s2c0_data_req              ),
    .s2c0_data_ready                (s2c0_data_ready            ),
    .s2c0_data_addr                 (s2c0_data_addr             ),
    .s2c0_data_bcount               (s2c0_data_bcount           ),
    .s2c0_data_en                   (s2c0_data_en               ),
    .s2c0_data_error                (s2c0_data_error            ),
    .s2c0_data_remain               (s2c0_data_remain           ),
    .s2c0_data_valid                (s2c0_data_valid            ),
    .s2c0_data_first_req            (s2c0_data_first_req        ),
    .s2c0_data_last_req             (s2c0_data_last_req         ),
    .s2c0_data_first_desc           (s2c0_data_first_desc       ),
    .s2c0_data_last_desc            (s2c0_data_last_desc        ),
    .s2c0_data_first_chain          (s2c0_data_first_chain      ),
    .s2c0_data_last_chain           (s2c0_data_last_chain       ),
    .s2c0_data_data                 (s2c0_data_data             ),
    .s2c0_data_user_control         (s2c0_data_user_control     ),
    .targ_wr_req                    (targ_wr_req                ),
    .targ_wr_core_ready             (targ_wr_core_ready         ),
    .targ_wr_user_ready             (targ_wr_user_ready         ),
    .targ_wr_cs                     (targ_wr_cs                 ),
    .targ_wr_start                  (targ_wr_start              ),
    .targ_wr_addr                   (targ_wr_addr               ),
    .targ_wr_count                  (targ_wr_count              ),
    .targ_wr_en                     (targ_wr_en                 ),
    .targ_wr_data                   (targ_wr_data               ),
    .targ_wr_be                     (targ_wr_be                 ),

    .targ_rd_req                    (targ_rd_req                ),
    .targ_rd_core_ready             (targ_rd_core_ready         ),
    .targ_rd_user_ready             (targ_rd_user_ready         ),
    .targ_rd_cs                     (targ_rd_cs                 ),
    .targ_rd_start                  (targ_rd_start              ),
    .targ_rd_addr                   (targ_rd_addr               ),
    .targ_rd_first_be               (targ_rd_first_be           ),
    .targ_rd_last_be                (targ_rd_last_be            ),
    .targ_rd_count                  (targ_rd_count              ),
    .targ_rd_en                     (targ_rd_en                 ),
    .targ_rd_data                   (targ_rd_data               ),

    .reg_wr_addr                    (reg_wr_addr                ),
    .reg_wr_en                      (reg_wr_en                  ),
    .reg_wr_be                      (reg_wr_be                  ),
    .reg_wr_data                    (reg_wr_data                ),
    .reg_rd_addr                    (reg_rd_addr                ),
    .reg_rd_be                      (reg_rd_be                  ),
    .reg_rd_data                    (reg_rd_data                )
);
 `ifndef SIMULATION
assign mst_req     = 1'b0;
assign mst_type    = 7'h0;
assign mst_data    = 32'h0;
assign mst_be      = 4'h0;
assign mst_addr    = 64'h0;
assign mst_msgcode = 8'h0;
 `endif

// ----------------------------
// Instantiate Reference Design

// Reference design is synchronous to core_clk
assign user_clk   = core_clk;
assign user_rst_n = link_up_i;

assign testmode = 1'b0;



assign ref_design_wr_cs[  0] = targ_wr_cs[0] | targ_wr_cs[1];
assign ref_design_wr_cs[4:1] = targ_wr_cs[5:2];
assign ref_design_rd_cs[  0] = targ_rd_cs[0] | targ_rd_cs[1];
assign ref_design_rd_cs[4:1] = targ_rd_cs[5:2];

sdram_dma_ref_design_pkt #(

    .DMA_DEST_ADDR_WIDTH            (DMA_DEST_BADDR_WIDTH       )

) sdram_dma_ref_design_pkt (

    .pcie_srst_n                    (user_rst_n                 ),
    .pcie_clk                       (user_clk                   ),

    .sdram_srst_n                   (user_rst_n                 ),
    .sdram_clk                      (user_clk                   ),

    .testmode                       (testmode                   ),

    .targ_wr_req                    (targ_wr_req                ),
    .targ_wr_core_ready             (targ_wr_core_ready         ),
    .targ_wr_user_ready             (targ_wr_user_ready         ),
    .targ_wr_cs                     (ref_design_wr_cs           ),
    .targ_wr_start                  (targ_wr_start              ),
    .targ_wr_addr                   (targ_wr_addr               ),
    .targ_wr_count                  (targ_wr_count              ),
    .targ_wr_en                     (targ_wr_en                 ),
    .targ_wr_data                   (targ_wr_data               ),
    .targ_wr_be                     (targ_wr_be                 ),

    .targ_rd_req                    (targ_rd_req                ),
    .targ_rd_core_ready             (targ_rd_core_ready         ),
    .targ_rd_user_ready             (targ_rd_user_ready         ),
    .targ_rd_cs                     (ref_design_rd_cs           ),
    .targ_rd_start                  (targ_rd_start              ),
    .targ_rd_addr                   (targ_rd_addr               ),
    .targ_rd_count                  (targ_rd_count              ),
    .targ_rd_en                     (targ_rd_en                 ),
    .targ_rd_data                   (targ_rd_data               ),

    .reg_wr_addr                    (reg_wr_addr                ),
    .reg_wr_en                      (reg_wr_en                  ),
    .reg_wr_be                      (reg_wr_be                  ),
    .reg_wr_data                    (reg_wr_data                ),
    .reg_rd_addr                    (reg_rd_addr                ),
    .reg_rd_data                    (reg_rd_data                ),

    .c2s0_desc_req                  (c2s0_desc_req              ),
    .c2s0_desc_ready                (c2s0_desc_ready            ),
    .c2s0_desc_ptr                  (c2s0_desc_ptr              ),
    .c2s0_desc_data                 (c2s0_desc_data             ),
    .c2s0_desc_abort                (c2s0_desc_abort            ),
    .c2s0_desc_abort_ack            (c2s0_desc_abort_ack        ),
    .c2s0_desc_rst_n                (c2s0_desc_rst_n            ),

    .c2s0_desc_done                 (c2s0_desc_done             ),
    .c2s0_desc_done_channel         (c2s0_desc_done_channel     ),
    .c2s0_desc_done_status          (c2s0_desc_done_status      ),

    .c2s0_cmd_rst_n                 (c2s0_cmd_rst_n             ),
    .c2s0_cmd_req                   (c2s0_cmd_req               ),
    .c2s0_cmd_ready                 (c2s0_cmd_ready             ),
    .c2s0_cmd_first_chain           (c2s0_cmd_first_chain       ),
    .c2s0_cmd_last_chain            (c2s0_cmd_last_chain        ),
    .c2s0_cmd_addr                  (c2s0_cmd_addr              ),
    .c2s0_cmd_bcount                (c2s0_cmd_bcount            ),
    .c2s0_cmd_user_control          (c2s0_cmd_user_control      ),
    .c2s0_cmd_abort                 (c2s0_cmd_abort             ),
    .c2s0_cmd_abort_ack             (c2s0_cmd_abort_ack         ),

    .c2s0_data_req                  (c2s0_data_req              ),
    .c2s0_data_ready                (c2s0_data_ready            ),
    .c2s0_data_req_remain           (c2s0_data_req_remain       ),
    .c2s0_data_req_last_desc        (c2s0_data_req_last_desc    ),
    .c2s0_data_addr                 (c2s0_data_addr             ),
    .c2s0_data_bcount               (c2s0_data_bcount           ),
    .c2s0_data_stop                 (c2s0_data_stop             ),
    .c2s0_data_stop_bcount          (c2s0_data_stop_bcount      ),

    .c2s0_data_en                   (c2s0_data_en               ),
    .c2s0_data_remain               (c2s0_data_remain           ),
    .c2s0_data_valid                (c2s0_data_valid            ),
    .c2s0_data_first_req            (c2s0_data_first_req        ),
    .c2s0_data_last_req             (c2s0_data_last_req         ),
    .c2s0_data_first_desc           (c2s0_data_first_desc       ),
    .c2s0_data_last_desc            (c2s0_data_last_desc        ),
    .c2s0_data_first_chain          (c2s0_data_first_chain      ),
    .c2s0_data_last_chain           (c2s0_data_last_chain       ),
    .c2s0_data_sop                  (c2s0_data_sop              ),
    .c2s0_data_eop                  (c2s0_data_eop              ),
    .c2s0_data_data                 (c2s0_data_data             ),
    .c2s0_data_user_status          (c2s0_data_user_status      ),
    .s2c0_desc_req                  (s2c0_desc_req              ),
    .s2c0_desc_ready                (s2c0_desc_ready            ),
    .s2c0_desc_ptr                  (s2c0_desc_ptr              ),
    .s2c0_desc_data                 (s2c0_desc_data             ),
    .s2c0_desc_abort                (s2c0_desc_abort            ),
    .s2c0_desc_abort_ack            (s2c0_desc_abort_ack        ),
    .s2c0_desc_rst_n                (s2c0_desc_rst_n            ),

    .s2c0_desc_done                 (s2c0_desc_done             ),
    .s2c0_desc_done_channel         (s2c0_desc_done_channel     ),
    .s2c0_desc_done_status          (s2c0_desc_done_status      ),

    .s2c0_cmd_rst_n                 (s2c0_cmd_rst_n             ),
    .s2c0_cmd_req                   (s2c0_cmd_req               ),
    .s2c0_cmd_ready                 (s2c0_cmd_ready             ),
    .s2c0_cmd_addr                  (s2c0_cmd_addr              ),
    .s2c0_cmd_bcount                (s2c0_cmd_bcount            ),
    .s2c0_cmd_user_control          (s2c0_cmd_user_control      ),
    .s2c0_cmd_abort                 (s2c0_cmd_abort             ),
    .s2c0_cmd_abort_ack             (s2c0_cmd_abort_ack         ),
    .s2c0_cmd_stop                  (s2c0_cmd_stop              ),
    .s2c0_cmd_stop_bcount           (s2c0_cmd_stop_bcount       ),

    .s2c0_data_req                  (s2c0_data_req              ),
    .s2c0_data_ready                (s2c0_data_ready            ),
    .s2c0_data_addr                 (s2c0_data_addr             ),
    .s2c0_data_bcount               (s2c0_data_bcount           ),
    .s2c0_data_en                   (s2c0_data_en               ),
    .s2c0_data_error                (s2c0_data_error            ),
    .s2c0_data_remain               (s2c0_data_remain           ),
    .s2c0_data_valid                (s2c0_data_valid            ),
    .s2c0_data_first_req            (s2c0_data_first_req        ),
    .s2c0_data_last_req             (s2c0_data_last_req         ),
    .s2c0_data_first_desc           (s2c0_data_first_desc       ),
    .s2c0_data_last_desc            (s2c0_data_last_desc        ),
    .s2c0_data_first_chain          (s2c0_data_first_chain      ),
    .s2c0_data_last_chain           (s2c0_data_last_chain       ),
    .s2c0_data_data                 (s2c0_data_data             ),
    .s2c0_data_user_control         (s2c0_data_user_control     ),
 `ifdef INSTANTIATE_MPFE
    .c2s0_l_r_req                   (l_r_req_p0                 ),
    .c2s0_l_w_req                   (l_w_req_p0                 ),
    .c2s0_l_req_pri                 (l_req_pri_p0               ),
    .c2s0_l_addr                    (l_addr_p0                  ),
    .c2s0_l_b_size                  (l_b_size_p0                ),
    .c2s0_l_auto_pch                (l_auto_pch_p0              ),
    .c2s0_l_busy                    (l_busy_p0                  ),
    .c2s0_l_r_valid                 (l_r_valid_p0               ),
    .c2s0_l_r_valid_last            (l_r_valid_last_p0          ),
    .c2s0_l_d_req                   (l_d_req_p0                 ),
    .c2s0_l_d_req_last              (l_d_req_last_p0            ),
    .c2s0_l_datain                  (l_datain_p0                ),
    .c2s0_l_dm_in                   (l_dm_in_p0                 ),
    .c2s0_l_dataout                 (l_dataout                  ),
 `endif
 `ifdef INSTANTIATE_MPFE
    .s2c0_l_r_req                   (l_r_req_p4                 ),
    .s2c0_l_w_req                   (l_w_req_p4                 ),
    .s2c0_l_req_pri                 (l_req_pri_p4               ),
    .s2c0_l_addr                    (l_addr_p4                  ),
    .s2c0_l_b_size                  (l_b_size_p4                ),
    .s2c0_l_auto_pch                (l_auto_pch_p4              ),
    .s2c0_l_busy                    (l_busy_p4                  ),
    .s2c0_l_r_valid                 (l_r_valid_p4               ),
    .s2c0_l_r_valid_last            (l_r_valid_last_p4          ),
    .s2c0_l_d_req                   (l_d_req_p4                 ),
    .s2c0_l_d_req_last              (l_d_req_last_p4            ),
    .s2c0_l_datain                  (l_datain_p4                ),
    .s2c0_l_dm_in                   (l_dm_in_p4                 ),
    .s2c0_l_dataout                 (l_dataout                  ),
 `endif
    .mgmt_core_ph                   (mgmt_core_ph               ),
    .mgmt_core_pd                   (mgmt_core_pd               ),
    .mgmt_core_nh                   (mgmt_core_nh               ),
    .mgmt_core_nd                   (mgmt_core_nd               ),
    .mgmt_core_ch                   (mgmt_core_ch               ),
    .mgmt_core_cd                   (mgmt_core_cd               ),

    .mgmt_chipset_ph                (mgmt_chipset_ph            ),
    .mgmt_chipset_pd                (mgmt_chipset_pd            ),
    .mgmt_chipset_nh                (mgmt_chipset_nh            ),
    .mgmt_chipset_nd                (mgmt_chipset_nd            ),
    .mgmt_chipset_ch                (mgmt_chipset_ch            ),
    .mgmt_chipset_cd                (mgmt_chipset_cd            ),

    .mgmt_user_version              (mgmt_user_version          ),
    .mgmt_core_version              (mgmt_core_version          ),
    .mgmt_be_version                (mgmt_be_version            ),
    .mgmt_msi_en                    (mgmt_msi_en                ),
    .mgmt_msix_en                   (mgmt_msix_en               ),
    .mgmt_interrupt                 (mgmt_interrupt             )

);
`ifdef INSTANTIATE_MPFE



// ----------------------------------------------------------------------------
// Instantiate SRAM DMA Target with Multi-Port Front End SDRAM Controller Ports

sram_mp #(

    .DMA_DEST_ADDR_WIDTH            (DMA_DEST_ADDR_WIDTH        ),
    .DSIZE                          (DSIZE                      ),
    .ADDR_WIDTH                     (ADDR_WIDTH                 )


) sram_mp (

    // clocks and reset
    .clk                            (user_clk                   ),
    .rst_n                          (user_rst_n                 ),

    //local port interfaces
    //port 0 interface
    .l_r_req_p0                     (l_r_req_p0                 ),
    .l_w_req_p0                     (l_w_req_p0                 ),
    .l_req_pri_p0                   (l_req_pri_p0               ),
    .l_addr_p0                      (l_addr_p0                  ),
    .l_b_size_p0                    (l_b_size_p0                ),
    .l_auto_pch_p0                  (l_auto_pch_p0              ),
    .l_busy_p0                      (l_busy_p0                  ),
    .l_r_valid_p0                   (l_r_valid_p0               ),
    .l_r_valid_last_p0              (l_r_valid_last_p0          ),
    .l_d_req_p0                     (l_d_req_p0                 ),
    .l_d_req_last_p0                (l_d_req_last_p0            ),
    .l_datain_p0                    (l_datain_p0                ),
    .l_dm_in_p0                     (l_dm_in_p0                 ),

    //port 1 interface
    .l_r_req_p1                     (l_r_req_p1                 ),
    .l_w_req_p1                     (l_w_req_p1                 ),
    .l_req_pri_p1                   (l_req_pri_p1               ),
    .l_addr_p1                      (l_addr_p1                  ),
    .l_b_size_p1                    (l_b_size_p1                ),
    .l_auto_pch_p1                  (l_auto_pch_p1              ),
    .l_busy_p1                      (l_busy_p1                  ),
    .l_r_valid_p1                   (l_r_valid_p1               ),
    .l_r_valid_last_p1              (l_r_valid_last_p1          ),
    .l_d_req_p1                     (l_d_req_p1                 ),
    .l_d_req_last_p1                (l_d_req_last_p1            ),
    .l_datain_p1                    (l_datain_p1                ),
    .l_dm_in_p1                     (l_dm_in_p1                 ),

    //port 2 interface
    .l_r_req_p2                     (l_r_req_p2                 ),
    .l_w_req_p2                     (l_w_req_p2                 ),
    .l_req_pri_p2                   (l_req_pri_p2               ),
    .l_addr_p2                      (l_addr_p2                  ),
    .l_b_size_p2                    (l_b_size_p2                ),
    .l_auto_pch_p2                  (l_auto_pch_p2              ),
    .l_busy_p2                      (l_busy_p2                  ),
    .l_r_valid_p2                   (l_r_valid_p2               ),
    .l_r_valid_last_p2              (l_r_valid_last_p2          ),
    .l_d_req_p2                     (l_d_req_p2                 ),
    .l_d_req_last_p2                (l_d_req_last_p2            ),
    .l_datain_p2                    (l_datain_p2                ),
    .l_dm_in_p2                     (l_dm_in_p2                 ),

    //port 3 interface
    .l_r_req_p3                     (l_r_req_p3                 ),
    .l_w_req_p3                     (l_w_req_p3                 ),
    .l_req_pri_p3                   (l_req_pri_p3               ),
    .l_addr_p3                      (l_addr_p3                  ),
    .l_b_size_p3                    (l_b_size_p3                ),
    .l_auto_pch_p3                  (l_auto_pch_p3              ),
    .l_busy_p3                      (l_busy_p3                  ),
    .l_r_valid_p3                   (l_r_valid_p3               ),
    .l_r_valid_last_p3              (l_r_valid_last_p3          ),
    .l_d_req_p3                     (l_d_req_p3                 ),
    .l_d_req_last_p3                (l_d_req_last_p3            ),
    .l_datain_p3                    (l_datain_p3                ),
    .l_dm_in_p3                     (l_dm_in_p3                 ),

    //port 4 interface
    .l_r_req_p4                     (l_r_req_p4                 ),
    .l_w_req_p4                     (l_w_req_p4                 ),
    .l_req_pri_p4                   (l_req_pri_p4               ),
    .l_addr_p4                      (l_addr_p4                  ),
    .l_b_size_p4                    (l_b_size_p4                ),
    .l_auto_pch_p4                  (l_auto_pch_p4              ),
    .l_busy_p4                      (l_busy_p4                  ),
    .l_r_valid_p4                   (l_r_valid_p4               ),
    .l_r_valid_last_p4              (l_r_valid_last_p4          ),
    .l_d_req_p4                     (l_d_req_p4                 ),
    .l_d_req_last_p4                (l_d_req_last_p4            ),
    .l_datain_p4                    (l_datain_p4                ),
    .l_dm_in_p4                     (l_dm_in_p4                 ),

    //port 5 interface
    .l_r_req_p5                     (l_r_req_p5                 ),
    .l_w_req_p5                     (l_w_req_p5                 ),
    .l_req_pri_p5                   (l_req_pri_p5               ),
    .l_addr_p5                      (l_addr_p5                  ),
    .l_b_size_p5                    (l_b_size_p5                ),
    .l_auto_pch_p5                  (l_auto_pch_p5              ),
    .l_busy_p5                      (l_busy_p5                  ),
    .l_r_valid_p5                   (l_r_valid_p5               ),
    .l_r_valid_last_p5              (l_r_valid_last_p5          ),
    .l_d_req_p5                     (l_d_req_p5                 ),
    .l_d_req_last_p5                (l_d_req_last_p5            ),
    .l_datain_p5                    (l_datain_p5                ),
    .l_dm_in_p5                     (l_dm_in_p5                 ),

    //port 6 interface
    .l_r_req_p6                     (l_r_req_p6                 ),
    .l_w_req_p6                     (l_w_req_p6                 ),
    .l_req_pri_p6                   (l_req_pri_p6               ),
    .l_addr_p6                      (l_addr_p6                  ),
    .l_b_size_p6                    (l_b_size_p6                ),
    .l_auto_pch_p6                  (l_auto_pch_p6              ),
    .l_busy_p6                      (l_busy_p6                  ),
    .l_r_valid_p6                   (l_r_valid_p6               ),
    .l_r_valid_last_p6              (l_r_valid_last_p6          ),
    .l_d_req_p6                     (l_d_req_p6                 ),
    .l_d_req_last_p6                (l_d_req_last_p6            ),
    .l_datain_p6                    (l_datain_p6                ),
    .l_dm_in_p6                     (l_dm_in_p6                 ),

    //port 7 interface
    .l_r_req_p7                     (l_r_req_p7                 ),
    .l_w_req_p7                     (l_w_req_p7                 ),
    .l_req_pri_p7                   (l_req_pri_p7               ),
    .l_addr_p7                      (l_addr_p7                  ),
    .l_b_size_p7                    (l_b_size_p7                ),
    .l_auto_pch_p7                  (l_auto_pch_p7              ),
    .l_busy_p7                      (l_busy_p7                  ),
    .l_r_valid_p7                   (l_r_valid_p7               ),
    .l_r_valid_last_p7              (l_r_valid_last_p7          ),
    .l_d_req_p7                     (l_d_req_p7                 ),
    .l_d_req_last_p7                (l_d_req_last_p7            ),
    .l_datain_p7                    (l_datain_p7                ),
    .l_dm_in_p7                     (l_dm_in_p7                 ),

    .l_dataout                      (l_dataout                  )

);


// Port 1 is unused
assign l_r_req_p1    = 1'b0;
assign l_w_req_p1    = 1'b0;
assign l_req_pri_p1  = 3'h0;
assign l_addr_p1     = {ADDR_WIDTH{1'b0}};
assign l_b_size_p1   = {BURST_SIZE_WIDTH{1'b0}};
assign l_auto_pch_p1 = 1'b0;
assign l_datain_p1   = {DSIZE{1'b0}};
assign l_dm_in_p1    = {(DSIZE/8){1'b0}};

// Port 2 is unused
assign l_r_req_p2    = 1'b0;
assign l_w_req_p2    = 1'b0;
assign l_req_pri_p2  = 3'h0;
assign l_addr_p2     = {ADDR_WIDTH{1'b0}};
assign l_b_size_p2   = {BURST_SIZE_WIDTH{1'b0}};
assign l_auto_pch_p2 = 1'b0;
assign l_datain_p2   = {DSIZE{1'b0}};
assign l_dm_in_p2    = {(DSIZE/8){1'b0}};

// Port 3 is unused
assign l_r_req_p3    = 1'b0;
assign l_w_req_p3    = 1'b0;
assign l_req_pri_p3  = 3'h0;
assign l_addr_p3     = {ADDR_WIDTH{1'b0}};
assign l_b_size_p3   = {BURST_SIZE_WIDTH{1'b0}};
assign l_auto_pch_p3 = 1'b0;
assign l_datain_p3   = {DSIZE{1'b0}};
assign l_dm_in_p3    = {(DSIZE/8){1'b0}};


// Port 5 is unused
assign l_r_req_p5    = 1'b0;
assign l_w_req_p5    = 1'b0;
assign l_req_pri_p5  = 3'h0;
assign l_addr_p5     = {ADDR_WIDTH{1'b0}};
assign l_b_size_p5   = {BURST_SIZE_WIDTH{1'b0}};
assign l_auto_pch_p5 = 1'b0;
assign l_datain_p5   = {DSIZE{1'b0}};
assign l_dm_in_p5    = {(DSIZE/8){1'b0}};

// Port 6 is unused
assign l_r_req_p6    = 1'b0;
assign l_w_req_p6    = 1'b0;
assign l_req_pri_p6  = 3'h0;
assign l_addr_p6     = {ADDR_WIDTH{1'b0}};
assign l_b_size_p6   = {BURST_SIZE_WIDTH{1'b0}};
assign l_auto_pch_p6 = 1'b0;
assign l_datain_p6   = {DSIZE{1'b0}};
assign l_dm_in_p6    = {(DSIZE/8){1'b0}};

// Port 7 is unused
assign l_r_req_p7    = 1'b0;
assign l_w_req_p7    = 1'b0;
assign l_req_pri_p7  = 3'h0;
assign l_addr_p7     = {ADDR_WIDTH{1'b0}};
assign l_b_size_p7   = {BURST_SIZE_WIDTH{1'b0}};
assign l_auto_pch_p7 = 1'b0;
assign l_datain_p7   = {DSIZE{1'b0}};
assign l_dm_in_p7    = {(DSIZE/8){1'b0}};
`endif // ifdef INSTANTIATE_MPFE



// ----
// LEDs

// Heart beat LED; flashes when primary PCIe core clock is present;
//   don't want to use reset so we can see when clock is present even
//   when reset is not working properly
always @(posedge user_clk)
begin
    led_ctr <= led_ctr + {{(LED_CTR_WIDTH-1){1'b0}}, 1'b1};
end
`ifdef SIMULATION

// Initialize for simulation
initial
begin
    led_ctr = {LED_CTR_WIDTH{1'b0}};
end
`endif

assign led[LED_LO  ] = link_up_i;                 // dl_link_up
assign led[LED_LO+1] = led_ctr[LED_CTR_WIDTH-1];  // Heart beat LED indicating primary PCIe Core clock is present
assign led[LED_LO+2] = mgmt_mst_en;               // Data Link Layer Up
endmodule
