#!/usr/bin/perl
# -------------------------------------------------------------------------
#  PROJECT: FPGA Build Script for ISE
#  COMPANY: Northwest Logic, Inc.
#
# ------------------------- CONFIDENTIAL ----------------------------------
#                 Copyright 2012 by Northwest Logic, Inc.
#
#  All rights reserved.  No part of this source code may be reproduced or
#  transmitted in any form or by any means, electronic or mechanical,
#  including photocopying, recording, or any information storage and
#  retrieval system, without permission in writing from Northest Logic, Inc.
#
#  Further, no use of this source code is permitted in any form or means
#  without a valid, written license agreement with Northwest Logic, Inc.
#
#                         Northwest Logic, Inc.
#                  1100 NW Compton Drive, Suite 100
#                      Beaverton, OR 97006, USA
#
#                       Ph.  +1 503 533 5800
#                       Fax. +1 503 533 5900
#                          www.nwlogic.com
# -------------------------------------------------------------------------

# o Create .xst and .prj on the fly
#    - Use projname.xcf if present, otherwise create default .xcf
#    - Use projname.xst if present, otherwise create default .xst
#    - Put created .prj files in results folder
#    - add -xst_opts default="..", -xst_opts projname="..." options (hash-type options)
#    - create .prj file for each project from a projname.f manifest
#    - allow .ncd in manifest for dependancy checking
#    - allow .f in manifest for alternate filelists
# o Add timing report parser to report all failing paths
#    - report longest path if none are failing
# o Add support for running X seeds of P&R, building a results.seednumber dir for each
#    - Move top.pcf and top.ngd (post-map) from results into results.seednumber before starting
#      each seed.

use Getopt::Long;
use File::Copy;
use File::Basename;
use File::Spec;
use warnings;
use strict;

# Parse Inputs

my $top       = "";
my $ucf       = "";

my $chipscope = "";
my $ngd_opts  = "";
my $map_opts  = "";
my $par_opts  = "";
my $trc_opts  = "";
my $sdf_opts  = "";
my $bgn_opts  = "";
my $prom      = "";
my $guide     = "";
my $optFile   = "";
my $xpl_opts  = "";
my $mppr      = "";
my $seed      = "";

my $do_syn    = 1;
my $do_trans  = 1;
my $do_map    = 1;
my $do_par    = 1;
my $do_trace  = 1;
my $do_xpl    = 0;
my $do_sdf    = 0;
my $do_bgn    = 1;
my $help      = 0;
my $recompile = 0;
my $no_pause  = 0;
my $smartx    = 0;
my $quiet     = 0;
my $synplify  = 0;
my @prjs;

# specifies the number of synthesis jobs
# allowed to run at the same time
my $NUM_CPUS  = 4;
my $debug = 0;

# allow specifing an options file with "-f or +f or --file <file>"
# other options on the line will be ignored and remain in @ARGV
Getopt::Long::Configure(qw(pass_through));
GetOptions('file=s' => \$optFile);
parseOptFile($optFile);

# now parse all of the options
Getopt::Long::Configure(qw(nopass_through));
my $goodopts = GetOptions('help|?'      => \$help,
                          'quiet'       => \$quiet,
                          'debug'       => \$debug,
                          'top=s'       => \$top,
                          'ucf=s'       => \$ucf,
                          'prj=s'       => \@prjs,
                          'synplify'    => \$synplify,
                          'chipscope=s' => \$chipscope,
                          'guide=s'     => \$guide,
                          'prom=s'      => \$prom,
                          'ngd_opts=s'  => \$ngd_opts,
                          'map_opts=s'  => \$map_opts,
                          'par_opts=s'  => \$par_opts,
                          'xpl_opts=s'  => \$xpl_opts,
                          'trc_opts=s'  => \$trc_opts,
                          'sdf_opts=s'  => \$sdf_opts,
                          'bgn_opts=s'  => \$bgn_opts,
                          'mppr=s'      => \$mppr,
                          'seed=s'      => \$seed,
                          'recompile'   => \$recompile,
                          'no_pause'    => \$no_pause,
                          'smartx'      => \$smartx,
                          'numcpus=s'   => \$NUM_CPUS,
                          'sdf'         => sub {$do_sdf=1},
                          'syn_only'    => sub {($do_syn,$do_trans,$do_map,$do_par,$do_trace,$do_bgn)=(1,0,0,0,0,0)},
                          'translate'   => sub {($do_syn,$do_trans,$do_map,$do_par,$do_trace,$do_bgn)=(0,1,1,1,1,1)},
                          'map'         => sub {($do_syn,$do_trans,$do_map,$do_par,$do_trace,$do_bgn)=(0,0,1,1,1,1)},
                          'par'         => sub {($do_syn,$do_trans,$do_map,$do_par,$do_trace,$do_bgn)=(0,0,0,1,1,1)},
                          'time'        => sub {($do_syn,$do_trans,$do_map,$do_par,$do_trace,$do_bgn)=(0,0,0,0,1,1)},
                          'bitgen'      => sub {($do_syn,$do_trans,$do_map,$do_par,$do_trace,$do_bgn)=(0,0,0,0,0,1)}
                         );
usage() if (not $goodopts or $help or not $top);

#
# Get the XILINX tools directory
# This little dance is needed due to strange behaviour of xilperl.exe
#
my $xilinx_tools = "";
if (not defined $ENV{XILINX}) {
  die "Cannot find XILINX environment variable.  Quitting\n";
}
my @xilinx_dirs = split(/;/,$ENV{XILINX});
my $xilinx_dir = $xilinx_dirs[-1];
if ($^O eq "linux") {
  $xilinx_tools = "$xilinx_dir/bin/lin64";
} else {
  $xilinx_tools = "$xilinx_dir/bin/nt64";
  if (not -d $xilinx_tools)	{
    $xilinx_tools = "$xilinx_dir/bin/nt";
  }
}
if (not -d $xilinx_tools) {
  die "Unable to find Xilinx tools directory $xilinx_tools.  Quitting\n";
}

my @synplicity_dirs;
my $synplicity_dir;
my $synplicity_tools = "";
if ($synplify) {
  if (not defined $ENV{SYNPLICITY}) {
    die "Cannot find SYNPLICITY environment variable.  Quitting\n";
  } else {
    @synplicity_dirs = split(/;/,$ENV{SYNPLICITY});
    $synplicity_dir = $synplicity_dirs[-1];
    if ($^O eq "linux") {
      $synplicity_tools = "$synplicity_dir";
    } else {
      $synplicity_tools = "$synplicity_dir";
    }
    if (not -d $synplicity_tools) {
      die "Unable to find synplify tools directory $synplicity_dir.  Quitting\n";
    }
  }
}


if ($^O ne "linux") {
  if ($smartx) {
    print "Warning: -smartx not supported unless a Linux machine is used\n";
  }
  $smartx = 0;
}



if ($mppr) {
  if ($^O eq "linux") {
    die "MPPR not available in Linux.  Quitting.\n";
  }

  $no_pause = 1;
  if (not $seed) {
    $seed = 1;
  }
  print "Running $mppr seed MPPR starting with seed $seed\n";
}


wintitle("FPGA Build Started");

# "ncd" is required when xst is used
# "edf" is required when synplify is used

# this perl 'glob' command syntax fills the array
# with files from the current directory
# matching the glob: "*.xst" or "*.prj"

my $syn_ext;
  if ($synplify) {
    $syn_ext = "edf";
    @prjs      = <*.prj>;
  } else {
    $syn_ext = "ngc";
    @prjs      = <*.xst>;
  }

my @projects;


# allow commas in project list on command line
@prjs = split(/,/,join(",",@prjs));

print "Build Summary\n";
print "=============\n";


if ($do_syn) {
print "Synthesis Projects\n";
print "-------------------------------------------------------\n";

  foreach my $prj (@prjs) {
    my $design = basename($prj,qw (.xst .prj));
    if (not $top) {
      $top = $design
    }
    if ($design eq $top) {
      print "Synthesizing (top)        : $design\n";
    } elsif ($recompile or not project_current($prj)) {
      push @projects,$design;
      print "Synthesizing (out of date): $design\n";
    } else {
      print "     SKIPPING (up to date): $design\n";
    }
  }
}

# find out if we can use threading to speed things up.
my $threadsOK;
if ($NUM_CPUS == 1) {
  # Don't bother trying for single-CPU systems
  $threadsOK = 0;
} else {
  $threadsOK = testThreads();
}

# install the control c handler so we can kill child threads if any
$SIG{INT} = \&got_controlC;


print "---------------------------------------------------------\n";
print "Using Top Level Constraints $ucf\n"       if ($ucf);
print "Using Chipscope File        $chipscope\n" if ($chipscope);
print "Using Special Prom Type     $prom\n"      if ($prom);
print "----------------------------------------------------------\n";
if ($smartx) {
  print "SmartXplore                 $xpl_opts\n"  if ($do_trans or $do_map or $do_par);
} else {
  print "Translate                   $ngd_opts\n"  if ($do_trans);
  print "Map                         $map_opts\n"  if ($do_map);
  print "Place & Route               $par_opts\n"  if ($do_par);
  print "Analyze Timing              $trc_opts\n"  if ($do_trace);
}
print "Create SDF                  $sdf_opts\n"  if ($do_sdf);
print "Build Bitstream             $bgn_opts\n"  if ($do_bgn);
&pause;
&date_and_time;

my $start_time = time;
my $end_synth_time;
my $end_trans_time;
my $end_map_time;
my $end_par_time;
my $elapsed_time;


mkdir "results" if (not -d "results");
chdir "results";


synthesize      ($top,\@projects)                                 if ($do_syn);
$end_synth_time = time;
if ($do_syn) {
  $elapsed_time = time_diff($start_time, $end_synth_time);
  print "Synthesis run time: $elapsed_time\n";
}
translate       ($top,$ucf,$ngd_opts,$syn_ext,$chipscope,$smartx) if ($do_trans or (($do_map or $do_par) and $smartx));
$end_trans_time = time;
if ($do_trans) {
  $elapsed_time = time_diff($end_synth_time, $end_trans_time);
  print "Translate run time: $elapsed_time\n";
}

if ($mppr) {
  # strip the seed number for the map options so it can be replaced with the MPPR value
  $map_opts =~ s/-t\s*\d+//;
  foreach my $n ($seed..$seed+$mppr-1) {
    print "Starting MPPR pass for seed $n\n";
    map_design      ($top,"$map_opts -t $n")      if ($do_map and not $smartx);
    place_and_route ($top,$par_opts)              if ($do_par and not $smartx);
    trace           ($top,$trc_opts)              if ($do_trace and not $smartx);
    create_sdf      ($top,$sdf_opts)              if ($do_sdf);
    bitgen          ($top,$bgn_opts,$prom)        if ($do_bgn);
    mkdir "../results_$n";
    system("xcopy *.* ..\\results_$n /Q /Y");
  }
} else {
  map_design      ($top,$map_opts)                if ($do_map and not $smartx);
  $end_map_time = time;
  $elapsed_time = time_diff($end_trans_time, $end_map_time);
  print "Map run time: $elapsed_time\n";
  place_and_route ($top,$par_opts)                if ($do_par and not $smartx);
  $end_par_time = time;
  $elapsed_time = time_diff($end_map_time, $end_par_time);
  print "PAR run time: $elapsed_time\n";
  trace           ($top,$trc_opts)                if ($do_trace and not $smartx);
  create_sdf      ($top,$sdf_opts)                if ($do_sdf);
  bitgen          ($top,$bgn_opts,$prom)          if ($do_bgn);
}


print "\nRun times:\n";
if ($do_syn) {
  $elapsed_time = time_diff($start_time, $end_synth_time);
  print "Synthesis: $elapsed_time\n";
}
if ($do_trans) {
  $elapsed_time = time_diff($end_synth_time, $end_trans_time);
  print "Translate: $elapsed_time\n";
}
if ($do_map) {
  $elapsed_time = time_diff($end_trans_time, $end_map_time);
  print "Map:       $elapsed_time\n";
}
if ($do_par) {
  $elapsed_time = time_diff($end_map_time, $end_par_time);
  print "PAR:       $elapsed_time\n";
}
  $elapsed_time = time_diff($start_time, time);
  print "TOTAL:     $elapsed_time\n";

# Exit gracefully if we've gotten through all steps without failing
&pass;

#####################################
#                                   #
# Subroutines Only Below this Point #
#                                   #
#####################################

sub synthesize {
  my $top = shift;
  my $projects_ref = shift;
  my @projects = @{$projects_ref};

  wintitle("Synthesizing");

  # make $top the last project to synthesize
  push @projects,$top;


  if ($synplify) {
  } else {
    # Set up directories referred to in .xst file
    mkdir "xst" if (not -d "xst");
    mkdir "xst/implement.tmp" if (not -d "xst/implement.tmp");
  }


  #
  # Run parallel synthesis jobs if threads are available
  #
  foreach my $design (@projects) {


    # Wait for previous children to exit before running the
    # last synthesis job (which is always the top level)
    if ($design eq $projects[-1]) {
      wait_all_children();
    } else {
      wait_cpu_available();
    }

    #delete the old log file if present
    if (-e "$design.srp") {unlink "$design.srp"}
    # Create a child process to run this synthesis job
    startSynth($design);
  }

  # Wait for all children to exit before leaving the subroutine
  wait_all_children();

}

sub translate {
  my $top = shift;
  my $ucf = shift;
  my $ngd_opts = shift;
  my $syn_ext = shift;
  my $chipscope = shift;
  my $smartx = shift;

  my $FH; # Filehandle

  # Use UCF if present
  my $ucf_opt = "-uc ../$ucf";

  if (not -e "../$ucf") {
    print "Warning: UCF ($ucf) not found for top level design.\n";
    $ucf_opt = "";
  }

  # Create Chipscope if Requested
  if ($chipscope) {
    # .edf or .ngc
    rename("$top.$syn_ext","${top}_orig.$syn_ext");
    print "Inserting Chipscope\n";
    my $command = "inserter -insert ../$chipscope -ngcbuild ${top}_orig.$syn_ext $top.$syn_ext > /dev/null";
    if (csystem($command)) {
      fail($command,"");
    }
  }

  if ($smartx) {
    wintitle("Running SmartXplorer");

    # make host list based on number of available CPUs
    my $hostlist_file = "smartx.hosts";
    my $netname = `uname -n`;
    chomp $netname;

    open($FH, ">$hostlist_file") or die "Can't open $hostlist_file for write.\n";
    foreach my $i (1..$NUM_CPUS) {
      print $FH "$netname\n";
    }
    close($FH);

    my $log = "smartxplorer.log";
    my $command = "${xilinx_tools}/smartxplorer -b -sd . $ucf_opt $xpl_opts -l smartx.hosts $top.$syn_ext";
    if (csystem($command)) {
      fail($command,$log);
    } else {
      # extract best result
      my $resultdir = "";
      open($FH, $log) or die "Can't open $log for read.\n";
      while (my $line = <$FH> ) {
        if ($line =~ /Run index +: +(\w+)/) {
          $resultdir = $1;
        }
      }
      close($FH);
      if ($resultdir ne "" and -d $resultdir) {
        copy("$resultdir/$top.ncd","$top.ncd");
        copy("$resultdir/$top.pcf","$top.pcf");
      } else {
        fail("Unable to find best result directory: \"$resultdir\" after running smartxplorer",$log);
      }
    }
  } else {
    wintitle("Translating");

    # Translate
    my $command = "${xilinx_tools}/ngdbuild -dd . -sd . $ucf_opt -intstyle xflow -quiet $ngd_opts $top.$syn_ext > /dev/null";
    if (csystem($command)) {
      fail($command,"$top.bld");
    }
  }
}

sub map_design {
  my $top = shift;
  my $map_opts = shift;

  wintitle("Mapping");

  my $command = "${xilinx_tools}/map -intstyle xflow -w $map_opts -o ${top}_mapped.ncd $top.ngd $top.pcf > /dev/null";
  if (csystem($command)) {
    fail($command,"${top}_mapped.mrp");
  }
}

sub place_and_route {
  my $top = shift;
  my $par_opts = shift;

  wintitle("Place and Route");

  my $command = "${xilinx_tools}/par $par_opts -w ${top}_mapped.ncd $top.ncd $top.pcf";
  if (csystem($command)) {
    fail($command,"$top.par");
  }
}

sub trace {
  my $top = shift;
  my $trc_opts = shift;

  wintitle("Timing Analysis");

  my $command = "${xilinx_tools}/trce $trc_opts -xml $top.twx -o $top.twr $top.ncd $top.pcf";
  if (csystem($command)) {
    fail($command,"$top.twr");
  }
}

sub create_sdf {
  my $top = shift;
  my $sdf_opts = shift;

  wintitle("Creating SDF");

  my $command = "${xilinx_tools}/netgen -sim -ofmt verilog -ne -w $sdf_opts -tm $top $top.ncd > /dev/null";
  if (csystem($command)) {
    fail($command,"");
  }
}

sub bitgen {
  my $top = shift;
  my $bgn_opts = shift;
  my $prom = shift;

  # Create BIT file for device programming
  # Use faster configuration clock {45 Mhz}
  # -m: Create Mask File for Verification

  my $version = date_and_time(1);

  wintitle("Running Bitgen");

  my $command = "${xilinx_tools}/bitgen -w $bgn_opts -g UserID:$version $top.ncd > /dev/null";
  if (csystem($command)) {
    fail($command,"$top.bgn");
  }

  if ($prom eq "ml555") {
    # This is the prom generation command when using the ML555 board
    # it requires 2 design revisions to be programmed into the prom
    #
    # How to Program the ML555 reference board PROM
    #
    # Programming mode should be set to "parallel" and Revision properties should be set up
    # selecting rev0 and rev1 to be selected and erased during prom programming
    # The first of the two proms should be selected in the JTAG chain when programming
    csystem("${xilinx_tools}/promgen -w -p mcs -c FF -o prom_parallel_2rev_$version -ver 0 $top.bit -ver 1 $top.bit -x xcf32p");
  } elsif ($prom eq "bpi") {
    # This is the prom generation command when using the HTG_PCIE_DDR3 board
    #   and other BPI PROM boards.  See Hitech Global HTG-PCIE-DDR3 User Guide
    #   for instructions on programming the resulting PROM file into the PROM
    csystem("${xilinx_tools}/promgen -w -p mcs -c FF -o prom_bpi_$version -u 00 $top.bit -x xcf128x -data_width 16");
  } elsif ($prom ne "none") {
    # Create prom file in mcs format loading up from address 0
    # All other standard options work for the proms on our ref boards
    csystem("${xilinx_tools}/promgen -w -p mcs -c FF -o prom_$version -u 00 $top.bit -x xcf32p");
  }
}

sub fail {
  my $command = shift;
  my $logfile = shift;

  wintitle("FPGA Build FAILED");

  print "\n$command\n";
  my $log = File::Spec->catfile("results",$logfile);
  print "Build FAILED - Review log $log for Errors\n";
  chdir "..";
  &date_and_time;
  &pause if ($^O ne "linux");
  &quit(1);
}

sub parse_xst_log {
  my $file = shift;
  open(LOG, "$file") or die "Can't open $file for parsing!";
  while (<LOG>) {
    if (/WARNING:|ERROR:/) {
      #  616 Invalid property XX: Did not attach to XX
      #  646 Signal XX is assigned but never used.
      #  647 Input XX is never used.
      #  916 Delay is ignored for synthesis
      #  413 Result of XX-bit expression is truncated to fit in XX-bit target. (common in NWL IP, usually OK)
      # 1127 Assignment to XX ignored, since the identifier is never used
      # 1290 Hierarchical block XX is unconnected in block XX
      # 1499 Empty module XX remains a black box.
      # 1710 FF/Latch XX (without init value) has a constant value of 0 in block XX.
      # 1780 Signal XX is never used or assigned.
      # 1895 Due to other FF/Latch trimming, FF/Latch XX (without init value) has a constant value of 0 in block XX.
      # 1898 Due to constant pushing, FF/Latch XX is unconnected in block XX
      # 1989 Unit XX: instances XX, XX of unit XX are equivalent, second instance is removed
      # 2211 Instantiating black box module XX
      # 2404 FFs/Latches XX (without init value) have a constant value of 0 in block XX.
      # 2677 Node XX of sequential type is unconnected in block XX
      if (not /\:(413|646|647|1780|2677|1710|1895|616|2211|2404|1898|1290|916|1989|1127|1499)\D/) {
        print "$file:$_";
      }
    }
  }
}

sub pass {

  wintitle("FPGA Build Completed");

  chdir "..";
  &date_and_time;
  &pause if ($^O ne "linux");
  &quit(0);
}

sub date_and_time {
  my $noprint = shift;
  # print date and time
  my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst)=localtime(time);
  if (not $noprint) {
    printf "\n\nCurrent Time: %4d-%02d-%02d %02d:%02d:%02d\n",$year+1900,$mon+1,$mday,$hour,$min,$sec;
  }
  return sprintf("%02d%02d%02d%02d",$mon+1,$mday,$year-100,$hour);
}

sub time_diff {
  my $start_time = shift;
  my $stop_time = shift;

  my $hour = 0;
  my $min = 0;
  my $sec = 0;

  my $diff_time = $stop_time - $start_time;

  if ($diff_time > 3600) {
    use integer;
    $hour = $diff_time / 3600;
    $diff_time = $diff_time - ($hour * 3600);
  }
  if ($diff_time > 60) {
    use integer;
    $min = $diff_time / 60;
    $diff_time = $diff_time - ($min * 60);
  }
  $sec = $diff_time;

  return sprintf("%02d:%02d:%02d",$hour,$min,$sec);
}

sub file_mtime {
  # returns time file was last modified in seconds since the epoch
  my $filename  = shift;
  my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,$atime,$mtime,$ctime,$blksize,$blocks) = stat($filename);
  return $mtime;
}

sub project_current {
  my $project = shift;
  my $basename = basename($project,qw (.xst .prj));

  # Compare date/time of lso file with all sources
  # for the current project.
  return(0) if ($synplify);

  if ((-e "results/${basename}.lso") and (-e "results/${basename}.ngc")) {
    my $up_to_date = 1;

    my $reftime = file_mtime("results/${basename}.lso");
    my $donetime = file_mtime("results/${basename}.ngc");

    my ($sF,$sT) = getTimes($project);
    my @sourceFiles = @$sF;
    my @sourceTimes = @$sT;
    print "project $project has ".@sourceTimes." source files\n" if $debug;
    for my $i (0..scalar(@sourceTimes)-1) {
      if ($sourceTimes[$i] > $reftime) {
        print "project $project: Source file out of date: $sourceFiles[$i]\n" if $debug;
        $up_to_date = 0;
        last;
      } else {
        print "project $project: Source file up to date: $sourceFiles[$i]\n" if $debug;
      }
    }
    if ($donetime <= $reftime) {
      print "project $project: Output file not up to date: results/$basename.ngc\n" if $debug;
      $up_to_date = 0;
    }
    return $up_to_date;
  } else {
    return 0;
  }
}

sub getTimes {
  my $project = shift;
  my($filename, $directories, $suffix) = fileparse($project,qw (.xst .prj));
  my ($xcfFile, $prjFile)  = parseXst($filename.$suffix);
  my @sourceFiles;
  my @sourceTimes;
  push @sourceTimes, file_mtime($xcfFile);
  push @sourceFiles, $xcfFile;
  my (@fileList) = parsePrj($prjFile);
  for my $fi (@fileList) {
    if (not -e $fi) {
      die "File $fi does not exist. File $fi found in $prjFile\n";
    }
    my $fi_time = file_mtime($fi);
    push @sourceTimes, $fi_time;
    push @sourceFiles, $fi;
  }
  return (\@sourceFiles,\@sourceTimes);
}


sub parseXst {
  my $xstFile = shift;
  my ($xcfFile,$prjFile);
  open(my $FH, $xstFile) or die "Can't open $xstFile for read.\n";
  while (my $line = <$FH> ) {
    if ($line =~ /-uc\s+(.*?)\r*$/ ) {
      chomp($xcfFile = $1);
    }
    if ($line =~ /-ifn\s+(.*?)\r*$/ ) {
      chomp($prjFile = $1);
    }
  }
  close($FH);
  if (defined($xcfFile) && defined($prjFile)) {
    return((trimDots($xcfFile), trimDots($prjFile)));
  } else {
    die "Could not find constraint file: \"$xcfFile\" file or project file \"$prjFile\" file referenced in $xstFile file.\n";
  }
}

sub parsePrj {
  my $prjFile = shift;
  my @fileList;
  open(my $FH, $prjFile) or die "Can't open $prjFile for read.\n";
  while (my $line = <$FH> ) {
    $line =~ s/\"//g;
    if ($line =~ /^\s*verilog\s+work\s+(\S+)/ ) {
      push(@fileList,$1);
    }
  }
  return(@fileList);
}

sub parseOptFile {
  my $optFile = shift;
  if ($optFile) {
    open(my $FH,$optFile) or die "Could not open $optFile for reading\n";
    while (my $line = <$FH>) {
      chomp($line);
      # remove DOS line ending if present. Should only matter on linux with dos file endings.
      $line =~ s#\015##;
      #remove quotes
      $line =~ s/"//g;
      # remove extra spaces at the end of the line
      $line =~ s/\s*$//;
      push @ARGV, split(" ",$line,2);
    }
    close($FH);
  }
}

sub trimDots {
  my ($file) = @_;
  ($file =~ s#\.\./##) or ($file =~ s#\.\.\\##);
  return($file);
}

sub wait_all_children {
  waitFor(0);
}

sub wait_cpu_available {
  waitFor($NUM_CPUS-1);
}

#
# This routine waits for a specified number of threads (or fewer) to be running.
# If threads are not available it just falls through.
#
sub waitFor {
  my($numThreadsAllowed) = @_;
  if ($threadsOK) {
    my($numthreads,$thrj,$thrd,@joinable);
    my @threads = threads->list;
    # loop until enough jobs are finished
    while (scalar(@threads) > $numThreadsAllowed) {
      print "Waiting for <= $numThreadsAllowed threads, currently at ".scalar(@threads)."\n" if $debug;
      # find out which threads are done and can bo joined.
      # The eval is here to hide the threads::joinable at compile time
      # because threads.pm isn't imported until the BEGIN block is
      # is run
      @joinable = eval('threads->list(threads::joinable)');
      if (scalar(@joinable) > 0 ) {
        $thrj = $joinable[0]->tid();
        my @results = $joinable[0]->join();
        run_on_finish($results[0],$results[1],$results[2]);
      } else {
        # too many jobs running. Sleep and wait for one to finish.
        sleep 2;
      }
      @threads = threads->list;
      $numthreads = scalar(@threads);
    }
  }
}

sub startSynth {
  my $design = shift;
  my $command;
  if ($synplify) {
    $command = "${synplicity_tools}/synplify -batch ../$design.prj -log $design.srp 2>&1 > /dev/null";
  } else {
    $command = "${xilinx_tools}/xst -intstyle xflow -ifn ../$design.xst -ofn $design.srp 2>&1 > /dev/null";
  }
  my $tid;
  print "Synthesizing $design\n";
  if ($threadsOK) {
    my ($thr) = threads->new(\&synthThread, ($command,$design));
  } else {
    my $errcode = csystem($command);
    if (-e "$design.srp") {
      parse_xst_log("$design.srp")
    }
    run_on_finish($errcode,$design,$command);
  }
  return();
}

sub synthThread {
  my($command,$design) = @_;
  my($threadnum) = threads->tid;
  my $errcode = csystem($command);
  if (-e "$design.srp") {
    parse_xst_log("$design.srp")
  }
  return(($errcode,$design,$command));
}

sub run_on_finish {
  my ($exit_code, $ident, $command) = @_;
  if ($exit_code) {
    killAllThreads();
    fail("Synthesis of $ident failed with exit code $exit_code\n","$ident.srp");
  } else {
    print "Completed Synthesis of $ident\n";
  }
}


sub killAllThreads {
  if ($threadsOK) {
    # if we are using threads then we need to stop any unfinished
    # jobs that are running when one job fails.
    my @threads = threads->list;
    for my $th (@threads) {
      $th->kill('KILL')->detach();
    }
  }
}


sub got_controlC {
  print "Caught a control C\n";
  killAllThreads();
  exit();
}


# See if thread work in the current environment
# try to create one inside an eval block and
# catch the error if it doesn't work.
sub testThreads {
  my $threads_ok = 1;
  eval {
    my $t = threads->new(sub {return 1}, ());
    $t->join();
  };
  if ($@) {
    $threads_ok = 0;
  }
  return($threads_ok);
}

sub pause {
  if (not $no_pause) {
    print "Press <Enter> to Continue:";<STDIN>;
  }
}

sub quit {
  my $return_code = shift;
  exit ($return_code);
}

sub csystem {
  my $command = shift;
  if ($^O ne "linux") {
    $command =~ s#/dev/null#NUL#g;
    $command =~ s#/#\\#g;
  }
  return system($command);
}

sub wintitle {
  my $title = shift;
  print "$title:\n";

  if ($^O ne "linux") {
    # Use TITLE Command for Windows
    system ("TITLE $title");
  } else {
    # Use Escape Sequence for Linux
    print "\e]0;$title\a";
  }
}

# Load the threads module if it is available
# Allow the script to keep running if it is not available
BEGIN {
  my $mod = "threads";
  if (eval "require $mod") {
    $mod->import();
  } else {
    print "Could not load module: \"$mod\". Multi-threaded synthesis mode will not be used.\n";
  }
}

sub usage {
  print "Usage:  $0 [options]\n";
  print "\n";
  print "   [options] are provided on command line\n";
  print "             or in a seperate file specified with -f (see below)\n";
  print "\n";
  print "   OPTION LIST\n";
  print "   --------------------------------------------------------------------------\n";
  print "   -h[elp] | -H | -?  : Help\n";
  print "   -f <option file>    Provide options in a seperate file\n";
  print "   -numcpus <int>      Maximum number of CPUs to use for multi-threading (default = 4)\n";
  print "\n";
  print "   -top <top_design>   REQUIRED top level module name (no file extension)\n";
  print "   -ucf <ucf file>     optional constraints file\n";
  print "   -prj <proj>         Use one or more times to specify synthesis projects.\n";
  print "                       Wildcards allowed\n";
  print "                       Defaults to use *.xst\n";
  print "\n";
  print "   -synplify           Use synplify for synthesis\n";
  print "                       Defaults to use *.prj\n";
  print "   -chipscope <file>   Insert Chipscope\n";
  print "   -guide <guide file> Optional Guide File\n";
  print "   -prom <type>        Specify special prom type (supports \"ml555\" or \"bpi\" or \"none\")\n";
  print "   -sdf                Generate SDF file\n";
  print "\n";
  print "   -ngd_opts \"opts\"    Options for ngdbuild\n";
  print "   -map_opts \"opts\"    Options for map\n";
  print "   -par_opts \"opts\"    Options for par\n";
  print "   -trc_opts \"opts\"    Options for trace\n";
  print "   -bgn_opts \"opts\"    Options for bitgen\n";
  print "\n";
  print "   -mppr <num seeds>   Run Multi-Pass Place-and-Route with <num seeds> iterations\n";
  print "   -seed <#>           Which seed to start with for MPPR (defaults to 1)\n";
  print "   -recompile          Forces synthesis of all modules\n";
  print "                       DEFAULT is to chack dependancies vs last run time\n";
  print "   -no_pause           Will not pause at end of route (useful for MPPR)\n";
  print "   -syn_only           Synthesis only\n";
  print "   -smartx             Use SmartXplorer for Map/Par\n";
  print "   -translate          Skip to translate step\n";
  print "   -map                Skip to mapping\n";
  print "   -par                Skip to place and route\n";
  print "   -time               Skip to timing analysis\n";
  print "   -bitgen             Skip to program file generation\n";
  print "\n";
  quit(0);
}
