//  -------------------------------------------------------------------------
//
// Logic Core:          PCI Express Bus Functional Model
// Module Name:         pcie_bfm
//
//
// MODULE DESCRIPTION:
//
// Implements a PCI Express Bus Functional Model using the
// Northwest Logic PCIe Core configured for operation as a Root Complex.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//
//             (c) Copyright 2009 by Northwest Logic, Inc.
//                        All rights reserved.
//
// Trade Secret of Northwest Logic, Inc.  Do not disclose.
//
// Use of this source code in any form or means is permitted only
// with a valid, written license agreement with Northwest Logic, Inc.
//
// Licensee shall keep all information contained herein confidential
// and shall protect same in whole or in part from disclosure and
// dissemination to all third parties.
//
//
//
//                        Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//
//                        Ph:  +1 503 533 5800
//                        Fax: +1 503 533 5900
//                      E-Mail: info@nwlogic.com
//                           www.nwlogic.com
//
//  -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module pcie_bfm_rp (

    rst_n,

    core_rst_n,
    core_clk,

    tx_p,
    tx_n,

    rx_p,
    rx_n,

    pl_link_up,
    dl_link_up,

    mgmt_training_mode, // Set for Root Complex; clear for Endpoint
    mgmt_pcie_status    // See PCIe Core Datasheet; contains a wealth of good debug info
);



// ----------------
// -- Parameters --
// ----------------

// NOTE: Only values defined using parameter are expected to be changed by the user;
//       Do not alter values defined using localparam

parameter   FINISH_STOP_N           = 0;    // On errors and simulation completion, $stop if FINISH_STOP_N==0 else $finish
parameter   STOP_ON_ERR             = 1;    // Set to 1 to $stop simulation on errors (for tests with optional $stop code);
                                            //   Note: During NWL command line regressions, $stop is automatically elevated to $finish
                                            //         by the regression flow for those simulators for which this is necessary

parameter   MIN_CPL_DELAY           = 1000; // Minimum delay between request and completion.

localparam  DUT_BUS_NUMBER          = 8'h01;

localparam  ROOT_PORT_IS_DS_SW      = 1;                        // Root Port is a Downstream Switch Port and needs to be configured before operation to DUT can begin
localparam  CFG0_BUS_NUMBER         = 8'h00;                    // Only used if BFM is Root Port
                                                                // Cfg Type 0 and all other busses with Cfg Type 1

parameter   BFM_MEM_SIZE_BITS       = 22;                       // Size of target BFM memory (default == 2^22 DWORDS == 16 MByte);
localparam  BFM_MEM_DEPTH           = 1 << BFM_MEM_SIZE_BITS;   //   the target BFM memory is accessed when the DUT issues
                                                                //   write and read requests

parameter   BFM0_BAR_SIZE           = 16;                       // Size of BFM BAR0          region (if present)
parameter   BFM1_BAR_SIZE           = 14;                       // Size of BFM BAR1          region (if present)
parameter   EXP_BAR_SIZE            = 12;                       // Size of BFM Expansion ROM region (if present)

// Read Completion Boundary; sets the address boundary on which a read request is
//   broken into multiple completions by the completer logic in this module
parameter   RCB_BYTES               = 128;  // Valid settings == {64, 128, 256, 512, 1024, 2048, or 4096 bytes}
localparam  RCB_REMAIN              =   (RCB_BYTES ==   64) ? 4 :
                                       ((RCB_BYTES ==  128) ? 5 :
                                       ((RCB_BYTES ==  256) ? 6 :
                                       ((RCB_BYTES ==  512) ? 7 :
                                       ((RCB_BYTES == 1024) ? 8 :
                                       ((RCB_BYTES == 2048) ? 9 : 10)))));

// Simulation time before a BFM issued non-posted request is considered as a Completion Timeout; initial value;
//   the register "cpl_timeout_limit" is initialized with this value at the beginning of the simulation. The value
//   is updated based on link width and speed but can then be overridden
parameter   CPL_TIMEOUT_LIMIT_INIT  = 50000000; // 50 uS

// DMA Descriptor Boundary; sets the address boundary on which a DMA is
//   broken into multiple descriptors by the DMA tasks
parameter   DMA_BYTES               = 32'd4096; // Valid settings == {2, 4, 8, 16, 32, 64, 128, 256, 512, 1024, 2048, 4096, 8192, 16384, 32768, or 65536 bytes}
localparam  DMA_REMAIN              =   (DMA_BYTES ==     2) ?  1 :
                                       ((DMA_BYTES ==     4) ?  2 :
                                       ((DMA_BYTES ==     8) ?  3 :
                                       ((DMA_BYTES ==    16) ?  4 :
                                       ((DMA_BYTES ==    32) ?  5 :
                                       ((DMA_BYTES ==    64) ?  6 :
                                       ((DMA_BYTES ==   128) ?  7 :
                                       ((DMA_BYTES ==   256) ?  8 :
                                       ((DMA_BYTES ==   512) ?  9 :
                                       ((DMA_BYTES ==  1024) ? 10 :
                                       ((DMA_BYTES ==  2048) ? 11 :
                                       ((DMA_BYTES ==  4096) ? 12 :
                                       ((DMA_BYTES ==  8192) ? 13 :
                                       ((DMA_BYTES == 16384) ? 14 :
                                       ((DMA_BYTES == 32768) ? 15 : 16))))))))))))));

parameter   BFM_ECRC_GEN_ON         = 1;    // ECRC Generation Enable - Default On when present
parameter   BFM_ECRC_CHECK_ON       = 1;    // ECRC Checking Enable   - Default On when present

parameter   BFM_PHY_LANE_MASK       = 16'h0000;

// NUM_LANES indicates the number of PCI Express lanes supported by the BFM; don't modify
localparam  NUM_LANES               = 8;

// Define the data width of the PCI Express BFM; don't modify
localparam  CORE_DATA_WIDTH         = 64;  // Width of input and output data
localparam  CORE_DW_WIDTH           = 2;   // Width of input and output data in 32-bit DWORDs
localparam  CORE_K_WIDTH            = 8;   // Width of input and output K
localparam  CORE_REMAIN_WIDTH       = 3;   // Set so 2^CORE_REMAIN_WIDTH == CORE_K_WIDTH

localparam  CMD_DATA_WIDTH          = 13;  // Width of cmd ports

// Pattern constants; 32-bit patterns used to auto generate data payloads; don't modify
localparam  PAT_CONSTANT            = 0;    // next = curr
localparam  PAT_ONES                = 1;    // next = all ones
localparam  PAT_ZEROS               = 2;    // next = all zeros
localparam  PAT_INC_NIB             = 3;    // for each nibble: next = curr + (pattern nibble width)
localparam  PAT_INC_BYTE            = 4;    // for each byte:   next = curr + (pattern byte width)
localparam  PAT_INC_WORD            = 5;    // for each word:   next = curr + (pattern word width)
localparam  PAT_INC_DWORD           = 6;    // for each dword:  next = curr + (pattern dword width)
localparam  PAT_L_SHIFT             = 7;    // for each dword:  next = curr << 1
localparam  PAT_R_SHIFT             = 8;    // for each dword:  next = curr >> 1
localparam  PAT_L_ROT               = 9;    // for each dword:  next = {curr[high_bit-1:0], curr[high_bit]}
localparam  PAT_R_ROT               = 10;   // for each dword:  next = {curr[0], curr[high_bit:1]}
localparam  PAT_DEC_NIB             = 11;   // for each nibble: next = curr - (pattern nibble width)
localparam  PAT_DEC_BYTE            = 12;   // for each byte:   next = curr - (pattern byte width)
localparam  PAT_FIB_NIB             = 13;   // Fibonacci sequence using data nibbles (psuedo random)

// Transaction type constants; used for PCIe core Local Bus; don't modify
localparam  TYPE_P                  = 2'h0; // Posted
localparam  TYPE_N                  = 2'h1; // Non-Posted
localparam  TYPE_C                  = 2'h2; // Completion

// Set size of PCIe BFM Transmit FIFOs; don't modify
localparam  FIFO_CMD_AWIDTH         = 4;
localparam  FIFO_CMD_WORDS          = (1 << FIFO_CMD_AWIDTH);

localparam  FIFO_DATA_AWIDTH        = 12; // Minimum recommended value is 12
localparam  FIFO_DATA_WORDS         = (1 << FIFO_DATA_AWIDTH);

localparam  DO_PKT_DMA_LOOPBACK_MAX_PKT_NUM_PACKETS = 4096; // Maximum number of packets that is supported for the DO_PKT_DMA_LOOPBACK task

parameter   SIM_EL_IDLE_TYPE        = 2'b10; // Electrical Idle Emulation: 11 == 1'b1 : Common Mode 1
                                             //                            10 == 1'b0 : Common Mode 0
                                             //                            01 == 1'bx : Undefined
                                             //                            00 == Reserved

parameter   RX_IDLE_ACTIVE_8G_ONLY_EIE = 0; // Set to 1 to cause rx_elec_idle to be 0 at 8G speed only when EIEOS == {8{8'h00, 8'ff}} is received;                                            0 == Use emulated analog comparator
parameter   RX_IDLE_ACTIVE_5G_ONLY_EIE = 0; // Set to 1 to cause rx_elec_idle to be 0 at 5G speed only when EIE == symbols are received back-back; alternating pattern of 5 zeros and 5 ones; 0 == Use emulated analog comparator


localparam  INT_MODE_DISABLED       = 2'h3;
localparam  INT_MODE_MSIX           = 2'h2;
localparam  INT_MODE_MSI            = 2'h1;
localparam  INT_MODE_LEGACY         = 2'h0;



// ----------------------
// -- Port Definitions --
// ----------------------

input                           rst_n;

output                          core_rst_n;
output                          core_clk;

output  [NUM_LANES-1:0]         tx_p;
output  [NUM_LANES-1:0]         tx_n;

input   [NUM_LANES-1:0]         rx_p;
input   [NUM_LANES-1:0]         rx_n;

output                          pl_link_up;
output                          dl_link_up;

input                           mgmt_training_mode;
output  [1535:0]                mgmt_pcie_status;

// ----------------
// -- Port Types --
// ----------------

wire                            rst_n;

wire                            core_rst_n;
wire                            core_clk;

wire    [NUM_LANES-1:0]         tx_p;
wire    [NUM_LANES-1:0]         tx_n;

wire    [NUM_LANES-1:0]         rx_p;
wire    [NUM_LANES-1:0]         rx_n;

wire                            pl_link_up;
wire                            dl_link_up;

wire                            mgmt_training_mode;
wire    [1535:0]                mgmt_pcie_status;

// -------------------
// -- Local Signals --
// -------------------

// -----------------------------------------------------------------
// These signals may be forced during testing to affect BFM behavior

// Set "slow_rx_loop" to artifically slow the BFM PCI Express receive loop
// Set "slow_rx_load" with the minimum number of clocks to wait between packet read starts;
//   slow_rx_load is only used when slow_rx_loop == 1'b1; slow_rx_load should be increased
//   if larger packet payload sizes are used
// The purpose of slow_rx_loop and slow_rx_load is to slow down consumption of
//   receive packets so that FIFOs feeding the receive buffer can have the opportunity
//   to back up and become full; this allows testing of different FIFO operation conditions
//   than possible with the typical flow which is to consume packets at maximum rate
reg                             slow_rx_loop    = 1'b0;     // Default is off
reg     [15:0]                  slow_rx_load    = 16'h00ff; // Default is to process one packet every 256 clocks when slow_rx_loop == 1'b1
// Set receive packet types that should be slowed; clear types not to slow;
//   note when slowed types are being received, they will still slow down other
//   traffic because their processing is being delayed which also hold up further
//   packets; default is to slow all packet types
reg                             slow_rx_mem_rd  = 1'b1;
reg                             slow_rx_mem_wr  = 1'b1;
reg                             slow_rx_io_rd   = 1'b1;
reg                             slow_rx_io_wr   = 1'b1;
reg                             slow_rx_cfg_rd  = 1'b1;
reg                             slow_rx_cfg_wr  = 1'b1;
reg                             slow_rx_msg     = 1'b1;
reg                             slow_rx_msgd    = 1'b1;
reg                             slow_rx_cpl     = 1'b1;
reg                             slow_rx_cpld    = 1'b1;

reg                             rp_is_ds_sw;
reg     [7:0]                   cfg0_bus_num;
reg     [7:0]                   host_bus_num;

// The correct interrupt mode of operation must be indicated on the following variables
//   for the BFM's interrupt detection logic to work properly; PCI Configuration logic
//   must set the following variables appropriately when enabling the DUT's MSI-X, MSI,
//   or Legacy Interrupt functionality
reg     [63:0]                  int_msix_addr            = {64{1'b1}};
reg     [12:0]                  int_msix_num_vectors     = 13'd4096;
reg     [4095:0]                int_int_msix_vector_hit  = 4096'h0;
wire    [4095:0]                int_msix_vector_hit;
reg     [4095:0]                int_msix_vector_hit_latch;
reg     [4095:0]                int_msix_vector_hit_clear;

reg     [63:0]                  int_msi_addr             = {64{1'b1}};
reg     [15:0]                  int_msi_data             = {16{1'b1}};
reg     [5:0]                   int_msi_num_vectors      = 6'd32;
reg     [31:0]                  int_int_msi_vector_hit   = 32'h0;
wire    [31:0]                  int_msi_vector_hit;
reg     [31:0]                  int_msi_vector_hit_latch;
reg     [31:0]                  int_msi_vector_hit_clear;

wire    [3:0]                   int_legi_vector_hit;
reg     [3:0]                   int_legi_vector_hit_latch;
reg     [3:0]                   int_legi_vector_hit_clear;

reg     [31:0]                  bfm_base_io_addr32       = {32{1'b0}};    // 32-bit I/O    space that the BFM will respond to ==  bfm_base_io_addr32      to (bfm_base_io_addr32      +2^(BFM_MEM_SIZE_BITS+2))-1
reg     [31:0]                  bfm_base_addr32          = {32{1'b0}};    // 32-bit Memory space that the BFM will respond to ==  bfm_base_addr32         to (bfm_base_addr32         +2^(BFM_MEM_SIZE_BITS+2))-1
reg     [31:0]                  bfm_base_addr64          = {32{1'b0}};    // 64-bit Memory space that the BFM will respond to == {bfm_base_addr64, 32'h0} to ({bfm_base_addr64, 32'h0}+2^(BFM_MEM_SIZE_BITS+2))-1
reg     [31:0]                  bfm_limit_addr64         = {32{1'b0}};    // 64-bit Memory Space limit address

// mgmt_gen_ecrc_errors cause ECRC errors to be generated in outgoing packets with the EP (poison) bit set
// mgmt_mask_poison causes the EP (poison) bit to be masked in outgoing packets
// These signals should be controlled with the set_completion_error_mode task.
reg                             mgmt_gen_ecrc_errors     = 1'b0;
reg                             mgmt_mask_poison         = 1'b0;

// End signals which may be forced during testing
// -----------------------------------------------------------------

reg     [31:0]                  bfm_mem_bsize;

// Used to slow the reciver loop when slow_rx_loop == 1'b1
reg     [15:0]                  slow_rx_ctr     = 16'h0000;
wire                            slow_ctr_en;
wire    [6:0]                   slow_fmt_type;
wire                            slow_fmt_type_en;
wire                            slow_en;

// Clock Source
wire                            clk;

// Transmit FIFO
reg                             i_tx_sop;
reg                             i_tx_eop;
wire                            i_tx_en;
reg     [31:0]                  i_tx_hdr0;
reg     [CORE_DATA_WIDTH-1:0]   i_tx_data;
reg     [31:0]                  i_tx_ecrc;

// Instantiate PCIe Model; Emulates a Root Complex Device
wire                            model_tx_sop;
wire                            model_tx_eop;
reg                             model_tx_eop_n;
wire                            model_tx_en;
wire    [CORE_DATA_WIDTH-1:0]   model_tx_data;

wire    [1:0]                   model_rx_sel;
wire    [CMD_DATA_WIDTH-1:0]    model_rx_cmd_data;
wire                            model_rx_err_ecrc;
wire                            model_rx_sop;
wire                            model_rx_eop;
wire                            model_rx_en;
wire    [CORE_DATA_WIDTH-1:0]   model_rx_data;

wire    [15:0]                  mgmt_cfg_id;
wire    [1023:0]                mgmt_cfg_constants;
wire    [511:0]                 mgmt_rp_status;

wire                            tx_req;
wire                            tx_ready;
wire    [31:0]                  tx_hdr0;
wire                            tx_sop;
wire                            tx_eop;
wire                            tx_en;
reg     [CORE_DATA_WIDTH-1:0]   tx_data;

wire    [1:0]                   rx_sel;
wire                            rx_sop;
wire                            rx_eop;
wire                            rx_en;
wire    [CORE_DATA_WIDTH-1:0]   rx_data;
wire    [CMD_DATA_WIDTH-1:0]    rx_cmd_data;
wire                            rx_err_ecrc;

// Receive PCI Express Packets
reg                             rx_busy;

wire                            has_payload;
wire    [9:0]                   dw_length;
wire    [10:0]                  max_dw_length;
wire    [12:0]                  payload_bytes;

wire    [12:0]                  hdr_bytes;
wire    [12:0]                  hdr_td_bytes;

reg     [12:0]                  r_rx_bcnt;

// Interrupts
wire                            inta;
wire                            intb;
wire                            intc;
wire                            intd;

wire    [15:0]                  bfm_bdf;
reg     [15:0]                  dut_bdf;

reg     [31:0]                  bfm_init_tag_status;                        // tag status for reads or non-posted write requests from BFM
reg     [2:0]                   bfm_init_tag_cpl_status[31:0];              // tag completion status
reg     [32767:0]               bfm_init_tag_payload[31:0];                 // tag received data payload
reg     [32767:0]               bfm_init_tag_expected_payload[31:0];        // tag expected data payload
reg     [31:0]                  bfm_init_tag_check_payload;                 // set if completion data should be checked against bfm_init_tag_expected_payload
reg     [31:0]                  bfm_init_tag_expect_timeout;                // set to inform BFM that a completion timeout is expected for this tag
time                            bfm_init_tag_req_time[31:0];                // Time of BFM request
reg     [10:0]                  bfm_init_tag_length[31:0];                  // tag remaining length expected in completions for reads or non-posted write requests from BFM
reg     [10:0]                  bfm_init_tag_st_length[31:0];               // tag Starting length expected in completions for reads or non-posted write requests from BFM
reg     [3:0]                   bfm_init_tag_fdw_be[31:0];                  // tag First Data Word Byte Enables.
reg     [3:0]                   bfm_init_tag_ldw_be[31:0];                  // tag Last Data Word Byte Enables.
reg     [10:0]                  bfm_init_tag_index[31:0];                   // tag current starting check memory index (increments with each completion dword)
reg                             bfm_init_tag_is_cfg_en;                     // (1) Enable Cfg Completion ID check; (0) Disable
reg                             bfm_init_tag_is_cfg_en_ur_0;                // (1) ID check passes on UR from function 0; (0) Normal ID check
reg     [31:0]                  bfm_init_tag_is_cfg;                        // 1 - Is a Cfg0 or Cfg 1 Write or Read Request; 0 - Otherwise
reg     [15:0]                  bfm_init_tag_is_cfg_id[31:0];               // Expected Completer ID; only used for completions to Cfg0 or Cfg 1 Write or Read Requests
reg     [4:0]                   bfm_init_next_tag;                          // next tag number to be assigned for read or non-posted write requests from BFM

time                            cpl_timeout_limit;
reg                             suppress_bfm_completion_timeout_reporting;  // 1 - Suppress completion timeout error reporting; 0 - Don't suppress

// Signals used for ECRC generation checking
reg                             bfm_check_td_en;
reg     [15:0]                  bfm_check_td_id;
reg                             bfm_check_td_present;

initial begin
    bfm_check_td_en = 1'b0;
    bfm_check_td_id = 16'h0100;
    bfm_check_td_present = 1'b0;
end

reg     [31:0]                  bfm_mem      [BFM_MEM_DEPTH-1:0];
reg     [3:0]                   bfm_cache_mem[BFM_MEM_DEPTH-1:0];
reg     [4:0]                   status_mem   [BFM_MEM_DEPTH-1:0];

wire    [31:0]                  bfm_mem00;
wire    [31:0]                  bfm_mem01;
wire    [31:0]                  bfm_mem02;
wire    [31:0]                  bfm_mem03;
wire    [31:0]                  bfm_mem04;
wire    [31:0]                  bfm_mem05;
wire    [31:0]                  bfm_mem06;
wire    [31:0]                  bfm_mem07;
wire    [31:0]                  bfm_mem08;
wire    [31:0]                  bfm_mem09;
wire    [31:0]                  bfm_mem10;
wire    [31:0]                  bfm_mem11;
wire    [31:0]                  bfm_mem12;
wire    [31:0]                  bfm_mem13;
wire    [31:0]                  bfm_mem14;
wire    [31:0]                  bfm_mem15;

// ATC Table - 16 entries per function
// Place 16 blocks in the range 8MB to 12MB for untranslated addresses (32 or 64 bit addresses)
// Place 16 blocks in the range 12MB to 16MB for translated addresses
// Use sizes from 4KB to 256KB
// Blocks can be read, write, or both.
// Blocks can allow translations or not.
// Once a block is requested, the active flag will be set
reg     [63:0]                  atc_tran_addr   [255:0][15:0];
reg     [63:0]                  atc_utran_addr  [255:0][15:0];
reg                             atc_active      [255:0][15:0];
reg                             atc_read        [255:0][15:0];
reg                             atc_write       [255:0][15:0];
reg                             atc_untran      [255:0][15:0];
reg     [5:0]                   atc_size        [255:0][15:0];
reg                             atc_inval       [255:0][15:0];
reg     [3:0]                   atc_itag_entry  [255:0][31:0];

reg     [4:0]                   dut_req_tag;                 // requester tag
reg     [7:0]                   dut_req_fmt_and_type [31:0]; // request packet format and type
reg     [15:0]                  dut_req_id           [31:0]; // requester id
reg     [2:0]                   dut_req_tc           [31:0]; // traffic class
reg     [2:0]                   dut_req_attr         [31:0]; // requester attr
reg     [10:0]                  dut_req_length       [31:0]; // requester length
reg     [3:0]                   dut_req_fdw_be       [31:0]; // requester first dword byte enables
reg     [3:0]                   dut_req_ldw_be       [31:0]; // requester last dword byte enabled
reg     [31:0]                  dut_req_addr0        [31:0]; // requester address 0
reg     [31:0]                  dut_req_addr1        [31:0]; // requester address 1
time                            dut_req_time         [31:0]; // time request came in
reg     [31:0]                  dut_req_tag_status;          // requst tag status

// Misc
integer                         xfer_keys = 1;      // Number of simultaneous calls allowed to task "xfer"; only 1 is allowed
reg     [31:0]                  next_prior = 32'h0; // For managing access to task xfer: Next priority to assign
reg     [31:0]                  curr_prior = 32'h0; // For managing access to task xfer: Current priority number

reg                             cpl_holdoff = 0;                    // Holdoff control to delay completions
reg                             single_completion_mode = 1;         // 1 == Intermingle completions from different non-posted requests; 0 == Don't intermingle completions from different non-posted requests
reg                             random_completion_order_mode = 1;   // 1 == Service non-posted requests in random order; 0 == Service non-posted requests in the order of the requests
reg                             random_rcb_length = 1;

integer                         i,j,k;

// Error Counter
integer                         ERROR_COUNT = 0;

// Information Display Statements
wire    [1:0]                   link_speed;
wire    [5:0]                   neg_link_width;
wire    [5:0]                   lts_state;

reg     [1:0]                   r_link_speed;
reg     [5:0]                   r_neg_link_width;

`ifdef SIMULATION
reg                             skip_eq_phase_2_3_override;
`else
wire                            skip_eq_phase_2_3_override = 1'b0;
`endif

reg                             tph_th;
reg     [1:0]                   tph_ph;
reg     [7:0]                   tph_st_lo;

initial begin
    tph_th    = 1'b0;
    tph_ph    = 2'b00;
    tph_st_lo = 8'h00;
end


// ---------------
// -- Equations --
// ---------------

// ------------
// Clock Source

assign clk = core_clk;

`ifdef SIMULATION
// Simulation Overrides
initial begin
    skip_eq_phase_2_3_override = 1'b0;
    model_tx_eop_n             = 1'b0;
end

`endif


// ------------------------------------------------------
// Instantiate PCIe Model; Emulates a Root Complex Device

pcie_model #(
    .SIM_EL_IDLE_TYPE               (SIM_EL_IDLE_TYPE           ),
    .BFM_PHY_LANE_MASK              (BFM_PHY_LANE_MASK          ),
    .RX_IDLE_ACTIVE_8G_ONLY_EIE     (RX_IDLE_ACTIVE_8G_ONLY_EIE ),
    .RX_IDLE_ACTIVE_5G_ONLY_EIE     (RX_IDLE_ACTIVE_5G_ONLY_EIE )
) pcie_model (

    .rst_n                  (rst_n                  ),

    .core_rst_n             (core_rst_n             ),
    .core_clk               (core_clk               ),

    .tx_p                   (tx_p                   ),
    .tx_n                   (tx_n                   ),

    .rx_p                   (rx_p                   ),
    .rx_n                   (rx_n                   ),

    .vc0_tx_sop             (model_tx_sop           ),
    .vc0_tx_eop             (model_tx_eop           ),
    .vc0_tx_eop_n           (model_tx_eop_n         ),
    .vc0_tx_en              (model_tx_en            ),
    .vc0_tx_data            (model_tx_data          ),

    .vc0_rx_sel             (model_rx_sel           ),
    .vc0_rx_cmd_data        (model_rx_cmd_data      ),
    .vc0_rx_err_ecrc        (model_rx_err_ecrc      ),
    .vc0_rx_sop             (model_rx_sop           ),
    .vc0_rx_eop             (model_rx_eop           ),
    .vc0_rx_en              (model_rx_en            ),
    .vc0_rx_data            (model_rx_data          ),

    .pl_link_up             (pl_link_up             ),
    .dl_link_up             (dl_link_up             ),

    .mgmt_cfg_id            (mgmt_cfg_id            ),  // When operating as an Endpoint, need to use this ID for Requestor/Completer ID
    .mgmt_cfg_constants     (mgmt_cfg_constants     ),
    .mgmt_rp_status         (mgmt_rp_status         ),
    .mgmt_rp_control        (32'h0                  ),
    .mgmt_pcie_status       (mgmt_pcie_status       )

);

assign mgmt_cfg_constants[ 15:  0] = 16'h19AA;      // vendor_id
assign mgmt_cfg_constants[ 31: 16] = 16'hBF02;      // device_id
assign mgmt_cfg_constants[ 39: 32] = 8'h00;         // revision_id
assign mgmt_cfg_constants[ 63: 40] = 24'h06_00_00;  // class_code                : Host Bridge
assign mgmt_cfg_constants[ 79: 64] = 16'h19AA;      // subsystem_vendor_id       : N/A for Root Port
assign mgmt_cfg_constants[ 95: 80] = 16'hE004;      // subsystem_id              : N/A for Root Port
assign mgmt_cfg_constants[127: 96] = 32'h00000000;  // cardbus_cis_ptr           : N/A for Root Port
assign mgmt_cfg_constants[    128] = 1'b1;          // int_enable                : Interrupts enabled
assign mgmt_cfg_constants[    129] = 1'b0;          // cap_enable                : No User Capabilities List
assign mgmt_cfg_constants[    130] = 1'b0;          // target only               : Master/Target
assign mgmt_cfg_constants[132:131] = 2'b00;         // Legacy interrupt pin used : 3=D, 2=C, 1=B, 0=A (A should always be used)
assign mgmt_cfg_constants[134:133] = 2'b0;          // Reserved
assign mgmt_cfg_constants[    135] = 1'b0;          // Unadjusted ACK TX Latency Limit Divider1
assign mgmt_cfg_constants[143:136] = 8'h00;         // cap_ptr                   : No User Capabilities List

  assign mgmt_cfg_constants[175:144] = mgmt_training_mode ? 32'h0 : 32'hffc00000;                         // BAR0          : 32-bit Memory, Not Prefetchable
  assign mgmt_cfg_constants[207:176] = mgmt_training_mode ? 32'h0 : 32'hffc00000;                         // BAR1          : 32-bit Memory, Not Prefetchable
assign mgmt_cfg_constants[239:208] = 32'hffff0001;                                                      // BAR2          : I/O memory when BFM configured as endpoint.
assign mgmt_cfg_constants[271:240] = 32'h00000000;                                                      // BAR3          : N/A for Root Port
assign mgmt_cfg_constants[303:272] = 32'h00000000;                                                      // BAR4          : N/A for Root Port
assign mgmt_cfg_constants[335:304] = 32'h00000000;                                                      // BAR5          : N/A for Root Port
assign mgmt_cfg_constants[367:336] = 32'h00000000;                                                      // Expansion ROM : Not implemented

 `ifdef SHORT_SIM
assign mgmt_cfg_constants[    368] = 1'b1;  // Speed up training for simulation
 `else
assign mgmt_cfg_constants[    368] = 1'b0;  // Spec. compliant training
 `endif
assign mgmt_cfg_constants[375:369] = 7'h0;  // Core test mode enables; Do not set!

assign mgmt_cfg_constants[    376] = 1'b1;  // l2_enable  : Set to enable core to enter L2;  clear to not support L2
assign mgmt_cfg_constants[    377] = 1'b1;  // l1_enable  : Set to enable core to enter L1;  clear to not support L1
assign mgmt_cfg_constants[    378] = 1'b1;  // l0s_enable : Set to enable core to enter L0s; clear to not support L0s

assign mgmt_cfg_constants[381:379] = 3'h0;  // Core test mode enables; Do not set!
assign mgmt_cfg_constants[    382] = 1'b0;  // Disable_Auotmatic_PME_TO_Ack_Message_Generation; 1 == User will do; 0 == Let core handle
 `ifdef SHORT_SIM
assign mgmt_cfg_constants[    383] = 1'b1;  // Quick_Simualtion_Reduce_Timeouts; 1 == Use shortened timeouts; 0 == Use PCIe compliant timeouts
 `else
assign mgmt_cfg_constants[    383] = 1'b0;  // Quick_Simualtion_Reduce_Timeouts; 1 == Use shortened timeouts; 0 == Use PCIe compliant timeouts
 `endif
assign mgmt_cfg_constants[399:384] = 16'h0; // ASPM L0s TX Entry Time; 0 == Use core default of 7uS

assign mgmt_cfg_constants[407:400] = 8'hF0; // Number of NFTS sets to request remote device to send when exiting L0s; must be
                                            //   set according to needs of the PCIe PHY

assign mgmt_cfg_constants[    408] = 1'b1; // Retry timout multiply: Double Replay Timer timeout value for high latency PHY configurations

assign mgmt_cfg_constants[    409] = 1'b1;  // mgmt_l1s_enable: 1 == Enable L1 ASPM Power State Functionality; 0 == Don't support

assign mgmt_cfg_constants[423:410] = 14'h0; // Replay Timer Rx_L0s_Adjustment

assign mgmt_cfg_constants[    424] = 1'b1;  // Unadjusted ACK TX Latency Limit Divider0
assign mgmt_cfg_constants[    425] = 1'b0;  // Include TX L0s Adjustment; set to include ACK Timer Tx_L0s_Adjustment in the Ack TX Latency Limit calculation
assign mgmt_cfg_constants[439:426] = 14'h0; // ACK Timer TX L0s Adjustment

assign mgmt_cfg_constants[455:440] = bfm_bdf;
assign mgmt_cfg_constants[    456] = 1'b0; // Root Port Mode   : Implement Root Port as Downstream Switch Port {Switch Port Mode, Root Port Mode} == 2'b10
assign mgmt_cfg_constants[    457] = 1'b0; // Advertise actual CH, CD credits
assign mgmt_cfg_constants[    458] = 1'b0; // PM_Force_RP_Mode == 0; don't force

assign mgmt_cfg_constants[    459] = mgmt_training_mode;  //1'b0 = endpoint, 1'b1 = Root-port.

assign mgmt_cfg_constants[461:460] = 2'h0;

assign mgmt_cfg_constants[    462] = 1'b0; // PCIe Gen 3 Cores Only: 1 == Advertise support for PCI Express 8GT/s, 5GT/s and 2.5 GT/s data rates; 0 == no 8GT/s support
assign mgmt_cfg_constants[    463] = 1'b0; // PCIe Gen 3 Cores Only: 1 == Direct link to 8Gbps when possible; 0 == let remote device determine link speed

assign mgmt_cfg_constants[    464] = 1'b1; // PCIe Gen 2 Cores Only: 1 == Advertise support for 5Gbps; 0 == Advertise 2.5Gbps only
assign mgmt_cfg_constants[    465] = 1'b1; // PCIe Gen 2 Cores Only: 1 == Direct link to 5Gbps when possible; 0 == let remote device determine link speed
assign mgmt_cfg_constants[    466] = 1'b1; // PCIe Gen 2 Cores Only: Selectable De-emphasis
assign mgmt_cfg_constants[469:467] = 3'h0;
assign mgmt_cfg_constants[    470] = 1'b0;
assign mgmt_cfg_constants[479:471] = 9'h0;
assign mgmt_cfg_constants[    480] = 1'b0;                 // Don't allow poison bits to pass to user interface
assign mgmt_cfg_constants[    481] = mgmt_gen_ecrc_errors; // Generate ECRC errors on packets with Poison Set
assign mgmt_cfg_constants[    482] = mgmt_mask_poison;     // Clear Poison bits on all outgoing user packets
assign mgmt_cfg_constants[511:483] = {(512-483){1'b0}};
assign mgmt_cfg_constants[517:512] = 6'h0;
assign mgmt_cfg_constants[541:518] = 24'h0;  // No hardcoded equalization
assign mgmt_cfg_constants[    542] = 1'b0;   // No polarity override
assign mgmt_cfg_constants[    543] = (BFM_PHY_LANE_MASK !== 0); // RX Detect Override
assign mgmt_cfg_constants[559:544] = 16'h0;  // No polarity override
assign mgmt_cfg_constants[575:560] = ~BFM_PHY_LANE_MASK[15:0];  // RX Detect Mask
assign mgmt_cfg_constants[578:576] = 3'h0;   // No max link width override
assign mgmt_cfg_constants[    579] = 1'b1;   // Disable inferred rx_idle in 8G
assign mgmt_cfg_constants[    580] = 1'b1;   // Disable inferred rx_idle in 5G
assign mgmt_cfg_constants[    581] = 1'b1;   // Disable inferred rx_idle in 2.5G
assign mgmt_cfg_constants[586:582] = 5'h0;   // No delay in rx_idle
assign mgmt_cfg_constants[591:587] = 5'h0;   // Reserved
assign mgmt_cfg_constants[607:592] = 16'h0;  // Control over DL Parity/ECC error handling/injection; Parity/ECC Capable Cores Only
assign mgmt_cfg_constants[623:608] = 16'h0;  // Control over TL Parity/ECC error handling/injection; Parity/ECC Capable Cores Only
assign mgmt_cfg_constants[639:624] = 16'h0;  // mgmt_aspm_threshold_l1: Number of microseconds to wait before entering ASPM L1; 0 == Use default of 1mS
assign mgmt_cfg_constants[703:640] = 64'h0;  // Master Loopback Control - All disabled; Master Loopback not present in all cores

assign mgmt_cfg_constants[    704] = 1'b0;   // TPH Enable
assign mgmt_cfg_constants[718:705] = 14'h0;  // Reserved
assign mgmt_cfg_constants[    719] = 1'b0;   // TPH Completer Enable
assign mgmt_cfg_constants[767:720] = 48'h0;  // Reserved
assign mgmt_cfg_constants[1023:768] = 256'h0;  // Reserved for SRIOV configuration


// Parse mgmt_rp_status Bus
assign inta = mgmt_rp_status[16] | mgmt_rp_status[20];
assign intb = mgmt_rp_status[17] | mgmt_rp_status[21];
assign intc = mgmt_rp_status[18] | mgmt_rp_status[22];
assign intd = mgmt_rp_status[19] | mgmt_rp_status[23];

// Delay I/O between behavioral PCIe Model code and
//   synthesizable PCIe Model code to avoid 0-time simualtion problems
assign #100 model_tx_sop    = i_tx_sop;
assign #100 model_tx_eop    = i_tx_eop;
assign #100 i_tx_en         = model_tx_en;
assign #100 model_tx_data   = i_tx_data;

assign #100 rx_sel          = model_rx_sel;
assign #100 rx_cmd_data     = model_rx_cmd_data;
assign #100 rx_err_ecrc     = model_rx_err_ecrc;
assign #100 rx_sop          = model_rx_sop;
assign #100 rx_eop          = model_rx_eop;
assign #100 model_rx_en     = rx_en;
assign #100 rx_data         = model_rx_data;

// initialize TX signals
initial
begin
    i_tx_sop  = 1'b0;
    i_tx_eop  = 1'b0;
    i_tx_hdr0 = 32'h0;
    i_tx_data = {CORE_DATA_WIDTH{1'b0}};
end

// ---------------------
// Lower Memory Contents

// Note: These wires exist solely to make it easy to check the
//   contents of the lower portion of the bfm memory for debugging
//   of test code; these values can be . referenced or viewed in the
//   waveform viewer of the simulator without needing to save the
//   contents of the entire memories
assign bfm_mem00 = bfm_mem[ 0];
assign bfm_mem01 = bfm_mem[ 1];
assign bfm_mem02 = bfm_mem[ 2];
assign bfm_mem03 = bfm_mem[ 3];
assign bfm_mem04 = bfm_mem[ 4];
assign bfm_mem05 = bfm_mem[ 5];
assign bfm_mem06 = bfm_mem[ 6];
assign bfm_mem07 = bfm_mem[ 7];
assign bfm_mem08 = bfm_mem[ 8];
assign bfm_mem09 = bfm_mem[ 9];
assign bfm_mem10 = bfm_mem[10];
assign bfm_mem11 = bfm_mem[11];
assign bfm_mem12 = bfm_mem[12];
assign bfm_mem13 = bfm_mem[13];
assign bfm_mem14 = bfm_mem[14];
assign bfm_mem15 = bfm_mem[15];



// -----
// Tasks

// All BFM PCI Express master traffic is generated via task calls.  There is one task "xfer_action"
//   which generates all PCI Express traffic, however, xfer_action may not be called directly; instead
//   call task "xfer" which manages access to xfer_action such that only one master can use the
//   BFM PCI Express transmit resources at a time; additonal tasks are defined to simplify the
//   generation of PCI Express traffic; these additional taks ultimately cause task xfer to be called

// The BFM automatically responds to read requests received from the DUT by transmitting the
//   approrpriate completions; this process occurs without any user intervention and calls
//   task xfer to transmit the completion packets

// Some non-PCI Express traffic generating tasks are also provided to simplify logging and pattern generation

// -------------------------------------------
// task: display_hdr
//   Display packet log header
//   No PCIe traffic is generated by this task

// Note: A fixed-width font should be used for best presentation

task display_hdr;
begin
    $display ("                                               F C    Cp        C     M            ");
    $display ("                                             L i p Cp B   C  C  f  C  s            ");
    $display ("                                             a r l L  C   f  f  g  f  g            ");
    $display ("                                             s s S A  o   g  g  F  g  C            ");
    $display ("                                             t t t d  u   B  D  u  R  o            ");
    $display ("                                             B B a d  n   u  e  n  e  d            ");
    $display ("                                             E E t r  t   s  v  c  g  e            ");
    $display ("                                                                                   ");
    $display ("     CMD  Tg RqID CpID Len  Addr64   Addr32  L F S LA Bct Bs Dv F Reg MC  Time(nS) ");
    $display ("    ----- -- ---- ---- --- -------- -------- - - - -- --- -- -- - --- -- ----------");
end
endtask



// -------------------------------------------
// task: display_short_hdr
//   Display only short-name packet log header
//   No PCIe traffic is generated by this task

// Note: A fixed-width font should be used for best presentation

task display_short_hdr;
begin
    $display ("     CMD  Tg RqID CpID Len  Addr64   Addr32  L F S LA Bct Bs Dv F Reg MC  Time(nS) ");
    $display ("    ----- -- ---- ---- --- -------- -------- - - - -- --- -- -- - --- -- ----------");
end
endtask



// ----------------------------------------------------------------
// task: xfer
//   Generates and transmits PCI Express packets;
//     called by all other tasks which generate PCI Express traffic

// Tasks to access Completion Holdoff control
//
task automatic set_cpl_holdoff;
begin
    cpl_holdoff = 1'b1;
end
endtask

task automatic clr_cpl_holdoff;
begin
    cpl_holdoff = 1'b0;
end
endtask

// Semaphore task used to ensure that only 1 xfer task call can
//   use the core transmit interface at one time; get_xfer_key may
//   only be called by task xfer
task automatic get_xfer_key;
begin
    wait (xfer_keys > 0)
        xfer_keys = xfer_keys - 1;
end
endtask

// Semaphore task used to ensure that only 1 xfer task call can
//   use the core transmit interface at one time; put_xfer_key may
//   only be called by task xfer
task automatic put_xfer_key;
begin
    xfer_keys = xfer_keys + 1;
end
endtask

// task xfer
task automatic xfer;

input   [2:0]       tc;                 // Traffic Class
input   [2:0]       attr;               // Attribute  Field
input   [6:0]       fmt_and_type;       // {Fmt, Type} of command to use
input   [9:0]       length;             // Length of the burst in DWORDs; 0x1 == 1, 0x2 == 2, ..., 0x3FF = 1023, 0x0 = 1024
input   [3:0]       first_dw_be;        // First payload DWORD's byte enables; Msg Only == MsgCode[3:0]
input   [3:0]       last_dw_be;         // Last payload DWORD's byte enables; Msg Only == MsgCode[7:4]
input   [63:0]      addr;               // Starting address for the command; Msg Only == Message bytes 8..15
input   [15:0]      req_id;             // Requester ID (for completions)
input   [7:0]       tag;                // Requester tag (for completions)
inout   [32767:0]   payload;            // For writes: payload data; for reads: input==expected payload data; output==data actually received
input               check_data;         // For read requests only, set to check completion data against payload, clear to not check
input               no_wait_for_cpl;    // For non-posted only, clear to wait for all completions to complete; set to not wait
input               expect_timeout;     // Set to 1 when a non-posted request is expected to timeout; 0 otherwise
input               use_payload_for_cpl;// Set to 1 for a completion which will use payload rather than bfm_mem for data

reg     [32767:0]   payload;

reg     [31:0]      my_prior;

begin
    // Check for completion issued while cpl_holdoff asserted
    //  In this case stall the completion until holdoff is deasserted

    if (((fmt_and_type == 7'b10_01010) | (fmt_and_type == 7'b00_01010)) && cpl_holdoff == 1'b1)
        while (cpl_holdoff == 1'b1)
            @(posedge clk);

    // The code between get_xfer_key and put_xfer_key can only be executed by
    //   one task call at a time; get_xfer_key and put_xfer_key are semaphores
    //   designed to ensure that only one task can execute the code between
    get_xfer_key;

    // Assign the priority of this task call
    my_prior = next_prior;

    // Increment priority for the next task call
    next_prior = next_prior + 1;

    // Move some time to be safe
    #1;

    // Give the key back
    put_xfer_key;

    // Wait for my turn to call xfer_action
    while (my_prior != curr_prior)
        @(posedge clk);

    // Now do the operation; Note once xfer_action may safely be used by another task call, curr_prior is incremented inside xfer_action
    //   to allow additional task calls access to PCI Express transmit resources
    xfer_action (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr, req_id, tag, payload, check_data, no_wait_for_cpl, 1'b0, 1'b0, expect_timeout, use_payload_for_cpl);
end
endtask

task automatic xfer_poison;

input   [2:0]       tc;                 // Traffic Class
input   [2:0]       attr;               // Attribute Field
input   [6:0]       fmt_and_type;       // {Fmt, Type} of command to use
input   [9:0]       length;             // Length of the burst in DWORDs; 0x1 == 1, 0x2 == 2, ..., 0x3FF = 1023, 0x0 = 1024
input   [3:0]       first_dw_be;        // First payload DWORD's byte enables; Msg Only == MsgCode[3:0]
input   [3:0]       last_dw_be;         // Last payload DWORD's byte enables; Msg Only == MsgCode[7:4]
input   [63:0]      addr;               // Starting address for the command; Msg Only == Message bytes 8..15
input   [15:0]      req_id;             // Requester ID (for completions)
input   [7:0]       tag;                // Requester tag (for completions)
inout   [32767:0]   payload;            // For writes: payload data; for reads: input==expected payload data; output==data actually received
input               check_data;         // For read requests only, set to check completion data against payload, clear to not check
input               no_wait_for_cpl;    // For non-posted only, clear to wait for all completions to complete; set to not wait
input               expect_timeout;     // Set to 1 when a non-posted request is expected to timeout; 0 otherwise
input               use_payload_for_cpl;// Set to 1 for a completion which will use payload rather than bfm_mem for data

reg     [32767:0]   payload;

reg     [31:0]      my_prior;

begin
    // Check for completion issued while cpl_holdoff asserted
    //  In this case stall the completion until holdoff is deasserted

    if (((fmt_and_type == 7'b10_01010) | (fmt_and_type == 7'b00_01010)) && cpl_holdoff == 1'b1)
        while (cpl_holdoff == 1'b1)
            @(posedge clk);

    // The code between get_xfer_key and put_xfer_key can only be executed by
    //   one task call at a time; get_xfer_key and put_xfer_key are semaphores
    //   designed to ensure that only one task can execute the code between
    get_xfer_key;

    // Assign the priority of this task call
    my_prior = next_prior;

    // Increment priority for the next task call
    next_prior = next_prior + 1;

    // Move some time to be safe
    #1;

    // Give the key back
    put_xfer_key;

    // Wait for my turn to call xfer_action
    while (my_prior != curr_prior)
        @(posedge clk);

    // Now do the operation; Note once xfer_action may safely be used by another task call, curr_prior is incremented inside xfer_action
    //   to allow additional task calls access to PCI Express transmit resources
    xfer_action (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr, req_id, tag, payload, check_data, no_wait_for_cpl, 1'b1, 1'b0, expect_timeout, use_payload_for_cpl);
end
endtask

task automatic xfer_ecrc;

input   [2:0]       tc;                 // Traffic Class
input   [2:0]       attr;               // Attribut Field
input   [6:0]       fmt_and_type;       // {Fmt, Type} of command to use
input   [9:0]       length;             // Length of the burst in DWORDs; 0x1 == 1, 0x2 == 2, ..., 0x3FF = 1023, 0x0 = 1024
input   [3:0]       first_dw_be;        // First payload DWORD's byte enables; Msg Only == MsgCode[3:0]
input   [3:0]       last_dw_be;         // Last payload DWORD's byte enables; Msg Only == MsgCode[7:4]
input   [63:0]      addr;               // Starting address for the command; Msg Only == Message bytes 8..15
input   [15:0]      req_id;             // Requester ID (for completions)
input   [7:0]       tag;                // Requester tag (for completions)
inout   [32767:0]   payload;            // For writes: payload data; for reads: input==expected payload data; output==data actually received
input               check_data;         // For read requests only, set to check completion data against payload, clear to not check
input               no_wait_for_cpl;    // For non-posted only, clear to wait for all completions to complete; set to not wait
input               use_payload_for_cpl;// Set to 1 for a completion which will use payload rather than bfm_mem for data

reg     [32767:0]   payload;

reg     [31:0]      my_prior;

begin
    // Check for completion issued while cpl_holdoff asserted
    //  In this case stall the completion until holdoff is deasserted

    if (((fmt_and_type == 7'b10_01010) | (fmt_and_type == 7'b00_01010)) && cpl_holdoff == 1'b1)
        while (cpl_holdoff == 1'b1)
            @(posedge clk);

    // The code between get_xfer_key and put_xfer_key can only be executed by
    //   one task call at a time; get_xfer_key and put_xfer_key are semaphores
    //   designed to ensure that only one task can execute the code between
    get_xfer_key;

    // Assign the priority of this task call
    my_prior = next_prior;

    // Increment priority for the next task call
    next_prior = next_prior + 1;

    // Move some time to be safe
    #1;

    // Give the key back
    put_xfer_key;

    // Wait for my turn to call xfer_action
    while (my_prior != curr_prior)
        @(posedge clk);

    // Now do the operation; Note once xfer_action may safely be used by another task call, curr_prior is incremented inside xfer_action
    //   to allow additional task calls access to PCI Express transmit resources
    xfer_action (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr, req_id, tag, payload, check_data, no_wait_for_cpl, 1'b0, 1'b1, 1'b0, use_payload_for_cpl);
end
endtask

// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
// NOTE! xfer_action may only be called from task xfer and must NEVER be called directly;
//   task xfer ensures that the shared BFM PCI Express transmit resources used by
//   xfer_action are only accessed by one task call at a time
// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

task automatic xfer_action;

input   [2:0]       tc;                 // Traffic Class
input   [2:0]       attr;               // Attribute Field
input   [6:0]       fmt_and_type;       // {Fmt, Type} of command to use
input   [9:0]       length;             // Length of the burst in DWORDs; 0x1 == 1, 0x2 == 2, ..., 0x3FF = 1023, 0x0 = 1024
input   [3:0]       first_dw_be;        // First payload DWORD's byte enables; Msg Only == MsgCode[3:0]
input   [3:0]       last_dw_be;         // Last payload DWORD's byte enables; Msg Only == MsgCode[7:4]
input   [63:0]      addr;               // Starting address for the command; Msg Only == Message bytes 8..15
input   [15:0]      req_id;             // Requester ID (for completions)
input   [7:0]       tag;                // Requester tag (for completions)
inout   [32767:0]   payload;            // For writes: payload data; for reads: input==expected payload data; output==data actually received
input               check_data;         // For read requests only, set to check completion data against payload, clear to not check
input               no_wait_for_cpl;    // For non-posted only, clear to wait for all completions to complete; set to not wait
input               poison;             // Request poisoned packet to be set 0 = no error, 1 = poisoned payload
input               add_ecrc;           // Request ECRC to be added to the packet by this task
input               expect_timeout;     // Set to 1 when a non-posted request is expected to timeout; 0 otherwise
input               use_payload_for_cpl;// Set to 1 for a completion which will use payload rather than bfm_mem for data

reg     [32767:0]   payload;            // Inout port

reg     [31:0]      pkt_ecrc;           // ECRC to append to packet

reg     [31:0]      ctime;              // Logging
reg     [7:0]       bs;                 //   ..
reg     [4:0]       dv;                 //   ..
reg     [2:0]       f;                  //   ..
reg     [11:0]      r;                  //   ..

reg                 valid;              // Set if fmt_and_type is a valid packet type
reg     [1:0]       ctype;              // Packet type (Non-posted, posted, completion)
reg                 has_payload;        // Set if transaction has payload data
reg                 req_c;              // Set if transaction requires completion
reg                 mem;                // Set if transaction is a memory access
reg                 io;                 // Set if transaction is an io access
reg                 cfg;                // Set if transaction is a configuration access
reg                 cpl;                // Set if transaction is a completion
reg                 msg;                // Set if transaction is a message
reg                 addr64;             // Set if transaction used 64-bit addressing
reg                 wr_rd_n;            // Set if transaction is a "write"; clear if a "read"

reg     [4:0]       my_tag;             // For transactions requiring completions, the tag which was assigned to the transaction

reg     [1:0]       cpl_lowest_addr;    // Lower address for completions
reg     [6:0]       cpl_lower_addr;     //   ..
reg     [11:0]      cpl_bcount;         // Byte count for completions

reg                 continue_i;         // Transaction tracking
reg     [31:0]      rcbr;               //   ..
reg     [10:0]      remain_length;      //   ..
reg     [9:0]       length_minus1;      //   ..
reg     [10:0]      length_plus_offset; //   ..
reg                 dw_sel;             //   ..
reg                 poisoned;
reg                 ecrc_error;
reg [2:0]           cpl_status;
reg [4:0]           status_mem_bits;

reg     [2:0]       max_payload_size;
reg     [31:0]      max_bytes;

integer             i;                  // Loop constants
integer             j;                  //   ..

begin
    my_tag = 5'b0;

    // NOTE: This routine uses several #1 delays to eliminate potentional simulator execution ordering issues;
    //       These delays are necessary for some simulators and should be preserved even though they have no
    //       other obvious function.

    casex (fmt_and_type)
        7'b00_00000 : begin valid = 1; ctype = TYPE_N; has_payload = 0; req_c = 1; mem = 1; io = 0; cfg = 0; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 0; end  // Memory Read Request (32-bit address)
        7'b01_00000 : begin valid = 1; ctype = TYPE_N; has_payload = 0; req_c = 1; mem = 1; io = 0; cfg = 0; cpl = 0; msg = 0; addr64 = 1; wr_rd_n = 0; end  // Memory Read Request (64-bit address)

        7'b00_00001 : begin valid = 1; ctype = TYPE_N; has_payload = 0; req_c = 1; mem = 1; io = 0; cfg = 0; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 0; end  // Memory Read Request-Locked (32-bit address); Endpoints do not support locked transactions
        7'b01_00001 : begin valid = 1; ctype = TYPE_N; has_payload = 0; req_c = 1; mem = 1; io = 0; cfg = 0; cpl = 0; msg = 0; addr64 = 1; wr_rd_n = 0; end  // Memory Read Request-Locked (64-bit address); Endpoints do not support locked transactions

        7'b10_00000 : begin valid = 1; ctype = TYPE_P; has_payload = 1; req_c = 0; mem = 1; io = 0; cfg = 0; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 1; end  // Memory Write Request (32-bit address)
        7'b11_00000 : begin valid = 1; ctype = TYPE_P; has_payload = 1; req_c = 0; mem = 1; io = 0; cfg = 0; cpl = 0; msg = 0; addr64 = 1; wr_rd_n = 1; end  // Memory Write Request (64-bit address)

        7'b00_00010 : begin valid = 1; ctype = TYPE_N; has_payload = 0; req_c = 1; mem = 0; io = 1; cfg = 0; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 0; end  // I/O Read Request (32-bit address)
        7'b10_00010 : begin valid = 1; ctype = TYPE_N; has_payload = 1; req_c = 1; mem = 0; io = 1; cfg = 0; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 1; end  // I/O Write Request (32-bit address)

        7'b00_00100 : begin valid = 1; ctype = TYPE_N; has_payload = 0; req_c = 1; mem = 0; io = 0; cfg = 1; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 0; end  // Configuration Read  (Type 0)
        7'b10_00100 : begin valid = 1; ctype = TYPE_N; has_payload = 1; req_c = 1; mem = 0; io = 0; cfg = 1; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 1; end  // Configuration Write (Type 0)

        7'b00_00101 : begin valid = 1; ctype = TYPE_N; has_payload = 0; req_c = 1; mem = 0; io = 0; cfg = 1; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 0; end  // Configuration Read  (Type 1)
        7'b10_00101 : begin valid = 1; ctype = TYPE_N; has_payload = 1; req_c = 1; mem = 0; io = 0; cfg = 1; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 1; end  // Configuration Write (Type 1)

        7'b01_10xxx : begin valid = 1; ctype = TYPE_P; has_payload = 0; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 1; addr64 = 1; wr_rd_n = 1; end  // Message Request without Data Payload; Broadcast from Root (Required - implemented)
        7'b11_10xxx : begin valid = 1; ctype = TYPE_P; has_payload = 1; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 1; addr64 = 1; wr_rd_n = 1; end  // Message Request with Data Payload; Broadcast from Root (Required - implemented)

        7'b00_01010 : begin valid = 1; ctype = TYPE_C; has_payload = 0; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 1; msg = 0; addr64 = 0; wr_rd_n = 1; end  // Completion without Data
        7'b10_01010 : begin valid = 1; ctype = TYPE_C; has_payload = 1; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 1; msg = 0; addr64 = 0; wr_rd_n = 1; end  // Completion with Data

        default     : begin valid = 0; ctype = TYPE_P; has_payload = 0; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 0; end  // Unsupported Fmt_and_Type
    endcase

    // If the request requires a completion, don't continue until there is an available tag
    if (req_c)
    begin
        // Wait if no tags available
        while (bfm_init_tag_status == 32'hffffffff)
        begin
            @(posedge clk);
            #1;
        end

        // Advance to first available tag
        while (bfm_init_tag_status[bfm_init_next_tag] == 1'b1)
            bfm_init_next_tag = bfm_init_next_tag + 1;

        // Save the granted tag
        my_tag = bfm_init_next_tag;

        // Roll tag number for the next requestor
        bfm_init_next_tag = bfm_init_next_tag + 1;
    end

    // Compute lower address and byte count for completions
    if (cpl)
    begin
        // Compute lower address for completions
        case (first_dw_be)
            4'b0000 : cpl_lowest_addr = 0; // Zero length read or write
            4'b0001 : cpl_lowest_addr = 0;
            4'b0010 : cpl_lowest_addr = 1;
            4'b0011 : cpl_lowest_addr = 0;
            4'b0100 : cpl_lowest_addr = 2;
            4'b0101 : cpl_lowest_addr = 0;
            4'b0110 : cpl_lowest_addr = 1;
            4'b0111 : cpl_lowest_addr = 0;
            4'b1000 : cpl_lowest_addr = 3;
            4'b1001 : cpl_lowest_addr = 0;
            4'b1010 : cpl_lowest_addr = 1;
            4'b1011 : cpl_lowest_addr = 0;
            4'b1100 : cpl_lowest_addr = 2;
            4'b1101 : cpl_lowest_addr = 0;
            4'b1110 : cpl_lowest_addr = 1;
            4'b1111 : cpl_lowest_addr = 0;
        endcase
        cpl_lower_addr = {addr[6:2], cpl_lowest_addr[1:0]};

        if ((last_dw_be == 0) | (length == 1)) // Single DWORD read
        begin
            case (first_dw_be)
                4'b0000 : cpl_bcount = 1; // Zero length read or write (special case: set to 1 per PCIe Spec.)
                4'b0001 : cpl_bcount = 1;
                4'b0010 : cpl_bcount = 1;
                4'b0011 : cpl_bcount = 2;
                4'b0100 : cpl_bcount = 1;
                4'b0101 : cpl_bcount = 3;
                4'b0110 : cpl_bcount = 2;
                4'b0111 : cpl_bcount = 3;
                4'b1000 : cpl_bcount = 1;
                4'b1001 : cpl_bcount = 4;
                4'b1010 : cpl_bcount = 3;
                4'b1011 : cpl_bcount = 4;
                4'b1100 : cpl_bcount = 2;
                4'b1101 : cpl_bcount = 4;
                4'b1110 : cpl_bcount = 3;
                4'b1111 : cpl_bcount = 4;
            endcase
        end
        else
        begin
            // Compute completion byte count
            length_minus1 = length - 1;
            cpl_bcount    = (length_minus1 << 2) + 4; // max byte count based only on length

            // subtract out dword misaligned start bytes
            case (first_dw_be)
                4'b0000 : cpl_bcount = cpl_bcount - 0; // undefined case
                4'b0001 : cpl_bcount = cpl_bcount - 0;
                4'b0010 : cpl_bcount = cpl_bcount - 1;
                4'b0011 : cpl_bcount = cpl_bcount - 0;
                4'b0100 : cpl_bcount = cpl_bcount - 2;
                4'b0101 : cpl_bcount = cpl_bcount - 0;
                4'b0110 : cpl_bcount = cpl_bcount - 1;
                4'b0111 : cpl_bcount = cpl_bcount - 0;
                4'b1000 : cpl_bcount = cpl_bcount - 3;
                4'b1001 : cpl_bcount = cpl_bcount - 0;
                4'b1010 : cpl_bcount = cpl_bcount - 1;
                4'b1011 : cpl_bcount = cpl_bcount - 0;
                4'b1100 : cpl_bcount = cpl_bcount - 2;
                4'b1101 : cpl_bcount = cpl_bcount - 0;
                4'b1110 : cpl_bcount = cpl_bcount - 1;
                4'b1111 : cpl_bcount = cpl_bcount - 0;
            endcase

            // subtract out dword misaligned end bytes
            case (last_dw_be)
                4'b0000 : cpl_bcount = cpl_bcount - 0; // undefined case
                4'b0001 : cpl_bcount = cpl_bcount - 3;
                4'b0010 : cpl_bcount = cpl_bcount - 2;
                4'b0011 : cpl_bcount = cpl_bcount - 2;
                4'b0100 : cpl_bcount = cpl_bcount - 1;
                4'b0101 : cpl_bcount = cpl_bcount - 1;
                4'b0110 : cpl_bcount = cpl_bcount - 1;
                4'b0111 : cpl_bcount = cpl_bcount - 1;
                4'b1000 : cpl_bcount = cpl_bcount - 0;
                4'b1001 : cpl_bcount = cpl_bcount - 0;
                4'b1010 : cpl_bcount = cpl_bcount - 0;
                4'b1011 : cpl_bcount = cpl_bcount - 0;
                4'b1100 : cpl_bcount = cpl_bcount - 0;
                4'b1101 : cpl_bcount = cpl_bcount - 0;
                4'b1110 : cpl_bcount = cpl_bcount - 0;
                4'b1111 : cpl_bcount = cpl_bcount - 0;
            endcase
        end
    end

    // If its a completion
    if (cpl)
    begin
        addr          = {32'h0, addr[31:2], 2'b00}; // Zero out lower 2 bits of address
        remain_length = (length == 0) ? 1024 : {1'b0, length}; // Check for special case of 0 == 1024
    end

    // Use a while loop to allow breaking a xfer command into several smaller packets
    continue_i = 1;
    while (continue_i == 1)
    begin
        // Break completions along RCB boundary; this is typical of actual chipset hardware when reading system memory
        if (cpl)
        begin
            // Compute the length for the current completion
            //   Max length is RCB bytes - starting address offset (Note: byte_addr[1:0] are always 00)
            max_payload_size = pcie_model.pcie_core_vc1.cfg_max_payload_size;
            case (max_payload_size)
                3'h0 : max_bytes = 128;
                3'h1 : max_bytes = 256;
                3'h2 : max_bytes = 512;
                3'h3 : max_bytes = 1024;
                3'h4 : max_bytes = 2048;
                3'h5 : max_bytes = 4096;
                default : $display("ERROR : Illegal max_payload setting");
            endcase

            if (max_bytes > RCB_BYTES)
                max_bytes = RCB_BYTES;

            case (max_bytes)
                32'd64   : length = 10'h010 - addr[5:2];
                32'd128  : length = 10'h020 - addr[6:2];
                32'd256  : length = 10'h040 - addr[7:2];
                32'd512  : length = 10'h080 - addr[8:2];
                32'd1024 : length = 10'h100 - addr[9:2];
                32'd2048 : length = 10'h200 - addr[10:2];
                32'd4096 : length = 10'h000 - addr[11:2];
                default  : $display ("ERROR : Illegal setting for max_bytes");
            endcase

            // Randomize RCB
            if (random_rcb_length)
            begin
                rcbr = $random;
                case (rcbr[1:0])
                    2'b00 :                  length = length;
                    2'b01 : if (length > 16) length = length - 16;
                    2'b10 : if (length > 32) length = length - 32;
                    2'b11 : if (length > 48) length = length - 48;
                endcase
            end

            length = (length > remain_length) ? remain_length : length;

            // Check for the final completion
            if (remain_length <= length)
                continue_i = 0;

            remain_length = remain_length - length;
        end
        else
        begin
            // Non-completions only do one itteration through the loop
            continue_i = 0;
        end

        // Log packet header information
        if (!$test$plusargs("pcie_traffic_msgs_off"))
        begin
            ctime = $time/1000; // nS

            // Parse address for configuration information
            bs = addr[31:24];
            dv = addr[23:19];
            f  = addr[18:16];
            r  = {addr[11:2], 2'b00};

            casex (fmt_and_type)
                //                                  Cmd                                                                      Tg      RqID     CpID     Length  Addr64       Addr32      L           F            CplStat, CplLowAddr      CplBcount  Bs  Dv  F  R  MsgCode                    Time
                7'b00_00000 : $display ("D : MRd32 %h %h ---- %h -------- %h %h %h - -- --- -- -- - --- -- %d",       my_tag, bfm_bdf,          length,              addr[31:0], last_dw_be, first_dw_be,                                                                              ctime);
                7'b01_00000 : $display ("D : MRd64 %h %h ---- %h %h %h %h %h - -- --- -- -- - --- -- %d",             my_tag, bfm_bdf,          length, addr[63:32], addr[31:0], last_dw_be, first_dw_be,                                                                              ctime);

                7'b00_00001 : $display ("D : MRL32 %h %h ---- %h -------- %h %h %h - -- --- -- -- - --- -- %d",       my_tag, bfm_bdf,          length,              addr[31:0], last_dw_be, first_dw_be,                                                                              ctime);
                7'b01_00001 : $display ("D : MRL64 %h %h ---- %h %h %h %h %h - -- --- -- -- - --- -- %d",             my_tag, bfm_bdf,          length, addr[63:32], addr[31:0], last_dw_be, first_dw_be,                                                                              ctime);

                7'b10_00000 : $display ("D : MWr32 -- %h ---- %h -------- %h %h %h - -- --- -- -- - --- -- %d",               bfm_bdf,          length,              addr[31:0], last_dw_be, first_dw_be,                                                                              ctime);
                7'b11_00000 : $display ("D : MWr64 -- %h ---- %h %h %h %h %h - -- --- -- -- - --- -- %d",                     bfm_bdf,          length, addr[63:32], addr[31:0], last_dw_be, first_dw_be,                                                                              ctime);

                7'b00_00010 : $display ("D : IoR32 %h %h ---- %h -------- %h %h %h - -- --- -- -- - --- -- %d",       my_tag, bfm_bdf,          length,              addr[31:0], last_dw_be, first_dw_be,                                                                              ctime);
                7'b10_00010 : $display ("D : IoW32 %h %h ---- %h -------- %h %h %h - -- --- -- -- - --- -- %d",       my_tag, bfm_bdf,          length,              addr[31:0], last_dw_be, first_dw_be,                                                                              ctime);

                7'b00_00100 : $display ("D : Cf0Rd %h %h ---- %h -------- -------- %h %h - -- --- %h %h %h %h -- %d", my_tag, bfm_bdf,          length,                          last_dw_be, first_dw_be,                                     bs, dv, f, r,                            ctime);
                7'b10_00100 : $display ("D : Cf0Wr %h %h ---- %h -------- -------- %h %h - -- --- %h %h %h %h -- %d", my_tag, bfm_bdf,          length,                          last_dw_be, first_dw_be,                                     bs, dv, f, r,                            ctime);

                7'b00_00101 : $display ("D : Cf1Rd %h %h ---- %h -------- -------- %h %h - -- --- %h %h %h %h -- %d", my_tag, bfm_bdf,          length,                          last_dw_be, first_dw_be,                                     bs, dv, f, r,                            ctime);
                7'b10_00101 : $display ("D : Cf1Wr %h %h ---- %h -------- -------- %h %h - -- --- %h %h %h %h -- %d", my_tag, bfm_bdf,          length,                          last_dw_be, first_dw_be,                                     bs, dv, f, r,                            ctime);

                7'b01_10xxx : $display ("D : Msg   -- %h ---- %h -------- -------- - - - -- --- -- -- - --- %h %d",           bfm_bdf,          length,                                                                                                     {last_dw_be, first_dw_be}, ctime);
                7'b11_10xxx : $display ("D : MsgD  -- %h ---- %h -------- -------- - - - -- --- -- -- - --- %h %d",           bfm_bdf,          length,                                                                                                     {last_dw_be, first_dw_be}, ctime);

                7'b00_01010 : $display ("D : Cpl   %h %h %h %h -------- -------- - - %h %h %h -- -- - --- -- %d",     tag,    req_id,  bfm_bdf, length,                                                   3'b000,  cpl_lower_addr, cpl_bcount,                                         ctime);
                7'b10_01010 : $display ("D : CplD  %h %h %h %h -------- -------- - - %h %h %h -- -- - --- -- %d",     tag,    req_id,  bfm_bdf, length,                                                   3'b000,  cpl_lower_addr, cpl_bcount,                                         ctime);

                default     : $display ("D : ERROR %h %h ---- %h -------- -------- - - - -- --- -- -- - --- -- %d",   my_tag, bfm_bdf,          length,                                                                                                                                ctime);
            endcase
        end

        // Manage Transaction Errors
        // Re-set to 0 on each transaction
        poisoned    = poison;
        ecrc_error  = 1'b0;
        cpl_status  = 3'b000;

        // if any of the accessed bfm memory for this TLP is marked
        // with en error bit, then set that error bit for this
        // TLP.  For completions only (since data comes from BFM mem)
        if (cpl & ~use_payload_for_cpl)
        begin
            for (i = 0; i < ((length == 0) ? 1024 : length); i = i + 1)
            begin
                status_mem_bits = status_mem[addr[BFM_MEM_SIZE_BITS+1:2] + i];
                if (status_mem_bits[0] === 1'b1)
                    poisoned   = 1'b1;
                if (status_mem_bits[1] === 1'b1)
                    ecrc_error = 1'b1;
                if (|status_mem_bits[4:2] === 1'b1)
                    cpl_status  = status_mem_bits[4:2];
            end
            // If completion status is not success (3'b0), there should be
            // no data in the packet, and the remaining packets are dropped
            if (cpl_status != 3'b0) begin
                continue_i    = 0;              // no more packets
                has_payload   = 0;              // no payload
                fmt_and_type  = 7'b00_01010;    // change format to completion without data
                cpl_bcount    = 0;
                length        = 1;
                remain_length = 0;              // no more data
            end
        end

        if (cpl & use_payload_for_cpl)      // If completion without data and use_payload -> UR
        begin
            if (fmt_and_type == 7'b00_01010)
                cpl_status = 3'b001;
        end

        // Carry out the request
        if (valid)
        begin
            // Drive the new packet at the falling edge of clock.
            // This is done instead of at the rising edge so back to back calls
            // will not end up with an extra dead cycle between transfers, and so
            // that we are always aligned properly to the BFMs clock, despite when
            // the task is originally called.
            @(negedge clk);
            i_tx_sop = 1'b1;
            pkt_ecrc = 32'hffffffff;        // ECRC init value

            // Show header requesting to transmit
            i_tx_hdr0[ 6: 0] = fmt_and_type;
            i_tx_hdr0[    7] = 1'b0;         // Reserved
            i_tx_hdr0[    8] = tph_th;
            i_tx_hdr0[   11] = 1'b0;         // Reserved
            i_tx_hdr0[   10] = attr[2];
            i_tx_hdr0[    9] = 1'b0;         // Reserved
            i_tx_hdr0[14:12] = tc;
            i_tx_hdr0[15   ] = 1'b0;         // Reserved
            i_tx_hdr0[17:16] = length[9:8];
            i_tx_hdr0[19:18] = 2'b0;         // Reserved
            i_tx_hdr0[21:20] = attr[1:0];    // Attr
            i_tx_hdr0[22   ] = poisoned | ecrc_error; // EP
            i_tx_hdr0[23   ] = add_ecrc;     // TD
            i_tx_hdr0[31:24] = length[7:0];


            // ----------------
            // 1st header DWORD

            i_tx_data[31:0] = i_tx_hdr0;
            pkt_ecrc = ecrc(pkt_ecrc, {i_tx_data[31:24],2'b11, i_tx_data[21:1], 1'b1});     // Set EP, TD, Type[0]



            // ----------------
            // 2nd header DWORD

            if (cpl)
            begin
                i_tx_data[39:32] = bfm_bdf[15:8];
                i_tx_data[47:40] = bfm_bdf[7:0];
                i_tx_data[51:48] = cpl_bcount[11:8];
                i_tx_data[52   ] = 1'b0;             // BCM
                i_tx_data[55:53] = cpl_status;       // Completion Status
                i_tx_data[63:56] = cpl_bcount[7:0];
            end
            else
            begin
                i_tx_data[39:32] = bfm_bdf[15:8];
                i_tx_data[47:40] = bfm_bdf[7:0];
                if (req_c)
                    i_tx_data[55:48] = {3'b0, my_tag[4:0]}; // Tag
                else
                    i_tx_data[55:48] = (mem &  wr_rd_n & tph_th) ? tph_st_lo : (msg ? tag : 8'b0);        // Tag
                i_tx_data[63:56]     = (mem & ~wr_rd_n & tph_th) ? tph_st_lo : {last_dw_be, first_dw_be}; // Note: Message code brought in on BE
            end
            pkt_ecrc = ecrc(pkt_ecrc, i_tx_data[63:32]);

            // Transfer first 64-bits of header
            @(posedge clk);
            while (~i_tx_en)
                @(posedge clk);
            i_tx_sop = 1'b0;



            // ----------------
            // 3rd header DWORD

            if (cpl)
            begin
                i_tx_data[ 7: 0] = req_id[15:8];
                i_tx_data[15: 8] = req_id[ 7:0];
                i_tx_data[23:16] = tag[7:0];
                i_tx_data[30:24] = cpl_lower_addr[6:0];
                i_tx_data[31   ] = 1'b0;            // Reserved
            end
            else if (addr64)    // Note: msg is always addr64
                i_tx_data[31:0] = {addr[39:32], addr[47:40], addr[55:48], addr[63:56]};
            else
                i_tx_data[31:0] = {addr[7:2], (tph_th ? tph_ph : 2'b00), addr[15:8], addr[23:16], addr[31:24]};
            pkt_ecrc = ecrc(pkt_ecrc, i_tx_data[31:0]);


            // ------------------------------
            // 4th header DWORD (if required)

            if (addr64)
            begin
                if (msg)
                    i_tx_data[63:32] = {addr[7:0], addr[15:8], addr[23:16], addr[31:24]};
                else
                    i_tx_data[63:32] = {addr[7:2], (tph_th ? tph_ph : 2'b00), addr[15:8], addr[23:16], addr[31:24]};
                dw_sel = 1'b0;
                pkt_ecrc = ecrc(pkt_ecrc, i_tx_data[63:32]);
            end
            else
            begin
                dw_sel = 1'b1; // start payload on 1st dword
            end

            // Output payload
            if (has_payload)
            begin
                // If a 4 DWORD header then we already completed the whole first word
                if (addr64)
                begin
                    @(posedge clk);
                    while (~i_tx_en)
                        @(posedge clk);
                end

                for (i=0; i< ((length == 0) ? 1024 : length); i=i+1)
                begin
                    if ((fmt_and_type == 7'b10_01010) & ~use_payload_for_cpl)
                    begin
                        //completion - assuming memory read completion
                        case (dw_sel)
                            1'b0 : i_tx_data[ 31: 0] = bfm_mem[addr[BFM_MEM_SIZE_BITS+1:2]+i];
                            1'b1 : i_tx_data[ 63:32] = bfm_mem[addr[BFM_MEM_SIZE_BITS+1:2]+i];
                        endcase
                        pkt_ecrc = ecrc(pkt_ecrc, bfm_mem[addr[BFM_MEM_SIZE_BITS+1:2]+i]);
                    end
                    else
                    begin
                        case (dw_sel)
                            1'b0 : for (j=0; j<32; j=j+1) i_tx_data[j+ 0] = payload[(i*32)+j];
                            1'b1 : for (j=0; j<32; j=j+1) i_tx_data[j+32] = payload[(i*32)+j];
                        endcase
                        case (dw_sel)
                            1'b0 : pkt_ecrc = ecrc(pkt_ecrc, i_tx_data[31: 0]);
                            1'b1 : pkt_ecrc = ecrc(pkt_ecrc, i_tx_data[63:32]);
                        endcase
                    end

                    // Log data payload information
                    if (!$test$plusargs("pcie_traffic_msgs_off"))
                    begin
                        if (cpl)
                        begin
                            case (dw_sel)
                                1'b0 : $display ("D :   Data (Tag 0x%x) == 0x%x", tag, i_tx_data[ 31: 0]);
                                1'b1 : $display ("D :   Data (Tag 0x%x) == 0x%x", tag, i_tx_data[ 63:32]);
                            endcase
                        end
                        else
                        begin
                            case (dw_sel)
                                1'b0 : $display ("D :   Data == 0x%x", i_tx_data[ 31: 0]);
                                1'b1 : $display ("D :   Data == 0x%x", i_tx_data[ 63:32]);
                            endcase
                        end
                    end

                    if ((i == ((length == 0) ? 1023 : (length-1))) & ~add_ecrc) // Final DWORD
                    begin
                        i_tx_eop = 1'b1;
                        @(posedge clk);
                        while (~i_tx_en)
                            @(posedge clk);
                    end
                    else if ((i == ((length == 0) ? 1023 : (length-1))) & add_ecrc)
                    begin
                        i_tx_ecrc = ~{pkt_ecrc[ 0], pkt_ecrc[ 1], pkt_ecrc[ 2], pkt_ecrc[ 3], pkt_ecrc[ 4], pkt_ecrc[ 5], pkt_ecrc[ 6], pkt_ecrc[ 7],
                                      pkt_ecrc[ 8], pkt_ecrc[ 9], pkt_ecrc[10], pkt_ecrc[11], pkt_ecrc[12], pkt_ecrc[13], pkt_ecrc[14], pkt_ecrc[15],
                                      pkt_ecrc[16], pkt_ecrc[17], pkt_ecrc[18], pkt_ecrc[19], pkt_ecrc[20], pkt_ecrc[21], pkt_ecrc[22], pkt_ecrc[23],
                                      pkt_ecrc[24], pkt_ecrc[25], pkt_ecrc[26], pkt_ecrc[27], pkt_ecrc[28], pkt_ecrc[29], pkt_ecrc[30], pkt_ecrc[31]};
                        case (dw_sel)
                            1'b0 : i_tx_data[63:32] = i_tx_ecrc;        // Add ECRC to last word
                            1'b1 : begin
                                @(posedge clk);
                                while (~i_tx_en)
                                    @(posedge clk);
                                i_tx_data = {32'h0, i_tx_ecrc};         // Put ECRC in next word
                            end
                        endcase
                        i_tx_eop = 1'b1;
                        @(posedge clk);
                        while (~i_tx_en)
                            @(posedge clk);
                    end
                    else if (dw_sel == 1'b1) // Last DWORD in current word
                    begin
                        @(posedge clk);
                        while (~i_tx_en)
                            @(posedge clk);
                    end

                    dw_sel = ~dw_sel;
                end
            end
            else
            begin
                i_tx_eop = 1'b1;
                @(posedge clk);
                while (~i_tx_en)
                    @(posedge clk);
            end

            // End transaction; tie all transmit ports inactive
            i_tx_sop = 1'b0; // Should already be 0, but make sure
            i_tx_eop = 1'b0; // Should already be 0, but make sure
            i_tx_data = {CORE_DATA_WIDTH{1'b0}};

        end
        else
        begin
            $display ("%m : ERROR : Invalid Fmt_and_Type received; request aborted (time %t)", $time);
            inc_errors;
        end

        // If its a message, decode and display the type
        if (!$test$plusargs("pcie_traffic_msgs_off"))
        begin
            if ( (fmt_and_type[6:3] == 4'b01_10) |
                 (fmt_and_type[6:3] == 4'b11_10) )
            begin
                case ({last_dw_be, first_dw_be})

                    8'h00   : $display ("BFM Transmitted Message : Unlock");

                    8'h14   : $display ("BFM Transmitted Message : PM_Active_State_Nak");
                    8'h18   : $display ("BFM Transmitted Message : PM_PME");
                    8'h19   : $display ("BFM Transmitted Message : PME_Turn_Off");
                    8'h1b   : $display ("BFM Transmitted Message : PME_TO_Ack");

                    8'h20   : $display ("BFM Transmitted Message : Assert_INTA");
                    8'h21   : $display ("BFM Transmitted Message : Assert_INTB");
                    8'h22   : $display ("BFM Transmitted Message : Assert_INTC");
                    8'h23   : $display ("BFM Transmitted Message : Assert_INTD");

                    8'h24   : $display ("BFM Transmitted Message : Deassert_INTA");
                    8'h25   : $display ("BFM Transmitted Message : Deassert_INTB");
                    8'h26   : $display ("BFM Transmitted Message : Deassert_INTC");
                    8'h27   : $display ("BFM Transmitted Message : Deassert_INTD");

                    8'h30   : $display ("BFM Transmitted Message : ERR_COR");
                    8'h31   : $display ("BFM Transmitted Message : ERR_NONFATAL");
                    8'h33   : $display ("BFM Transmitted Message : ERR_FATAL");

                    8'h40   : $display ("BFM Transmitted Message : Ignored_Message - Only supported in PCIe 1.0a Specification");
                    8'h41   : $display ("BFM Transmitted Message : Ignored_Message - Only supported in PCIe 1.0a Specification");
                    8'h43   : $display ("BFM Transmitted Message : Ignored_Message - Only supported in PCIe 1.0a Specification");
                    8'h44   : $display ("BFM Transmitted Message : Ignored_Message - Only supported in PCIe 1.0a Specification");
                    8'h45   : $display ("BFM Transmitted Message : Ignored_Message - Only supported in PCIe 1.0a Specification");
                    8'h47   : $display ("BFM Transmitted Message : Ignored_Message - Only supported in PCIe 1.0a Specification");
                    8'h48   : $display ("BFM Transmitted Message : Ignored_Message - Only supported in PCIe 1.0a Specification");

                    8'h50   : $display ("BFM Transmitted Message : Set_Slot_Power_Limit");

                    8'h7e   : $display ("BFM Transmitted Message : Vendor_Defined_Type_0");
                    8'h7f   : $display ("BFM Transmitted Message : Vendor_Defined_Type_1");

                    default : $display ("BFM Transmitted Message : Unknown message code");

                endcase
            end
        end

        // If its a completion, then update for the transfer which just occurred
        if (cpl)
        begin
            cpl_bcount = cpl_bcount - ( ((length-1) << 2) + (4-cpl_lower_addr[1:0]) );
            cpl_lower_addr  = 0; // Always zero after 1st completion
            addr = addr + (length << 2);
            payload[9:0] = remain_length;       // return remaining word count
            if (single_completion_mode == 1'b1)
                continue_i = 0;
        end
    end // while

    // allow next call to this task to occur
    curr_prior = curr_prior + 1;

    if (req_c) // The request requires a completion response
    begin
        bfm_init_tag_req_time[my_tag]         = $time;          // Time of request
        bfm_init_tag_status[my_tag]           = 1'b1;           // Mark tag as active
        bfm_init_tag_cpl_status[my_tag]       = 3'h0;           // Default to success
        bfm_init_tag_expected_payload[my_tag] = payload;        // Save expected payload
        bfm_init_tag_check_payload[my_tag]    = check_data;     // Save whether to check completion data
        bfm_init_tag_expect_timeout[my_tag]   = expect_timeout; // Save expect timeout information
        bfm_init_tag_payload[my_tag]          = 0;              // Zero received data
        bfm_init_tag_fdw_be[my_tag]           = first_dw_be;    // Save off the First DWord BE
        bfm_init_tag_ldw_be[my_tag]           = last_dw_be;     // Save off the Last DWord BE
        bfm_init_tag_is_cfg[my_tag]           = cfg;            // Save whether it is a Cfg0 or Cfg 1 Write or Read Request
        bfm_init_tag_is_cfg_id[my_tag]        = addr[31:16];    // Save expected Completer ID (for Cfg transactions, this is task input addr[31:16])

        // Payload is undefined until overwritten by completion of the request with no_wait_for_cpl == 1'b0
        payload = {32768{1'bx}};

        // Save expected length of completions
        if ((io | cfg) & wr_rd_n) // I/O and Cfg Writes are a special case and expect 0 length completions
            bfm_init_tag_length[my_tag] = 0;
        else
            bfm_init_tag_length[my_tag] = (length == 0) ? 1024 : {1'b0, length}; // Check for special case of 0 == 1024

        bfm_init_tag_st_length[my_tag] = (length == 0) ? 1024 : {1'b0, length}; // Save off the starting length

         // Set check index to beginning
        bfm_init_tag_index[my_tag] = 0;

        // Wait for completion if transaction requires one
        if (no_wait_for_cpl == 1'b0) // set to override waiting
        begin
            while (bfm_init_tag_status[my_tag] == 1'b1)
            begin
                @(posedge clk);
                #1;
            end

            if (bfm_init_tag_is_cfg[my_tag]) // If it is a configuration request completion
            begin
                case (bfm_init_tag_cpl_status[my_tag]) // Check configuration completions status and convert data so proper behavior is synthesized
                    3'b000  : payload = bfm_init_tag_payload[my_tag]; // Successful; return received payload
                    3'b001  : payload = 'hFFFFFFFF;                   // Unsupported Request
                    3'b010  : payload = 'hFFFF0001;                   // CRS response
                    3'b100  : payload = 'hFFFFFFFF;                   // Completer Abort
                    default : payload = 'hFFFFFFFF;                   // Unsupported Request is assumed if an undefined completion status is received
                endcase
            end
            else
            begin
                // Copy completion data into payload for output from task
                payload = bfm_init_tag_payload[my_tag];
            end

            // If the transaction is not a Configuration Request, then write all unwritten payload data to X;
            //   Configuration writes and reads must return all 1s or all 0s so the PCIe Enumeration routine can
            //     recognize that a request to find a device was terminated with UR (no data was returned)
            if ( ~( (fmt_and_type == 7'b00_00100) |   // Cf0Rd
                    (fmt_and_type == 7'b10_00100) |   // Cf0Wr
                    (fmt_and_type == 7'b00_00101) |   // Cf1Rd
                    (fmt_and_type == 7'b10_00101) ) ) // Cf1Wr
            begin
                if (bfm_init_tag_index[my_tag] < 1024)
                begin : mark_bad_payload_data
                    integer      k;
                    for (k=bfm_init_tag_index[my_tag]*32; k<32768; k=k+1)
                        payload[k] = 1'bx;
                end
            end
        end
        else
            payload[4:0] = my_tag;
    end
end
endtask

function automatic [31:0] ecrc;

input   [31:0]      ci;
input   [31:0]      d;

reg     [31:0]      co_1;
reg     [31:0]      co_2;

begin
    co_1[0] = ci[0] ^ ci[6] ^ ci[9] ^ ci[10] ^ ci[12] ^ ci[16] ^ ci[24] ^ ci[25] ^ ci[26] ^ ci[28] ^ ci[29] ^ ci[30] ^ ci[31];
    co_1[1] = ci[0] ^ ci[1] ^ ci[6] ^ ci[7] ^ ci[9] ^ ci[11] ^ ci[12] ^ ci[13] ^ ci[16] ^ ci[17] ^ ci[24] ^ ci[27] ^ ci[28];
    co_1[2] = ci[0] ^ ci[1] ^ ci[2] ^ ci[6] ^ ci[7] ^ ci[8] ^ ci[9] ^ ci[13] ^ ci[14] ^ ci[16] ^ ci[17] ^ ci[18] ^ ci[24] ^ ci[26] ^ ci[30] ^ ci[31];
    co_1[3] = ci[1] ^ ci[2] ^ ci[3] ^ ci[7] ^ ci[8] ^ ci[9] ^ ci[10] ^ ci[14] ^ ci[15] ^ ci[17] ^ ci[18] ^ ci[19] ^ ci[25] ^ ci[27] ^ ci[31];
    co_1[4] = ci[0] ^ ci[2] ^ ci[3] ^ ci[4] ^ ci[6] ^ ci[8] ^ ci[11] ^ ci[12] ^ ci[15] ^ ci[18] ^ ci[19] ^ ci[20] ^ ci[24] ^ ci[25] ^ ci[29] ^ ci[30] ^ ci[31];
    co_1[5] = ci[0] ^ ci[1] ^ ci[3] ^ ci[4] ^ ci[5] ^ ci[6] ^ ci[7] ^ ci[10] ^ ci[13] ^ ci[19] ^ ci[20] ^ ci[21] ^ ci[24] ^ ci[28] ^ ci[29];
    co_1[6] = ci[1] ^ ci[2] ^ ci[4] ^ ci[5] ^ ci[6] ^ ci[7] ^ ci[8] ^ ci[11] ^ ci[14] ^ ci[20] ^ ci[21] ^ ci[22] ^ ci[25] ^ ci[29] ^ ci[30];
    co_1[7] = ci[0] ^ ci[2] ^ ci[3] ^ ci[5] ^ ci[7] ^ ci[8] ^ ci[10] ^ ci[15] ^ ci[16] ^ ci[21] ^ ci[22] ^ ci[23] ^ ci[24] ^ ci[25] ^ ci[28] ^ ci[29];
    co_1[8] = ci[0] ^ ci[1] ^ ci[3] ^ ci[4] ^ ci[8] ^ ci[10] ^ ci[11] ^ ci[12] ^ ci[17] ^ ci[22] ^ ci[23] ^ ci[28] ^ ci[31];
    co_1[9] = ci[1] ^ ci[2] ^ ci[4] ^ ci[5] ^ ci[9] ^ ci[11] ^ ci[12] ^ ci[13] ^ ci[18] ^ ci[23] ^ ci[24] ^ ci[29];
    co_1[10] = ci[0] ^ ci[2] ^ ci[3] ^ ci[5] ^ ci[9] ^ ci[13] ^ ci[14] ^ ci[16] ^ ci[19] ^ ci[26] ^ ci[28] ^ ci[29] ^ ci[31];
    co_1[11] = ci[0] ^ ci[1] ^ ci[3] ^ ci[4] ^ ci[9] ^ ci[12] ^ ci[14] ^ ci[15] ^ ci[16] ^ ci[17] ^ ci[20] ^ ci[24] ^ ci[25] ^ ci[26] ^ ci[27] ^ ci[28] ^ ci[31];
    co_1[12] = ci[0] ^ ci[1] ^ ci[2] ^ ci[4] ^ ci[5] ^ ci[6] ^ ci[9] ^ ci[12] ^ ci[13] ^ ci[15] ^ ci[17] ^ ci[18] ^ ci[21] ^ ci[24] ^ ci[27] ^ ci[30] ^ ci[31];
    co_1[13] = ci[1] ^ ci[2] ^ ci[3] ^ ci[5] ^ ci[6] ^ ci[7] ^ ci[10] ^ ci[13] ^ ci[14] ^ ci[16] ^ ci[18] ^ ci[19] ^ ci[22] ^ ci[25] ^ ci[28] ^ ci[31];
    co_1[14] = ci[2] ^ ci[3] ^ ci[4] ^ ci[6] ^ ci[7] ^ ci[8] ^ ci[11] ^ ci[14] ^ ci[15] ^ ci[17] ^ ci[19] ^ ci[20] ^ ci[23] ^ ci[26] ^ ci[29];
    co_1[15] = ci[3] ^ ci[4] ^ ci[5] ^ ci[7] ^ ci[8] ^ ci[9] ^ ci[12] ^ ci[15] ^ ci[16] ^ ci[18] ^ ci[20] ^ ci[21] ^ ci[24] ^ ci[27] ^ ci[30];
    co_1[16] = ci[0] ^ ci[4] ^ ci[5] ^ ci[8] ^ ci[12] ^ ci[13] ^ ci[17] ^ ci[19] ^ ci[21] ^ ci[22] ^ ci[24] ^ ci[26] ^ ci[29] ^ ci[30];
    co_1[17] = ci[1] ^ ci[5] ^ ci[6] ^ ci[9] ^ ci[13] ^ ci[14] ^ ci[18] ^ ci[20] ^ ci[22] ^ ci[23] ^ ci[25] ^ ci[27] ^ ci[30] ^ ci[31];
    co_1[18] = ci[2] ^ ci[6] ^ ci[7] ^ ci[10] ^ ci[14] ^ ci[15] ^ ci[19] ^ ci[21] ^ ci[23] ^ ci[24] ^ ci[26] ^ ci[28] ^ ci[31];
    co_1[19] = ci[3] ^ ci[7] ^ ci[8] ^ ci[11] ^ ci[15] ^ ci[16] ^ ci[20] ^ ci[22] ^ ci[24] ^ ci[25] ^ ci[27] ^ ci[29];
    co_1[20] = ci[4] ^ ci[8] ^ ci[9] ^ ci[12] ^ ci[16] ^ ci[17] ^ ci[21] ^ ci[23] ^ ci[25] ^ ci[26] ^ ci[28] ^ ci[30];
    co_1[21] = ci[5] ^ ci[9] ^ ci[10] ^ ci[13] ^ ci[17] ^ ci[18] ^ ci[22] ^ ci[24] ^ ci[26] ^ ci[27] ^ ci[29] ^ ci[31];
    co_1[22] = ci[0] ^ ci[9] ^ ci[11] ^ ci[12] ^ ci[14] ^ ci[16] ^ ci[18] ^ ci[19] ^ ci[23] ^ ci[24] ^ ci[26] ^ ci[27] ^ ci[29] ^ ci[31];
    co_1[23] = ci[0] ^ ci[1] ^ ci[6] ^ ci[9] ^ ci[13] ^ ci[15] ^ ci[16] ^ ci[17] ^ ci[19] ^ ci[20] ^ ci[26] ^ ci[27] ^ ci[29] ^ ci[31];
    co_1[24] = ci[1] ^ ci[2] ^ ci[7] ^ ci[10] ^ ci[14] ^ ci[16] ^ ci[17] ^ ci[18] ^ ci[20] ^ ci[21] ^ ci[27] ^ ci[28] ^ ci[30];
    co_1[25] = ci[2] ^ ci[3] ^ ci[8] ^ ci[11] ^ ci[15] ^ ci[17] ^ ci[18] ^ ci[19] ^ ci[21] ^ ci[22] ^ ci[28] ^ ci[29] ^ ci[31];
    co_1[26] = ci[0] ^ ci[3] ^ ci[4] ^ ci[6] ^ ci[10] ^ ci[18] ^ ci[19] ^ ci[20] ^ ci[22] ^ ci[23] ^ ci[24] ^ ci[25] ^ ci[26] ^ ci[28] ^ ci[31];
    co_1[27] = ci[1] ^ ci[4] ^ ci[5] ^ ci[7] ^ ci[11] ^ ci[19] ^ ci[20] ^ ci[21] ^ ci[23] ^ ci[24] ^ ci[25] ^ ci[26] ^ ci[27] ^ ci[29];
    co_1[28] = ci[2] ^ ci[5] ^ ci[6] ^ ci[8] ^ ci[12] ^ ci[20] ^ ci[21] ^ ci[22] ^ ci[24] ^ ci[25] ^ ci[26] ^ ci[27] ^ ci[28] ^ ci[30];
    co_1[29] = ci[3] ^ ci[6] ^ ci[7] ^ ci[9] ^ ci[13] ^ ci[21] ^ ci[22] ^ ci[23] ^ ci[25] ^ ci[26] ^ ci[27] ^ ci[28] ^ ci[29] ^ ci[31];
    co_1[30] = ci[4] ^ ci[7] ^ ci[8] ^ ci[10] ^ ci[14] ^ ci[22] ^ ci[23] ^ ci[24] ^ ci[26] ^ ci[27] ^ ci[28] ^ ci[29] ^ ci[30];
    co_1[31] = ci[5] ^ ci[8] ^ ci[9] ^ ci[11] ^ ci[15] ^ ci[23] ^ ci[24] ^ ci[25] ^ ci[27] ^ ci[28] ^ ci[29] ^ ci[30] ^ ci[31];

    co_2[0] = d[0] ^ d[1] ^ d[2] ^ d[3] ^ d[5] ^ d[6] ^ d[7] ^ d[15] ^ d[19] ^ d[21] ^ d[22] ^ d[25] ^ d[31];
    co_2[1] = d[3] ^ d[4] ^ d[7] ^ d[14] ^ d[15] ^ d[18] ^ d[19] ^ d[20] ^ d[22] ^ d[24] ^ d[25] ^ d[30] ^ d[31];
    co_2[2] = d[0] ^ d[1] ^ d[5] ^ d[7] ^ d[13] ^ d[14] ^ d[15] ^ d[17] ^ d[18] ^ d[22] ^ d[23] ^ d[24] ^ d[25] ^ d[29] ^ d[30] ^ d[31];
    co_2[3] = d[0] ^ d[4] ^ d[6] ^ d[12] ^ d[13] ^ d[14] ^ d[16] ^ d[17] ^ d[21] ^ d[22] ^ d[23] ^ d[24] ^ d[28] ^ d[29] ^ d[30];
    co_2[4] = d[0] ^ d[1] ^ d[2] ^ d[6] ^ d[7] ^ d[11] ^ d[12] ^ d[13] ^ d[16] ^ d[19] ^ d[20] ^ d[23] ^ d[25] ^ d[27] ^ d[28] ^ d[29] ^ d[31];
    co_2[5] = d[2] ^ d[3] ^ d[7] ^ d[10] ^ d[11] ^ d[12] ^ d[18] ^ d[21] ^ d[24] ^ d[25] ^ d[26] ^ d[27] ^ d[28] ^ d[30] ^ d[31];
    co_2[6] = d[1] ^ d[2] ^ d[6] ^ d[9] ^ d[10] ^ d[11] ^ d[17] ^ d[20] ^ d[23] ^ d[24] ^ d[25] ^ d[26] ^ d[27] ^ d[29] ^ d[30];
    co_2[7] = d[2] ^ d[3] ^ d[6] ^ d[7] ^ d[8] ^ d[9] ^ d[10] ^ d[15] ^ d[16] ^ d[21] ^ d[23] ^ d[24] ^ d[26] ^ d[28] ^ d[29] ^ d[31];
    co_2[8] = d[0] ^ d[3] ^ d[8] ^ d[9] ^ d[14] ^ d[19] ^ d[20] ^ d[21] ^ d[23] ^ d[27] ^ d[28] ^ d[30] ^ d[31];
    co_2[9] = d[2] ^ d[7] ^ d[8] ^ d[13] ^ d[18] ^ d[19] ^ d[20] ^ d[22] ^ d[26] ^ d[27] ^ d[29] ^ d[30];
    co_2[10] = d[0] ^ d[2] ^ d[3] ^ d[5] ^ d[12] ^ d[15] ^ d[17] ^ d[18] ^ d[22] ^ d[26] ^ d[28] ^ d[29] ^ d[31];
    co_2[11] = d[0] ^ d[3] ^ d[4] ^ d[5] ^ d[6] ^ d[7] ^ d[11] ^ d[14] ^ d[15] ^ d[16] ^ d[17] ^ d[19] ^ d[22] ^ d[27] ^ d[28] ^ d[30] ^ d[31];
    co_2[12] = d[0] ^ d[1] ^ d[4] ^ d[7] ^ d[10] ^ d[13] ^ d[14] ^ d[16] ^ d[18] ^ d[19] ^ d[22] ^ d[25] ^ d[26] ^ d[27] ^ d[29] ^ d[30] ^ d[31];
    co_2[13] = d[0] ^ d[3] ^ d[6] ^ d[9] ^ d[12] ^ d[13] ^ d[15] ^ d[17] ^ d[18] ^ d[21] ^ d[24] ^ d[25] ^ d[26] ^ d[28] ^ d[29] ^ d[30];
    co_2[14] = d[2] ^ d[5] ^ d[8] ^ d[11] ^ d[12] ^ d[14] ^ d[16] ^ d[17] ^ d[20] ^ d[23] ^ d[24] ^ d[25] ^ d[27] ^ d[28] ^ d[29];
    co_2[15] = d[1] ^ d[4] ^ d[7] ^ d[10] ^ d[11] ^ d[13] ^ d[15] ^ d[16] ^ d[19] ^ d[22] ^ d[23] ^ d[24] ^ d[26] ^ d[27] ^ d[28];
    co_2[16] = d[1] ^ d[2] ^ d[5] ^ d[7] ^ d[9] ^ d[10] ^ d[12] ^ d[14] ^ d[18] ^ d[19] ^ d[23] ^ d[26] ^ d[27] ^ d[31];
    co_2[17] = d[0] ^ d[1] ^ d[4] ^ d[6] ^ d[8] ^ d[9] ^ d[11] ^ d[13] ^ d[17] ^ d[18] ^ d[22] ^ d[25] ^ d[26] ^ d[30];
    co_2[18] = d[0] ^ d[3] ^ d[5] ^ d[7] ^ d[8] ^ d[10] ^ d[12] ^ d[16] ^ d[17] ^ d[21] ^ d[24] ^ d[25] ^ d[29];
    co_2[19] = d[2] ^ d[4] ^ d[6] ^ d[7] ^ d[9] ^ d[11] ^ d[15] ^ d[16] ^ d[20] ^ d[23] ^ d[24] ^ d[28];
    co_2[20] = d[1] ^ d[3] ^ d[5] ^ d[6] ^ d[8] ^ d[10] ^ d[14] ^ d[15] ^ d[19] ^ d[22] ^ d[23] ^ d[27];
    co_2[21] = d[0] ^ d[2] ^ d[4] ^ d[5] ^ d[7] ^ d[9] ^ d[13] ^ d[14] ^ d[18] ^ d[21] ^ d[22] ^ d[26];
    co_2[22] = d[0] ^ d[2] ^ d[4] ^ d[5] ^ d[7] ^ d[8] ^ d[12] ^ d[13] ^ d[15] ^ d[17] ^ d[19] ^ d[20] ^ d[22] ^ d[31];
    co_2[23] = d[0] ^ d[2] ^ d[4] ^ d[5] ^ d[11] ^ d[12] ^ d[14] ^ d[15] ^ d[16] ^ d[18] ^ d[22] ^ d[25] ^ d[30] ^ d[31];
    co_2[24] = d[1] ^ d[3] ^ d[4] ^ d[10] ^ d[11] ^ d[13] ^ d[14] ^ d[15] ^ d[17] ^ d[21] ^ d[24] ^ d[29] ^ d[30];
    co_2[25] = d[0] ^ d[2] ^ d[3] ^ d[9] ^ d[10] ^ d[12] ^ d[13] ^ d[14] ^ d[16] ^ d[20] ^ d[23] ^ d[28] ^ d[29];
    co_2[26] = d[0] ^ d[3] ^ d[5] ^ d[6] ^ d[7] ^ d[8] ^ d[9] ^ d[11] ^ d[12] ^ d[13] ^ d[21] ^ d[25] ^ d[27] ^ d[28] ^ d[31];
    co_2[27] = d[2] ^ d[4] ^ d[5] ^ d[6] ^ d[7] ^ d[8] ^ d[10] ^ d[11] ^ d[12] ^ d[20] ^ d[24] ^ d[26] ^ d[27] ^ d[30];
    co_2[28] = d[1] ^ d[3] ^ d[4] ^ d[5] ^ d[6] ^ d[7] ^ d[9] ^ d[10] ^ d[11] ^ d[19] ^ d[23] ^ d[25] ^ d[26] ^ d[29];
    co_2[29] = d[0] ^ d[2] ^ d[3] ^ d[4] ^ d[5] ^ d[6] ^ d[8] ^ d[9] ^ d[10] ^ d[18] ^ d[22] ^ d[24] ^ d[25] ^ d[28];
    co_2[30] = d[1] ^ d[2] ^ d[3] ^ d[4] ^ d[5] ^ d[7] ^ d[8] ^ d[9] ^ d[17] ^ d[21] ^ d[23] ^ d[24] ^ d[27];
    co_2[31] = d[0] ^ d[1] ^ d[2] ^ d[3] ^ d[4] ^ d[6] ^ d[7] ^ d[8] ^ d[16] ^ d[20] ^ d[22] ^ d[23] ^ d[26];

    ecrc = co_1 ^ co_2;
end
endfunction

// --------------------------------------
// task: xferb
//   Byte-length general purpose transfer

// Similar to task xfer except takes byte length and computes first and last DWORD byte enables for the user

task automatic xferb;

input   [2:0]       tc;             // Traffic Class
input   [2:0]       attr;           // Attribute Field
input   [6:0]       fmt_and_type;   // {Fmt, Type} of command to use
input   [11:0]      blength;        // Length of the burst in bytes; 0x1 == 1, 0x2 == 2, ..., 0xFFF = 4095, 0x0 = 4096
input   [63:0]      addr;           // Starting address for the command
input   [15:0]      req_id;         // Requester ID (for completions)
input   [7:0]       tag;            // Requester tag (for completions)
inout   [32767:0]   payload;        // For writes: payload data; for reads: input==expected payload data to receive with completions; output==data actually received
input               check_data;     // For read requests only, set to check completion data against payload, clear to not check
input               no_wait_for_cpl;    // For non-posted only, clear to wait for all completions to complete; set to not wait

reg     [3:0]       first_dw_be;
reg     [3:0]       last_dw_be;
reg     [11:0]      blength_minus1;
reg     [63:0]      end_addr;
reg     [9:0]       length;

begin
    // Compute first and last DWORD byte enables
    case (addr[1:0])
        2'b00 : first_dw_be = 4'hf;
        2'b01 : first_dw_be = 4'he;
        2'b10 : first_dw_be = 4'hc;
        2'b11 : first_dw_be = 4'h8;
    endcase

    blength_minus1 = blength - 1;
    end_addr       = addr + blength_minus1;

    case (end_addr[1:0])
        2'b00 : last_dw_be = 4'h1;
        2'b01 : last_dw_be = 4'h3;
        2'b10 : last_dw_be = 4'h7;
        2'b11 : last_dw_be = 4'hf;
    endcase

    if ((addr[1:0] + blength_minus1) < 4) // single DWORD transfer
    begin
        first_dw_be = first_dw_be & last_dw_be;
        last_dw_be  = 4'h0;
    end

    // Compute length
    length = ((blength_minus1 + addr[1:0]) >> 2) + 1;

    // Do transfer
    xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr, req_id, tag, payload, check_data, no_wait_for_cpl, 1'b0, 1'b0);
end
endtask



// -------------------------------------------------------
// task: hot_reset_dut_hierarchy
//   Resets a DUT hierarchy by writing the Secondary Bus Reset Configuration Register of a downstream PCIe port

task automatic hot_reset_dut_hierarchy;

    // The following inputs are the device that will initiate the reset of the PCIe hierarchy; the ds_port device
    //   will not be reset, but all devices dowstream of this device's downstream port will be reset
    input   [15:0]      ds_port_bdf;            // Downstream Port: Bus, Device, Function #
    input   [11:0]      ds_port_pcie_cap_addr;  // Downstream Port: Start of PCIe Capability address offset

    input   [31:0]      reset_active_us;        // uS to hold reset asserted
    input   [31:0]      delay_access_us;        // uS to wait after reset de-assertion before returning from the task (emulate delay to first access time)

    input               exit_on_dl_link_up;     // 1 - Exit either after delay_access_us or when DUT Data Link Layer is back up
                                                // When 1, the DS Port is checked for Data Link Layer Link Active Reporting capability; if not present, this input is ignored
                                                //   since DS port lacks the necessary capability to report Data Link Layer Link Active Status

    reg     [31:0]      read_data;
    reg     [31:0]      write_data;
    reg     [31:0]      ctr;
    reg                 dll_active_rep_cap;
    reg                 exit_while;

    begin
        // Read-Modify-Write to set Secondary Bus Reset
        //         dut_bdf,     addr,    be,   rd_data
        cfg_rd_bdf(ds_port_bdf, 12'h03c, 4'hf, read_data);
        write_data = read_data | 32'h00400000;
        //         dut_bdf,     addr,    be,   wr_data
        cfg_wr_bdf(ds_port_bdf, 12'h03c, 4'h4, write_data);

        // Wait until BFM reaches Hot Reset
        while (lts_state != 6'h7)
            @(posedge clk);

        // Hold reset condition for specified time
        ctr = reset_active_us;
        while (ctr != 32'd0)
        begin
            #1000000; // 1uS
            ctr = ctr - 32'd1;
        end

        // Read-Modify-Write to clear Secondary Bus Reset
        //         dut_bdf,     addr,    be,   rd_data
        cfg_rd_bdf(ds_port_bdf, 12'h03c, 4'hf, read_data);
        write_data = read_data & 32'hffbfffff;
        //         dut_bdf,     addr,    be,   wr_data
        cfg_wr_bdf(ds_port_bdf, 12'h03c, 4'h4, write_data);

        if ((delay_access_us != 32'h0) & exit_on_dl_link_up)
        begin
            // Read DS Port: Link Capabilities to determine whether DS Port supports Data Link Layer Link Active Reporting
            cfg_rd_bdf(ds_port_bdf, (ds_port_pcie_cap_addr + 12'h00C), 4'hf, read_data);
            dll_active_rep_cap = read_data[20];
        end
        else
        begin
            dll_active_rep_cap = 1'b0;
        end

        ctr = delay_access_us;
        exit_while = 1'b0;
        while ((ctr != 32'd0) & (exit_while == 1'b0))
        begin
            if (exit_on_dl_link_up & dll_active_rep_cap) // Wait for dl_link_up before continuing
            begin
                fork
                    begin : poll
                        // Poll Data Link Layer Active every 1uS
                        cfg_rd_bdf(ds_port_bdf, (ds_port_pcie_cap_addr + 12'h010), 4'hf, read_data);

                        if (read_data[29])
                            exit_while = 1'b1;
                    end
                    begin : fixed_delay
                        #1000000; // 1uS
                    end
                join
            end
            else
            begin
                #1000000; // 1uS
            end
            ctr = ctr - 32'd1;
        end
    end
endtask



// -------------------------------------------------------
// task: hot_reset_dut_wait_dll_active
//   Resets a DUT hierarchy and waits for Data Link Layer to become active (or 400 uS) before returning

task automatic hot_reset_dut_wait_dll_active;
    input   [15:0]      ds_port_bdf;            // Downstream Port: Bus, Device, Function #
    input   [11:0]      ds_port_pcie_cap_addr;  // Downstream Port: Start of PCIe Capability address offset

    begin
        //                       ds_port_bdf, ds_port_pcie_cap_addr, reset_active_us, delay_access_us, exit_on_dl_link_up
        hot_reset_dut_hierarchy (ds_port_bdf, ds_port_pcie_cap_addr, 32'h1,           32'h400,         1'b1);
    end
endtask



// -------------------------------------------------------
// task: hot_reset_dut
//   Resets a DUT hierarchy and returns 1 uS after releasing reset

task automatic hot_reset_dut;
    input   [15:0]      ds_port_bdf;            // Downstream Port: Bus, Device, Function #
    input   [11:0]      ds_port_pcie_cap_addr;  // Downstream Port: Start of PCIe Capability address offset

    begin
        //                       ds_port_bdf, ds_port_pcie_cap_addr, reset_active_us, delay_access_us, exit_on_dl_link_up
        hot_reset_dut_hierarchy (ds_port_bdf, ds_port_pcie_cap_addr, 32'h1,           32'h1,           1'b0);
    end
endtask


// -------------------------------------------------------
// task: cfg_rd_bdf
//   Issues a single configuration read to a device
//   Note: Waits for completion before returning from task
//   Uses Type 0 if Target Bus == 0, else Type 1 (Assumes Host Bus is Bus 0)

task automatic cfg_rd_bdf;

input   [15:0]      dut_bdf;    // DUT ID: {Bus[7:0], Device[4:0], Function[2:0]}
input   [11:0]      addr;       // Cfg Register Byte Address; DWORD resolution; addr[1:0] ignored
input   [3:0]       be;         // Byte enables
output  [31:0]      rd_data;    // Return data that was read

reg     [63:0]      pkt_addr;
reg     [6:0]       fmt_and_type;
reg     [7:0]       tag;

reg     [32767:0]   payload;

begin
    payload = 0;

    pkt_addr = {32'h0, dut_bdf[15:0], 4'h0, addr[11:8], addr[7:2], 2'b00};

    if (dut_bdf[15:8] == cfg0_bus_num)
        fmt_and_type = 7'b00_00100; // Cfg Rd Type 0
    else
        fmt_and_type = 7'b00_00101; // Cfg Rd Type 1

    // xfer (tc,     attr, fmt_and_type, length, first_dw_be, last_dw_be, addr,     req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
    xfer    (3'b000, 3'h0, fmt_and_type, 1,      be,          4'h0,       pkt_addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0,           1'b0);

    rd_data = payload[31:0];
end
endtask

task automatic cfg_rd_bdf_null;

input   [15:0]      dut_bdf;    // DUT ID: {Bus[7:0], Device[4:0], Function[2:0]}
input   [11:0]      addr;       // Cfg Register Byte Address; DWORD resolution; addr[1:0] ignored
input   [3:0]       be;         // Byte enables
output  [31:0]      rd_data;    // Return data that was read

reg     [63:0]      pkt_addr;
reg     [6:0]       fmt_and_type;
reg     [7:0]       tag;

reg     [32767:0]   payload;

begin
    payload = 0;

    pkt_addr = {32'h0, dut_bdf[15:0], 4'h0, addr[11:8], addr[7:2], 2'b00};

    if (dut_bdf[15:8] == cfg0_bus_num)
        fmt_and_type = 7'b00_00100; // Cfg Rd Type 0
    else
        fmt_and_type = 7'b00_00101; // Cfg Rd Type 1

    // xfer (tc,     attr, fmt_and_type, length, first_dw_be, last_dw_be, addr,     req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
    xfer    (3'b000, 3'h0, fmt_and_type, 1,      be,          4'h0,       pkt_addr, 0,      0,   payload, 1'b0,       1'b1,            1'b0,           1'b0);

    rd_data = 32'h0;
end
endtask


// -------------------------------------------------------
// task: cfg_wr_bdf
//   Issues a single configuration write to a device
//   Note: Waits for completion before returning from task
//   Uses Type 0 if Target Bus == 0, else Type 1 (Assumes Host Bus is Bus 0)

task automatic cfg_wr_bdf;

input   [15:0]      dut_bdf;    // DUT ID: {Bus[7:0], Device[4:0], Function[2:0]}
input   [11:0]      addr;       // Cfg Register Byte Address; DWORD resolution; addr[1:0] ignored
input   [3:0]       be;         // Byte enables
input   [31:0]      wr_data;    // Return data that was read

reg     [63:0]      pkt_addr;
reg     [6:0]       fmt_and_type;

reg     [32767:0]   payload;

begin
    payload = wr_data;

    pkt_addr = {32'h0, dut_bdf[15:0], 4'h0, addr[11:8], addr[7:2], 2'b00};

    if (dut_bdf[15:8] == cfg0_bus_num)
        fmt_and_type = 7'b10_00100; // Cfg Wr Type 0
    else
        fmt_and_type = 7'b10_00101; // Cfg Wr Type 1

    // xfer (tc,     attr, fmt_and_type, length, first_dw_be, last_dw_be, addr,     req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
    xfer    (3'b000, 3'h0, fmt_and_type, 1,      be,          4'h0,       pkt_addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0,           1'b0);
end
endtask

task automatic cfg_wr_bdf_null;

input   [15:0]      dut_bdf;    // DUT ID: {Bus[7:0], Device[4:0], Function[2:0]}
input   [11:0]      addr;       // Cfg Register Byte Address; DWORD resolution; addr[1:0] ignored
input   [3:0]       be;         // Byte enables
input   [31:0]      wr_data;    // Return data that was read

reg     [63:0]      pkt_addr;
reg     [6:0]       fmt_and_type;

reg     [32767:0]   payload;

begin
    payload = wr_data;

    pkt_addr = {32'h0, dut_bdf[15:0], 4'h0, addr[11:8], addr[7:2], 2'b00};

    if (dut_bdf[15:8] == cfg0_bus_num)
        fmt_and_type = 7'b10_00100; // Cfg Wr Type 0
    else
        fmt_and_type = 7'b10_00101; // Cfg Wr Type 1

    // xfer (tc,     attr, fmt_and_type, length, first_dw_be, last_dw_be, addr,     req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
    xfer    (3'b000, 3'h0, fmt_and_type, 1,      be,          4'h0,       pkt_addr, 0,      0,   payload, 1'b0,       1'b1,            1'b0,           1'b0);
end
endtask

// ---------------------------------------------------
// task: transmit_msg
//   Use to transmit Messages (with or wihout payload)

task automatic transmit_msg;

input   [2:0]       tc;             // Traffic Class
input               msgd_msg_n;     // 1 == msgd, 0 == msg
input   [7:0]       msg_len;
input   [2:0]       msg_routing;
input   [7:0]       msg_code;
input   [7:0]       msg_tag;
input   [63:0]      msg_info;       // Placed into packet in the same format as ADDR[63:0]
input   [32767:0]   msg_data;       // Only used if type is msgd

reg     [32767:0]   payload;
reg     [6:0]       fmt_and_type;
reg     [9:0]       length;


begin
    payload      = 0;
    fmt_and_type = msgd_msg_n ? {4'b11_10, msg_routing} : {4'b01_10, msg_routing};
    length       = msgd_msg_n ? msg_len                 : 10'h0;

    if (msgd_msg_n)
        payload = msg_data;

    // xfer (tc, attr, fmt_and_type, length, first_dw_be,   last_dw_be,    addr,           req_id, tag,     payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
    xfer    (tc, 3'h0, fmt_and_type, length, msg_code[3:0], msg_code[7:4], msg_info[63:0], 0,      msg_tag, payload, 1'b0,       1'b0,            1'b0,           1'b0);
end
endtask



// -----------------------------------------
// task: mem_write_dword_addr32
//   Issue 32-bit address single dword write

task automatic mem_write_dword_addr32;

input   [2:0]       tc;      // Traffic Class
input   [31:0]      addr;    // DWORD Address
input   [31:0]      data;    // Data
input   [3:0]       be;      // Byte enables

reg     [32767:0]   payload;

begin
    payload = data;

    // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
    xfer    (tc, 3'h0, 7'b10_00000,  1,      be,          4'h0,       addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0,           1'b0);
end
endtask



// ------------------------------------------------------------------------------
// task: mem_read_dword_addr32
//   Issue 32-bit address single dword read and optionally compare to expect_data
//   Note: Waits for completion before returning from task

task automatic mem_read_dword_addr32;

input   [2:0]       tc;          // Traffic Class
input   [31:0]      addr;        // DWORD Address
input   [31:0]      expect_data; // Expect data
input   [3:0]       be;          // bytes to check
output  [31:0]      read_data;   // Read data

reg     [32767:0]   payload;

begin
    payload = expect_data;

    // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
    xfer    (tc, 3'h0, 7'b00_00000,  1,      4'hf,        4'h0,       addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0,           1'b0);

    if ((payload[ 7: 0] !== expect_data[ 7: 0] && be[0] == 1'b1) ||
        (payload[15: 8] !== expect_data[15: 8] && be[1] == 1'b1) ||
        (payload[23:16] !== expect_data[23:16] && be[2] == 1'b1) ||
        (payload[31:24] !== expect_data[31:24] && be[3] == 1'b1) )
    begin
        $display ("%m : ERROR : Read data does not match expected : Address == 0x%x : Expected Data == 0x%x, Read Data == 0x%x (time %t)", addr, expect_data, payload[31:0], $time);
        inc_errors;
    end

    read_data = payload[31:0];
end
endtask



// ---------------------------------------------
// task: mem_write_dword
//   Issues 32/64-bit address single dword write

task automatic mem_write_dword;

input   [2:0]       tc;         // Traffic Class
input   [63:0]      addr;       // DWORD Address
input   [31:0]      data;       // Data
input   [3:0]       be;         // Byte enables

reg     [32767:0]   payload;    // Payload

begin
    payload = data;

    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b11_00000,  1,      be,          4'h0,       addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0,           1'b0);
    end
    else
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b10_00000,  1,      be,          4'h0,       addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0,           1'b0);
    end
end
endtask


// ---------------------------------------------
// task: mem_write_dword_poison
//   Issues 32/64-bit address single dword write

task automatic mem_write_dword_poison;

input   [2:0]       tc;         // Traffic Class
input   [63:0]      addr;       // DWORD Address
input   [31:0]      data;       // Data
input   [3:0]       be;         // Byte enables

reg     [32767:0]   payload;    // Payload

begin
    payload = data;

    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        // xfer        (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer_poison    (tc, 3'h0, 7'b11_00000,  1,      be,          4'h0,       addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0,           1'b0);
    end
    else
    begin
        // xfer        (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer_poison    (tc, 3'h0, 7'b10_00000,  1,      be,          4'h0,       addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0,           1'b0);
    end
end
endtask


// ---------------------------------------------
// task: mem_write_dword_ecrc
//   Issues 32/64-bit address single dword write

task automatic mem_write_dword_ecrc;

input   [2:0]       tc;         // Traffic Class
input   [63:0]      addr;       // DWORD Address
input   [31:0]      data;       // Data
input   [3:0]       be;         // Byte enables

reg     [32767:0]   payload;    // Payload

begin
    payload = data;

    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        // xfer      (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, payload_for_cpl
        xfer_ecrc    (tc, 3'h0, 7'b11_00000,  1,      be,          4'h0,       addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0);
    end
    else
    begin
        // xfer      (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, payload_for_cpl
        xfer_ecrc    (tc, 3'h0, 7'b10_00000,  1,      be,          4'h0,       addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0);
    end
end
endtask



// ------------------------------------------
// task: io_write_dword
//   Issues 32-bit address single dword write

task automatic io_write_dword;

input   [2:0]       tc;      // Traffic Class
input   [63:0]      addr;    // DWORD Address
input   [31:0]      data;    // Data
input   [3:0]       be;      // Byte enables

reg     [32767:0]   payload;

begin
    payload = data;

    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        $display ("%m: ERROR : Illegal call with > 32-bit address (time %t)", $time);
        inc_errors;
    end
    else
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload,  check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b10_00010,  1,      be,          4'h0,       addr, 0,      0,   payload,  1'b0,       1'b0,            1'b0,           1'b0);
    end
end
endtask



// ---------------------------------------------------------------------------------
// task: mem_read_dword
//   Issue 32/64-bit address single dword read and optionally compare to expect_data
//   Note: Waits for completion before returning from task

task automatic mem_read_dword;

input   [2:0]       tc;             // Traffic Class
input   [63:0]      addr;           // DWORD Address
input   [31:0]      expect_data;    // Expect data
input   [3:0]       be;             // bytes to check
output  [31:0]      read_data;      // Read data

reg     [32767:0]   payload;

begin
    payload = expect_data;

    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b01_00000,  1,      4'hf,        4'h0,       addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0,           1'b0);
    end
    else
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b00_00000,  1,      4'hf,        4'h0,       addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0,           1'b0);
    end

    if ((payload[ 7: 0] !== expect_data[ 7: 0] && be[0] == 1'b1) ||
        (payload[15: 8] !== expect_data[15: 8] && be[1] == 1'b1) ||
        (payload[23:16] !== expect_data[23:16] && be[2] == 1'b1) ||
        (payload[31:24] !== expect_data[31:24] && be[3] == 1'b1) )
    begin
        $display ("%m : ERROR : Read data does not match expected : Address == 0x%x : Expected Data == 0x%x, Read Data == 0x%x (time %t)", addr, expect_data, payload[31:0], $time);
        inc_errors;
    end

    read_data = payload[31:0];
end
endtask

// ---------------------------------------------------------------------------------
// task: mem_read_dword_attr
//   Issue 32/64-bit address single dword read and optionally compare to expect_data
//   Note: Waits for completion before returning from task
//   Allows the attributes to be specified

task automatic mem_read_dword_attr;

input   [2:0]       tc;             // Traffic Class
input   [63:0]      addr;           // DWORD Address
input   [31:0]      expect_data;    // Expect data
input   [3:0]       be;             // bytes to check
input   [2:0]       attr;           // attributes
output  [31:0]      read_data;      // Read data

reg     [32767:0]   payload;

begin
    payload = expect_data;

    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, attr, 7'b01_00000,  1,      4'hf,        4'h0,       addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0,           1'b0);
    end
    else
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, attr, 7'b00_00000,  1,      4'hf,        4'h0,       addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0,           1'b0);
    end

    if ((payload[ 7: 0] !== expect_data[ 7: 0] && be[0] == 1'b1) ||
        (payload[15: 8] !== expect_data[15: 8] && be[1] == 1'b1) ||
        (payload[23:16] !== expect_data[23:16] && be[2] == 1'b1) ||
        (payload[31:24] !== expect_data[31:24] && be[3] == 1'b1) )
    begin
        $display ("%m : ERROR : Read data does not match expected : Address == 0x%x : Expected Data == 0x%x, Read Data == 0x%x (time %t)", addr, expect_data, payload[31:0], $time);
        inc_errors;
    end

    read_data = payload[31:0];
end
endtask


// ---------------------------------------------------------------------------------
// task: mem_read_dword_ro
//   Issue 32/64-bit address single dword read and optionally compare to expect_data
//   Note: Waits for completion before returning from task
//   Requests Relaxed Ordering on the read and its completion

task automatic mem_read_dword_ro;

input   [2:0]       tc;             // Traffic Class
input   [63:0]      addr;           // DWORD Address
input   [31:0]      expect_data;    // Expect data
input   [3:0]       be;             // bytes to check
output  [31:0]      read_data;      // Read data

reg     [32767:0]   payload;

begin
    payload = expect_data;

    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h2, 7'b01_00000,  1,      4'hf,        4'h0,       addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0,           1'b0);
    end
    else
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h2, 7'b00_00000,  1,      4'hf,        4'h0,       addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0,           1'b0);
    end

    if ((payload[ 7: 0] !== expect_data[ 7: 0] && be[0] == 1'b1) ||
        (payload[15: 8] !== expect_data[15: 8] && be[1] == 1'b1) ||
        (payload[23:16] !== expect_data[23:16] && be[2] == 1'b1) ||
        (payload[31:24] !== expect_data[31:24] && be[3] == 1'b1) )
    begin
        $display ("%m : ERROR : Read data does not match expected : Address == 0x%x : Expected Data == 0x%x, Read Data == 0x%x (time %t)", addr, expect_data, payload[31:0], $time);
        inc_errors;
    end

    read_data = payload[31:0];
end
endtask

// ---------------------------------------------------------------------------------
// task: mem_read_dword_timeout
//   Issue 32/64-bit address single dword read
//   Note: expected to abort and timeout

task automatic mem_read_dword_timeout;

input   [2:0]       tc;             // Traffic Class
input   [63:0]      addr;           // DWORD Address

reg     [32767:0]   payload;

begin
    payload = 0;

    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b01_00000,  1,      4'hf,        4'h0,       addr, 0,      0,   payload, 1'b0,       1'b1,            1'b1,           1'b0);
    end
    else
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b00_00000,  1,      4'hf,        4'h0,       addr, 0,      0,   payload, 1'b0,       1'b1,            1'b1,           1'b0);
    end
end
endtask

// ---------------------------------------------------------------------------------
// task: mem_read_dword_poison
//   Issue 32/64-bit address single dword read
//   Note: Read should be dropped aborted (completer-abort)

task automatic mem_read_dword_poison;

input   [2:0]       tc;             // Traffic Class
input   [63:0]      addr;           // DWORD Address
input   [31:0]      expect_data;    // Expect data
input   [3:0]       be;             // bytes to check
input               expect_timeout; // 1 if the transaction is expected to timeout
output  [31:0]      read_data;      // Read data

reg     [32767:0]   payload;

begin
    payload = expect_data;

    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        // xfer     (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer_poison (tc, 3'h0, 7'b01_00000,  1,      4'hf,        4'h0,       addr, 0,      0,   payload, 1'b0,   expect_timeout,  expect_timeout, 1'b0);
    end
    else
    begin
        // xfer     (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer_poison (tc, 3'h0, 7'b00_00000,  1,      4'hf,        4'h0,       addr, 0,      0,   payload, 1'b0,   expect_timeout,  expect_timeout, 1'b0);
    end

    if ((payload[ 7: 0] !== expect_data[ 7: 0] && be[0] == 1'b1) ||
        (payload[15: 8] !== expect_data[15: 8] && be[1] == 1'b1) ||
        (payload[23:16] !== expect_data[23:16] && be[2] == 1'b1) ||
        (payload[31:24] !== expect_data[31:24] && be[3] == 1'b1) )
    begin
        $display ("%m : ERROR : Read data does not match expected : Address == 0x%x : Expected Data == 0x%x, Read Data == 0x%x (time %t)", addr, expect_data, payload[31:0], $time);
        inc_errors;
    end

    read_data = payload[31:0];
end
endtask


// ---------------------------------------------------------------------------------
// task: mem_read_dword_fast
//   Issue 32/64-bit address single dword read and optionally compare to expect_data
//   Note: Does not wait for completion before returning from task

task automatic mem_read_dword_fast;

input   [2:0]       tc;             // Traffic Class
input   [63:0]      addr;           // DWORD Address
input   [31:0]      expect_data;    // Expected data
input               check_data;     // Set to check data against expect_data

reg     [32767:0]   payload;

begin
    payload = expect_data;

    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b01_00000,  1,      4'hf,        4'h0,       addr, 0,      0,   payload, check_data, 1'b1,            1'b0,           1'b0);
    end
    else
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b00_00000,  1,      4'hf,        4'h0,       addr, 0,      0,   payload, check_data, 1'b1,            1'b0,           1'b0);
    end
end
endtask


// ---------------------------------------------------------------------------------
// task: mem_read_dword_ro_fast
//   Issue 32/64-bit address single dword read and optionally compare to expect_data
//   Note: Does not wait for completion before returning from task

task automatic mem_read_dword_ro_fast;

input   [2:0]       tc;             // Traffic Class
input   [63:0]      addr;           // DWORD Address
input   [31:0]      expect_data;    // Expected data
input               check_data;     // Set to check data against expect_data

reg     [32767:0]   payload;

begin
    payload = expect_data;

    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h2, 7'b01_00000,  1,      4'hf,        4'h0,       addr, 0,      0,   payload, check_data, 1'b1,            1'b0,           1'b0);
    end
    else
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h2, 7'b00_00000,  1,      4'hf,        4'h0,       addr, 0,      0,   payload, check_data, 1'b1,            1'b0,           1'b0);
    end
end
endtask


// ---------------------------------------------------------------------------------
// task: mem_read_dword_fast_wait
//   Issue 32/64-bit address single dword read and optionally compare to expect_data
//   Note: Does not wait for completion before returning from task

task automatic mem_read_dword_fast_wait;

input   [2:0]       tc;             // Traffic Class
input   [63:0]      addr;           // DWORD Address
input   [31:0]      expect_data;    // Expected data
input               check_data;     // Set to check data against expect_data

reg     [32767:0]   payload;
reg     [4:0]       my_tag;

begin
    payload = expect_data;

    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b01_00000,  1,      4'hf,        4'h0,       addr, 0,      0,   payload, check_data, 1'b1,            1'b0,           1'b0);
    end
    else
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b00_00000,  1,      4'hf,        4'h0,       addr, 0,      0,   payload, check_data, 1'b1,            1'b0,           1'b0);
    end
    my_tag = payload[4:0];
    while (bfm_init_tag_status[my_tag] == 1'b1)
    begin
        @(posedge clk);
        #1;
    end

end
endtask



// ------------------------------------------------------------------------------
// task: io_read_dword
//   Issue 32-bit address single dword read and optionally compare to expect_data
//   Note: Waits for completion before returning from task

task automatic io_read_dword;

input   [2:0]       tc;             // Traffic Class
input   [63:0]      addr;           // DWORD Address
input   [31:0]      expect_data;    // Expect data
input   [3:0]       be;             // bytes to check
output  [31:0]      read_data;      // Read data

reg     [32767:0]   payload;

begin
    payload = expect_data;

    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        $display ("%m: ERROR : Illegal call with > 32-bit address (time %t)", $time);
        inc_errors;
    end
    else
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b00_00010,  1,      4'hf,        4'h0,       addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0,           1'b0);
    end

    if ((payload[ 7: 0] !== expect_data[ 7: 0] && be[0] == 1'b1) ||
        (payload[15: 8] !== expect_data[15: 8] && be[1] == 1'b1) ||
        (payload[23:16] !== expect_data[23:16] && be[2] == 1'b1) ||
        (payload[31:24] !== expect_data[31:24] && be[3] == 1'b1) )
    begin
        $display ("%m : ERROR : Read data does not match expected : Address == 0x%x : Expected Data == 0x%x, Read Data == 0x%x (time %t)", addr, expect_data, payload[31:0], $time);
        inc_errors;
    end

    read_data = payload[31:0];
end
endtask



// --------------------------
// task: mem_write_burst
//   Issue burst memory write

task automatic mem_write_burst;

input   [2:0]       tc;                 // Traffic Class
input   [63:0]      addr;               // Starting address
input   [9:0]       length;             // length in dwords
input   [3:0]       first_dw_be;        // first dword byte enable
input   [3:0]       last_dw_be;         // last dword byte enable
input   [32767:0]   payload;            // Data payload to write

begin
    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b11_00000,  length, first_dw_be, last_dw_be, addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0,           1'b0);
    end
    else
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b10_00000,  length, first_dw_be, last_dw_be, addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0,           1'b0);
    end
end
endtask



// ---------------------------------------------------------------
// task: mem_write_burst_pattern
//   Issue burst memory write using a data pattern for the payload

task automatic mem_write_burst_pattern;

input   [2:0]       tc;                 // Traffic Class
input   [63:0]      addr;               // Starting address
input   [9:0]       length;             // length in dwords
input   [3:0]       first_dw_be;        // first dword byte enable
input   [3:0]       last_dw_be;         // last dword byte enable
input   [31:0]      start_data;         // DWORD starting seed value for the pattern
input   [31:0]      pattern;            // Pattern used to generate subsequent DWORDs

reg     [32767:0]   payload;

begin
    init_payload (start_data, pattern, payload);

    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b11_00000,  length, first_dw_be, last_dw_be, addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0,           1'b0);
    end
    else
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b10_00000,  length, first_dw_be, last_dw_be, addr, 0,      0,   payload, 1'b0,       1'b0,            1'b0,           1'b0);
    end
end
endtask

// ---------------------------------------------------------------
// task: mem_write_burst_pattern_ecrc
//   Issue burst memory write using a data pattern for the payload

task automatic mem_write_burst_pattern_ecrc;

input   [2:0]       tc;                 // Traffic Class
input   [63:0]      addr;               // Starting address
input   [9:0]       length;             // length in dwords
input   [3:0]       first_dw_be;        // first dword byte enable
input   [3:0]       last_dw_be;         // last dword byte enable
input   [31:0]      start_data;         // DWORD starting seed value for the pattern
input   [31:0]      pattern;            // Pattern used to generate subsequent DWORDs

reg     [32767:0]   payload;

begin
    init_payload (start_data, pattern, payload);

    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        // xfer   (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, payload_for_cpl
        xfer_ecrc (tc, 3'h0, 7'b11_00000,  length, first_dw_be, last_dw_be, addr, 0,      0,   payload, 1'b0,       1'b0,       1'b0);
    end
    else
    begin
        // xfer   (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, payload_for_cpl
        xfer_ecrc (tc, 3'h0, 7'b10_00000,  length, first_dw_be, last_dw_be, addr, 0,      0,   payload, 1'b0,       1'b0,       1'b0);
    end
end
endtask



// --------------------------------------------------------------
// task: mem_read_burst
//   Issue burst memory read and optionally check completion data
//   Note: Waits for completion before returning from task

task automatic mem_read_burst;

input   [2:0]       tc;                 // Traffic Class
input   [63:0]      addr;               // Starting address
input   [9:0]       length;             // length in dwords
input               check_data;         // causes compare of read data with expect data
input   [3:0]       first_dw_be;        // first dword byte enable for comparison
input   [3:0]       last_dw_be;         // last dword byte enable for comparison
inout   [32767:0]   payload;            // As input == Expected data; as output == Data Received

begin
    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b01_00000,  length, first_dw_be, last_dw_be, addr, 0,      0,   payload, check_data, 1'b0,            1'b0,           1'b0);
    end
    else
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b00_00000,  length, first_dw_be, last_dw_be, addr, 0,      0,   payload, check_data, 1'b0,            1'b0,           1'b0);
    end
end
endtask



// --------------------------------------------------------------
// task: mem_read_burst_fast
//   Issue burst memory read and optionally check completion data
//   Note: Does not wait for completion before returning from task
//     Still checks data if check_data == 1; returned payload is invalid

task automatic mem_read_burst_fast;

input   [2:0]       tc;                 // Traffic Class
input   [63:0]      addr;               // Starting address
input   [9:0]       length;             // length in dwords
input               check_data;         // causes compare of read data with expect data
input   [3:0]       first_dw_be;        // first dword byte enable for comparison
input   [3:0]       last_dw_be;         // last dword byte enable for comparison
inout   [32767:0]   payload;            // As input == Expected data; as output == Data Received

begin
    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b01_00000,  length, first_dw_be, last_dw_be, addr, 0,      0,   payload, check_data, 1'b1,            1'b0,           1'b0);
    end
    else
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b00_00000,  length, first_dw_be, last_dw_be, addr, 0,      0,   payload, check_data, 1'b1,            1'b0,           1'b0);
    end

    payload = {32768{1'bx}};
end
endtask



// -------------------------------------------------------------------------------------
// task: mem_read_burst_pattern
//   Issue burst memory read and optionally check completion data against a data pattern
//   Note: Waits for completion before returning from task

task automatic mem_read_burst_pattern;

input   [2:0]       tc;                 // Traffic Class
input   [63:0]      addr;               // Starting address
input   [9:0]       length;             // length in dwords
input               check_data;         // causes compare of read data with expect data
input   [3:0]       first_dw_be;        // first dword byte enable for comparison
input   [3:0]       last_dw_be;         // last dword byte enable for comparison
input   [31:0]      start_data;         // DWORD starting seed value for the pattern
input   [31:0]      pattern;            // Pattern used to generate subsequent DWORDs

reg     [32767:0]   payload;

begin
    init_payload (start_data, pattern, payload);

    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b01_00000,  length, first_dw_be, last_dw_be, addr, 0,      0,   payload, check_data, 1'b0,            1'b0,           1'b0);
    end
    else
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b00_00000,  length, first_dw_be, last_dw_be, addr, 0,      0,   payload, check_data, 1'b0,            1'b0,           1'b0);
    end
end
endtask

// -------------------------------------------------------------------------------------
// task: mem_read_burst_pattern
//   Issue burst memory read and optionally check completion data against a data pattern
//   Note: Waits for completion before returning from task

task automatic mem_read_burst_pattern_option_fast;

input   [2:0]       tc;                 // Traffic Class
input   [63:0]      addr;               // Starting address
input   [9:0]       length;             // length in dwords
input               check_data;         // causes compare of read data with expect data
input   [3:0]       first_dw_be;        // first dword byte enable for comparison
input   [3:0]       last_dw_be;         // last dword byte enable for comparison
input   [31:0]      start_data;         // DWORD starting seed value for the pattern
input   [31:0]      pattern;            // Pattern used to generate subsequent DWORDs
input               option_fast;        // 1= don't wait for cpl. 0=wait for completion


reg     [32767:0]   payload;

begin
    init_payload (start_data, pattern, payload);

    if (addr[63:32] != 32'h0) // Check for 64-bit address
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b01_00000,  length, first_dw_be, last_dw_be, addr, 0,      0,   payload, check_data, option_fast,     1'b0,           1'b0);
    end
    else
    begin
        // xfer (tc, attr, fmt_and_type, length, first_dw_be, last_dw_be, addr  req_id, tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
        xfer    (tc, 3'h0, 7'b00_00000,  length, first_dw_be, last_dw_be, addr, 0,      0,   payload, check_data, option_fast,     1'b0,           1'b0);
    end
end
endtask



// ---------------------------------------------------------
// task: init_payload
//   Initializes payload inout port with the desired pattern
//   No PCIe traffic is generated by this task

task automatic init_payload;

input   [31:0]      start_data; // First DWORD of the pattern
input   [31:0]      pattern;    // Data pattern to use
output  [32767:0]   payload;    // Pattern output formatted as 4096 bytes of payload data

reg     [31:0]      curr_data;
reg     [31:0]      next_data;

integer             i;
integer             j;

begin : init_payload

    curr_data = start_data;
    next_data = 0;

    // Step through each DWORD and setup its contents
    for (i=0; i<1024; i=i+1)
    begin
        // Step through each bit in the DWORD and setup the pattern
        for (j=0; j<32; j=j+1)
        begin
            payload[(i*32)+j] = curr_data[j];
        end

        // Increment to the next value in the pattern
        get_pattern (pattern, curr_data, next_data);
        curr_data = next_data;
    end
end
endtask

// ------------------------------------------------------------
// task: set_completion_error_mode (ecrc_errors, poison_errors)
//   defined the error mode used when completions are sent
//   with the EP (poison) bit set
//

task set_completion_error_mode;
input ecrc_errors;
input poison_errors;
    begin
        // mgmt_gen_ecrc_errors cause ECRC errors to be generated in outgoing packets with the EP (poison) bit set
        mgmt_gen_ecrc_errors = ecrc_errors;
        // mgmt_mask_poison causes the EP (poison) bit to be masked in outgoing packets
        mgmt_mask_poison     = ~poison_errors;
    end
endtask

// ------------------------------------------------------------
// task: init_status_mem
//   Initializes a block of status memory with status bits
//   to indicate if there should be errors generated on packets
//   generated from those addresses in bfm memory

task automatic init_status_mem;

input   [63:0]  start_addr;       // Starting byte address
input   [31:0]  byte_count;       // Number of bytes to setup starting at address offset start_addr
input [3:0]     cpl_status;       // Desired completion status for this area of memory
input           ecrc_error;       // Generate ECRC errors for this area of memory
input           poisoned;         // Generate Poison Bits for this area of memory

reg     [32:0]  mod_byte_count;
reg     [31:0]  dword_count;
reg     [63:0]  dword_addr;

integer         i;

    begin: init_status_mem
        // Get number of DWORDs to setup
        mod_byte_count = byte_count + start_addr[1:0];
        dword_count    = (mod_byte_count+3)/4; // Divide by 4 rounding up

        // Step through each DWORD and setup its contents
        for (i=0; i<dword_count; i=i+1)
        begin
            // Compute the address of the current DWORD
            dword_addr = (start_addr/4) + i;

            // Write DWORD data
            status_mem[dword_addr[BFM_MEM_SIZE_BITS-1:0]] = {cpl_status, ecrc_error, poisoned};

        end
    end
endtask

// ------------------------------------------------------------
// task: init_bfm_mem
//   Initializes a block of bfm memory with the desired pattern
//   No PCIe traffic is generated by this task

task automatic init_bfm_mem;

input   [63:0]  start_addr;         // Starting byte address
input   [31:0]  byte_count;         // Number of bytes to setup starting at address offset start_addr
input   [31:0]  start_data;         // Write data for first xfer; subsequent write data uses start_data_pattern
input   [31:0]  pattern;            // Data pattern to use; see bfm_const.v for options

reg     [63:0]  end_addr;
reg     [32:0]  mod_byte_count;
reg     [31:0]  dword_count;

reg     [63:0]  curr_addr;
reg     [31:0]  curr_data;
reg     [31:0]  next_data;

reg     [63:0]  dword_addr;

reg     [3:0]   temp_be;

integer         i;
integer         j;

begin : init_bfm_mem

    // Get last address to setup
    end_addr = (start_addr + {32'h0, byte_count}) - 64'h1;

    // Get number of DWORDs to setup
    mod_byte_count = byte_count + start_addr[1:0];
    dword_count    = (mod_byte_count+3)/4; // Divide by 4 rounding up

    // Initialize for start of loop
    curr_addr = {start_addr[63:2], 2'b00};
    curr_data = start_data;
    next_data = 32'h0;

    // Step through each DWORD and setup its contents
    for (i=0; i<dword_count; i=i+1)
    begin
        // Step through each byte and compute byte enables
        for (j=0; j<4; j=j+1)
        begin
            temp_be[j] = ((curr_addr >= start_addr) & (curr_addr <= end_addr));
            curr_addr  = curr_addr + 1;
        end

        // Compute the address of the current DWORD
        dword_addr = (start_addr/4) + i;

        // Write DWORD data
        bfm_mem[dword_addr][ 7: 0] = temp_be[0] ? curr_data[ 7: 0] : bfm_mem[dword_addr][ 7: 0];
        bfm_mem[dword_addr][15: 8] = temp_be[1] ? curr_data[15: 8] : bfm_mem[dword_addr][15: 8];
        bfm_mem[dword_addr][23:16] = temp_be[2] ? curr_data[23:16] : bfm_mem[dword_addr][23:16];
        bfm_mem[dword_addr][31:24] = temp_be[3] ? curr_data[31:24] : bfm_mem[dword_addr][31:24];

        // Increment to the next pattern if it's a 32-bit pattern
        get_pattern (pattern, curr_data, next_data);
        curr_data = next_data;
    end
end
endtask



// ---------------------------------------------------------
// task: get_pattern
//   Returns the next value in a pattern when given the
//   pattern type (pattern) and current pattern value (curr)
//   No PCIe traffic is generated by this task

task automatic get_pattern;

input   [31:0]  pattern;    // Pattern type (see list in bfm_const.v)
input   [31:0]  curr;       // Current pattern word
output  [31:0]  next;       // Next word in pattern

begin : get_pattern

    case (pattern)

        PAT_CONSTANT    :   next = curr;

        PAT_ONES        :   next = 32'hffffffff;

        PAT_ZEROS       :   next = 32'h00000000;

        PAT_INC_NIB     :   begin
            next[31:28] = curr[31:28] + 8;
            next[27:24] = curr[27:24] + 8;
            next[23:20] = curr[23:20] + 8;
            next[19:16] = curr[19:16] + 8;
            next[15:12] = curr[15:12] + 8;
            next[11: 8] = curr[11: 8] + 8;
            next[ 7: 4] = curr[ 7: 4] + 8;
            next[ 3: 0] = curr[ 3: 0] + 8;
        end

        PAT_INC_BYTE    :   begin
            next[31:24] = curr[31:24] + 4;
            next[23:16] = curr[23:16] + 4;
            next[15: 8] = curr[15: 8] + 4;
            next[ 7: 0] = curr[ 7: 0] + 4;
        end

        PAT_INC_WORD    :   begin
            next[31:16] = curr[31:16] + 2;
            next[15: 0] = curr[15: 0] + 2;
        end

        PAT_INC_DWORD   :   next[31: 0] = curr[31: 0] + 1;

        PAT_L_SHIFT     :   next[31: 0] = {curr[30: 0], 1'b0};

        PAT_R_SHIFT     :   next[31: 0] = {1'b0, curr[31: 1]};

        PAT_L_ROT       :   next[31: 0] = {curr[30: 0], curr[31]};

        PAT_R_ROT       :   next[31: 0] = {curr[ 0], curr[31: 1]};

        PAT_FIB_NIB     :   begin
              next[3:0]    = curr[3:0]   + curr[7:4];
              next[7:4]    = curr[7:4]   + next[3:0];
              next[11:8]   = curr[11:8]  + next[7:4];
              next[15:12]  = curr[15:12] + next[11:8];
              next[19:16]  = curr[19:16] + next[15:12];
              next[23:20]  = curr[23:20] + next[19:16];
              next[27:24]  = curr[27:24] + next[23:20];
              next[31:28]  = curr[31:28] + next[27:24];
        end

        default         :   next = curr;

    endcase
end
endtask



// ---------------------------------------------------------
// task: get_packet_pattern
//   Returns the next value in a pattern when given the
//   pattern type (pattern) and current pattern value (curr)
//   No PCIe traffic is generated by this task

task automatic get_packet_pattern;

input   [2:0]   pattern;    // Pattern type (see list in bfm_const.v)
input   [31:0]  curr;       // Current pattern word
output  [31:0]  next;       // Next word in pattern

begin : get_packet_pattern

    case (pattern)

        // PAT_CONSTANT
        3'h0    :   next = curr;

        // PAT_INC_BYTE
        3'h1    :   begin
                        next[31:24] = curr[31:24] + 4;
                        next[23:16] = curr[23:16] + 4;
                        next[15: 8] = curr[15: 8] + 4;
                        next[ 7: 0] = curr[ 7: 0] + 4;
                    end

        // PAT_LFSR
        3'h2    :   next[31: 0] = {curr[30:0], ~(^(curr[31:0] & 32'b1000_0000_0010_0000_0000_0100_0000_0011))};

        // PAT_INC_DWORD
        3'h3    :   next[31: 0] = curr[31: 0] + 1;

        default :   next = curr;

    endcase
end
endtask



// ---------------------------------------------------------
// task: get_byte_pattern
//   Returns the next value in a pattern when given the
//   pattern type (data_pat) and current pattern value (data)

task automatic get_byte_pattern;

input   [7:0]   data;       // Pattern seed
input   [7:0]   data_pat;   // Pattern used to generate next value
output  [7:0]   next;

begin : get_byte_pattern

    case (data_pat)

        PAT_CONSTANT    :   next = data;

        PAT_ONES        :   next = 8'hff;

        PAT_ZEROS       :   next = 8'h00;

        PAT_INC_NIB     :   begin
            next[7:4] = data[7:4] + 4'h2;
            next[3:0] = data[3:0] + 4'h2;
        end

        PAT_DEC_NIB     :   begin
            next[7:4] = data[7:4] - 4'h2;
            next[3:0] = data[3:0] - 4'h2;
        end

        PAT_INC_BYTE    :   begin
            next[7:0] = data[7:0] + 8'h1;
        end

        PAT_DEC_BYTE    :   begin
            next[7:0] = data[7:0] - 8'h1;
        end

        PAT_L_SHIFT     :   next[7:0] = {data[6:0], 1'b0};

        PAT_R_SHIFT     :   next[7:0] = {1'b0, data[7:1]};

        PAT_L_ROT       :   next[7:0] = {data[6:0], data[7]};

        PAT_R_ROT       :   next[7:0] = {data[0], data[7:1]};

        PAT_FIB_NIB     :   begin
            next[3:0] = data[3:0] + data[7:4];
            next[7:4] = data[7:4] + next[3:0];
        end

        default         :   next = data;

    endcase
end
endtask
// -------------------------------------------------------------
// task: do_multi_dma_g3
//
// Generates and completes a DMA transaction
// Requires the PCI Express Back-End DMA functionality
//   to be present in the DUT
// Larger DMAs are broken into multiple descriptors along
//   the address boundary defined by the parameter DMA_BYTES

task automatic do_multi_dma_g3; // Generation 3 DMA Engine Task

input   [63:0]  reg_com_bar;    // Common DMA Register Base Address (BAR0 Base Address + offset)
input   [63:0]  reg_dma_bar;    // DMA Register Base Address for DMA Engine to use (BAR0 Base Address + offset)
input   [1:0]   int_mode_msix_msi_legacy; // 2==MSI-X, 1==MSI, 0==Legacy
input   [11:0]  int_vector;     // MSI-X/MSI Vector used by the DMA Engine
input   [63:0]  system_addr;    // DMA Starting System Address
input   [63:0]  card_addr;      // DMA Starting Card Address
input   [31:0]  bcount;         // Total Byte Count to trabsfer in the DMA operation
input   [63:0]  desc_ptr;       // Starting system address where the Descriptors required to complete the operation will be generated
input           done_wait;      // 1 == Wait for DMA completion interrupt before exiting task; 0 == dont wait; if not waiting the interrupt must be handled by the user
output  [31:0]  num_desc;       // Number of descriptors that were used for the DMA operation; num_desc*32 == number of bytes consumed in system memory by the generated descriptors

reg     [63:0]  dma_reg_cap;
reg     [63:0]  dma_reg_cst;
reg     [63:0]  dma_reg_dpl;
reg     [63:0]  dma_reg_dph;
reg     [63:0]  dma_reg_hwt;
reg     [63:0]  dma_reg_lbc;
reg     [63:0]  dma_reg_ldl;
reg     [63:0]  dma_reg_ldh;

reg     [63:0]  card_addr_size;

reg     [31:0]  read_cap;
reg     [31:0]  read_cst;
reg     [31:0]  read_lbc;
reg     [31:0]  read_hwt;
reg     [63:0]  throughput;

reg     [63:0]  curr_sys_addr;
reg     [63:0]  curr_card_addr;
reg     [31:0]  curr_bcount;
reg     [63:0]  curr_desc_ptr;
reg     [31:0]  num_desc;

reg     [31:0]  max_bcount;
reg     [31:0]  xfer_bcount;
reg             continue_i;
reg             sequence_i;
reg     [31:0]  write_control;
reg     [31:0]  bfm_dword_addr;
reg             done;

begin
    if (!$test$plusargs("pcie_dma_msgs_off"))
         $display ("%m : Starting DMA: reg_com_bar=0x%x, reg_dma_bar=0x%x, system_addr=0x%x, card_addr=0x%x, bcount=0x%x, desc_ptr=0x%x (time %t)", reg_com_bar, reg_dma_bar, system_addr, card_addr, bcount, desc_ptr, $time);

    dma_reg_cap = reg_dma_bar + 'h00;
    dma_reg_cst = reg_dma_bar + 'h04;
    dma_reg_dpl = reg_dma_bar + 'h08;
    dma_reg_dph = reg_dma_bar + 'h0C;
    dma_reg_hwt = reg_dma_bar + 'h10;
    dma_reg_lbc = reg_dma_bar + 'h14;
    dma_reg_ldl = reg_dma_bar + 'h18;
    dma_reg_ldh = reg_dma_bar + 'h1C;

    // Get DMA Engine Capabilities
    //mem_read_dword (tc,     addr,        expect_data, check_be, read_data);
    mem_read_dword   (3'b000, dma_reg_cap, 32'h0,       0,        read_cap);

    card_addr_size = ((1<<read_cap[23:16])/1024);

    if (desc_ptr[4:0] != 5'h0)
        $display ("%m : WARNING : desc_addr is not aligned to a 32 byte address boundary; the DMA Engine will refuse to process this DMA chain; desc_ptr == 0x%x", desc_ptr);

    // If an engine is found, then display the type to the log
    if (read_cap[0] == 1'b1 && !$test$plusargs("pcie_dma_msgs_off"))
    begin
        if (read_cap[1] == 1'b0)
        begin
            if (!$test$plusargs("pcie_dma_msgs_off"))
                $display ("%m : INFO : DMA Capabilities: Present, System to Card, Engine#%d, Card Address Size %1dKBytes", read_cap[15:8], card_addr_size);
        end
        else
        begin
            if (!$test$plusargs("pcie_dma_msgs_off"))
                $display ("%m : INFO : DMA Capabilities: Present, Card to System, Engine#%d, Card Address Size %1dKBytes", read_cap[15:8], card_addr_size);
        end
    end

    // Process the DMA if no errors are detected
    if (read_cap[0] == 1'b0)
    begin
        $display ("%m : ERROR : DMA Engine does not exist at the request location.  Capabilities Read=0x%x (time %t)", read_cap, $time);
        inc_errors;
    end
    else if (bcount == 32'h0)
    begin
        $display ("%m : ERROR : Called with invalid byte_count (time %t)", $time);
        inc_errors;
    end
    else
    begin
        // Enable Global DMA Interrupts
        //mem_write_dword (tc,     addr,         data,  be);
        mem_write_dword   (3'b000, reg_com_bar, 32'h1, 4'h1);

        // Create the Descriptors
        curr_sys_addr  = system_addr;
        curr_card_addr = card_addr;
        curr_bcount    = bcount;
        curr_desc_ptr  = desc_ptr;
        num_desc       = 32'h0;

        // Break large DMAs along DMA_BYTES address boundaries
        while (curr_bcount != 32'h0)
        begin
            max_bcount  = DMA_BYTES - {{(32-DMA_REMAIN){1'b0}}, curr_sys_addr[DMA_REMAIN-1:0]};
            xfer_bcount = (curr_bcount > max_bcount) ? max_bcount : curr_bcount;

            // Check to see if this is the first Descriptor in a contiguous Card Address DMA
            if (num_desc == 32'h0) // First Descriptor
                continue_i = 1'b0;
            else
                continue_i = 1'b1;

            // Check to see if this is the last Descriptor in a contiguos Card Address set
            if (xfer_bcount == curr_bcount) // Last Descriptor
            begin
                sequence_i = 1'b0;
                write_control = {22'h0, continue_i, sequence_i, 4'h0, 4'hf}; // Interrupt on all events
            end
            else
            begin
                sequence_i = 1'b1;
                write_control = {22'h0, continue_i, sequence_i, 4'h0, 4'he}; // Interrupt on all events except complete
            end

            // Write the DMA Descriptor info into BFM memory (system memory)
            bfm_dword_addr = curr_desc_ptr[BFM_MEM_SIZE_BITS+1:2];// Drop highest address bits since BFM memory is indexed without a base address
            bfm_mem[bfm_dword_addr+0] = write_control;            // Control[31:0]
            bfm_mem[bfm_dword_addr+1] = xfer_bcount;              // Byte Count[31:0]
            bfm_mem[bfm_dword_addr+2] = curr_sys_addr[31: 0];     // System Address[31: 0]
            bfm_mem[bfm_dword_addr+3] = curr_sys_addr[63:32];     // System Address[63:32]
            bfm_mem[bfm_dword_addr+4] = curr_card_addr[31: 0];    // Card Address[31: 0]
            bfm_mem[bfm_dword_addr+5] = curr_card_addr[63:32];    // Card Address[63:32]

            if (xfer_bcount == curr_bcount) // Last Descriptor
                curr_desc_ptr = 64'h0; // mark end of Descriptor Chain
            else
                curr_desc_ptr = curr_desc_ptr + 32; // Advance to the next Descriptor

            bfm_mem[bfm_dword_addr+6] = curr_desc_ptr[31: 0];     // Desc_Ptr[31: 0]
            bfm_mem[bfm_dword_addr+7] = curr_desc_ptr[63:32];     // Desc_Ptr[63:32]

            // Increment for the next Descrptor
            curr_sys_addr  = curr_sys_addr  + xfer_bcount;
            curr_card_addr = curr_card_addr + xfer_bcount;
            curr_bcount    = curr_bcount    - xfer_bcount;
            num_desc       = num_desc       + 32'h1;
        end

        //mem_write_dword (tc,     addr,        data,            be);
        mem_write_dword   (3'b000, dma_reg_dpl, desc_ptr[31:0],  4'hf); // Write Descriptor Pointer
        mem_write_dword   (3'b000, dma_reg_dph, desc_ptr[63:32], 4'hf); //   ..
        mem_write_dword   (3'b000, dma_reg_cst, 32'h00000101,    4'hf); // Enable interrupts and start the DMA

        // Wait for the DMA to Complete if done_wait was set
        if (done_wait)
        begin
            done = 1'b0;

            while (done == 1'b0)
                        begin
                if ( ((int_mode_msix_msi_legacy == 2'b00) & (int_legi_vector_hit[int_vector[ 1:0]] === 1'b1)) | // Legacy Interrupt in Legacy Mode
                     ((int_mode_msix_msi_legacy == 2'b01) & (int_msi_vector_hit [int_vector[ 4:0]] === 1'b1)) | // MSI    Interrupt in MSI    Mode
                     ((int_mode_msix_msi_legacy == 2'b10) & (int_msix_vector_hit[int_vector[11:0]] === 1'b1)) ) // MSI-X  Interrupt in MSI-X  Mode
                begin
                    // Check to see if the interrupt is ours
                    //mem_read_dword (tc,     addr,        expect_data, check_be, read_data);
                    mem_read_dword   (3'b000, dma_reg_cst, 32'h0,       0,        read_cst);

                    // Check for DMA Complete Status
                    if (read_cst[1]) // Check to see if the interrupt is ours
                    begin
                        // Clear the interrupt & complete bit
                        //mem_write_dword (tc,     addr,        data,  be);
                        mem_write_dword   (3'b000, dma_reg_cst, 32'h0802, 4'hf);
                        done = 1'b1;

                        if (read_cst[11])
                        begin
                            // Check that DMA Completed Byte Count is the same as the byte count for the DMA
                            //mem_read_dword (tc,     addr,        expect_data, check_be, read_data);
                            mem_read_dword   (3'b000, dma_reg_lbc, bcount,      4'hf,     read_lbc);

                            // Get hardware time
                            //mem_read_dword (tc,     addr,        expect_data, check_be, read_data);
                            mem_read_dword   (3'b000, dma_reg_hwt, 32'h0,       4'h0,     read_hwt);

                            throughput = (read_lbc * 954); // Convert ns to S and bytes to MBytes: 1e9/(1024*1024)
                            throughput = throughput / read_hwt;

                            if (!$test$plusargs("pcie_dma_msgs_off"))
                                $display ("%m : INFO : DMA Done: Completed Byte Count == %d bytes, Hardware Time == %d ns, Throughput == %d MB/s", read_lbc, read_hwt, throughput[63:0]);
                        end
                        else
                        begin
                            $display ("%m : INFO : DMA Interrupted, but did not complete successfully");
                            end
                        //wait for Legacy interrupt to clear before continuing
                        while ((int_mode_msix_msi_legacy == 2'b00) & (int_legi_vector_hit[int_vector[ 1:0]] === 1'b1))
                            begin
                            @(posedge clk);
                            end
                        end
                            end
                else
                        begin
                    @(posedge clk);
                        end
                    end
                end
            end
        end
endtask



// --------------------------------------------------
//  task: do_pkt_dma_chain
//    Generates a Packet DMA Transaction for
//      the standard Packet DMA stream reference design.
//    Requires that a pattern generator/checker module
//      is connected to the DMA Engine.
//    A contiguous Descriptor table, starting at desc_ptr,
//      is created for the Descriptors required to transfer
//      the specified packets.
//    DMA data is written/read from system memory starting
//      at system_addr
//    This task implements a finite DMA chain ending
//      with NextDescPtr == 0
//    This task polls Descriptors to determine completion
//      status and does not use interrupts

task automatic do_pkt_dma_chain;

input   [63:0]                  system_addr;            // DMA Starting System Address
input   [63:0]                  system_max_bsize;       // DMA byte size starting at system_addr that is reserved for this task; the task will verify that this size is not exceeded
input   [63:0]                  desc_ptr;               // DMA Starting System Address where the Descriptor chain will be located
input   [63:0]                  desc_max_bsize;         // DMA byte size starting at desc_ptr that is reserved for the Descriptors required by this task; the task will verify that this size is not exceeded
input   [63:0]                  desc_bsize;             // Descriptor size in bytes; Descriptors will be created on address boundaries that are multiples of desc_bsize; must be a power of 2
input   [63:0]                  reg_com_bar;            // DMA Common Register Block Base Address; needed to set global enables
input   [63:0]                  reg_dma_bar;            // DMA Engine Register Block Base Address; choose which engine will be used
input   [63:0]                  reg_pat_bar;            // DMA Engine Pattern Register Block Base Address; must be the pattern generator connected to used DMA Engine
input   [11:0]                  int_vector;             // MSI-X/MSI Vector used by the DMA Engine
input   [1:0]                   pat_length_entries;     // Number of pat_length[] table entries to use; 0-3
input   [2:0]                   pat_data_type;          // Data pattern: 0==CONSTANT, 1==INC BYTE, 2==LFSR; 3==INC_DWORD
input                           pat_data_cont;          // 1 == Continue the data pattern across packet boundaries 0 == Restart with data_seed for each packet
input   [2:0]                   pat_user_type;          // User status/control pattern: 0==CONSTANT, 1==INC BYTE, 2==LFSR; 3==INC_DWORD
input                           pat_user_cont;          // 1 == Continue the user_status/control pattern across packet boundaries 0 == Restart with user_status_seed for each packet
input   [7:0]                   pat_active_clocks;      // Packet Generator data ready rate control
input   [7:0]                   pat_inactive_clocks;    //   ..
input   [31:0]                  pat_num_packets;        // Number of packets to generate or 0 for infinite
input   [31:0]                  pat_data_seed;          // Data pattern seed; first DWORD of data pattern; subsequent words are generated according to pattern type
input   [31:0]                  pat_user_seed;          // User status/control pattern seed; first DWORD of user_status/control pattern; subsequent words are generated according to pattern type
input   [19:0]                  pat_length0;            // Packet Length Table[0] - Sets packet sizes in bytes
input   [19:0]                  pat_length1;            // Packet Length Table[1] - Sets packet sizes in bytes
input   [19:0]                  pat_length2;            // Packet Length Table[2] - Sets packet sizes in bytes
input   [19:0]                  pat_length3;            // Packet Length Table[3] - Sets packet sizes in bytes
input   [31:0]                  verbose;                // Log Display Control : 0 == Only errors; 1 == Limited messages; 2=Expanded messages; 3=Verbose including passing
input                           on_err_stop;            // Set to stop when an error is detected
input                           timeout_en;             // Set to enable DMA timeout functionality
input   [31:0]                  timeout_clocks;         // If timeout_en==1 and the DMA fails to make progress for timeout_clocks clock cycles, the DMA operation is aborted and an error generated;
                                                        //   Every time a Descriptor completes (DMA making progress), the timeout counter is reset to timeout_clocks
input                           check_status;           // Enable checking of user status
output  [31:0]                  total_bcount;           // Total number of bytes transferred

reg     [31:0]                  my_verbose;

reg     [31:0]                  pat_control;
reg     [8:0]                   sum_clocks;

reg     [63:0]                  dma_reg_cap;
reg     [63:0]                  dma_reg_cst;
reg     [63:0]                  dma_reg_ndp;
reg     [63:0]                  dma_reg_sdp;
reg     [63:0]                  dma_reg_cdp;

reg     [63:0]                  dma_pat_ctl;
reg     [63:0]                  dma_pat_npk;
reg     [63:0]                  dma_pat_dsd;
reg     [63:0]                  dma_pat_usd;
reg     [63:0]                  dma_pat_err;
reg     [63:0]                  dma_pat_pl0;
reg     [63:0]                  dma_pat_pl1;
reg     [63:0]                  dma_pat_pl2;
reg     [63:0]                  dma_pat_pl3;

reg                             done_wait;

reg     [31:0]                  read_cap;
reg                             eng_type_c2s_s2c_n;
reg     [63:0]                  card_addr_size;
reg     [63:0]                  max_desc_bsize;
reg     [63:0]                  desc_bsize_mask;

reg     [63:0]                  curr_sys_addr;
reg     [63:0]                  curr_desc_ptr;
reg     [63:0]                  last_desc_ptr;
reg     [1:0]                   curr_table_offset;
integer                         num_desc;
integer                         desc_limit_error;
integer                         p;
reg     [63:0]                  curr_bcount;

reg     [63:0]                  max_bcount;
reg     [63:0]                  xfer_bcount;
reg     [63:0]                  used_xfer_bcount;
reg     [7:0]                   control_flags;
reg     [31:0]                  bfm_dword_addr;

reg     [BFM_MEM_SIZE_BITS-1:0] dword_offset_addr;
integer                         j;

reg                             expected_sop;
reg                             expected_eop;
reg     [19:0]                  expected_bcount;
reg                             expected_short;
reg                             abort_issued;
reg     [31:0]                  expected_user_lo;
reg     [31:0]                  expected_user_hi;
reg     [31:0]                  this_expected_user_lo;
reg     [31:0]                  this_expected_user_hi;

reg     [31:0]                  expected_data;
reg     [31:0]                  next_expected_data;

reg     [31:0]                  check_primary_status;
reg     [31:0]                  check_user_status_lo;
reg     [31:0]                  check_user_status_hi;

reg     [63:0]                  check_sys_addr;
reg     [31:0]                  mem; // Data captured in the BFM memory
reg     [31:0]                  exp;

reg                             check_exit;
reg     [31:0]                  timeout_ctr;
reg     [31:0]                  read_cst;
reg     [31:0]                  read_ctl;
reg     [31:0]                  read_err;

reg                             drop_data;

begin
    // Optional over-ride to turn off DMA messages
    if ($test$plusargs("pcie_dma_msgs_off"))
        my_verbose = 0;
    else
        my_verbose = verbose;

    pat_control = {pat_inactive_clocks[7:0],
                   pat_active_clocks[7:0],
                   pat_user_cont, pat_user_type[2:0], pat_data_cont, pat_data_type[2:0],
                   2'h0, pat_length_entries[1:0], 3'h0, 1'b1};

    if (my_verbose > 0)
    begin
        $display ("%m : Starting Packet DMA: reg_dma_bar=0x%x, reg_pat_bar=0x%x, reg_com_bar=0x%x, (time %t)",
                                                    reg_dma_bar,      reg_pat_bar,      reg_com_bar,      $time);

        $display ("%m : Starting Packet DMA: reg_dma_bar=0x%x, system_addr=0x%x, desc_ptr=0x%x desc_bsize=0x%x, (time %t)",
                                                    reg_dma_bar,      system_addr,      desc_ptr,     desc_bsize,      $time);

        $display ("%m : Starting Packet DMA: reg_dma_bar=0x%x, pat_num_packets=0x%x, pat_length_entries=%d; pat_length_table[0,1,2,3]=[0x%x,0x%x,0x%x,0x%x]",
                                                    reg_dma_bar,      pat_num_packets,      pat_length_entries,    pat_length0, pat_length1, pat_length2, pat_length3);

        $display ("%m : Starting Packet DMA: reg_dma_bar=0x%x, pat_data_seed=0x%x, pat_data_type=0x%x, pat_data_cont=%d, (time %t)",
                                                    reg_dma_bar,      pat_data_seed,      pat_data_type,      pat_data_cont,    $time);

        $display ("%m : Starting Packet DMA: reg_dma_bar=0x%x, pat_user_seed=0x%x, pat_user_type=0x%x, pat_user_cont=%d, (time %t)",
                                                    reg_dma_bar,      pat_user_seed,      pat_user_type,      pat_user_cont,    $time);

        sum_clocks = pat_active_clocks[7:0] + pat_inactive_clocks[7:0];

        $display ("%m : Starting Packet DMA: reg_dma_bar=0x%x, Packet data ready rate=(%d/%d) of maximum rate", reg_dma_bar, pat_active_clocks, sum_clocks);
    end

    // Get DMA Engine Register Addresses
    dma_reg_cap = reg_dma_bar + 'h00;
    dma_reg_cst = reg_dma_bar + 'h04;
    dma_reg_ndp = reg_dma_bar + 'h08;
    dma_reg_sdp = reg_dma_bar + 'h0c;
    dma_reg_cdp = reg_dma_bar + 'h10;

    // Get DMA Engine Pattern Generator/Checker Addresses
    dma_pat_ctl = reg_pat_bar + 'h00;
    dma_pat_npk = reg_pat_bar + 'h04;
    dma_pat_dsd = reg_pat_bar + 'h08;
    dma_pat_usd = reg_pat_bar + 'h0c;
    dma_pat_err = reg_pat_bar + 'h10; // Pattern checker only
    // Reserved 'h1f-'h14
    dma_pat_pl0 = reg_pat_bar + 'h20;
    dma_pat_pl1 = reg_pat_bar + 'h24;
    dma_pat_pl2 = reg_pat_bar + 'h28;
    dma_pat_pl3 = reg_pat_bar + 'h2c;

    done_wait = 1; // Wait for DMA to Complete, so we can check data; don't change

    // Initialize loop exit variables
    check_exit  = 1'b0;
    timeout_ctr = timeout_clocks;
    total_bcount = 32'b0;

    // Get DMA Engine Capabilities
    //mem_read_dword (tc,     addr,        expect_data, check_be, read_data);
    mem_read_dword   (3'b000, dma_reg_cap, 32'h0,       0,        read_cap);

    // Save engine type; 1 == C2S, 0 == S2C
    eng_type_c2s_s2c_n = read_cap[1];

    card_addr_size = ((1<<read_cap[22:16])/1024);
    max_desc_bsize =  (1<<read_cap[29:24])-1;

    // Want to create a mask which can be ANDed with values to limit them to desc_bsize or smaller;
    //   desc_bsize must be a power of 2 for the line below to work properly
    desc_bsize_mask = desc_bsize - 1;

    // If an engine is found, then display the type to the log
    if (my_verbose > 0)
    begin
        if (read_cap[0] == 1'b1)
        begin
            if (read_cap[1] == 1'b0)
                $display ("%m : INFO : reg_dma_bar=0x%x : DMA Capabilities: Present, System to Card, Engine#%d, Card Address Size %1d KBytes, MaxDescriptorSize=%1d Bytes", reg_dma_bar, read_cap[15:8], card_addr_size, max_desc_bsize);
            else
                $display ("%m : INFO : reg_dma_bar=0x%x : DMA Capabilities: Present, Card to System, Engine#%d, Card Address Size %1d KBytes, MaxDescriptorSize=%1d Bytes", reg_dma_bar, read_cap[15:8], card_addr_size, max_desc_bsize);
        end
    end

    // This is an error, but is allowed to continue so that the DMA Engine's response to this error can be tested
    if (desc_ptr[4:0] != 5'h0)
        $display ("%m : INFO : reg_dma_bar=0x%x : desc_addr is not aligned to a 32 byte address boundary; the DMA Engine will refuse to process this DMA chain and will timeout; desc_ptr == 0x%x", reg_dma_bar, desc_ptr);

    // Check task inputs for validity
    if (pat_num_packets == 32'h0)
    begin
        $display ("%m : ERROR : reg_dma_bar=0x%x : Packet generator pat_num_packets == 0 (infinite); this task requires a finite length DMA chain and does not support pat_num_packets != 0 (time %t)", reg_dma_bar, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (desc_bsize > max_desc_bsize)
    begin
        $display ("%m : ERROR : reg_dma_bar=0x%x : Requested descriptor size (desc_bsize == %d) > DMA Engine max supported descriptor size (max_desc_bsize == %d) (time %t)", reg_dma_bar, desc_bsize, max_desc_bsize, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if ((desc_bsize % 2) != 0)
    begin
        $display ("%m : ERROR : reg_dma_bar=0x%x : Requested descriptor size (desc_bsize == %d) must be a power of 2 (time %t)", reg_dma_bar, desc_bsize, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (read_cap[0] == 1'b0)
    begin
        $display ("%m : ERROR : reg_dma_bar=0x%x : DMA Engine does not exist at the request location.  Capabilities Read=0x%x (time %t)", reg_dma_bar, read_cap, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (read_cap[7:4] == 4'h0)
    begin
        $display ("%m : ERROR : reg_dma_bar=0x%x : DMA Engine is not a Packet DMA Engine.  Capabilities Read=0x%x (time %t)", reg_dma_bar, read_cap, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (pat_data_type > 3'h3)
    begin
        $display ("%m : ERROR : reg_dma_bar=0x%x : Invalid data pattern type; pat_data_type=0x%x; max value supported=0x%x (time %t)", reg_dma_bar, pat_data_type, 3'h3, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (pat_user_type > 3'h3)
    begin
        $display ("%m : ERROR : reg_dma_bar=0x%x : Invalid data pattern type; pat_user_type=0x%x; max value supported=0x%x (time %t)", reg_dma_bar, pat_data_type, 3'h3, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (desc_ptr[63:32] != 32'h0)
    begin
        $display ("%m : ERROR : reg_dma_bar=0x%x : Packet DMA Engines do not support Descriptors being located at 64-bit addresses; desc_ptr == 0x%x (time %t)", reg_dma_bar, desc_ptr, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (pat_length0[1:0] != 2'h0)
    begin
        $display ("%m : ERROR : reg_dma_bar=0x%x : (pat_length0 == 0x%x) must be a multiple of 4 bytes in bytes because 4 byte patterns are used in this test (time %t)", reg_dma_bar, pat_length0, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (pat_length1[1:0] != 2'h0)
    begin
        $display ("%m : ERROR : reg_dma_bar=0x%x : (pat_length1 == 0x%x) must be a multiple of 4 bytes in bytes because 4 byte patterns are used in this test (time %t)", reg_dma_bar, pat_length1, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (pat_length2[1:0] != 2'h0)
    begin
        $display ("%m : ERROR : reg_dma_bar=0x%x : (pat_length2 == 0x%x) must be a multiple of 4 bytes in bytes because 4 byte patterns are used in this test (time %t)", reg_dma_bar, pat_length2, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (pat_length3[1:0] != 2'h0)
    begin
        $display ("%m : ERROR : reg_dma_bar=0x%x : (pat_length3 == 0x%x) must be a multiple of 4 bytes in bytes because 4 byte patterns are used in this test (time %t)", reg_dma_bar, pat_length3, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (system_addr[1:0] != 2'h0)
    begin
        $display ("%m : ERROR : reg_dma_bar=0x%x : (system_addr == 0x%x) must be a multiple of 4 bytes in bytes because 4 byte patterns are used in this test (time %t)", reg_dma_bar, system_addr, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else
    begin
        // ----------------------
        // Create the Descriptors

        curr_sys_addr     = system_addr;
        curr_desc_ptr     = desc_ptr;
        curr_table_offset = 2'h0;
        num_desc          = 0;
        desc_limit_error  = 0;

        // Setup initial seed values for S2C Packet DMA
        if (eng_type_c2s_s2c_n == 1'b0) // S2C
        begin
            // Initial value for user_control
            expected_user_lo = pat_user_seed;
            get_packet_pattern (pat_user_type, expected_user_lo, expected_user_hi);

            // Initial value for data
            expected_data = pat_data_seed;
        end

        // Step through each packet that will be transferred and create Descriptors for it
        for (p=0; p<pat_num_packets; p=p+1)
        begin
            if (desc_limit_error == 0)
            begin
                case (curr_table_offset[1:0])
                    2'b00 : curr_bcount = pat_length0;
                    2'b01 : curr_bcount = pat_length1;
                    2'b10 : curr_bcount = pat_length2;
                    2'b11 : curr_bcount = pat_length3;
                endcase

                // Increment to next packet size in table in preparation for next packet
                if (curr_table_offset == pat_length_entries)
                    curr_table_offset = 2'h0;
                else
                    curr_table_offset = curr_table_offset + 2'h1;

                // Initalize packet data and control information for S2C DMA
                if (eng_type_c2s_s2c_n == 1'b0) // S2S
                begin
                    // Set sop for each new packet
                    expected_sop = 1;

                    // Reinitialize user_control pattern if in non-continuous pattern mode
                    if (pat_user_cont == 1'b0)
                    begin
                        // Initial value for user_control
                        expected_user_lo = pat_user_seed;
                        get_packet_pattern (pat_user_type, expected_user_lo, expected_user_hi);
                    end

                    // Reinitialize pattern if in non-continuous pattern mode
                    if (pat_data_cont == 1'b0)
                        expected_data = pat_data_seed;
                end

                // Break large packets into multiple Descriptors along desc_bsize address boundaries
                while (curr_bcount != 32'h0)
                begin
                    if (desc_limit_error == 0)
                    begin
                        // Don't exeed desc_bsize address boundaries
                        max_bcount  = desc_bsize - (desc_bsize_mask & curr_sys_addr);
                        xfer_bcount = (desc_bsize > max_bcount) ? max_bcount : desc_bsize;

                        // Use max size Descriptors except for the last Descriptor use remaining size
                        used_xfer_bcount = (curr_bcount > max_bcount) ? max_bcount : curr_bcount;

                        if (eng_type_c2s_s2c_n) // C2S
                        begin
                            control_flags = 8'h0; // Not using interrupts

                            // Write the DMA Descriptor info into BFM memory (system memory)
                            bfm_dword_addr = curr_desc_ptr[BFM_MEM_SIZE_BITS+1:2];// Drop highest address bits since BFM memory is indexed without a base address
                            bfm_mem[bfm_dword_addr+0] = 32'h0; // {C2SDescStatusFlags[7:0], Reserved[3:0], C2SDescByteCount[19:0]} - Status so clear
                            bfm_mem[bfm_dword_addr+1] = 32'h0; // C2SDescUserStatus[31: 0] - Status so clear
                            bfm_mem[bfm_dword_addr+2] = 32'h0; // C2SDescUserStatus[63:32] - Status so clear
                            bfm_mem[bfm_dword_addr+3] = 32'h0; // DescCardAddr; unused
                            bfm_mem[bfm_dword_addr+4] = {control_flags[7:0], 4'h0, xfer_bcount[19:0]}; // {C2SDescControlFlags[7:0], DescCardAddr[35:32], DescByteCount[19:0]} - Control
                            bfm_mem[bfm_dword_addr+5] = curr_sys_addr[31: 0];     // System Address[31: 0]
                            bfm_mem[bfm_dword_addr+6] = curr_sys_addr[63:32];     // System Address[63:32]
                        end
                        else // S2C
                        begin
                            // Set eop for Descriptors containing the end of a packet
                            expected_eop = (used_xfer_bcount == curr_bcount);

                            // S2C Control Flags; not using interrupts
                            control_flags = {expected_sop, expected_eop, 6'h0};

                            // Write the DMA Descriptor info into BFM memory (system memory)
                            bfm_dword_addr = curr_desc_ptr[BFM_MEM_SIZE_BITS+1:2];// Drop highest address bits since BFM memory is indexed without a base address
                            bfm_mem[bfm_dword_addr+0] = {8'h0, 4'h0, used_xfer_bcount[19:0]}; // {S2CDescStatusFlags[7:0], Reserved[3:0], S2CDescByteCount[19:0]} - Status so clear except bcount which is bytes to use in this descriptor
                            bfm_mem[bfm_dword_addr+1] = expected_user_lo; // S2CDescUserControl[31: 0]
                            bfm_mem[bfm_dword_addr+2] = expected_user_hi; // S2CDescUserControl[63:32]
                            bfm_mem[bfm_dword_addr+3] = 32'h0; // DescCardAddr; unused
                            bfm_mem[bfm_dword_addr+4] = {control_flags[7:0], 4'h0, xfer_bcount[19:0]}; // {S2CDescControlFlags[7:0], DescCardAddr[35:32], DescByteCount[19:0]} - Control
                            bfm_mem[bfm_dword_addr+5] = curr_sys_addr[31: 0];     // System Address[31: 0]
                            bfm_mem[bfm_dword_addr+6] = curr_sys_addr[63:32];     // System Address[63:32]

                            // Roll user_control pattern whenever completing a Descriptor with EOP asserted
                            if (expected_eop)
                            begin
                                get_packet_pattern (pat_user_type, expected_user_hi, expected_user_lo);
                                get_packet_pattern (pat_user_type, expected_user_lo, expected_user_hi);
                            end
                        end

                        if (my_verbose > 1)
                        begin
                            if (eng_type_c2s_s2c_n) // C2S
                                $display ("%m : INFO : reg_dma_bar=0x%x : Creating Descriptor[%4d] : sys_addr = 0x%x, desc_addr = 0x%x, bcnt = 0x%x, used_bcnt = 0x%x", reg_dma_bar, num_desc, curr_sys_addr, curr_desc_ptr[31:0], xfer_bcount[19:0], used_xfer_bcount[19:0]);
                            else // S2C
                                $display ("%m : INFO : reg_dma_bar=0x%x : Creating Descriptor[%4d] : sys_addr = 0x%x, desc_addr = 0x%x, bcnt = 0x%x, used_bcnt = 0x%x, sop=%d, eop=%d", reg_dma_bar, num_desc, curr_sys_addr, curr_desc_ptr[31:0], xfer_bcount[19:0], used_xfer_bcount[19:0], expected_sop, expected_eop);
                        end

                        last_desc_ptr = curr_desc_ptr;

                        if ((xfer_bcount >= curr_bcount) & (p == (pat_num_packets-1))) // Last Descriptor in chain
                        begin
                            curr_desc_ptr = 64'h0; // Mark end of Descriptor Chain
                        end
                        else
                        begin
                            curr_desc_ptr = curr_desc_ptr + 32; // Advance to the next Descriptor address; descriptors are 32 bytes each
                        end

                        bfm_mem[bfm_dword_addr+7] = curr_desc_ptr[31: 0];     // {DescNextDescPtr[31:5], 5b00000}

                        // -----------------------------------------
                        // Initialize the DMA destination BFM memory

                        // Get offset into BFM memory where DMA data will be written
                        dword_offset_addr = curr_sys_addr[BFM_MEM_SIZE_BITS+1:2];

                        if (eng_type_c2s_s2c_n) // C2S
                        begin
                            // Initialize the BFM memory region used by this DMA operation with all ones;
                            //   After the DMA is done all locations that the DMA should not have accessed
                            //   should still be all 1s; a check can be made to ensure that the DMA did
                            //   not write any extra bytes
                            for (j=dword_offset_addr; j<(dword_offset_addr+(xfer_bcount/4)); j=j+1)
                                bfm_mem[j] = 32'hffffffff;
                        end
                        else // S2C
                        begin
                            // Initialize the BFM memory region with the desired pattern
                            for (j=dword_offset_addr; j<(dword_offset_addr+(used_xfer_bcount/4)); j=j+1)
                            begin
                                bfm_mem[j] = expected_data;
                                get_packet_pattern (pat_data_type, bfm_mem[j], expected_data);
                            end
                            // Fill any unused portion of the Descriptor with all 1s
                            for (j=dword_offset_addr+(used_xfer_bcount/4); j<(dword_offset_addr+((xfer_bcount-used_xfer_bcount)/4)); j=j+1)
                            begin
                                bfm_mem[j] = 32'hffffffff;
                            end
                        end

                        // Increment for the next Descrptor
                        curr_sys_addr  = curr_sys_addr  + xfer_bcount;
                        curr_bcount    = (xfer_bcount > curr_bcount) ? 0 : (curr_bcount - xfer_bcount);
                        num_desc       = num_desc       + 1;
                        expected_sop   = 0;
                        expected_eop   = 0;

                        if (curr_sys_addr > (system_addr + system_max_bsize)) // While creating Descriptors, system_max_bsize was exceeded
                        begin
                            desc_limit_error = 1;
                            $display ("%m : ERROR : reg_dma_bar=0x%x : (system_max_bsize == 0x%x) is too small for the amount of system memory required to receive the specified packets (time %t)", reg_dma_bar, system_max_bsize, $time);
                            $display ("%m : ERROR : reg_dma_bar=0x%x : (Current Descriptor being created ended at sys_addr == 0x%x) > (sys_addr address limit == 0x%x)", reg_dma_bar, (curr_sys_addr-1), ((system_addr + system_max_bsize)-1));
                            $display ("%m : ERROR : reg_dma_bar=0x%x : Increase system_max_bsize, decrease (pat_num_packets == 0x%x), decrease packet lengths, or reduce (desc_bsize == %d) so there is less waste betweeen packets", reg_dma_bar, pat_num_packets, desc_bsize);
                            inc_errors;
                            if (on_err_stop)
                            begin
                                report_status;
                                if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                            end
                        end

                        if ((num_desc * 32) > desc_max_bsize) // While creating Descriptors, system_max_bsize was exceeded
                        begin
                            desc_limit_error = 1;
                            $display ("%m : ERROR : reg_dma_bar=0x%x : (desc_max_bsize == 0x%x) is not large enough for the amount of Descriptors required to be created to receive the specified packets (time %t)", reg_dma_bar, desc_max_bsize, $time);
                            $display ("%m : ERROR : reg_dma_bar=0x%x : (Descriptors created thus far require 0x%x bytes) > (desc_max_bsize == 0x%x)", reg_dma_bar, (num_desc * 32), desc_max_bsize);
                            $display ("%m : ERROR : reg_dma_bar=0x%x : Increase desc_max_bsize, decrease (pat_num_packets == 0x%x), decrease packet lengths, or increase (desc_bsize == %d) so fewer Descriptors are required per packet", reg_dma_bar, pat_num_packets, desc_bsize);
                            inc_errors;
                            if (on_err_stop)
                            begin
                                report_status;
                                if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                            end
                        end
                    end
                end
            end
        end

        if (desc_limit_error == 0) // Do the DMA operation if still no errors
        begin

            // --------------------------------------------------
            // Initialize and start the Pattern Generator/Checker

            if (my_verbose > 0)
                $display ("%m : INFO : reg_dma_bar=0x%x : Initializing and starting the pattern generator", reg_dma_bar);

            //mem_write_dword (tc,     addr,        data,            be);
            mem_write_dword   (3'b000, dma_pat_pl3, pat_length3,     4'hf);
            mem_write_dword   (3'b000, dma_pat_pl2, pat_length2,     4'hf);
            mem_write_dword   (3'b000, dma_pat_pl1, pat_length1,     4'hf);
            mem_write_dword   (3'b000, dma_pat_pl0, pat_length0,     4'hf);
            mem_write_dword   (3'b000, dma_pat_usd, pat_user_seed,   4'hf);
            mem_write_dword   (3'b000, dma_pat_dsd, pat_data_seed,   4'hf);
            mem_write_dword   (3'b000, dma_pat_npk, pat_num_packets, 4'hf);
            mem_write_dword   (3'b000, dma_pat_ctl, pat_control,     4'hf); // Starts pattern generator/checker

            // -----------------------------------
            // Initialize and start the DMA Engine

            if (my_verbose > 0)
                $display ("%m : INFO : reg_dma_bar=0x%x : Initializing and starting DMA Engine", reg_dma_bar);

            //mem_write_dword (tc,     addr,        data,            be);
            mem_write_dword   (3'b000, dma_reg_cdp, 32'h0,           4'hf); // Zero completed Descriptor Pointer
            mem_write_dword   (3'b000, dma_reg_sdp, 32'h0,           4'hf); // End of chain is marked with ndp==0, so this register is not used
            mem_write_dword   (3'b000, dma_reg_ndp, desc_ptr[31: 0], 4'hf); // Start of DMA chain
            mem_write_dword   (3'b000, dma_reg_cst, 32'h00000101,    4'hf); // Start the DMA engine; enable interrupts

            // ---------------------------
            // Check for correct operation

            // if (done_wait != 0), wait for Descriptors to complete, and check their values
            if (done_wait != 0)
            begin
                // Initialize abort status
                abort_issued = 1'b0;

                if (eng_type_c2s_s2c_n) // Check for expected C2S operation
                begin
                    // Set to starting values
                    curr_sys_addr     = system_addr;
                    curr_desc_ptr     = desc_ptr;
                    curr_table_offset = 2'h0;

                    // Initialize expected value for user_status
                    expected_user_lo = pat_user_seed;
                    get_packet_pattern (pat_user_type, expected_user_lo, expected_user_hi);

                    // Initialize expected value for data
                    expected_data = pat_data_seed;

                    // Step through the number of Descriptors that were created and check each one
                    //   for proper operation
                    check_exit = 1'b0;
                    while ((num_desc != 0) & (check_exit == 1'b0))
                    begin
                        expected_sop = 1;

                        // Initialize expected value for user_status if pattern is not continuous from packet to packet
                        if (pat_user_cont == 1'b0)
                        begin
                            expected_user_lo = pat_user_seed;
                            get_packet_pattern (pat_user_type, expected_user_lo, expected_user_hi);
                        end

                        // Initialize expected value for data if pattern is not continuous from packet to packet
                        if (pat_data_cont == 1'b0)
                            expected_data = pat_data_seed;

                        case (curr_table_offset[1:0])
                            2'b00 : curr_bcount = pat_length0;
                            2'b01 : curr_bcount = pat_length1;
                            2'b10 : curr_bcount = pat_length2;
                            2'b11 : curr_bcount = pat_length3;
                        endcase

                        // Increment to next packet size in table in preparation for next packet
                        if (curr_table_offset == pat_length_entries)
                            curr_table_offset = 2'h0;
                        else
                            curr_table_offset = curr_table_offset + 2'h1;

                        // Break large packets along desc_bsize address boundaries
                        while ((curr_bcount != 32'h0) & (check_exit == 1'b0))
                        begin
                            // Don't exeed a desc_bsize address boundary
                            max_bcount  = desc_bsize - (desc_bsize_mask & curr_sys_addr);
                            xfer_bcount = (desc_bsize > max_bcount) ? max_bcount : desc_bsize;

                            expected_eop    = (xfer_bcount >= curr_bcount);
                            expected_bcount = (xfer_bcount >= curr_bcount) ? curr_bcount : xfer_bcount;
                            // If a software abort occurs, all descriptors in transit will be tagged as short
                            expected_short  = (xfer_bcount >  curr_bcount) || (abort_issued == 1'b1);

                            bfm_dword_addr = curr_desc_ptr[BFM_MEM_SIZE_BITS+1:2];// Drop highest address bits since BFM memory is indexed without a base address

                            // Reset timeout counter to initial value
                            timeout_ctr = timeout_clocks;

                            while ((bfm_mem[bfm_dword_addr+0] == 0) & (check_exit == 1'b0)) // Descriptor status byte not updated
                            begin
                                @(posedge clk);

                                if (timeout_en)
                                begin
                                    timeout_ctr = timeout_ctr - 32'h1;
                                    if (timeout_ctr == 32'h0)
                                        check_exit = 1'b1; // Exit while loop
                                end
                            end

                            if (timeout_en & (timeout_ctr == 32'h0))   // DMA Timeout Occurred
                            begin
                                check_exit = 1'b1; // Exit while loop
                                $display ("%m : %0s : reg_dma_bar=0x%x : DMA operation failed to make progress for 0x%x clocks.  DMA may be locked up.  Aborting test and resetting DMA Engine.  (time %t)",
                                           on_err_stop ? "ERROR" : "INFO", reg_dma_bar, timeout_clocks, $time);
                                if (on_err_stop)
                                begin
                                    inc_errors;
                                    report_status;
                                    if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                end

                                // Reset DMA Engine
                                //mem_write_dword (tc,     addr,        data,            be);
                                mem_write_dword   (3'b000, dma_reg_cst, 32'h00008000,    4'hf);

                                // Poll until it is seen that reset is completed
                                read_cst = 32'h00008000; // Not done status
                                while (read_cst[15] == 1'b1)
                                begin
                                    //mem_read_dword (tc,     addr,        expect_data, check_be, read_data);
                                    mem_read_dword   (3'b000, dma_reg_cst, 32'h0,       0,        read_cst);
                                end
                            end
                            else
                            begin

                                // -----------------------------------
                                // Check for correct Descriptor Status

                                check_primary_status = bfm_mem[bfm_dword_addr+0];
                                if (check_primary_status[25] == 1'b1) begin                         // short
                                    mem_read_dword (3'b000, dma_reg_cst, 32'h0, 4'h0, read_cst);    // Check DMA status
                                    if (read_cst[10] == 1'b0)                                       // If DMA engine has stopped, we have aborted
                                        check_exit = 1'b1;
                                    if (read_cst[14] == 1'b1) begin                                 // If reset_request is set, this is an expected short
                                        abort_issued = 1'b1;                                        //  because an abort has been issued
                                        expected_short = 1'b1;
                                    end
                                    if (abort_issued & (check_primary_status[19:0] != expected_bcount)) // Abort has stopped data transfer
                                    begin
                                        xfer_bcount = xfer_bcount - expected_bcount;            // remove data from total since it is not valid
                                        expected_bcount = check_primary_status[19:0];           // change expected count to avoid error report
                                    end
                                end
                                total_bcount = total_bcount + check_primary_status[19:0];

                                if (check_primary_status[31] != expected_sop)
                                begin
                                    $display ("%m : ERROR : reg_dma_bar=0x%x : Desc_Ptr = 0x%x, Expected status (sop == %d) but got the opposite (time %t)", reg_dma_bar, curr_desc_ptr, expected_sop, $time);
                                    inc_errors;
                                    if (on_err_stop)
                                    begin
                                        report_status;
                                        if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                    end
                                end

                                if (check_primary_status[30] != expected_eop)
                                begin
                                    $display ("%m : ERROR : reg_dma_bar=0x%x : Desc_Ptr = 0x%x, Expected status (eop == %d) but got the opposite (time %t)", reg_dma_bar, curr_desc_ptr, expected_eop, $time);
                                    inc_errors;
                                    if (on_err_stop)
                                    begin
                                        report_status;
                                        if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                    end
                                end

                                // check_primary_status[29:28] - reserved

                                if (check_primary_status[30] == 1'b1) // EOP == 1; user_status is zero except when eop is asserted
                                begin
                                    this_expected_user_lo = expected_user_lo;
                                    this_expected_user_hi = expected_user_hi;
                                end
                                else
                                begin
                                    this_expected_user_lo = 32'h0;
                                    this_expected_user_hi = 32'h0;
                                end

                                if (check_primary_status[28] == 1'b1)
                                begin
                                    $display ("%m : ERROR : reg_dma_bar=0x%x : Desc_Ptr = 0x%x, Error on completion (time %t)", reg_dma_bar, curr_desc_ptr, $time);
                                end
                                if ((check_primary_status[27] != (this_expected_user_hi == 32'h0)) & check_status)
                                begin
                                    $display ("%m : ERROR : reg_dma_bar=0x%x : Desc_Ptr = 0x%x, Expected status (user_status_hi_is0 == %d) but got the opposite (time %t)", reg_dma_bar, curr_desc_ptr, (this_expected_user_hi == 32'h0), $time);
                                    inc_errors;
                                    if (on_err_stop)
                                    begin
                                        report_status;
                                        if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                    end
                                end

                                if ((check_primary_status[26] != (this_expected_user_lo == 32'h0)) & check_status)
                                begin
                                    $display ("%m : ERROR : reg_dma_bar=0x%x : Desc_Ptr = 0x%x, Expected status (user_status_lo_is0 == %d) but got the opposite (time %t)", reg_dma_bar, curr_desc_ptr, (this_expected_user_lo == 32'h0), $time);
                                    inc_errors;
                                    if (on_err_stop)
                                    begin
                                        report_status;
                                        if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                    end
                                end
                                if (check_primary_status[25] != expected_short)
                                begin
                                    $display ("%m : ERROR : reg_dma_bar=0x%x : Desc_Ptr = 0x%x, Expected status (short == %d) but got the opposite (time %t)", reg_dma_bar, curr_desc_ptr, expected_short, $time);
                                    inc_errors;
                                    if (on_err_stop)
                                    begin
                                        report_status;
                                        if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                    end
                                end

                                if (check_primary_status[24] != 1'b1)
                                begin
                                    $display ("%m : ERROR : reg_dma_bar=0x%x : Desc_Ptr = 0x%x, Expected status (done == 1'b1) but got the opposite (time %t)", reg_dma_bar, curr_desc_ptr, $time);
                                    inc_errors;
                                    if (on_err_stop)
                                    begin
                                        report_status;
                                        if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                    end
                                end

                                if (check_primary_status[19:0] != expected_bcount[19:0])
                                begin
                                    $display ("%m : ERROR : reg_dma_bar=0x%x : Desc_Ptr = 0x%x, Expected status (bcount_completed == 0x%x) but got 0x%x (time %t)", reg_dma_bar, curr_desc_ptr, expected_bcount[19:0], check_primary_status[19:0], $time);
                                    inc_errors;
                                    if (on_err_stop)
                                    begin
                                        report_status;
                                        if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                    end
                                end

                                // Follow protocol to check user_status[31:0]
                                if (check_primary_status[26] & check_status) // user_status_lo_is0
                                begin
                                    check_user_status_lo = bfm_mem[bfm_dword_addr+1];
                                    if (check_user_status_lo != 32'h0)
                                    begin
                                        $display ("%m : ERROR : reg_dma_bar=0x%x : Desc_Ptr = 0x%x, Status (user_status_lo_is0 == 1) indicating that user_status[31: 0] == 0, but user_status[31: 0] == 0x%x (time %t)", reg_dma_bar, curr_desc_ptr, check_user_status_lo, $time);
                                        inc_errors;
                                        if (on_err_stop)
                                        begin
                                            report_status;
                                            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                        end
                                    end
                                end
                                else if (check_status)
                                begin
                                    while (bfm_mem[bfm_dword_addr+1] == 0)
                                        @(posedge clk);

                                    check_user_status_lo = bfm_mem[bfm_dword_addr+1];

                                    if (check_user_status_lo != this_expected_user_lo)
                                    begin
                                        $display ("%m : ERROR : reg_dma_bar=0x%x : Desc_Ptr = 0x%x, Expected status (user_status[31: 0] == 0x%x) but got 0x%x (time %t)", reg_dma_bar, curr_desc_ptr, this_expected_user_lo, check_user_status_lo, $time);
                                        inc_errors;
                                        if (on_err_stop)
                                        begin
                                            report_status;
                                            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                        end
                                    end
                                end

                                // Follow protocol to check user_status[63:32]
                                if (check_primary_status[27] & check_status) // user_status_hi_is0
                                begin
                                    check_user_status_hi = bfm_mem[bfm_dword_addr+2];
                                    if (check_user_status_hi != 32'h0)
                                    begin
                                        $display ("%m : ERROR : reg_dma_bar=0x%x : Desc_Ptr = 0x%x, Status (user_status_hi_is0 == 1) indicating that user_status[63:32] == 0, but user_status[63:32] == 0x%x (time %t)", reg_dma_bar, curr_desc_ptr, check_user_status_hi, $time);
                                        inc_errors;
                                        if (on_err_stop)
                                        begin
                                            report_status;
                                            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                        end
                                    end
                                end
                                else if (check_status)
                                begin
                                    while (bfm_mem[bfm_dword_addr+2] == 0)
                                        @(posedge clk);

                                    check_user_status_hi = bfm_mem[bfm_dword_addr+2];

                                    if (check_user_status_hi != this_expected_user_hi)
                                    begin
                                        $display ("%m : ERROR : reg_dma_bar=0x%x : Desc_Ptr = 0x%x, Expected status (user_status[63:32] == 0x%x) but got 0x%x (time %t)", reg_dma_bar, curr_desc_ptr, this_expected_user_hi, check_user_status_hi, $time);
                                        inc_errors;
                                        if (on_err_stop)
                                        begin
                                            report_status;
                                            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                        end
                                    end
                                end

                                // --------------------------
                                // Check for correct DMA data

                                // Do a read to the card to flush writes to destination so that we can be sure that all data
                                //   has arrived at its destination; read has no other purpose
                                //mem_read_dword (tc,     addr,        expect_data, check_be, read_data);
                                mem_read_dword   (3'b000, dma_reg_cap, 32'h0,       0,        read_cap);

                                check_sys_addr = curr_sys_addr;

                                // Aborts with Very small values of byte count are frequently wrong.
                                // Skip the data checking
                                if (abort_issued & (expected_bcount < 20'd32)) begin
                                    $display ("%m : INFO : Dropping data check on last %0d bytes", expected_bcount);
                                    drop_data = 1'h1;
                                end
                                else
                                    drop_data = 1'h0;


                                // Walk through the system memory buffer for the Descriptor and verify that it contains the correct data
                                for (i=0; i<xfer_bcount; i=i+4)
                                begin
                                    bfm_dword_addr = check_sys_addr[BFM_MEM_SIZE_BITS+1:2];// Drop highest address bits since BFM memory is indexed without a base address
                                    mem = bfm_mem[bfm_dword_addr];

                                    if (i<expected_bcount) // Check against expected_data
                                    begin
                                        // Don't check data before check_sys_addr[1:0] as this was not included in the DMA operation
                                        case(check_sys_addr[1:0])
                                            2'b00 : exp =  expected_data[31: 0];
                                            2'b01 : exp = {expected_data[31: 8],      8'hxx};
                                            2'b10 : exp = {expected_data[31:16],   16'hxxxx};
                                            2'b11 : exp = {expected_data[31:24], 24'hxxxxxx};
                                        endcase

                                        if ((((mem[31:24] !== expected_data[31:24])                            ) | // Only check bytes that were part of the transfer
                                             ((mem[23:16] !== expected_data[23:16]) & (check_sys_addr[1:0] < 3)) |
                                             ((mem[15: 8] !== expected_data[15: 8]) & (check_sys_addr[1:0] < 2)) |
                                             ((mem[ 7: 0] !== expected_data[ 7: 0]) & (check_sys_addr[1:0] < 1)) ) & ~drop_data)
                                        begin
                                            $display ("%m : ERROR : reg_dma_bar=0x%x : BfmMemOffset == 0x%x : Expected Data == 0x%x, Received Data == 0x%x (time %t)", reg_dma_bar, check_sys_addr, exp, mem, $time);
                                            inc_errors;
                                            if (on_err_stop)
                                            begin
                                                report_status;
                                                if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                            end
                                        end
                                        else
                                        begin
                                            if (my_verbose > 2)
                                                $display ("%m : PASS : reg_dma_bar=0x%x : BfmMemOffset == 0x%x : Expected Data == 0x%x, Received Data == 0x%x", reg_dma_bar, check_sys_addr, exp, mem);
                                        end

                                        // Get next data value in the pattern
                                        get_packet_pattern (pat_data_type, expected_data, next_expected_data);
                                        expected_data = next_expected_data;
                                    end
                                    else
                                    begin // Unused data buffer should not have been modified, was earlier filled with 1s, and thus should still contain all 1s
                                        // Don't check data before check_sys_addr[1:0] as this was not included in the DMA operation
                                        case(check_sys_addr[1:0])
                                            2'b00 : exp =  32'hffffffff;
                                            2'b01 : exp = {24'hffffff,  8'hxx};
                                            2'b10 : exp = {16'hffff, 16'hxxxx};
                                            2'b11 : exp = { 8'hff, 24'hxxxxxx};
                                        endcase

                                        if ( ((mem[31:24] !== 8'hff)                            ) | // Only check bytes that were part of the transfer
                                             ((mem[23:16] !== 8'hff) & (check_sys_addr[1:0] < 3)) |
                                             ((mem[15: 8] !== 8'hff) & (check_sys_addr[1:0] < 2)) |
                                             ((mem[ 7: 0] !== 8'hff) & (check_sys_addr[1:0] < 1)) )
                                        begin
                                            $display ("%m : ERROR : reg_dma_bar=0x%x : BfmMemOffset == 0x%x : Expected Data == 0x%x, Received Data == 0x%x (time %t)", reg_dma_bar, check_sys_addr, exp, mem, $time);
                                            inc_errors;
                                            if (on_err_stop)
                                            begin
                                                report_status;
                                                if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                            end
                                        end
                                        else
                                        begin
                                            if (my_verbose > 2)
                                                $display ("%m : PASS : reg_dma_bar=0x%x : BfmMemOffset == 0x%x : Expected Data == 0x%x, Received Data == 0x%x", reg_dma_bar, check_sys_addr, exp, mem);
                                        end
                                    end

                                    // Get next check address; move 1 DWORD at a time
                                    check_sys_addr = check_sys_addr + (4 - check_sys_addr[1:0]);
                                end
                            end

                            // --------------------------------
                            // Increment for the next Descrptor

                            curr_desc_ptr = curr_desc_ptr + 32; // Advance to the next Descriptor address; descriptors are 32 bytes each
                            curr_sys_addr = curr_sys_addr + xfer_bcount;
                            curr_bcount   = (xfer_bcount > curr_bcount) ? 0 : (curr_bcount - xfer_bcount);
                            num_desc      = num_desc       - 1;
                            expected_sop  = 0;
                        end

                        // Roll user_status pattern if continuous user_status is requested
                        if (pat_user_cont == 1'b1)
                        begin
                            get_packet_pattern (pat_user_type, expected_user_hi, expected_user_lo);
                            get_packet_pattern (pat_user_type, expected_user_lo, expected_user_hi);
                        end
                    end
                    if (abort_issued)
                        $display ("%m : INFO : C2S DMA aborted. Total bytes transferred: %1d", total_bcount);
                end
                else // Check for expected S2C operation
                begin
                    // Wait for all Descriptors to complete
                    curr_desc_ptr = desc_ptr;
                    check_exit = 1'b0;

                    // Reset timeout counter to initial value
                    timeout_ctr = timeout_clocks;

                    while (check_exit == 1'b0)
                    begin
                        bfm_dword_addr = curr_desc_ptr[BFM_MEM_SIZE_BITS+1:2];// Drop highest address bits since BFM memory is indexed without a base address

                        // Wait for all Descriptors in the task to complete
                        while (((bfm_mem[bfm_dword_addr+0] & 32'hFF000000) == 0) & (check_exit == 1'b0)) // Descriptor status byte not updated
                        begin
                            @(posedge clk);

                            if (timeout_en)
                            begin
                                timeout_ctr = timeout_ctr - 32'h1;
                                if (timeout_ctr == 32'h0)
                                    check_exit = 1'b1; // Exit while loop
                            end
                        end

                        if (curr_desc_ptr == last_desc_ptr) begin       // Last Descriptor Completed
                            check_primary_status = bfm_mem[bfm_dword_addr+0];
                            total_bcount = total_bcount + check_primary_status[19:0];
                            check_exit = 1'b1; // Exit while loop
                        end
                        else if (timeout_en & (timeout_ctr == 32'h0))   // DMA Timeout Occurred
                        begin
                            check_exit = 1'b1; // Exit while loop
                            $display ("%m : %0s : reg_dma_bar=0x%x : DMA operation failed to make progress for 0x%x clocks.  DMA may be locked up.  Aborting test and resetting DMA Engine.  (time %t)",
                                       on_err_stop ? "ERROR" : "INFO", reg_dma_bar, timeout_clocks, $time);
                            if (on_err_stop)
                            begin
                                inc_errors;
                                report_status;
                                if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                            end

                            // Reset DMA Engine
                            //mem_write_dword (tc,     addr,        data,            be);
                            mem_write_dword   (3'b000, dma_reg_cst, 32'h00008000,    4'hf);

                            // Poll until it is seen that reset is completed
                            read_cst = 32'h00008000; // Not done status
                            while (read_cst[15] == 1'b1)
                            begin
                                //mem_read_dword (tc,     addr,        expect_data, check_be, read_data);
                                mem_read_dword   (3'b000, dma_reg_cst, 32'h0,       0,        read_cst);
                            end
                        end
                        else // A Descriptor completed that was not the last Descriptor
                        begin
                            curr_desc_ptr = curr_desc_ptr + 32; // Get next Descriptor

                            // Check for abort
                            check_primary_status = bfm_mem[bfm_dword_addr+0];
                            if (check_primary_status[25] == 1'b1) begin                         // Short
                                mem_read_dword ( 3'b0, dma_reg_cst, 32'h0, 4'h0, read_cst);     // Check status
                                if (read_cst[10] == 1'b0) begin                                 // If DMA engine is stopped, we have aborted
                                    check_exit = 1'b1;
                                    abort_issued = 1'b1;
                                end
                            end
                            total_bcount = total_bcount + check_primary_status[19:0];

                            // Rest timeout_ctr when making progress
                            timeout_ctr = timeout_clocks;
                        end
                    end
                    if (abort_issued)
                        $display ("%m : INFO : S2C DMA aborted. Total bytes transferred: %1d", total_bcount);

                    // Read Pattern Checker control register until Pattern Checker indicates it is idle;
                    //   Don't want to read error register or allow Pattern Checker to be reused until
                    //   all data from the previous operation completed
                    if (~abort_issued) begin                                                    // If DMA was not aborted, wait for checker to finish
                        read_ctl = 32'h1; // Not done status
                        while (read_ctl[0] == 1'b1)
                        begin
                            //mem_read_dword (tc,     addr,        expect_data, check_be, read_data);
                            mem_read_dword   (3'b000, dma_pat_ctl, 32'h0,       0,        read_ctl);
                        end
                    end
                    // Read Pattern Checker Error register
                    //mem_read_dword (tc,     addr,        expect_data, check_be, read_data);
                    mem_read_dword   (3'b000, dma_pat_err, 32'h0,       0,        read_err);

                    if (read_err !== 0) // One or more errors where received
                    begin
                        if (read_err[31:8] !== 24'h0) // err_ctr[23:0] != 0 indicates one or more errors occurred
                        begin
                            if (read_err[31:8] === 24'hffffff)
                                $display ("%m : ERROR : reg_dma_bar=0x%x : Task completed, but %d Errors were detected; Error count is saturated, there may have been even more errors (time %t)", reg_dma_bar, read_err[31:8], $time);
                            else
                                $display ("%m : ERROR : reg_dma_bar=0x%x : Task completed, but %d Errors were detected (time %t)", reg_dma_bar, read_err[31:8], $time);
                        end

                        if (read_err[5] !== 1'b0) // err_user_control
                            $display ("%m : ERROR : reg_dma_bar=0x%x : err_user_control==1 : Pattern Checker received one or more user_control errors.  (time %t)", reg_dma_bar, $time);
                        else
                            $display ("%m : INFO  : reg_dma_bar=0x%x : err_user_control==0 : No errors of this type.                                    (time %t)", reg_dma_bar, $time);

                        if (read_err[4] !== 1'b0) // err_data_valid
                            $display ("%m : ERROR : reg_dma_bar=0x%x : err_data_valid==1   : Pattern Checker received one or more data_valid errors.    (time %t)", reg_dma_bar, $time);
                        else
                            $display ("%m : INFO  : reg_dma_bar=0x%x : err_data_valid==0   : No errors of this type.                                    (time %t)", reg_dma_bar, $time);

                        if (read_err[3] !== 1'b0) // err_data
                            $display ("%m : ERROR : reg_dma_bar=0x%x : err_data==1         : Pattern Checker received one or more data errors.          (time %t)", reg_dma_bar, $time);
                        else
                            $display ("%m : INFO  : reg_dma_bar=0x%x : err_data==0         : No errors of this type.                                    (time %t)", reg_dma_bar, $time);

                        if (read_err[2] !== 1'b0) // err_cpl
                            $display ("%m : ERROR : reg_dma_bar=0x%x : err_cpl==1          : Pattern Checker received one or more completion errors.    (time %t)", reg_dma_bar, $time);
                        else
                            $display ("%m : INFO  : reg_dma_bar=0x%x : err_cpl==0          : No errors of this type.                                    (time %t)", reg_dma_bar, $time);

                        if (read_err[1] !== 1'b0) // err_eop
                            $display ("%m : ERROR : reg_dma_bar=0x%x : err_eop==1          : Pattern Checker received one or more eop errors.           (time %t)", reg_dma_bar, $time);
                        else
                            $display ("%m : INFO  : reg_dma_bar=0x%x : err_eop==0          : No errors of this type.                                    (time %t)", reg_dma_bar, $time);

                        if (read_err[0] !== 1'b0) // err_sop
                            $display ("%m : ERROR : reg_dma_bar=0x%x : err_sop==1          : Pattern Checker received one or more sop errors.           (time %t)", reg_dma_bar, $time);
                        else
                            $display ("%m : INFO  : reg_dma_bar=0x%x : err_sop==0          : No errors of this type.                                    (time %t)", reg_dma_bar, $time);

                        // Increment error counter by 1 for each error that occurred
                        for (i=0; i<read_err[31:8]; i=i+1)
                            inc_errors;

                        if (on_err_stop)
                        begin
                            report_status;
                            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                        end

                        // Clear Packet Checker error register
                        //mem_write_dword (tc,     addr,        data,         be);
                        mem_write_dword   (3'b000, dma_pat_err, 32'h00000080, 4'h1);

                        // Read Pattern Checker Error register; verify that is cleared
                        //mem_read_dword (tc,     addr,        expect_data, check_be, read_data);
                        mem_read_dword   (3'b000, dma_pat_err, 32'h0,       4'hf,     read_err);

                        if (read_err !== 32'h0)
                        begin
                            inc_errors;
                            $display ("%m : ERROR : reg_dma_bar=0x%x : Failed to be able to clear Packet Check error register. (time %t)", reg_dma_bar, $time);
                            if (on_err_stop)
                            begin
                                report_status;
                                if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                            end
                        end
                    end
                    else if (my_verbose > 2) // If no errors and verbose, report pass
                    begin
                        $display ("%m : PASS : reg_dma_bar=0x%x : Task completed with no errors : (time %t)", reg_dma_bar, $time);
                    end
                end
            end
        end
    end
end
endtask



// --------------------------------------------------
//  task: do_pkt_dma_loopback
//    Puts a Packet Generator/Checker pair in loopback
//      mode.  Packets that are transmitted on the S2C
//      DMA Engine are received on the C2S DMA Engine.
//      Received packets are verified against transmitted
//      packets.  A circular buffer is created for each of
//      the S2C and C2S Packet DMA Engines.  The circular
//      buffers are randomized indpendently to create
//      unique buffers to put stress on packet recombination.
//    This task uses interrupts to read DMA Engine Completed
//      Descriptor pointers to determine completion status.

task automatic do_pkt_dma_loopback;

input   [63:0]                  s2c_rbuf_system_addr;       // S2C DMA Ring Buffer : Starting System Address
input   [63:0]                  s2c_rbuf_system_bsize;      // S2C DMA Ring Buffer : Buffer size in bytes
input   [63:0]                  s2c_rbuf_desc_ptr;          // S2C DMA Ring Buffer : Starting System Address where Descriptors will be located
input   [31:0]                  s2c_rbuf_desc_max_bsize;    // S2C DMA Ring Buffer : Maximum size of each Descriptor in bytes
input   [31:0]                  s2c_rbuf_desc_min_bsize;    // S2C DMA Ring Buffer : Maximum size of each Descriptor in bytes
input   [31:0]                  s2c_rbuf_desc_max_num;      // S2C DMA Ring Buffer : Maximum number of Descriptors to implement in Circular buffer

input   [63:0]                  c2s_rbuf_system_addr;       // C2S DMA Ring Buffer : Starting System Address
input   [63:0]                  c2s_rbuf_system_bsize;      // C2S DMA Ring Buffer : Buffer size in bytes
input   [63:0]                  c2s_rbuf_desc_ptr;          // C2S DMA Ring Buffer : Starting System Address where Descriptors will be located
input   [31:0]                  c2s_rbuf_desc_max_bsize;    // C2S DMA Ring Buffer : Maximum size of each Descriptor in bytes
input   [31:0]                  c2s_rbuf_desc_min_bsize;    // C2S DMA Ring Buffer : Maximum size of each Descriptor in bytes
input   [31:0]                  c2s_rbuf_desc_max_num;      // C2S DMA Ring Buffer : Maximum number of Descriptors to implement in Circular buffer

input   [19:0]                  byte_alignment;             // Byte alignment to enforce for Packet Length, Descriptor System Addresses, and Descriptor Byte Counts; must be a positive power of 2;
                                                            //   for example, if alignment == 16 then Packet Length[3:0], Descriptor System Address[3:0] and Descriptor Byte Count[3:0] are required to equal 0000;
                                                            //   If a hardware design has a minimum alignment that it can handle, then alignment should be set to that limit; alignment is also useful for
                                                            //   testing various buffer configurations

input   [63:0]                  reg_com_bar;                // DMA Common Register Block Base Address; needed to set global enables
input   [1:0]                   int_mode;                   // 2==MSI-X, 1==MSI, 0==Legacy

input   [63:0]                  reg_s2c_dma_bar;            // S2C DMA Engine Register Block Base Address; chooses which engine will be used; must use a valid pair connected in loopback
input   [63:0]                  reg_s2c_pat_bar;            // S2C DMA Engine Pattern Register Block Base Address; must be the pattern generator connected to used DMA Engine
input   [11:0]                  s2c_int_vector;             // MSI-X/MSI Vector used by the DMA Engine

input   [63:0]                  reg_c2s_dma_bar;            // C2S DMA Engine Register Block Base Address; chooses which engine will be used; must use a valid pair connected in loopback
input   [63:0]                  reg_c2s_pat_bar;            // C2S DMA Engine Pattern Register Block Base Address; must be the pattern generator connected to used DMA Engine
input   [11:0]                  c2s_int_vector;             // MSI-X/MSI Vector used by the DMA Engine

input   [31:0]                  pkt_max_bsize;              // Maximum packet size in bytes; packet size is randomized up to this size
input   [31:0]                  pkt_num_packets;            // Number of packets to transmit/receive; this many packets are transmitted and received before ending the task (max value = DO_PKT_DMA_LOOPBACK_MAX_PKT_NUM_PACKETS)

input   [31:0]                  pat_user_seed;              // Seed value for user_status/control pattern; subsequent data continues from packet to packet
input   [2:0]                   pat_user_type;              // User status/control pattern: 0==CONSTANT, 1==INC BYTE, 2==LFSR; 3==INC_DWORD

input   [31:0]                  pat_data_seed;              // Seed value for data pattern; subsequent data continues from packet to packet
input   [2:0]                   pat_data_type;              // Data pattern:                0==CONSTANT, 1==INC BYTE, 2==LFSR; 3==INC_DWORD        // Initial value for user_control/status

input   [31:0]                  verbose;                    // Log Display Control : 0 == Only errors; 1 == Limited messages; 2=Expanded messages; 3=Verbose
input                           on_err_stop;                // Set to stop when an error is detected
input                           timeout_en;                 // Set to enable DMA timeout functionality
input   [31:0]                  timeout_clocks;             // If timeout_en==1 and the DMA fails to make progress for timeout_clocks clock cycles, the DMA operation is aborted and an error generated;

input                           check_status;
input   [31:0]                  axi_pcie_n;                 // Unused, tie to 0

reg     [31:0]                  my_verbose;

reg     [63:0]                  alignment;
reg     [63:0]                  alignment_mask;

// DMA Engine Registers
reg     [63:0]                  dma_s2c_reg_cap;
reg     [63:0]                  dma_s2c_reg_cst;
reg     [63:0]                  dma_s2c_reg_ndp;
reg     [63:0]                  dma_s2c_reg_sdp;
reg     [63:0]                  dma_s2c_reg_cdp;

reg     [63:0]                  dma_c2s_reg_cap;
reg     [63:0]                  dma_c2s_reg_cst;
reg     [63:0]                  dma_c2s_reg_ndp;
reg     [63:0]                  dma_c2s_reg_sdp;
reg     [63:0]                  dma_c2s_reg_cdp;

reg     [63:0]                  dma_s2c_pat_ctl;

reg     [63:0]                  dma_c2s_pat_ctl;

reg     [31:0]                  read_cap;

reg                             s2c_eng_present;
reg                             s2c_eng_type_c2s_s2c_n;
reg                             s2c_eng_pkt_block_n;
reg     [7:0]                   s2c_eng_num;
reg     [63:0]                  s2c_card_addr_size;
reg     [63:0]                  s2c_max_desc_bsize;

reg                             c2s_eng_present;
reg                             c2s_eng_type_c2s_s2c_n;
reg                             c2s_eng_pkt_block_n;
reg     [7:0]                   c2s_eng_num;
reg     [63:0]                  c2s_card_addr_size;
reg     [63:0]                  c2s_max_desc_bsize;

reg                             check_exit;
reg                             done_wait;

reg     [31:0]                  pkt_max_bsize_mask;

// Create Circular Buffers
integer                         i;
reg     [63:0]                  curr_sys_addr;
reg     [63:0]                  limit_sys_addr;
reg     [31:0]                  desc_ptr;
reg     [31:0]                  curr_desc_ptr;

reg     [31:0]                  desc_max_bsize;
reg     [31:0]                  desc_min_bsize;
reg     [31:0]                  desc_limit;

reg     [31:0]                  desc_range;
reg     [31:0]                  desc_range_mask;

reg     [63:0]                  addr;
reg     [31:0]                  bfm_dword_addr;
reg     [31:0]                  read_mem;
integer                         num_desc;
reg     [19:0]                  xfer_bcount;
reg     [31:0]                  r;
reg     [31:0]                  next_desc_ptr;
integer                         j;
reg     [31:0]                  s2c_rbuf_last_desc_ptr;
reg     [31:0]                  c2s_rbuf_last_desc_ptr;

// Generate random lengths for the requested number of packets
integer                         avg_pkt_size;
integer                         p;
reg     [31:0]                  pkt_size    [DO_PKT_DMA_LOOPBACK_MAX_PKT_NUM_PACKETS-1:0];

// Put Packet Generator/Checker Pair in Loopback
reg     [31:0]                  read_lpe;

// Enable Global DMA Interrupts
reg                             transmit_exit;
reg                             receive_exit;
integer                         c2s_int_check;
integer                         s2c_int_check;
integer                         c2s_int_handled;
integer                         s2c_int_handled;

reg     [31:0]                  s2c_cdp;
reg     [31:0]                  c2s_cdp;

// Reset DMA Engines
reg     [31:0]                  read_cst;

begin
    // Optional over-ride to turn off DMA messages
    if ($test$plusargs("pcie_dma_msgs_off"))
        my_verbose = 0;
    else
        my_verbose = verbose;

    if (my_verbose > 0)
    begin
        $display ("%m : Starting Packet DMA: reg_s2c_dma_bar=0x%x, reg_com_bar=0x%x, s2c_int_vector=0x%x, c2s_int_vector=0x%x (time %t)",                         reg_s2c_dma_bar, reg_com_bar, s2c_int_vector, c2s_int_vector,                            $time);
        $display ("%m : Starting Packet DMA: reg_s2c_dma_bar=0x%x, reg_c2s_dma_bar=0x%x, reg_s2c_pat_bar=0x%x, reg_c2s_pat_bar=0x%x (time %t)",                   reg_s2c_dma_bar, reg_c2s_dma_bar,      reg_s2c_pat_bar,         reg_c2s_pat_bar,         $time);
        $display ("%m : Starting Packet DMA: reg_s2c_dma_bar=0x%x, s2c_rbuf_system_addr=0x%x, s2c_rbuf_system_bsize=0x%x (time %t)",                              reg_s2c_dma_bar, s2c_rbuf_system_addr, s2c_rbuf_system_bsize,                            $time);
        $display ("%m : Starting Packet DMA: reg_s2c_dma_bar=0x%x, s2c_rbuf_desc_ptr=0x%x, s2c_rbuf_desc_max_bsize=0x%x, s2c_rbuf_desc_min_bsize=0x%x (time %t)", reg_s2c_dma_bar, s2c_rbuf_desc_ptr,    s2c_rbuf_desc_max_bsize, s2c_rbuf_desc_min_bsize, $time);
        $display ("%m : Starting Packet DMA: reg_s2c_dma_bar=0x%x, c2s_rbuf_system_addr=0x%x, c2s_rbuf_system_bsize=0x%x (time %t)",                              reg_s2c_dma_bar, c2s_rbuf_system_addr, c2s_rbuf_system_bsize,                            $time);
        $display ("%m : Starting Packet DMA: reg_s2c_dma_bar=0x%x, c2s_rbuf_desc_ptr=0x%x, c2s_rbuf_desc_max_bsize=0x%x, c2s_rbuf_desc_min_bsize=0x%x (time %t)", reg_s2c_dma_bar, c2s_rbuf_desc_ptr,    c2s_rbuf_desc_max_bsize, c2s_rbuf_desc_min_bsize, $time);
        $display ("%m : Starting Packet DMA: reg_s2c_dma_bar=0x%x, pkt_max_bsize=0x%x, pkt_num_packets=0x%x (time %t)",                                           reg_s2c_dma_bar, pkt_max_bsize,        pkt_num_packets,                                  $time);
    end

    // Alignment to use for Circular Buffer System Address Pointers and Descriptor Sizes and
    //   random packet sizes; extend to 64-bits since need to mask up to 64-bit quantities
    alignment = {44'h0, byte_alignment};

    // Create a mask that can be used to limit byte counts to
    //   the requested alignment; mask is setup to be ANDed
    //   with a byte quantity to truncate the byte quantity
    //   to an aligned value
    alignment_mask = alignment - 64'h1;
    alignment_mask = ~alignment_mask;



    // --------------------
    // DMA Engine Registers

    // Get S2C DMA Engine Register Addresses
    dma_s2c_reg_cap = reg_s2c_dma_bar + 'h00;
    dma_s2c_reg_cst = reg_s2c_dma_bar + 'h04;
    dma_s2c_reg_ndp = reg_s2c_dma_bar + 'h08;
    dma_s2c_reg_sdp = reg_s2c_dma_bar + 'h0c;
    dma_s2c_reg_cdp = reg_s2c_dma_bar + 'h10;

    // Get C2S DMA Engine Register Addresses
    dma_c2s_reg_cap = reg_c2s_dma_bar + 'h00;
    dma_c2s_reg_cst = reg_c2s_dma_bar + 'h04;
    dma_c2s_reg_ndp = reg_c2s_dma_bar + 'h08;
    dma_c2s_reg_sdp = reg_c2s_dma_bar + 'h0c;
    dma_c2s_reg_cdp = reg_c2s_dma_bar + 'h10;

    // Get S2C DMA Engine Pattern Checker Addresses
    dma_s2c_pat_ctl = reg_s2c_pat_bar + 'h00;

    // Get C2S DMA Engine Pattern Generator Addresses
    dma_c2s_pat_ctl = reg_c2s_pat_bar + 'h00;

    // Get S2C DMA Engine Capabilities
    //mem_read_dword (tc,     addr,            expect_data, check_be, read_data);
    mem_read_dword   (3'b000, dma_s2c_reg_cap, 32'h0,       0,        read_cap);

    s2c_eng_present        =      read_cap[    0];
    s2c_eng_type_c2s_s2c_n =      read_cap[    1];
    s2c_eng_pkt_block_n    =      read_cap[    4];
    s2c_eng_num            =      read_cap[15: 8];
    s2c_card_addr_size     = ((1<<read_cap[22:16])/1024);
    s2c_max_desc_bsize     =  (1<<read_cap[29:24])-1;

    // Get C2S DMA Engine Capabilities
    //mem_read_dword (tc,     addr,            expect_data, check_be, read_data);
    mem_read_dword   (3'b000, dma_c2s_reg_cap, 32'h0,       0,        read_cap);
    c2s_eng_present        =      read_cap[    0];
    c2s_eng_type_c2s_s2c_n =      read_cap[    1];
    c2s_eng_pkt_block_n    =      read_cap[    4];
    c2s_eng_num            =      read_cap[15: 8];
    c2s_card_addr_size     = ((1<<read_cap[22:16])/1024);
    c2s_max_desc_bsize     =  (1<<read_cap[29:24])-1;

    if (my_verbose > 0)
    begin
        // If an engine is found, then display the capabilities to the log
        if (s2c_eng_present)
        begin
            if (s2c_eng_type_c2s_s2c_n == 1'b0)
                $display ("%m : INFO : reg_s2c_dma_bar=0x%x : DMA Capabilities: Present, System to Card, Engine#%d, Card Address Size %1d KBytes, MaxDescriptorSize=%1d Bytes", reg_s2c_dma_bar, s2c_eng_num, s2c_card_addr_size, s2c_max_desc_bsize);
            else
                $display ("%m : INFO : reg_s2c_dma_bar=0x%x : DMA Capabilities: Present, Card to System, Engine#%d, Card Address Size %1d KBytes, MaxDescriptorSize=%1d Bytes", reg_s2c_dma_bar, s2c_eng_num, s2c_card_addr_size, s2c_max_desc_bsize);
        end

        // If an engine is found, then display the capabilities to the log
        if (c2s_eng_present)
        begin
            if (c2s_eng_type_c2s_s2c_n == 1'b0)
                $display ("%m : INFO : reg_s2c_dma_bar=0x%x : DMA Capabilities: Present, System to Card, Engine#%d, Card Address Size %1d KBytes, MaxDescriptorSize=%1d Bytes", reg_c2s_dma_bar, c2s_eng_num, c2s_card_addr_size, c2s_max_desc_bsize);
            else
                $display ("%m : INFO : reg_s2c_dma_bar=0x%x : DMA Capabilities: Present, Card to System, Engine#%d, Card Address Size %1d KBytes, MaxDescriptorSize=%1d Bytes", reg_c2s_dma_bar, c2s_eng_num, c2s_card_addr_size, c2s_max_desc_bsize);
        end
    end

    // Check task inputs for validity
    if (!((s2c_eng_present == 1'b1) && (c2s_eng_present == 1'b1)))
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : At least one System to Card and one Card to System DMA Engine must be present (time %t)", reg_s2c_dma_bar, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (!((s2c_eng_type_c2s_s2c_n == 1'b0) && (c2s_eng_type_c2s_s2c_n == 1'b1)))
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : reg_s2c_dma_bar must be a System to Card DMA Engine and reg_c2s_dma_bar must be a Card to System DMA Engine (time %t)", reg_s2c_dma_bar, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (!((s2c_eng_pkt_block_n == 1'b1) && (c2s_eng_pkt_block_n == 1'b1)))
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : reg_s2c_dma_bar and reg_c2s_dma_bar must be packet DMA Engines (time %t)", reg_s2c_dma_bar, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if ((s2c_rbuf_desc_ptr[4:0] != 5'h0) | (c2s_rbuf_desc_ptr[4:0] != 5'h0))
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : Descriptor pointers must be aligned to a 32 byte address boundary; s2c_rbuf_desc_ptr=0x%x, c2s_rbuf_desc_ptr=0x%x (time %t)", reg_s2c_dma_bar, s2c_rbuf_desc_ptr, c2s_rbuf_desc_ptr, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (pkt_num_packets == 32'h0)
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : pkt_num_packets=0x%x must be non-zero (time %t)", reg_s2c_dma_bar, pkt_num_packets, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (pkt_num_packets > DO_PKT_DMA_LOOPBACK_MAX_PKT_NUM_PACKETS)
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : pkt_num_packets=0x%x must be <= 0x%x (time %t)", reg_s2c_dma_bar, pkt_num_packets, DO_PKT_DMA_LOOPBACK_MAX_PKT_NUM_PACKETS, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if ( (( c2s_rbuf_system_addr                          <= s2c_rbuf_system_addr) & ((c2s_rbuf_system_addr + c2s_rbuf_system_bsize) >= (s2c_rbuf_system_addr + s2c_rbuf_system_bsize))) | // C2S Range includes S2C Range
              (( c2s_rbuf_system_addr                          >= s2c_rbuf_system_addr) & ( c2s_rbuf_system_addr                          <  (s2c_rbuf_system_addr + s2c_rbuf_system_bsize))) | // C2S Start Inside S2C Range
              (((c2s_rbuf_system_addr + c2s_rbuf_system_bsize) >  s2c_rbuf_system_addr) & ((c2s_rbuf_system_addr + c2s_rbuf_system_bsize) <= (s2c_rbuf_system_addr + s2c_rbuf_system_bsize))) | // C2S End Inside S2C Range
              (( s2c_rbuf_system_addr                          <= c2s_rbuf_system_addr) & ((s2c_rbuf_system_addr + s2c_rbuf_system_bsize) >= (c2s_rbuf_system_addr + c2s_rbuf_system_bsize))) | // S2C Range includes C2S Range
              (( s2c_rbuf_system_addr                          >= c2s_rbuf_system_addr) & ( s2c_rbuf_system_addr                          <  (c2s_rbuf_system_addr + c2s_rbuf_system_bsize))) | // S2C Start Inside C2S Range
              (((s2c_rbuf_system_addr + s2c_rbuf_system_bsize) >  c2s_rbuf_system_addr) & ((s2c_rbuf_system_addr + s2c_rbuf_system_bsize) <= (c2s_rbuf_system_addr + c2s_rbuf_system_bsize)))   // S2C End Inside C2S Range
            )
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : s2c_rbuf_system_addr range overlaps with c2s_rbuf_system_addr range (time %t)", reg_s2c_dma_bar, $time);
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x :   s2c_rbuf_system_addr=0x%x; s2c_rbuf_system_bsize=0x%x", reg_s2c_dma_bar, s2c_rbuf_system_addr, s2c_rbuf_system_bsize);
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x :   c2s_rbuf_system_addr=0x%x; c2s_rbuf_system_bsize=0x%x", reg_s2c_dma_bar, c2s_rbuf_system_addr, c2s_rbuf_system_bsize);
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x :     s2c_rbuf_system_addr range[0x%x : 0x%x]", reg_s2c_dma_bar, s2c_rbuf_system_addr, (s2c_rbuf_system_addr + s2c_rbuf_system_bsize - 1));
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x :     c2s_rbuf_system_addr range[0x%x : 0x%x]", reg_s2c_dma_bar, c2s_rbuf_system_addr, (c2s_rbuf_system_addr + c2s_rbuf_system_bsize - 1));
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (s2c_rbuf_desc_max_bsize > s2c_max_desc_bsize)
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : Requested max descriptor size (s2c_rbuf_desc_max_bsize=0x%x) > DMA Engine max supported descriptor size (s2c_max_desc_bsize=0x%x) (time %t)", reg_s2c_dma_bar, s2c_rbuf_desc_max_bsize, s2c_max_desc_bsize, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (s2c_rbuf_desc_min_bsize > s2c_max_desc_bsize)
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : Requested min descriptor size (s2c_rbuf_desc_min_bsize=0x%x) > DMA Engine max supported descriptor size (s2c_max_desc_bsize=0x%x) (time %t)", reg_s2c_dma_bar, s2c_rbuf_desc_min_bsize, s2c_max_desc_bsize, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (c2s_rbuf_desc_max_bsize > c2s_max_desc_bsize)
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : Requested max descriptor size (c2s_rbuf_desc_max_bsize=0x%x) > DMA Engine max supported descriptor size (c2s_max_desc_bsize=0x%x) (time %t)", reg_s2c_dma_bar, c2s_rbuf_desc_max_bsize, c2s_max_desc_bsize, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (c2s_rbuf_desc_min_bsize > c2s_max_desc_bsize)
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : Requested min descriptor size (c2s_rbuf_desc_min_bsize=0x%x) > DMA Engine max supported descriptor size (c2s_max_desc_bsize=0x%x) (time %t)", reg_s2c_dma_bar, c2s_rbuf_desc_min_bsize, c2s_max_desc_bsize, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if ((s2c_rbuf_desc_max_bsize % 2) != 32'h0)
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : Requested max descriptor size (s2c_rbuf_desc_max_bsize=0x%x) must be a power of 2 (time %t)", reg_s2c_dma_bar, s2c_rbuf_desc_max_bsize, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if ((s2c_rbuf_desc_min_bsize % 2) != 32'h0)
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : Requested min descriptor size (s2c_rbuf_desc_min_bsize=0x%x) must be a power of 2 (time %t)", reg_s2c_dma_bar, s2c_rbuf_desc_min_bsize, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if ((c2s_rbuf_desc_max_bsize % 2) != 32'h0)
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : Requested max descriptor size (c2s_rbuf_desc_max_bsize=0x%x) must be a power of 2 (time %t)", reg_s2c_dma_bar, c2s_rbuf_desc_max_bsize, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if ((c2s_rbuf_desc_min_bsize % 2) != 32'h0)
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : Requested min descriptor size (c2s_rbuf_desc_min_bsize=0x%x) must be a power of 2 (time %t)", reg_s2c_dma_bar, c2s_rbuf_desc_min_bsize, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if ((pkt_max_bsize % 2) != 32'h0)
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : Requested max packet size (pkt_max_bsize=0x%x) must be a power of 2 (time %t)", reg_s2c_dma_bar, pkt_max_bsize, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if ((s2c_rbuf_desc_ptr[63:32] != 32'h0) | (c2s_rbuf_desc_ptr[63:32] != 32'h0))
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : Packet DMA Engines do not support Descriptors being located at 64-bit addresses; s2c_rbuf_desc_ptr=0x%x, c2s_rbuf_desc_ptr=0x%x (time %t)", reg_s2c_dma_bar, s2c_rbuf_desc_ptr, c2s_rbuf_desc_ptr, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (s2c_rbuf_system_bsize == 0)
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : (s2c_rbuf_system_bsize=0x%x) must be non-zero (time %t)", reg_s2c_dma_bar, s2c_rbuf_system_bsize, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (c2s_rbuf_system_bsize == 0)
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : (c2s_rbuf_system_bsize=0x%x) must be non-zero (time %t)", reg_s2c_dma_bar, c2s_rbuf_system_bsize, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (s2c_rbuf_desc_max_num == 0)
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : (s2c_rbuf_desc_max_num=0x%x) must be non-zero (time %t)", reg_s2c_dma_bar, s2c_rbuf_desc_max_num, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (c2s_rbuf_desc_max_num == 0)
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : (c2s_rbuf_desc_max_num=0x%x) must be non-zero (time %t)", reg_s2c_dma_bar, c2s_rbuf_desc_max_num, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if ((s2c_rbuf_system_addr & alignment_mask) != s2c_rbuf_system_addr)
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : (s2c_rbuf_system_addr=0x%x) must be aligned to (alignment=0x%x) (time %t)", reg_s2c_dma_bar, s2c_rbuf_system_addr, alignment, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if ((c2s_rbuf_system_addr & alignment_mask) != c2s_rbuf_system_addr)
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : (c2s_rbuf_system_addr=0x%x) must be aligned to (alignment=0x%x) (time %t)", reg_s2c_dma_bar, c2s_rbuf_system_addr, alignment, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (!( (alignment ==    1) || (alignment ==    2) || (alignment ==     4) || (alignment ==     8) ||
                (alignment ==   16) || (alignment ==   32) || (alignment ==    64) || (alignment ==   128) ||
                (alignment ==  256) || (alignment ==  512) || (alignment ==  1024) || (alignment ==  2048) ||
                (alignment == 4096) || (alignment == 8192) || (alignment == 16384) || (alignment == 32768) ) )
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : (alignment=0x%x) must be a positive power of 2 between 1 and 32768 (time %t)", reg_s2c_dma_bar, alignment, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (s2c_rbuf_desc_max_bsize < alignment)
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : (s2c_rbuf_desc_max_bsize=0x%x) must be greater than or equal to (alignment=0x%x) (time %t)", reg_s2c_dma_bar, s2c_rbuf_desc_max_bsize, alignment, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else if (c2s_rbuf_desc_max_bsize < alignment)
    begin
        $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : (c2s_rbuf_desc_max_bsize=0x%x) must be greater than or equal to (alignment=0x%x) (time %t)", reg_s2c_dma_bar, c2s_rbuf_desc_max_bsize, alignment, $time);
        inc_errors;
        if (on_err_stop)
        begin
            report_status;
            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
        end
    end
    else // Accept task call
    begin
        // Initialize loop exit
        check_exit = 1'b0;

        // Wait for DMA to Complete, so we can check data; don't change
        done_wait = 1;

        pkt_max_bsize_mask = pkt_max_bsize - 32'h1;



        // -----------------------
        // Create Circular Buffers

        // Create two circular buffers; first for S2C and then for C2S
        for (i=0; i<2; i=i+1)
        begin
            if (i == 0) // S2C
            begin
                curr_sys_addr  = s2c_rbuf_system_addr;
                limit_sys_addr = s2c_rbuf_system_addr + s2c_rbuf_system_bsize;
                desc_ptr       = s2c_rbuf_desc_ptr[31:0];
                curr_desc_ptr  = s2c_rbuf_desc_ptr[31:0];
                desc_max_bsize = s2c_rbuf_desc_max_bsize;
                desc_min_bsize = s2c_rbuf_desc_min_bsize;
                desc_limit     = s2c_rbuf_desc_max_num;
            end
            else
            begin
                curr_sys_addr  = c2s_rbuf_system_addr;
                limit_sys_addr = c2s_rbuf_system_addr + s2c_rbuf_system_bsize;
                desc_ptr       = c2s_rbuf_desc_ptr[31:0];
                curr_desc_ptr  = c2s_rbuf_desc_ptr[31:0];
                desc_max_bsize = c2s_rbuf_desc_max_bsize;
                desc_min_bsize = c2s_rbuf_desc_min_bsize;
                desc_limit     = c2s_rbuf_desc_max_num;
            end

            desc_range = desc_max_bsize - desc_min_bsize;

            // For masking to work properly desc_range must be a binary multiple
            if (desc_range == 32'h0)
                desc_range_mask = 32'h0;
            else
                desc_range_mask = desc_range - 32'h1;

            // Initialize BFM DMA memory to avoid X issues
            for (addr = {curr_sys_addr[63:2], 2'b00}; addr < limit_sys_addr; addr = addr + 4)
            begin
                bfm_dword_addr = addr[BFM_MEM_SIZE_BITS+1:2];// Drop highest address bits since BFM memory is indexed without a base address

                read_mem = bfm_mem[bfm_dword_addr];
                for (j=0; j<4; j=j+1)
                begin
                    // Clear bytes which are part of the buffer
                    if (((addr + j) >= curr_sys_addr) & ((addr + j) <= limit_sys_addr))
                    begin
                        case (j)
                            0 : read_mem[ 7: 0] = 8'h0;
                            1 : read_mem[15: 8] = 8'h0;
                            2 : read_mem[23:16] = 8'h0;
                            3 : read_mem[31:24] = 8'h0;
                        endcase
                    end
                end
                bfm_mem[bfm_dword_addr] = read_mem;
            end

            num_desc = 0;

            // Create Descriptors to fill requested circular buffer
            while ((curr_sys_addr < limit_sys_addr) & (num_desc < desc_limit))
            begin
                // Get random, aligned Descriptor size between alignment and desc_max_bsize
                r = $random;
                xfer_bcount = r[19:0] & desc_range_mask[19:0];      // Randomize Descriptor size over range
                xfer_bcount = xfer_bcount + desc_min_bsize;         // Add min Descriptor size to random range
                xfer_bcount = (xfer_bcount & alignment_mask[19:0]); // Truncate to requested alignment

                if (xfer_bcount < alignment[19:0])
                    xfer_bcount = alignment[19:0];

                // Don't let Descriptors be created which exceed the allocated space; if Desc limit is hit, use remaining buffer space for final Descriptor
                if (((curr_sys_addr + xfer_bcount) >= limit_sys_addr) || ((num_desc + 1) == desc_limit)) // Last Descriptor in circular buffer
                begin
                    xfer_bcount = limit_sys_addr - curr_sys_addr;

                    next_desc_ptr = desc_ptr; // Point back to beginning of chain

                    // Save the position of the last element in the circular buffer
                    if (i == 0) // S2C
                        s2c_rbuf_last_desc_ptr = curr_desc_ptr;
                    else // C2S
                        c2s_rbuf_last_desc_ptr = curr_desc_ptr;
                end
                else
                begin
                    next_desc_ptr = curr_desc_ptr + 32; // Advance to the next Descriptor address; descriptors are 32 bytes each
                end

                // Truncate xfer_bcount to an aligned value; make sure its non-zero
                xfer_bcount = xfer_bcount & alignment_mask[19:0];
                if (xfer_bcount < alignment[19:0])
                    xfer_bcount = alignment[19:0];

                // Put the DMA Descriptor info into BFM memory (system memory)
                bfm_dword_addr = curr_desc_ptr[BFM_MEM_SIZE_BITS+1:2];// Drop highest address bits since BFM memory is indexed without a base address
                bfm_mem[bfm_dword_addr+0] = 32'h0;                            // This field setup per packet
                bfm_mem[bfm_dword_addr+1] = 32'h0;                            // This field setup per packet
                bfm_mem[bfm_dword_addr+2] = 32'h0;                            // This field setup per packet
                bfm_mem[bfm_dword_addr+3] = 32'h0;                            // DescCardAddr[31:0]
                bfm_mem[bfm_dword_addr+4] = {8'h01, 4'h0, xfer_bcount[19:0]}; // {ControlFlags[7:0] setup per packet, DescCardAddr[35:32], DescByteCount[19:0]} - Control; enable interrupts for each Descriptor
                bfm_mem[bfm_dword_addr+5] = curr_sys_addr[31: 0];             // System Address[31: 0]
                bfm_mem[bfm_dword_addr+6] = curr_sys_addr[63:32];             // System Address[63:32]
                bfm_mem[bfm_dword_addr+7] = next_desc_ptr[31: 0];             // {DescNextDescPtr[31:5], 5b00000}

                if (my_verbose > 1)
                begin
                    if (i == 0) // S2C
                        $display ("%m : INFO : reg_s2c_dma_bar=0x%x : Creating S2C Circular Buffer Descriptor[%4d] : sys_addr=0x%x, desc_addr=0x%x, bcnt=0x%x, next_desc=0x%x", reg_s2c_dma_bar, num_desc, curr_sys_addr, curr_desc_ptr[31:0], xfer_bcount[19:0], next_desc_ptr[31:0]);
                    else // C2S
                        $display ("%m : INFO : reg_s2c_dma_bar=0x%x : Creating C2S Circular Buffer Descriptor[%4d] : sys_addr=0x%x, desc_addr=0x%x, bcnt=0x%x, next_desc=0x%x", reg_s2c_dma_bar, num_desc, curr_sys_addr, curr_desc_ptr[31:0], xfer_bcount[19:0], next_desc_ptr[31:0]);
                end

                // Increment for the next Descrptor
                curr_desc_ptr = next_desc_ptr;
                curr_sys_addr = curr_sys_addr + xfer_bcount;
                num_desc      = num_desc      + 1;
            end
        end



        // -----------------------------------------------------------
        // Generate random lengths for the requested number of packets

        // Generating random packet sizes; the receiver and transmitter
        //   needs to use the same packet sizes so packets can be checked;
        //   pre-generate all packet sizes before starting transmit and receive
        //   processes
        avg_pkt_size = 0;
        for (p=0; p<pkt_num_packets; p=p+1)
        begin
            // Get random, aligned packet size between alignment and pkt_max_bsize
            r = $random;
            r = (r & pkt_max_bsize_mask) + 32'h1;
            r = (r & alignment_mask[31:0]);
            if (r < alignment[31:0])
                r = alignment[31:0];

            pkt_size[p]  =                r;
            avg_pkt_size = avg_pkt_size + r;

            if (my_verbose > 1)
                $display ("%m : INFO : reg_s2c_dma_bar=0x%x : Random packet size[%4d]=0x%x", reg_s2c_dma_bar, p, pkt_size[p]);
        end

        avg_pkt_size = avg_pkt_size / pkt_num_packets;



        // ---------------------------------------------
        // Put Packet Generator/Checker Pair in Loopback

        if (my_verbose > 0)
            $display ("%m : INFO : reg_s2c_dma_bar=0x%x : Putting Packet Generator and Packet Checker into Loopback mode", reg_s2c_dma_bar);

        //mem_write_dword (tc,     addr,            data,         be);
        mem_write_dword   (3'b000, dma_s2c_pat_ctl, 32'h00000002, 4'hf); // Set Loopback_Enable
        mem_write_dword   (3'b000, dma_c2s_pat_ctl, 32'h00000002, 4'hf); // Set Loopback_Enable

        // Now read to make sure loopback mode was enabled

        if (my_verbose > 0)
            $display ("%m : INFO : reg_s2c_dma_bar=0x%x : Verifying that Loopback mode was successfully entered", reg_s2c_dma_bar);

        //mem_read_dword  (tc,     addr,            expect_data,  check_be, read_data);
        mem_read_dword    (3'b000, dma_s2c_pat_ctl, 32'h00000002, 4'h0,     read_lpe);
        if (read_lpe != 32'h00000002)
        begin
            $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : The Packet Checker could not be put into loopback mode; the test cannot continue; aborting (time %t)", reg_s2c_dma_bar, $time);
            $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : The Packet Checker may still be busy from a previous operation; read dma_s2c_pat_ctl=0x%x",  reg_s2c_dma_bar, read_lpe);
            inc_errors;
            if (on_err_stop)
            begin
                report_status;
                if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
            end
            check_exit = 1'b1;
        end

        //mem_read_dword  (tc,     addr,            expect_data,  check_be, read_data);
        mem_read_dword    (3'b000, dma_c2s_pat_ctl, 32'h00000002, 4'h0,     read_lpe);
        if (read_lpe != 32'h00000002)
        begin
            $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : The Packet Generator could not be put into loopback mode; the test cannot continue; aborting (time %t)", reg_s2c_dma_bar, $time);
            $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : The Packet Generator may still be busy from a previous operation; read dma_c2s_pat_ctl=0x%x",  reg_s2c_dma_bar, read_lpe);
            inc_errors;
            if (on_err_stop)
            begin
                report_status;
                if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
            end
            check_exit = 1'b1;
        end

        // If no errors, then continue
        if (check_exit == 1'b0)
        begin
            // ----------------------------
            // Enable Global DMA Interrupts

            //mem_write_dword (tc,     addr,        data,  be);
            mem_write_dword   (3'b000, reg_com_bar, 32'h1, 4'h1);



            // ----------------------------
            // Transmit and Receive Packets

            transmit_exit      = 1'b0;
            receive_exit       = 1'b0;

            c2s_int_check   = 0;
            c2s_int_handled = 0;

            s2c_int_check   = 0;
            s2c_int_handled = 0;

            s2c_cdp = 32'h0;
            c2s_cdp = 32'h0;

            // Split execution into multiple threads
            fork

                // ----------------------
                // Performance Monitoring

                begin : performance_regs

                    integer         t;
                    reg [31:0]      s2c_run_time;
                    reg [31:0]      s2c_wait_time;
                    reg [31:0]      s2c_completed_bcount;
                    reg [31:0]      c2s_run_time;
                    reg [31:0]      c2s_wait_time;
                    reg [31:0]      c2s_completed_bcount;

                    t = 0;
                    while ((transmit_exit == 1'b0) || (receive_exit == 1'b0))
                    begin
 `ifdef SIMULATION
                        if (t >= 9999) // Report performance every 10 uS (update period for simulation)
 `else
                        if (t >= 999999999) // Report performance every 1 S (update period for hardware)
 `endif
                        begin
                            // mem_read_dword (tc,     addr,                 expect_data, check_be, read_data);
                            mem_read_dword    (3'b000, reg_s2c_dma_bar+'h14, 32'h0,       0,        s2c_run_time);

                            // mem_read_dword (tc,     addr,                 expect_data, check_be, read_data);
                            mem_read_dword    (3'b000, reg_s2c_dma_bar+'h18, 32'h0,       0,        s2c_wait_time);

                            // mem_read_dword (tc,     addr,                 expect_data, check_be, read_data);
                            mem_read_dword    (3'b000, reg_s2c_dma_bar+'h1c, 32'h0,       0,        s2c_completed_bcount);

                            // mem_read_dword (tc,     addr,                 expect_data, check_be, read_data);
                            mem_read_dword    (3'b000, reg_c2s_dma_bar+'h14, 32'h0,       0,        c2s_run_time);

                            // mem_read_dword (tc,     addr,                 expect_data, check_be, read_data);
                            mem_read_dword    (3'b000, reg_c2s_dma_bar+'h18, 32'h0,       0,        c2s_wait_time);

                            // mem_read_dword (tc,     addr,                 expect_data, check_be, read_data);
                            mem_read_dword    (3'b000, reg_c2s_dma_bar+'h1c, 32'h0,       0,        c2s_completed_bcount);

                            if (my_verbose > 1)
                            begin
                                $display ("%m : INFO : reg_s2c_dma_bar=0x%x : S2C        HW_Run_Time=0x%x : SampleCtr=0x%x (time %t)", reg_s2c_dma_bar, {s2c_run_time[31:2],         2'b00}, s2c_run_time[1:0],         $time);
                                $display ("%m : INFO : reg_s2c_dma_bar=0x%x : S2C       HW_Wait_Time=0x%x : SampleCtr=0x%x (time %t)", reg_s2c_dma_bar, {s2c_wait_time[31:2],        2'b00}, s2c_wait_time[1:0],        $time);
                                $display ("%m : INFO : reg_s2c_dma_bar=0x%x : S2C CompletedByteCount=0x%x : SampleCtr=0x%x (time %t)", reg_s2c_dma_bar, {s2c_completed_bcount[31:2], 2'b00}, s2c_completed_bcount[1:0], $time);

                                $display ("%m : INFO : reg_s2c_dma_bar=0x%x : C2S        HW_Run_Time=0x%x : SampleCtr=0x%x (time %t)", reg_s2c_dma_bar, {c2s_run_time[31:2],         2'b00}, c2s_run_time[1:0],         $time);
                                $display ("%m : INFO : reg_s2c_dma_bar=0x%x : C2S       HW_Wait_Time=0x%x : SampleCtr=0x%x (time %t)", reg_s2c_dma_bar, {c2s_wait_time[31:2],        2'b00}, c2s_wait_time[1:0],        $time);
                                $display ("%m : INFO : reg_s2c_dma_bar=0x%x : C2S CompletedByteCount=0x%x : SampleCtr=0x%x (time %t)", reg_s2c_dma_bar, {c2s_completed_bcount[31:2], 2'b00}, c2s_completed_bcount[1:0], $time);

 `ifdef SIMULATION
                                // The sample period is 10 uS when simulating
                                $display ("%m : INFO : reg_s2c_dma_bar=0x%x : S2C MB/s=%0d.%0d : Run%%=%0d : Wait%%=%0d : Avg Packet Size=%0d (time %t)", reg_s2c_dma_bar, ({s2c_completed_bcount[31:2], 2'b00} / 10),      ({s2c_completed_bcount[31:2], 2'b00} % 10),      ({s2c_run_time[31:2], 2'b00}/100),   ({s2c_wait_time[31:2], 2'b00}/100),   avg_pkt_size, $time);
                                $display ("%m : INFO : reg_s2c_dma_bar=0x%x : C2S MB/s=%0d.%0d : Run%%=%0d : Wait%%=%0d : Avg Packet Size=%0d (time %t)", reg_s2c_dma_bar, ({c2s_completed_bcount[31:2], 2'b00} / 10),      ({c2s_completed_bcount[31:2], 2'b00} % 10),      ({c2s_run_time[31:2], 2'b00}/100),   ({c2s_wait_time[31:2], 2'b00}/100),   avg_pkt_size, $time);
 `else
                                // The sample period is 1 S when targeted for hardware
                                $display ("%m : INFO : reg_s2c_dma_bar=0x%x : S2C MB/s=%0d.%0d : Run%%=%0d : Wait%%=%0d : Avg Packet Size=%0d (time %t)", reg_s2c_dma_bar, ({s2c_completed_bcount[31:2], 2'b00} / 1000000), ({s2c_completed_bcount[31:2], 2'b00} % 1000000), ({s2c_run_time[31:2], 2'b00}/10000), ({s2c_wait_time[31:2], 2'b00}/10000), avg_pkt_size, $time);
                                $display ("%m : INFO : reg_s2c_dma_bar=0x%x : C2S MB/s=%0d.%0d : Run%%=%0d : Wait%%=%0d : Avg Packet Size=%0d (time %t)", reg_s2c_dma_bar, ({c2s_completed_bcount[31:2], 2'b00} / 1000000), ({c2s_completed_bcount[31:2], 2'b00} % 1000000), ({c2s_run_time[31:2], 2'b00}/10000), ({c2s_wait_time[31:2], 2'b00}/10000), avg_pkt_size, $time);
 `endif
                            end

                            t = 0;
                        end
                        else
                        begin
                            #1000; // Wait 1 nS
                            t = t + 1;
                        end
                    end

                    if (my_verbose > 1)
                        $display ("%m : INFO : reg_s2c_dma_bar=0x%x : Performance monitoring process ended", reg_s2c_dma_bar);
                end

                // ----------------------
                // C2S Interrupt Detector

                // Watch the interrupt line for interrupts and schedule a process to handle them;
                //   MSI-X/MSI interrupts are only a single pulse, so we need to be watching every
                //   clock so we don't miss one
                begin : c2s_interrupt_detector

                    reg     [31:0]  timeout_ctr;
                    reg     [31:0]  read_data;

                    // Initialize timeout ctr
                    timeout_ctr = timeout_clocks;

                    // Wait for interrupt to assert
                    while (receive_exit == 1'b0)
                    begin
                        case (int_mode[1:0])

                            2'b00   :   begin // Legacy Interrupts in use
                                            if (int_legi_vector_hit[c2s_int_vector[1:0]] === 1'b1)
                                            begin
                                                // Increment to schedule the interrupt handler to run
                                                c2s_int_check = c2s_int_check + 1;
                                                if (my_verbose > 1) $display ("%m : reg_c2s_dma_bar=0x%x : Legacy Interrupt Detected : Scheduling Interrupt Handler : NumPended=0x%x (time %t)", reg_c2s_dma_bar, c2s_int_check, $time);

                                                // Let 100 clocks pass between each interrupt check
                                                //   since Legacy interrupts are level based and
                                                //   will be constantly asserted until cleared
                                                repeat (99) @(posedge clk);

                                                // Initialize timeout ctr
                                                timeout_ctr = timeout_clocks;
                                            end
                                        end

                            2'b01   :   begin // MSI Interrupts in use
                                            if (int_msi_vector_hit[c2s_int_vector[4:0]] === 1'b1)
                                            begin
                                                // Increment to schedule the interrupt handler to run
                                                c2s_int_check = c2s_int_check + 1;

                                                if (my_verbose > 1) $display ("%m : reg_c2s_dma_bar=0x%x : MSI Interrupt Detected : Scheduling Interrupt Handler : NumPended=0x%x (time %t)", reg_c2s_dma_bar, c2s_int_check, $time);

                                                // Initialize timeout ctr
                                                timeout_ctr = timeout_clocks;
                                            end
                                        end

                            default :   begin // MSI-X Interrupts in use
                                            if (int_msix_vector_hit[c2s_int_vector[11:0]] === 1'b1)
                                            begin
                                                // Increment to schedule the interrupt handler to run
                                                c2s_int_check = c2s_int_check + 1;

                                                if (my_verbose > 1) $display ("%m : reg_c2s_dma_bar=0x%x : MSI-X Interrupt Detected : Scheduling Interrupt Handler : NumPended=0x%x (time %t)", reg_c2s_dma_bar, c2s_int_check, $time);

                                                // Initialize timeout ctr
                                                timeout_ctr = timeout_clocks;
                                            end
                                        end
                        endcase
                        // Move 1 clock of time between interrupt checks
                        @(posedge clk);
                        // Check for timeouts
                        if (timeout_en)
                        begin
                            if (timeout_ctr < 32'h1)
                                timeout_ctr = 32'h0;
                            else
                                timeout_ctr = timeout_ctr - 32'h1;

                            if (timeout_ctr == 32'h0) // Timeout occurred
                            begin
                                // End DMA processes
                                receive_exit  = 1'b1;

                                $display ("%m : %0s : reg_c2s_dma_bar=0x%x : C2S DMA operation failed to make progress for 0x%x clocks.  DMA may be locked up.  Aborting test and resetting DMA Engine.  (time %t)",
                                           on_err_stop ? "ERROR" : "INFO", reg_c2s_dma_bar, timeout_clocks, $time);
                                if (on_err_stop)
                                begin
                                    inc_errors;
                                    report_status;
                                    if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                end

                                // --------------------
                                // Reset C2S DMA Engine

                                //mem_write_dword (tc,     addr,            data,         be);
                                mem_write_dword   (3'b000, dma_c2s_reg_cst, 32'h00008000, 4'hf);

                                // Poll until it is seen that reset is completed
                                read_data = 32'h00008000; // Not done status
                                while (read_data[15] == 1'b1)
                                begin
                                    //mem_read_dword (tc,     addr,            expect_data, check_be, read_data);
                                    mem_read_dword   (3'b000, dma_c2s_reg_cst, 32'h0,       0,        read_data);
                                end
                            end
                        end
                    end
                end

                // ----------------------
                // S2C Interrupt Detector

                // Watch the interrupt line for interrupts and schedule a process to handle them;
                //   MSI-X/MSI interrupts are only a single pulse, so we need to be watching every
                //   clock so we don't miss one
                begin : s2c_interrupt_detector

                    reg     [31:0]  timeout_ctr;
                    reg     [31:0]  read_data;

                    // Initialize timeout ctr
                    timeout_ctr = timeout_clocks;

                    // Wait for interrupt to assert
                    while (transmit_exit == 1'b0)
                    begin
                        case (int_mode[1:0])

                            2'b00   :   begin // Legacy Interrupts in use
                                            if (int_legi_vector_hit[s2c_int_vector[1:0]] === 1'b1)
                                            begin
                                                // Increment to schedule the interrupt handler to run
                                                s2c_int_check = s2c_int_check + 1;
                                                if (my_verbose > 1) $display ("%m : reg_s2c_dma_bar=0x%x : Legacy Interrupt Detected : Scheduling Interrupt Handler : NumPended=0x%x (time %t)", reg_s2c_dma_bar, s2c_int_check, $time);

                                                // Let 100 clocks pass between each interrupt check
                                                //   since Legacy interrupts are level based and
                                                //   will be constantly asserted until cleared
                                                repeat (99) @(posedge clk);

                                                // Initialize timeout ctr
                                                timeout_ctr = timeout_clocks;
                                            end
                                        end

                            2'b01   :   begin // MSI Interrupts in use
                                            if (int_msi_vector_hit[s2c_int_vector[4:0]] === 1'b1)
                                            begin
                                                // Increment to schedule the interrupt handler to run
                                                s2c_int_check = s2c_int_check + 1;

                                                if (my_verbose > 1) $display ("%m : reg_s2c_dma_bar=0x%x : MSI Interrupt Detected : Scheduling Interrupt Handler : NumPended=0x%x (time %t)", reg_s2c_dma_bar, s2c_int_check, $time);

                                                // Initialize timeout ctr
                                                timeout_ctr = timeout_clocks;
                                            end
                                        end

                            default :   begin // MSI-X Interrupts in use
                                            if (int_msix_vector_hit[s2c_int_vector[11:0]] === 1'b1)
                                            begin
                                                // Increment to schedule the interrupt handler to run
                                                s2c_int_check = s2c_int_check + 1;

                                                if (my_verbose > 1) $display ("%m : reg_s2c_dma_bar=0x%x : MSI-X Interrupt Detected : Scheduling Interrupt Handler : NumPended=0x%x (time %t)", reg_s2c_dma_bar, s2c_int_check, $time);

                                                // Initialize timeout ctr
                                                timeout_ctr = timeout_clocks;
                                            end
                                        end
                        endcase
                        // Move 1 clock of time between interrupt checks
                        @(posedge clk);

                        // Check for timeouts
                        if (timeout_en)
                        begin
                            if (timeout_ctr < 32'h1)
                                timeout_ctr = 32'h0;
                            else
                                timeout_ctr = timeout_ctr - 32'h1;

                            if (timeout_ctr == 32'h0) // Timeout occurred
                            begin
                                // End DMA processes
                                transmit_exit      = 1'b1;

                                $display ("%m : %0s : reg_s2c_dma_bar=0x%x : S2C DMA operation failed to make progress for 0x%x clocks.  DMA may be locked up.  Aborting test and resetting DMA Engine.  (time %t)",
                                           on_err_stop ? "ERROR" : "INFO", reg_s2c_dma_bar, timeout_clocks, $time);
                                if (on_err_stop)
                                begin
                                    inc_errors;
                                    report_status;
                                    if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                end

                                // --------------------
                                // Reset S2C DMA Engine

                                //mem_write_dword (tc,     addr,            data,            be);
                                mem_write_dword   (3'b000, dma_s2c_reg_cst, 32'h00008000, 4'hf);

                                // Poll until it is seen that reset is completed
                                read_data = 32'h00008000; // Not done status
                                while (read_data[15] == 1'b1)
                                begin
                                    //mem_read_dword (tc,     addr,            expect_data, check_be, read_data);
                                    mem_read_dword   (3'b000, dma_s2c_reg_cst, 32'h0,       0,        read_data);
                                end
                            end
                        end
                    end
                end

                // ---------------------
                // C2S Interrupt Handler

                // Process C2S interrupts; if interrupts are being shared, the interrupt may not belong to us
                begin : c2s_interrupt_handler

                    reg             int_active;
                    reg     [31:0]  read_data;
                    integer         num_pending;

                    while (receive_exit == 1'b0)
                    begin
                        if (c2s_int_handled != c2s_int_check) // A new interrupt needs to be handled
                        begin
                            num_pending = c2s_int_check - c2s_int_handled;

                            if (my_verbose > 1)
                                $display ("%m : Interrupt Received : Entering C2S Interrupt Handler : NumPended=0x%x, NumHandled=0x%x, Handling 0x%x Interrupt Event", c2s_int_check, c2s_int_handled, num_pending);

                            // --------------------------------------------
                            // Check to see if the interrupt is for C2S DMA

                            // mem_read_dword (tc,     addr,            expect_data, check_be, read_data);
                            mem_read_dword    (3'b000, dma_c2s_reg_cst, 32'h0,       0,        read_data);

                            int_active = 1'b0;

                            // Check for DMA Complete Status
                            if (read_data[1]) // Check to see if the interrupt is ours
                            begin
                                int_active = 1'b1; // Interrupt is ours

                                if (my_verbose > 1)
                                    $display ("%m : Processing Interrupt - C2S DMA Engine Interrupt Status is set");

                                // Clear the interrupt and other status bits; leave interrupts enabled
                                // mem_write_dword (tc,     addr,            data,         be);
                                mem_write_dword    (3'b000, dma_c2s_reg_cst, 32'h000000FF, 4'h1);
                            end
                            else
                            begin
                                if (my_verbose > 1)
                                    $display ("%m : Processing Interrupt - C2S DMA Engine Interrupt Status is not set; not our interrupt");
                            end

                            // ----------------------
                            // Process C2S Interrupts

                            if (int_active)
                            begin
                                // Get completed Descriptor pointer
                                // mem_read_dword (tc,     addr,            expect_data, check_be, read_data);
                                mem_read_dword    (3'b000, dma_c2s_reg_cdp, 32'h0,       0,        read_data);

                                // Update c2s_cdp to advance circular buffer limit by amount of consumed Descriptors;
                                //   the receive task uses c2s_cdp to know how many Descriptors are available for
                                //   processing
                                if (read_data != 32'h0) // 0 means no Descriptors have completed
                                    c2s_cdp = read_data;

                                int_active = 1'b0;
                            end

                            // Increment to indicate that a pending interrupt was handled
                            c2s_int_handled = c2s_int_handled + num_pending;
                        end
                        else
                        begin
                            // Wait 1 clock between checks for something to do
                            @(posedge clk);
                        end
                    end

                    if (my_verbose > 1)
                        $display ("%m : INFO : reg_c2s_dma_bar=0x%x : C2S Interrupt handler process ended", reg_c2s_dma_bar);
                end

                // ---------------------
                // S2C Interrupt Handler

                // Process S2C interrupts; if interrupts are being shared, the interrupt may not belong to us
                begin : s2c_interrupt_handler

                    reg             int_active;
                    reg     [31:0]  read_data;
                    integer         num_pending;

                    while (transmit_exit == 1'b0)
                    begin
                        if (s2c_int_handled != s2c_int_check) // A new interrupt needs to be handled
                        begin
                            num_pending = s2c_int_check - s2c_int_handled;

                            if (my_verbose > 1)
                                $display ("%m : Interrupt Received : Entering S2C Interrupt Handler : NumPended=0x%x, NumHandled=0x%x, Handling 0x%x Interrupt Event", s2c_int_check, s2c_int_handled, num_pending);

                            // --------------------------------------------
                            // Check to see if the interrupt is for S2C DMA

                            // mem_read_dword (tc,     addr,            expect_data, check_be, read_data);
                            mem_read_dword    (3'b000, dma_s2c_reg_cst, 32'h0,       0,        read_data);

                            int_active = 1'b0;

                            // Check for DMA Complete Status
                            if (read_data[1]) // Check to see if the interrupt is ours
                            begin
                                int_active = 1'b1; // Interrupt is ours

                                if (my_verbose > 1)
                                    $display ("%m : Processing Interrupt - S2C DMA Engine Interrupt Status is set");

                                // Clear the interrupt and other status bits; leave interrupts enabled
                                // mem_write_dword (tc,     addr,            data,         be);
                                mem_write_dword    (3'b000, dma_s2c_reg_cst, 32'h000000FF, 4'h1);
                            end
                            else
                            begin
                                if (my_verbose > 1)
                                    $display ("%m : Processing Interrupt - S2C DMA Engine Interrupt Status is not set; not our interrupt");
                            end

                            // ----------------------
                            // Process S2C Interrupts

                            if (int_active)
                            begin
                                // Get completed Descriptor pointer
                                // mem_read_dword (tc,     addr,            expect_data, check_be, read_data);
                                mem_read_dword    (3'b000, dma_s2c_reg_cdp, 32'h0,       0,        read_data);

                                // Update s2c_cdp to advance circular buffer limit by amount of consumed Descriptors;
                                //   the receive task uses s2c_cdp to know how many Descriptors are available for
                                //   processing
                                if (read_data != 32'h0) // 0 means no Descriptors have completed
                                    s2c_cdp = read_data;

                                int_active = 1'b0;
                            end

                            // Increment to indicate that a pending interrupt was handled
                            s2c_int_handled = s2c_int_handled + num_pending;
                        end
                        else
                        begin
                            // Wait 1 clock between checks for something to do
                            @(posedge clk);
                        end
                    end

                    if (my_verbose > 1)
                        $display ("%m : INFO : reg_s2c_dma_bar=0x%x : S2C Interrupt handler process ended", reg_s2c_dma_bar);
                end

                // ---------------
                // Packet Transmit

                begin : pkt_transmit

                    reg     [31:0]  s2c_ptr;

                    reg     [31:0]  data;
                    reg     [1:0]   data_index;
                    reg     [31:0]  user_lo;
                    reg     [31:0]  user_hi;

                    integer         d;
                    integer         p;
                    reg     [31:0]  curr_bcount;
                    reg             curr_sop;

                    reg     [31:0]  bfm_desc_addr;
                    reg     [31:0]  mem;
                    reg     [19:0]  desc_bcount;
                    reg     [63:0]  desc_addr;
                    reg     [19:0]  used_bcount;
                    reg             curr_eop;
                    reg     [7:0]   control_flags;
                    reg     [63:0]  addr;
                    reg     [31:0]  bfm_dword_addr;
                    reg     [7:0]   data_byte;
                    reg     [31:0]  next_data;
                    reg     [31:0]  read_desc_addr;
                    reg     [31:0]  dummy_data;

                    integer         j;

                    // ------------------------------------------------
                    // Initialize and start the Transmit S2C DMA Engine

                    if (my_verbose > 0)
                        $display ("%m : INFO : reg_s2c_dma_bar=0x%x : Initializing and starting S2C DMA Engine", reg_s2c_dma_bar);

                    // Assign transmit task ownership of the whole buffer during initialization;
                    //   as transmit Descriptors are processed by the DMA Engine, the interrupt
                    //   handler will update s2c_cdp to return Descriptors to the transmit task
                    s2c_ptr = s2c_rbuf_desc_ptr[31:0];      // Next Desc to use
                    s2c_cdp = s2c_rbuf_last_desc_ptr[31:0]; // Limit Desc; don't use

                    //mem_write_dword (tc,     addr,            data,                    be);
                    mem_write_dword   (3'b000, dma_s2c_reg_cdp, 32'h0,                   4'hf); // Zero completed Descriptor Pointer

                    //mem_write_dword (tc,     addr,            data,                    be);
                    mem_write_dword   (3'b000, dma_s2c_reg_sdp, s2c_rbuf_desc_ptr[31:0], 4'hf); // Assign sdp pointer same value as ndp pointer == no Descriptors allocated to DMA Engine yet

                    //mem_write_dword (tc,     addr,            data,                    be);
                    mem_write_dword   (3'b000, dma_s2c_reg_ndp, s2c_rbuf_desc_ptr[31:0], 4'hf); // Start of DMA circular buffer

                    //mem_write_dword (tc,     addr,            data,                    be);
                    mem_write_dword   (3'b000, dma_s2c_reg_cst, 32'h00000101,            4'hf); // Start the DMA engine; enable interrupts

                    // Get initial packet seed values
                    data       = pat_data_seed;
                    data_index = 2'h0;

                    user_lo = pat_user_seed;
                    get_packet_pattern (pat_user_type, user_lo, user_hi);

                    // Transmit requested packets
                    d = 0; // Descriptor counter
                    p = 0; // Packet counter
                    while ((p < pkt_num_packets) && (transmit_exit == 1'b0))
                    begin
                        // Get packet size in bytes for the current packet
                        curr_bcount = pkt_size[p];
                        curr_sop    = 1'b1;

                        // Break large packets over multiple Descriptors if required
                        while ((curr_bcount != 32'h0) && (transmit_exit == 1'b0))
                        begin
                            // Fill Descriptors with transmit packet data as they become available
                            if ((s2c_ptr != s2c_cdp) && (transmit_exit == 1'b0)) // Hold and don't use the s2c_cdp Descriptor
                            begin
                                // Get address in BFM memory (system memory) of Descriptor to fill in
                                bfm_desc_addr = s2c_ptr[BFM_MEM_SIZE_BITS+1:2];// Drop highest address bits since BFM memory is indexed without a base address

                                // Get Descriptor's byte count and control flags
                                mem         = bfm_mem[bfm_desc_addr+4];
                                desc_bcount = mem[19:0];

                                // Get Descriptor's system address
                                desc_addr   = {bfm_mem[bfm_desc_addr+6], bfm_mem[bfm_desc_addr+5]};

                                if (curr_bcount > desc_bcount)
                                begin
                                    used_bcount = desc_bcount;
                                    curr_eop    = 1'b0;
                                end
                                else
                                begin
                                    used_bcount = curr_bcount;
                                    curr_eop    = 1'b1;
                                end

                                // S2C Control Flags
                                control_flags = {curr_sop, curr_eop, 5'h0, 1'b1}; // Enable interrupts for all Descriptors; multiple interrupt events will get merged

                                // Only update Descriptor fields which need to change; put in same desc_bcount read out
                                bfm_mem[bfm_desc_addr+0] = {8'h0, 4'h0, used_bcount[19:0]}; // {S2CDescStatusFlags[7:0], Reserved[3:0], S2CDescByteCount[19:0]} - Status so clear except bcount which is bytes to use in this descriptor
                                bfm_mem[bfm_desc_addr+1] = curr_sop ? user_lo : 32'h0; // S2CDescUserControl[31: 0]
                                bfm_mem[bfm_desc_addr+2] = curr_sop ? user_hi : 32'h0; // S2CDescUserControl[63:32]
                                bfm_mem[bfm_desc_addr+4] = {control_flags[7:0], 4'h0, desc_bcount[19:0]}; // {S2CDescControlFlags[7:0], DescCardAddr[35:32], DescByteCount[19:0]} - Control

                                // Step through buffer pointed to by Descriptor and setup requested data pattern
                                for (addr = {desc_addr[63:2], 2'b00}; addr < (desc_addr + used_bcount); addr = addr + 4)
                                begin
                                    bfm_dword_addr = addr[BFM_MEM_SIZE_BITS+1:2];// Drop highest address bits since BFM memory is indexed without a base address

                                    // Only modify bytes which are part of the buffer; do read-modify-write
                                    mem = bfm_mem[bfm_dword_addr];
                                    for (j=0; j<4; j=j+1)
                                    begin
                                        // Only modify a byte that is part of the buffer
                                        if (((addr + j) >= desc_addr) & ((addr + j) < (desc_addr + used_bcount)))
                                        begin
                                            // Get current data byte in pattern
                                            case (data_index)
                                                2'h0    : data_byte = data[ 7: 0];
                                                2'h1    : data_byte = data[15: 8];
                                                2'h2    : data_byte = data[23:16];
                                                default : data_byte = data[31:24];
                                            endcase

                                            case (j)
                                                0       : mem[ 7: 0] = data_byte;
                                                1       : mem[15: 8] = data_byte;
                                                2       : mem[23:16] = data_byte;
                                                default : mem[31:24] = data_byte;
                                            endcase

                                            // Advance the pattern when the last byte in the pattern is used
                                            if (data_index == 2'h3)
                                            begin
                                                get_packet_pattern (pat_data_type, data, next_data);
                                                data = next_data;
                                            end

                                            // Advance to next byte
                                            data_index = data_index + 2'h1;
                                        end
                                    end
                                    bfm_mem[bfm_dword_addr] = mem;
                                end

                                if (my_verbose > 1)
                                begin
                                    if ((p == (pkt_num_packets - 1)) && (curr_eop == 1'b1)) // Last Descriptor in last packet& task
                                        $display ("%m : INFO : reg_s2c_dma_bar=0x%x : TX Pended Descriptor[%4d] for Packet[%4d] : DescAddr=0x%x, used_bcount=0x%x : Last Descriptor for Packet : Last Descriptor for Task", reg_s2c_dma_bar, d, p, s2c_ptr, used_bcount);
                                    else if (curr_eop == 1'b1) // Last Descriptor in last packet
                                        $display ("%m : INFO : reg_s2c_dma_bar=0x%x : TX Pended Descriptor[%4d] for Packet[%4d] : DescAddr=0x%x, used_bcount=0x%x : Last Descriptor for Packet", reg_s2c_dma_bar, d, p, s2c_ptr, used_bcount);
                                    else
                                        $display ("%m : INFO : reg_s2c_dma_bar=0x%x : TX Pended Descriptor[%4d] for Packet[%4d] : DescAddr=0x%x, used_bcount=0x%x", reg_s2c_dma_bar, d, p, s2c_ptr, used_bcount);
                                end

                                // Move s2c_ptr to next Descriptor in the circular buffer
                                s2c_ptr = bfm_mem[bfm_desc_addr+7];

                                // Give filled in Descriptor to the DMA Engine
                                //mem_write_dword (tc,     addr,            data,          be);
                                mem_write_dword   (3'b000, dma_s2c_reg_sdp, s2c_ptr[31:0], 4'hf);

                                // Prepare for the next Descriptor
                                curr_sop = 0;
                                curr_bcount = curr_bcount - used_bcount;

                                // Advance Descriptor count
                                d = d + 1;
                            end
                            else
                            begin
                                // Wait for new Descriptors to become available for transmission
                                @(posedge clk);
                            end
                        end

                        // Advance packet count
                        p = p + 1;

                        // Roll user_control for next packer
                        get_packet_pattern (pat_user_type, user_hi, user_lo);
                        get_packet_pattern (pat_user_type, user_lo, user_hi);
                    end

                    // When done, allow interrupt handler to exit also
                    transmit_exit = 1'b1;

                    if (my_verbose > 1)
                        $display ("%m : INFO : reg_s2c_dma_bar=0x%x : Transmit process ended", reg_s2c_dma_bar);
                end

                // --------------
                // Packet Receive

                begin : pkt_receive

                    reg     [31:0]  c2s_ptr;

                    reg     [31:0]  data;
                    reg     [1:0]   data_index;
                    reg     [31:0]  user_lo;
                    reg     [31:0]  user_hi;

                    integer         d;
                    integer         p;
                    reg     [31:0]  curr_bcount;
                    reg             curr_sop;
                    reg     [31:0]  last_processed_c2s_cdp;

                    reg     [31:0]  bfm_desc_addr;
                    reg     [31:0]  mem;
                    reg     [19:0]  desc_bcount;
                    reg     [63:0]  desc_addr;
                    reg     [19:0]  used_bcount;
                    reg             curr_eop;
                    reg             curr_short;

                    reg     [31:0]  check_primary_status;
                    reg     [31:0]  this_user_lo;
                    reg     [31:0]  this_user_hi;
                    reg     [31:0]  check_user_status_lo;
                    reg     [31:0]  check_user_status_hi;

                    reg     [63:0]  addr;
                    reg     [31:0]  bfm_dword_addr;
                    reg             check_fail;
                    reg     [31:0]  exp;
                    reg     [7:0]   data_byte;
                    integer         j;
                    reg     [31:0]  next_data;
                    reg     [31:0]  read_desc_addr;
                    reg     [31:0]  dummy_data;

                    // -----------------------------------------------
                    // Initialize and start the Receive C2S DMA Engine

                    if (my_verbose > 0)
                        $display ("%m : INFO : reg_s2c_dma_bar=0x%x : Initializing and starting C2S DMA Engine", reg_s2c_dma_bar);

                    // Default to no descriptors completed; expecting next Descriptor at head of circular buffer
                    c2s_ptr                = 32'h0; // used to walk the Descriptor list and processes received packets
                    c2s_cdp                = 32'h0; // used to track the last completed Descriptor; c2s_ptr cannot pass c2s_cdp
                    last_processed_c2s_cdp = 32'h0; // last processed Descriptor

                    //mem_write_dword (tc,     addr,            data,                         be);
                    mem_write_dword   (3'b000, dma_c2s_reg_cdp, 32'h0,                        4'hf); // Zero completed Descriptor Pointer

                    //mem_write_dword (tc,     addr,            data,                         be);
                    mem_write_dword   (3'b000, dma_c2s_reg_sdp, c2s_rbuf_last_desc_ptr[31:0], 4'hf); // Assign sdp pointer to the last Descriptor == all Descriptors but last one allocated to DMA Engine

                    //mem_write_dword (tc,     addr,            data,                         be);
                    mem_write_dword   (3'b000, dma_c2s_reg_ndp, c2s_rbuf_desc_ptr[31:0],      4'hf); // Start of DMA circular buffer

                    //mem_write_dword (tc,     addr,            data,                         be);
                    mem_write_dword   (3'b000, dma_c2s_reg_cst, 32'h00000101,                 4'hf); // Start the DMA engine; enable interrupts

                    // Get initial packet seed values
                    data       = pat_data_seed;
                    data_index = 2'h0;

                    user_lo = pat_user_seed;
                    get_packet_pattern (pat_user_type, user_lo, user_hi);

                    // Receive requested packets
                    d = 0; // Descriptor counter
                    p = 0; // Packet counter
                    while ((p < pkt_num_packets) && (receive_exit == 1'b0))
                    begin
                        // Get packet size in bytes for the current packet
                        curr_bcount = pkt_size[p];
                        curr_sop    = 1'b1;

                        // Break large packets over multiple Descriptors if required
                        while ((curr_bcount != 32'h0) && (receive_exit == 1'b0))
                        begin
                            // Check Descriptors against expected control and data as they become available
                            if ((c2s_cdp != last_processed_c2s_cdp) && (receive_exit == 1'b0)) // Wait for Descriptors to complete before they can be checked
                            begin

                                // If first time to receive a packet, set pointer to start of the loop
                                if (c2s_ptr == 32'h0)
                                    c2s_ptr = c2s_rbuf_desc_ptr[31:0];  // Next Desc to receive

                                // Get address in BFM memory (system memory) of Descriptor to fill in
                                bfm_desc_addr = c2s_ptr[BFM_MEM_SIZE_BITS+1:2];// Drop highest address bits since BFM memory is indexed without a base address

                                // Get Descriptor's byte count
                                mem         = bfm_mem[bfm_desc_addr+4];
                                desc_bcount = mem[19:0];

                                // Get Descriptor's system address
                                desc_addr = {bfm_mem[bfm_desc_addr+6], bfm_mem[bfm_desc_addr+5]};

                                // Get expected bcount and EOP
                                if (curr_bcount > desc_bcount)
                                begin
                                    used_bcount = desc_bcount;
                                    curr_eop    = 1'b0;
                                end
                                else
                                begin
                                    used_bcount = curr_bcount;
                                    curr_eop    = 1'b1;
                                end

                                // Get expected short
                                if (curr_bcount >= desc_bcount)
                                    curr_short  = 1'b0;
                                else
                                    curr_short  = 1'b1;



                                // -----------------------------------
                                // Check for correct Descriptor Status

                                check_primary_status = bfm_mem[bfm_desc_addr+0];

                                if (curr_sop != check_primary_status[31])
                                begin
                                    $display ("%m : ERROR : reg_c2s_dma_bar=0x%x : Desc_Ptr = 0x%x, Expected status (sop=%d) but got the opposite (time %t)", reg_c2s_dma_bar, c2s_ptr, curr_sop, $time);
                                    inc_errors;
                                    if (on_err_stop)
                                    begin
                                        report_status;
                                        if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                    end
                                end

                                if (curr_eop != check_primary_status[30])
                                begin
                                    $display ("%m : ERROR : reg_c2s_dma_bar=0x%x : Desc_Ptr = 0x%x, Expected status (eop=%d) but got the opposite (time %t)", reg_c2s_dma_bar, c2s_ptr, curr_eop, $time);
                                    inc_errors;
                                    if (on_err_stop)
                                    begin
                                        report_status;
                                        if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                    end
                                end

                                // check_primary_status[29:28] - reserved

                                if (check_primary_status[30] == 1'b1) // EOP == 1; user_status is zero except when eop is asserted
                                begin
                                    this_user_lo = user_lo;
                                    this_user_hi = user_hi;
                                end
                                else
                                begin
                                    this_user_lo = 32'h0;
                                    this_user_hi = 32'h0;
                                end

                                if ((check_primary_status[27] != (this_user_hi == 32'h0)) & check_status)
                                begin
                                    $display ("%m : ERROR : reg_c2s_dma_bar=0x%x : Desc_Ptr=0x%x, Expected status (user_status_hi_is0=%d) but got the opposite (time %t)", reg_c2s_dma_bar, c2s_ptr, (this_user_hi == 32'h0), $time);
                                    inc_errors;
                                    if (on_err_stop)
                                    begin
                                        report_status;
                                        if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                    end
                                end

                                if ((check_primary_status[26] != (this_user_lo == 32'h0)) & check_status)
                                begin
                                    $display ("%m : ERROR : reg_c2s_dma_bar=0x%x : Desc_Ptr=0x%x, Expected status (user_status_lo_is0=%d) but got the opposite (time %t)", reg_c2s_dma_bar, c2s_ptr, (this_user_lo == 32'h0), $time);
                                    inc_errors;
                                    if (on_err_stop)
                                    begin
                                        report_status;
                                        if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                    end
                                end

                                if (check_primary_status[25] != curr_short)
                                begin
                                    $display ("%m : ERROR : reg_c2s_dma_bar=0x%x : Desc_Ptr = 0x%x, Expected status (short=%d) but got the opposite (time %t)", reg_c2s_dma_bar, c2s_ptr, curr_short, $time);
                                    inc_errors;
                                    if (on_err_stop)
                                    begin
                                        report_status;
                                        if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                    end
                                end

                                if (check_primary_status[24] != 1'b1)
                                begin
                                    $display ("%m : ERROR : reg_c2s_dma_bar=0x%x : Desc_Ptr = 0x%x, Expected status (done=1'b1) but got the opposite (time %t)", reg_c2s_dma_bar, c2s_ptr, $time);
                                    inc_errors;
                                    if (on_err_stop)
                                    begin
                                        report_status;
                                        if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                    end
                                end

                                if (check_primary_status[19:0] != used_bcount[19:0])
                                begin
                                    $display ("%m : ERROR : reg_c2s_dma_bar=0x%x : Desc_Ptr = 0x%x, Expected status (bcount_completed=0x%x) but got 0x%x (time %t)", reg_c2s_dma_bar, c2s_ptr, used_bcount[19:0], check_primary_status[19:0], $time);
                                    inc_errors;
                                    if (on_err_stop)
                                    begin
                                        report_status;
                                        if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                    end
                                end

                                // Follow protocol to check user_status[31:0]
                                if (check_primary_status[26] & check_status) // user_status_lo_is0
                                begin
                                    check_user_status_lo = bfm_mem[bfm_desc_addr+1];
                                    if (check_user_status_lo != 32'h0)
                                    begin
                                        $display ("%m : ERROR : reg_c2s_dma_bar=0x%x : Desc_Ptr = 0x%x, Status (user_status_lo_is0=1) indicating that user_status[31: 0]=0, but user_status[31: 0]=0x%x (time %t)", reg_c2s_dma_bar, curr_desc_ptr, check_user_status_lo, $time);
                                        inc_errors;
                                        if (on_err_stop)
                                        begin
                                            report_status;
                                            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                        end
                                    end
                                end
                                else if (check_status)
                                begin
                                    while (bfm_mem[bfm_desc_addr+1] == 0)
                                        @(posedge clk);

                                    check_user_status_lo = bfm_mem[bfm_desc_addr+1];

                                    if (check_user_status_lo != this_user_lo)
                                    begin
                                        $display ("%m : ERROR : reg_c2s_dma_bar=0x%x : Desc_Ptr = 0x%x, Expected status (user_status[31: 0]=0x%x) but got 0x%x (time %t)", reg_c2s_dma_bar, curr_desc_ptr, this_user_lo, check_user_status_lo, $time);
                                        inc_errors;
                                        if (on_err_stop)
                                        begin
                                            report_status;
                                            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                        end
                                    end
                                end

                                // Follow protocol to check user_status[63:32]
                                if (check_primary_status[27] & check_status) // user_status_hi_is0
                                begin
                                    check_user_status_hi = bfm_mem[bfm_desc_addr+2];
                                    if (check_user_status_hi != 32'h0)
                                    begin
                                        $display ("%m : ERROR : reg_c2s_dma_bar=0x%x : Desc_Ptr = 0x%x, Status (user_status_hi_is0=1) indicating that user_status[63:32]=0, but user_status[63:32]=0x%x (time %t)", reg_c2s_dma_bar, curr_desc_ptr, check_user_status_hi, $time);
                                        inc_errors;
                                        if (on_err_stop)
                                        begin
                                            report_status;
                                            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                        end
                                    end
                                end
                                else if (check_status)
                                begin
                                    while (bfm_mem[bfm_desc_addr+2] == 0)
                                        @(posedge clk);

                                    check_user_status_hi = bfm_mem[bfm_desc_addr+2];

                                    if (check_user_status_hi != this_user_hi)
                                    begin
                                        $display ("%m : ERROR : reg_c2s_dma_bar=0x%x : Desc_Ptr = 0x%x, Expected status (user_status[63:32]=0x%x) but got 0x%x (time %t)", reg_c2s_dma_bar, curr_desc_ptr, this_user_hi, check_user_status_hi, $time);
                                        inc_errors;
                                        if (on_err_stop)
                                        begin
                                            report_status;
                                            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                        end
                                    end
                                end



                                // ---------------------
                                // Check Descriptor Data

                                // Step through buffer pointed to by Descriptor and check the data against the expected pattern
                                for (addr = {desc_addr[63:2], 2'b00}; addr < (desc_addr + used_bcount); addr = addr + 4)
                                begin
                                    bfm_dword_addr = addr[BFM_MEM_SIZE_BITS+1:2];// Drop highest address bits since BFM memory is indexed without a base address

                                    // Only modify bytes which are part of the buffer; do read-modify-write
                                    mem = bfm_mem[bfm_dword_addr];
                                    check_fail = 1'b0;
                                    exp        = 32'hxxxxxxxx;
                                    for (j=0; j<4; j=j+1)
                                    begin
                                        if (((addr + j) >= desc_addr) & ((addr + j) < (desc_addr + used_bcount)))
                                        begin
                                            // Get current data byte in pattern
                                            case (data_index)
                                                2'h0    : data_byte = data[ 7: 0];
                                                2'h1    : data_byte = data[15: 8];
                                                2'h2    : data_byte = data[23:16];
                                                default : data_byte = data[31:24];
                                            endcase

                                            // Check byte for errors
                                            case (j)
                                                0       : begin if (mem[ 7: 0] != data_byte) check_fail = 1'b1; exp[ 7: 0] = data_byte; end
                                                1       : begin if (mem[15: 8] != data_byte) check_fail = 1'b1; exp[15: 8] = data_byte; end
                                                2       : begin if (mem[23:16] != data_byte) check_fail = 1'b1; exp[23:16] = data_byte; end
                                                default : begin if (mem[31:24] != data_byte) check_fail = 1'b1; exp[31:24] = data_byte; end
                                            endcase

                                            // Advance the pattern when the last byte in the pattern is used
                                            if (data_index == 2'h3)
                                            begin
                                                get_packet_pattern (pat_data_type, data, next_data);
                                                data = next_data;
                                            end

                                            // Advance to next byte
                                            data_index = data_index + 2'h1;
                                        end
                                    end

                                    if (check_fail)
                                    begin
                                        $display ("%m : ERROR : reg_c2s_dma_bar=0x%x : Data Error : Desc_Ptr = 0x%x, SysAddr = 0x%x, expected 0x%x but got 0x%x (time %t)", reg_c2s_dma_bar, curr_desc_ptr, addr, exp, mem, $time);
                                        inc_errors;
                                        if (on_err_stop)
                                        begin
                                            report_status;
                                            if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                                        end
                                    end
                                end

                                if (my_verbose > 1)
                                begin
                                    if ((p == (pkt_num_packets - 1)) && (curr_eop == 1'b1)) // Last Descriptor in last packet
                                        $display ("%m : INFO : reg_c2s_dma_bar=0x%x : RX Checked Descriptor[%4d] for Packet[%4d] : DescAddr=0x%x, used_bcount=0x%x : Last Descriptor for Packet : Last Descriptor for Task", reg_c2s_dma_bar, d, p, c2s_ptr, used_bcount);
                                    else if (curr_eop == 1'b1) // Last Descriptor in this packet
                                        $display ("%m : INFO : reg_c2s_dma_bar=0x%x : RX Checked Descriptor[%4d] for Packet[%4d] : DescAddr=0x%x, used_bcount=0x%x : Last Descriptor for Packet", reg_c2s_dma_bar, d, p, c2s_ptr, used_bcount);
                                    else
                                        $display ("%m : INFO : reg_c2s_dma_bar=0x%x : RX Checked Descriptor[%4d] for Packet[%4d] : DescAddr=0x%x, used_bcount=0x%x", reg_c2s_dma_bar, d, p, c2s_ptr, used_bcount);
                                end



                                // ------------------
                                // Recycle Descriptor

                                // Clear status fields
                                bfm_mem[bfm_desc_addr+0] = 32'h0;
                                bfm_mem[bfm_desc_addr+1] = 32'h0;
                                bfm_mem[bfm_desc_addr+2] = 32'h0;

                                // Save last processed Descriptor
                                last_processed_c2s_cdp = c2s_ptr;

                                // Prepare for the next Descriptor
                                curr_sop = 0;
                                curr_bcount = curr_bcount - used_bcount;

                                // Advance software Desc. Ptr to the current descriptor
                                // (maintain one unused descriptor in the buffer to prevent overrun)
                                //mem_write_dword (tc,     addr,            data,          be);
                                mem_write_dword   (3'b000, dma_c2s_reg_sdp, c2s_ptr[31:0], 4'hf);

                                // Move c2s_ptr to next Descriptor in the circular buffer
                                c2s_ptr                = bfm_mem[bfm_desc_addr+7];

                                // Advance Descriptor count
                                d = d + 1;
                            end
                            else
                            begin
                                // Wait for new Descriptors to become available for transmission
                                @(posedge clk);
                            end
                        end

                        // Advance packet count
                        p = p + 1;

                        // Roll user_status for next packet
                        get_packet_pattern (pat_user_type, user_hi, user_lo);
                        get_packet_pattern (pat_user_type, user_lo, user_hi);
                    end

                    // When done, allow interrupt handler to exit also
                    receive_exit = 1'b1;

                    if (my_verbose > 1)
                        $display ("%m : INFO : reg_c2s_dma_bar=0x%x : Receive process ended", reg_c2s_dma_bar);
                end
            join



            // --------------------------------------------------
            // Take Packet Generator/Checker Pair out of Loopback

            if (my_verbose > 0)
                $display ("%m : INFO : reg_s2c_dma_bar=0x%x : Test finished.  Taking Packet Generator and Packet Checker out of Loopback mode", reg_s2c_dma_bar);

            //mem_write_dword (tc,     addr,            data,         be);
            mem_write_dword   (3'b000, dma_s2c_pat_ctl, 32'h00000000, 4'hf); // Clear Loopback_Enable
            mem_write_dword   (3'b000, dma_c2s_pat_ctl, 32'h00000000, 4'hf); // Clear Loopback_Enable

            // Now read to make sure loopback mode was disabled

            if (my_verbose > 0)
                $display ("%m : INFO : reg_s2c_dma_bar=0x%x : Verifying that Loopback mode was successfully disabled", reg_s2c_dma_bar);

            //mem_read_dword  (tc,     addr,            expect_data,  check_be, read_data);
            mem_read_dword    (3'b000, dma_s2c_pat_ctl, 32'h00000000, 4'h0,     read_lpe);
            if (read_lpe != 32'h00000000)
            begin
                $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : Could not disable Loopback mode for the Packet Checker, read dma_s2c_pat_ctl=0x%x (time %t)",  reg_s2c_dma_bar, read_lpe, $time);
                inc_errors;
                if (on_err_stop)
                begin
                    report_status;
                    if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                end
            end

            //mem_read_dword  (tc,     addr,            expect_data,  check_be, read_data);
            mem_read_dword    (3'b000, dma_c2s_pat_ctl, 32'h00000000, 4'h0,     read_lpe);
            if (read_lpe != 32'h00000000)
            begin
                $display ("%m : ERROR : reg_s2c_dma_bar=0x%x : Could not disable Loopback mode for the Packet Generator, read dma_c2s_pat_ctl=0x%x (time %t)",  reg_s2c_dma_bar, read_lpe, $time);
                inc_errors;
                if (on_err_stop)
                begin
                    report_status;
                    if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                end
            end


            // -----------------
            // Reset DMA Engines

            // Reset S2C DMA Engine

            //mem_write_dword (tc,     addr,            data,            be);
            mem_write_dword   (3'b000, dma_s2c_reg_cst, 32'h00008000,    4'hf);

            // Poll until it is seen that reset is completed
            read_cst = 32'h00008000; // Not done status
            while (read_cst[15] == 1'b1)
            begin
                //mem_read_dword (tc,     addr,            expect_data, check_be, read_data);
                mem_read_dword   (3'b000, dma_s2c_reg_cst, 32'h0,       0,        read_cst);
            end

            // Reset C2S DMA Engine

            //mem_write_dword (tc,     addr,            data,            be);
            mem_write_dword   (3'b000, dma_c2s_reg_cst, 32'h00008000,    4'hf);

            // Poll until it is seen that reset is completed
            read_cst = 32'h00008000; // Not done status
            while (read_cst[15] == 1'b1)
            begin
                //mem_read_dword (tc,     addr,            expect_data, check_be, read_data);
                mem_read_dword   (3'b000, dma_c2s_reg_cst, 32'h0,       0,        read_cst);
            end
        end
    end
end
endtask



// -------------------------------------------
// task: inc_errors
//   Increments the error count by 1
//   No PCIe traffic is generated by this task

task inc_errors;
begin
    ERROR_COUNT = ERROR_COUNT + 1;
end
endtask // inc_errors



// -------------------------------------------
// task: clear_errors
//   Clears the error count
//   No PCIe traffic is generated by this task

task clear_errors;
begin
    ERROR_COUNT = 0;
end
endtask // clear_errors



// -------------------------------------------
// task: report_status
//   Reports the error count or pass status
//   No PCIe traffic is generated by this task

task report_status;
begin
    if (ERROR_COUNT == 0)
        $display ("%m: SIMULATION STATUS: ALL TESTS PASS");
    else
        $display ("%m: SIMULATION STATUS: ********* %d ERRORS DETECTED **********", ERROR_COUNT);
end
endtask



// ----------------------
// task: set_pcie_bfm_mem

// Set PCIe BFM Slave memory to the desired contents; directly modifies the memory array; no bus accesses are done
//    bfm_mem[(bcount-1)+addr:addr] = payload

task automatic set_pcie_bfm_mem;

input   [63:0]                      addr;
input   [9:0]                       bcount;
input   [3:0]                       cache;
input   [4095:0]                    payload; // Supports up to a 512 Byte/4096-bit write

reg     [4095:0]                    data;

reg     [63:0]                      i;

reg     [BFM_MEM_SIZE_BITS-1:0]     rd_addr;
reg     [31:0]                      rd_data;

reg     [1:0]                       boffset;
reg     [31:0]                      bdata;
reg     [31:0]                      bbe;

begin : set_pcie_bfm_mem

    if (bcount > 512)
        $display ("%m : ERROR : bcount == %d; This task supports a maximum of 512 bytes", bcount);
    else
    begin
        data = payload;

        // Walk through requested bytes and do read-modify-write on Memory Array
        for (i=addr; i<(addr+bcount); i=i+1)
        begin
            //
            rd_addr = i[BFM_MEM_SIZE_BITS+1:2]; // BFM Mem is a DWORD array
            rd_data = bfm_mem[rd_addr];

//            if (!$test$plusargs("pcie_dma_msgs_off"))
//                $display ("%m : DEBUG : Direct Slave Memory Read : addr==0x%x, data==0x%x", rd_addr, rd_data);

            boffset = i[1:0];
            bdata = data[7:0] << (boffset*8);
            bbe   = 8'hff     << (boffset*8);

//            if (!$test$plusargs("pcie_dma_msgs_off"))
//                $display ("%m : DEBUG : Direct Slave Memory Write : boffset==0x%x, bdata==0x%x, bbe==0x%x", boffset, bdata, bbe);

//            if (!$test$plusargs("pcie_dma_msgs_off"))
//                $display ("%m : DEBUG : Direct Slave Memory Write : addr==0x%x, data==0x%x", i, data[7:0]);

            bfm_mem[rd_addr] = (bbe & bdata) | (~bbe & rd_data);
            bfm_cache_mem[rd_addr] = cache;

            data = data >> 8;
        end
    end
end
endtask

task automatic get_pcie_bfm_mem;

input   [63:0]                      addr;
input   [9:0]                       bcount;
input   [3:0]                       cache;
output  [4095:0]                    payload; // Supports up to a 512 Byte/4096-bit write

reg     [4095:0]                    data;

reg     [63:0]                      i;

reg     [BFM_MEM_SIZE_BITS-1:0]     rd_addr;
reg     [31:0]                      rd_data;

reg     [1:0]                       boffset;
reg     [7:0]                       byteval;
reg     [9:0]                       count;

begin : get_pcie_bfm_mem

    if (bcount > 512)
        $display ("%m : ERROR : bcount == %d; This task supports a maximum of 512 bytes", bcount);
    else
    begin
        data = 0;

        count = bcount;
        // Walk through requested bytes and do read-modify-write on Memory Array
        while (count != 0) begin
            i = addr + count - 1;
            rd_addr = i[BFM_MEM_SIZE_BITS+1:2]; // BFM Mem is a DWORD array
            boffset = i[1:0];

            rd_data = bfm_mem[rd_addr];
            byteval    = (rd_data >> (boffset * 8));

            data = (data << 8) | byteval;
            count = count - 1;
            if (bfm_cache_mem[rd_addr] != cache)
                $display ("%m : ERROR : get_pcie_bfm_mem : Cache value does not match at address %h", rd_addr);
        end
        payload = data;
    end
end
endtask



// -------------------------------
// Initialize Simulation Variables

assign bfm_bdf = mgmt_cfg_id;

initial
begin
    rp_is_ds_sw        = ROOT_PORT_IS_DS_SW;
    cfg0_bus_num       = CFG0_BUS_NUMBER;
    host_bus_num       = mgmt_cfg_id;  //Currently unused by ref_design_ts.

    // Assign device IDs
    dut_bdf = {DUT_BUS_NUMBER,       5'h0, 3'h0}; // Device 0, Func 0

    // Initialize Miscellaneous
    bfm_init_next_tag = 5'h0;


    suppress_bfm_completion_timeout_reporting = 1'b0; // Don't suppress

    bfm_init_tag_is_cfg_en      = 1'b1; // Enable cfg completion ID checking
    bfm_init_tag_is_cfg_en_ur_0 = 1'b0; // No exception for UR from function 0

    for (j = 0 ; j < 32 ; j = j + 1)
    begin
        bfm_init_tag_status[j]           = 1'b0;
        bfm_init_tag_cpl_status[j]       = 3'h0;
        bfm_init_tag_payload[j]          = 32768'h0;
        bfm_init_tag_expected_payload[j] = 32768'h0;
        bfm_init_tag_check_payload[j]    = 1'b0;
        bfm_init_tag_expect_timeout[j]   = 1'b0;
        bfm_init_tag_req_time[j]         = $time;
        bfm_init_tag_length[j]           = 11'h0;
        bfm_init_tag_index[j]            = 11'h0;
        bfm_init_tag_is_cfg[j]           = 1'b0;
        bfm_init_tag_is_cfg_id[j]        = 16'h0;

        dut_req_time[j]                  = $time;
        dut_req_tag_status[j]            = 1'b0;
    end

    // BFM memory is sized to BFM_MEM_SIZE_BITS DWORDs; express this in bytes
    bfm_mem_bsize = (1 << (BFM_MEM_SIZE_BITS + 2));

    // Initialize BFM memory
    for (j = 0; j < (1 << BFM_MEM_SIZE_BITS); j = j + 1)
        bfm_mem[j] = 32'h0;
end

// Setup ATC tables
initial
begin
    for (j=0; j<256; j=j+1) begin       // loop over functions
        for (k=0; k<16; k=k+1) begin    // loop over table entries
            atc_tran_addr   [j][k] = 64'h6400000000000000;
            atc_utran_addr  [j][k] = 64'h0000000032000000;
            atc_active      [j][k] = 1'b0;
            atc_read        [j][k] = 1'b0;
            atc_write       [j][k] = 1'b0;
            atc_untran      [j][k] = 1'b0;
            atc_size        [j][k] = 6'd12;     // 4KB
            atc_inval       [j][k] = 1'b0;
        end
        for (k=0; k<32; k=k+1) begin    // loop over itags
            atc_itag_entry  [j][k] = 4'h0;
        end
    end
end

task atc_add_entry;
input   [7:0]   func;
input   [3:0]   entry;
input   [63:0]  utran_addr;
input   [63:0]  tran_addr;
input           read;
input           write;
input           untran;
input   [5:0]   size;

reg     [63:0]  addr_mask;

begin
    addr_mask = ~((64'h1 << size) - 1);
    atc_tran_addr   [func][entry] = tran_addr  & addr_mask;
    atc_utran_addr  [func][entry] = utran_addr & addr_mask;
    atc_active      [func][entry] = 1'b0;
    atc_read        [func][entry] = read;
    atc_write       [func][entry] = write;
    atc_untran      [func][entry] = untran;
    atc_size        [func][entry] = size;
end
endtask

task ats_request_cpl;
input   [2:0]       tc;
input   [7:0]       tag;
input   [15:0]      reqid;
input   [63:0]      addr;

reg     [7:0]       func;
integer             i;
reg     [63:0]      addr_mask;
reg                 match;
reg     [3:0]       entry;
reg     [63:0]      tran_addr;
reg     [32767:0]   payload;
reg                 read;
reg                 write;
reg                 untran;
reg     [5:0]       size;
reg                 s;
reg     [63:0]      dummy_addr;

begin
    func = reqid[7:0];
    match = 1'b0;
    entry = 0;
    payload = 0;
    for (i=0; i<16; i=i+1) begin
        addr_mask = ~((64'h1 << atc_size[func][i]) - 64'h1);
        if ((atc_utran_addr[func][i] & addr_mask) == (addr & addr_mask)) begin
            entry = i;
            match = 1'b1;
        end
    end
    if (match) begin
        tran_addr   = atc_tran_addr [func][entry];
        read        = atc_read      [func][entry];
        write       = atc_write     [func][entry];
        untran      = atc_untran    [func][entry];
        size        = atc_size      [func][entry];
        s = (size > 5'd12);
        addr_mask = ~((64'h1 << size) - 64'h1);
        tran_addr = (tran_addr & addr_mask) | ((64'h1 << (size - 1)) - 1);    // Set lower bits to indicate size
        atc_active[func][entry] = 1'b1;
        payload[63:32] = {5'h0, untran, write, read, tran_addr[15:12], s, 1'b0, 2'h0, tran_addr[23:16], tran_addr[31:24]};
        payload[31:0]  = {tran_addr[39:32], tran_addr[47:40], tran_addr[55:48], tran_addr[63:56]};
        dummy_addr = RCB_BYTES - 8;
        xfer (tc, 3'h0, 7'b10_01010, 10'h2, 4'hf, 4'hf, dummy_addr, reqid, tag, payload, 1'b0, 1'b0, 1'b0, 1'b1);
    end
    else begin
        xfer (tc, 3'h0, 7'b00_01010, 10'h2, 4'hf, 4'hf, 64'h0,      reqid, tag, payload, 1'b0, 1'b0, 1'b0, 1'b1);
    end
end
endtask

task ats_invalidate;
input   [7:0]   func;
input   [3:0]   entry;
input   [4:0]   itag;

reg     [63:0]  addr;
reg     [5:0]   size;
reg             s;
reg     [15:0]  f_bdf;
reg     [7:0]   msg_tag;
reg     [63:0]  addr_mask;

begin
    if (atc_active[func][entry]) begin
        f_bdf = dut_bdf + func;
        addr  = atc_utran_addr [func][entry];
        size  = atc_size       [func][entry];
        s     = (size > 5'd12);
        addr_mask = ~((64'h1 << size) - 1);
        addr  = (addr & addr_mask) | ((64'h1 << (size - 1)) - 1);   // Set lower bits to indicate size
        msg_tag = {3'h0, itag};
        // Issue an Invalidate for the translation
        //            TC    MSGD  LEN   ROUTE MSGCODE TAG      ADDR            DATA
        transmit_msg (3'h0, 1'b1, 8'h2, 3'h2,  8'h1,  msg_tag, {f_bdf, 48'h0}, {8'h0,        addr[15:12], s, 3'h0, addr[23:16], addr[31:24],
                                                                                addr[39:32], addr[47:40],          addr[55:48], addr[63:56]});
        atc_itag_entry[func][itag] = entry;
        atc_inval     [func][entry] = 1'b1;
    end
end
endtask

task ats_invalidate_ack_check;
input   [159:0]     msg;

reg     [6:0]       fmt_and_type;
reg     [7:0]       msg_code;
reg     [7:0]       func;
reg     [3:0]       entry;
reg     [31:0]      itag_vector;
integer             i;
reg     [4:0]       itag;

begin
    fmt_and_type = msg[6:0];
    msg_code     = msg[63:56];
    func         = msg[47:40];
    itag_vector  = {msg[103:96], msg[111:104], msg[119:112], msg[127:120]};
    for (i=0; i<32; i=i+1) begin
        if (itag_vector[i] == 1'b1)
            itag = i;
    end
    // Check only ATS Invalidate Completion Messges
    if ((fmt_and_type == 7'b01_10010) & (msg_code == 8'h02)) begin
        entry = atc_itag_entry[func][itag];
        if (atc_active[func][entry] & atc_inval[func][entry])   // Look for invalidated table entry
        begin
            atc_active[func][entry] = 1'b0;                     // De-activate it or Flag Error
            atc_inval[func][entry]  = 1'b0;
        end
        else begin
            $display ("ERROR : ATS Invalidate Acknowledgement to Inactive Entry at time = %0t", $time);
            inc_errors;
        end
    end
end
endtask

task ats_memory_access_check;
input   [63:0]      addr;
input               wr_rd_n;
input   [7:0]       func;
input   [1:0]       at;

integer             i;
reg     [63:0]      addr_mask;
reg                 tran_match;
reg                 utran_match;
reg     [3:0]       tran_entry;
reg     [3:0]       utran_entry;

begin
    tran_match = 0;
    utran_match = 0;
    for (i=0; i<16; i=i+1) begin
        addr_mask = ~((64'h1 << atc_size[func][i]) - 64'h1);
        if (((addr & addr_mask) == (atc_tran_addr[func][i] & addr_mask)) & atc_active[func][i] & ~atc_untran[func][i]) begin
            tran_match = 1'b1;
            tran_entry = i;
        end
        if (((addr & addr_mask) == (atc_utran_addr[func][i] & addr_mask)) & atc_active[func][i] & atc_untran[func][i]) begin
            utran_match = 1'b1;
            utran_entry = i;
        end
    end
    if ((at == 2'b10) & ~tran_match) begin
        $display ("ERROR : translated address received which does not match ATC entry");
        inc_errors;
    end
    if ((at == 2'b10) & tran_match) begin
        if (wr_rd_n & ~atc_write[func][tran_entry]) begin
            $display ("ERROR : translated address write received to entry which does not allow writes");
            inc_errors;
        end
        if (~wr_rd_n & ~atc_read[func][tran_entry]) begin
            $display ("ERROR : translated address read received to entry which does not allow reads");
            inc_errors;
        end
    end
    if ((at == 2'b00) & utran_match) begin
        if (wr_rd_n & ~atc_write[func][utran_entry] & atc_read[func][utran_entry]) begin
            $display ("ERROR : untranslated address write received to entry which does not allow writes");
            inc_errors;
        end
        if (~wr_rd_n & ~atc_read[func][utran_entry] & atc_write[func][utran_entry]) begin
            $display ("ERROR : untranslated address read received to entry which does not allow reads");
            inc_errors;
        end
    end
end
endtask

always @(posedge clk)
begin
    if (pcie_model.msg_en) begin
        ats_invalidate_ack_check(pcie_model.msg_data);

        if (pcie_model.msg_data[6:0] === 7'h30) begin
            if (pcie_model.msg_data[63:56] == 8'h30)
                $display ("%m: WARNING : Correctable_Error Message received from DUT at time %t", $time);
            else if (pcie_model.msg_data[63:56] == 8'h31)
                $display ("%m: WARNING : Non-Fatal_Error Message received from DUT at time %t", $time);
            else if (pcie_model.msg_data[63:56] == 8'h33)
                $display ("%m: WARNING : Fatal_Error Message received from DUT at time %t", $time);
        end
    end
end

// Set the CPL timeout value based on link width and speed; increase timeout values as link gets smaller and slower
always @*
begin
    case ({link_speed, neg_link_width})

        8'b00_000000    : cpl_timeout_limit = CPL_TIMEOUT_LIMIT_INIT * 4;  // Gen1, x1; cap at * 4
        8'b00_000001    : cpl_timeout_limit = CPL_TIMEOUT_LIMIT_INIT * 4;  // Gen1, x1; cap at * 4
        8'b00_000010    : cpl_timeout_limit = CPL_TIMEOUT_LIMIT_INIT * 4;  // Gen1, x2; cap at * 4
        8'b00_000100    : cpl_timeout_limit = CPL_TIMEOUT_LIMIT_INIT * 4;  // Gen1, x4
        8'b00_001000    : cpl_timeout_limit = CPL_TIMEOUT_LIMIT_INIT * 4;  // Gen1, x8
        8'b00_010000    : cpl_timeout_limit = CPL_TIMEOUT_LIMIT_INIT * 4;  // Gen1, x16

        8'b01_000000    : cpl_timeout_limit = CPL_TIMEOUT_LIMIT_INIT * 4;  // Gen2, x1; cap at * 4
        8'b01_000001    : cpl_timeout_limit = CPL_TIMEOUT_LIMIT_INIT * 4;  // Gen2, x1; cap at * 4
        8'b01_000010    : cpl_timeout_limit = CPL_TIMEOUT_LIMIT_INIT * 4;  // Gen2, x2
        8'b01_000100    : cpl_timeout_limit = CPL_TIMEOUT_LIMIT_INIT * 4;  // Gen2, x4
        8'b01_001000    : cpl_timeout_limit = CPL_TIMEOUT_LIMIT_INIT * 4;  // Gen2, x8
        8'b01_010000    : cpl_timeout_limit = CPL_TIMEOUT_LIMIT_INIT * 2;  // Gen2, x16

        8'b10_000000    : cpl_timeout_limit = CPL_TIMEOUT_LIMIT_INIT * 4;  // Gen3, x1
        8'b10_000001    : cpl_timeout_limit = CPL_TIMEOUT_LIMIT_INIT * 4;  // Gen3, x1
        8'b10_000010    : cpl_timeout_limit = CPL_TIMEOUT_LIMIT_INIT * 4;  // Gen3, x2
        8'b10_000100    : cpl_timeout_limit = CPL_TIMEOUT_LIMIT_INIT * 4;  // Gen3, x4
        8'b10_001000    : cpl_timeout_limit = CPL_TIMEOUT_LIMIT_INIT * 2;  // Gen3, x8
        8'b10_010000    : cpl_timeout_limit = CPL_TIMEOUT_LIMIT_INIT * 1;  // Gen3, x16

        default         : cpl_timeout_limit = CPL_TIMEOUT_LIMIT_INIT;
    endcase
end


// ----------------------------------------------------
// Implement BFM Non-Posted Request Completion Timeouts

genvar jj;
generate for (jj=0; jj<32; jj=jj+1)
    begin : gen_bfm_cpl_timeout
        always @(posedge clk)
        begin
            if ((rst_n === 1'b1) & (bfm_init_tag_status[jj] === 1'b1))
            begin
                if (($time - bfm_init_tag_req_time[jj]) >= cpl_timeout_limit)
                begin
                    if (~suppress_bfm_completion_timeout_reporting)
                    begin
                        if (bfm_init_tag_expect_timeout[jj] != 1'b1)
                        begin
                            $display ("%m : ERROR : Completion Timeout : BFM Non-Posted Request failed to be completed before timeout; Tag=0x%x, StartTime=%t, TimedOutTime=%t, Delta=%t", jj, bfm_init_tag_req_time[jj], $time, ($time - bfm_init_tag_req_time[jj]));
                            inc_errors;
                        end
                        else
                        begin
                            $display ("%m : INFO : Completion Timeout : BFM Non-Posted Request received Completion Timeout Status as expected; Tag=0x%x (time %t)", jj, $time);
                        end
                    end

                    bfm_init_tag_status[jj]           = 1'h0;
                    bfm_init_tag_cpl_status[jj]       = 3'h0;
                    bfm_init_tag_payload[jj]          = 32768'h0;
                    bfm_init_tag_expected_payload[jj] = 32768'h0;
                    bfm_init_tag_check_payload[jj]    = 1'b0;
                    bfm_init_tag_expect_timeout[jj]   = 1'b0;
                    bfm_init_tag_req_time[jj]         = $time;
                end
            end
        end
    end
endgenerate



// ---------------------------
// Receive PCI Express Packets

always @(posedge clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
        rx_busy <= 1'b0;
    else
    begin
        if (rx_sop & ~rx_eop & rx_en)
            rx_busy <= 1'b1;
        else if (rx_eop & rx_en)
            rx_busy <= 1'b0;
    end
end

// If slow_rx_loop == 1'b1, then wait slow_rx_load clocks between selected types of packets starts
//   to enable testing buffer full conditions; for normal operation set slow_rx_loop == 1'b0
always @(posedge clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
        slow_rx_ctr <= 16'h0000;
    else
    begin
        if (rx_sop & rx_en & (slow_rx_loop == 1'b1))
            slow_rx_ctr <= slow_rx_load;
        else if (slow_rx_ctr != 16'h00)
            slow_rx_ctr <= slow_rx_ctr - 16'h01;
    end
end

assign slow_ctr_en = (slow_rx_ctr == 16'h0000);

assign slow_fmt_type = rx_data[6:0];

// Slow only selected types of packets; note unselected packet types will still be slowed when selected slow types are being received
assign slow_fmt_type_en = (((slow_fmt_type[6:1] == 6'b00_0000 ) | (slow_fmt_type[6:1] == 6'b01_0000 )) & slow_rx_mem_rd) |  // Mem Rd
                          (((slow_fmt_type[6:0] == 7'b10_00000) | (slow_fmt_type[6:0] == 7'b10_00000)) & slow_rx_mem_wr) |  // Mem Wr
                          (((slow_fmt_type[6:0] == 7'b00_00010)                                      ) & slow_rx_io_rd ) |  // IO Rd
                          (((slow_fmt_type[6:0] == 7'b10_00010)                                      ) & slow_rx_io_wr ) |  // IO Wr
                          (((slow_fmt_type[6:0] == 7'b00_00100)                                      ) & slow_rx_cfg_rd) |  // Cfg Rd (Type 0)
                          (((slow_fmt_type[6:0] == 7'b10_00100)                                      ) & slow_rx_cfg_wr) |  // Cfg Wr (Type 0)
                          (((slow_fmt_type[6:4] == 3'b01_1    )                                      ) & slow_rx_msg   ) |  // Message without Data Payload
                          (((slow_fmt_type[6:4] == 3'b11_1    )                                      ) & slow_rx_msgd  ) |  // Message with Data Payload
                          (((slow_fmt_type[6:0] == 7'b00_01010)                                      ) & slow_rx_cpl   ) |  // Completion without Data
                          (((slow_fmt_type[6:0] == 7'b10_01010)                                      ) & slow_rx_cpld  );   // Completion with Data

assign slow_en = slow_fmt_type_en ? slow_ctr_en : 1'b1;

assign rx_en = rst_n & ((rx_sop & slow_en) | rx_busy);

// Compute number of bytes in the packet
assign has_payload   =  rx_data[    6];                  // 1 == has payload
assign dw_length     = {rx_data[17:16], rx_data[31:24]}; // length[9:0]
assign max_dw_length = {(dw_length == 10'h0), dw_length};
assign payload_bytes = has_payload ? {max_dw_length, 2'b00} : 13'h0;

assign hdr_bytes    = rx_data[ 5] ? 13'd16 : 13'd12; // 4 DWORD Header : 3 DWORD Header
assign hdr_td_bytes = rx_data[23] ? 13'd4  : 13'd0;  // TLP Digest Present : Not Present

// r_rx_bcnt is used 1 clock after rx_sop
always @(posedge clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
        r_rx_bcnt <= 13'h0;
    else if (rx_sop & rx_en)
        r_rx_bcnt <= hdr_bytes + hdr_td_bytes + payload_bytes;
end

function automatic [BFM_MEM_SIZE_BITS-1:0] bfm_addr_from_mem_addr;
input  [63:0]                  mem_addr;
begin
    bfm_addr_from_mem_addr = mem_addr[BFM_MEM_SIZE_BITS+1:2];
end
endfunction

// Process received packets
initial
begin : receive_initial_block

    reg     [127:0]                     rx_hdr;

    reg     [6:0]                       fmt_and_type;
    reg     [CMD_DATA_WIDTH-1:0]        cmd_data;
    reg                                 td_present;
    reg     [15:0]                      cpl_id_capt;
    reg     [7:0]                       cpl_tag_capt;

    reg     [7:0]                       msg_code_capt;

    reg     [1:0]                       at_field;

    reg     [7:0]                       dut_req_tag_capt;
    reg     [15:0]                      dut_req_id_capt;
    reg     [2:0]                       dut_req_attr_capt;
    reg     [15:0]                      dut_cpl_req_id_capt;
    reg     [2:0]                       dut_cpl_status;
    reg     [11:0]                      dut_cpl_bcount_capt;
    reg     [6:0]                       dut_cpl_low_addr_capt;
    reg     [7:0]                       dut_cpl_req_tag_capt;
    reg     [2:0]                       dut_req_tc_capt;
    reg                                 dut_req_length_msb;
    reg     [10:0]                      dut_req_length_capt;
    reg     [3:0]                       dut_req_fdw_be_capt;
    reg     [3:0]                       dut_req_ldw_be_capt;
    reg     [31:0]                      dut_req_addr0_capt;
    reg     [31:0]                      dut_req_addr1_capt;

    reg     [31:0]                      ctime;
    reg     [7:0]                       bs;
    reg     [4:0]                       dv;
    reg     [2:0]                       f;
    reg     [11:0]                      r;

    reg                                 valid;
    reg     [1:0]                       ctype;              // Packet type (Non-posted, posted, completion)
    reg                                 has_payload;
    reg                                 req_c;              // Set if transaction requires completion
    reg                                 mem;
    reg                                 io;
    reg                                 cfg;
    reg                                 cpl;                // Set if completion packet
    reg                                 msg;
    reg                                 addr64;
    reg                                 wr_rd_n;

    reg     [12:0]                      bcnt;            // Running Byte Count
    reg     [12:0]                      st_bcnt;         // Starting Byte Count

    reg     [3:0]                       first_dw_be;
    reg     [3:0]                       last_dw_be;
    reg     [31:0]                      data_dword;
    reg     [3:0]                       data_be;
    reg     [63:0]                      mem_addr;
    reg     [BFM_MEM_SIZE_BITS-1:0]     bfm_addr;

    reg                                 hit_msix;
    reg     [11:0]                      hit_msix_vector;
    reg                                 hit_msix_valid;

    reg                                 hit_msi;
    reg     [4:0]                       hit_msi_vector;
    reg     [15:0]                      check_msi_vector;
    reg     [15:0]                      expected_msi_vector;
    reg                                 hit_msi_valid;

    reg                                 hit_bfm_mem32;
    reg                                 hit_bfm_mem64;
    reg                                 hit_mem_user;
    reg                                 hit_io_user;
    reg                                 hit_user;

    reg                                 dw_sel;

    integer                             cpl_mem_index;
    integer                             j;

    reg     [32767:0]                   cpl_payload;
    reg     [32767:0]                   exp_cpl_payload;

    reg                                 data_error;
    reg     [31:0]                      exp_data;
    reg     [31:0]                      rcv_data;
    reg     [3:0]                       mask_dw_be;      // Byte Enables to use to generate the Data Comparison Mask (mask_data)
    reg     [31:0]                      mask_dw_data;    // Mask used to determine which bits, i.e. bytes should be used during comparisons.

    @(posedge clk);

    // Pull packets out of the FIFO as fast as possible (use slow_rx_loop to throttle speed)
    while (1)
    begin
        // Wait for the link to come up before attempting to receive TLPs
        while (rst_n !== 1'b1)
            @(posedge clk);

        // Wait for a packet read to start
        while (~(rx_sop & rx_en))
            @(posedge clk);

        // Get full receive packet header
        rx_hdr[ 63: 0] = rx_data[63:0];
        @(posedge clk);
        rx_hdr[127:64] = rx_data[63:0];

        // Capture transaction format and type from header
        fmt_and_type = rx_hdr[ 6: 0]; // {Fmt[1:0], Type[4:0]}

        // Capture decode information
        cmd_data = rx_cmd_data;

        // Capture whether TLP Digest (ECRC) is present
        td_present   = rx_hdr[23];

        // Capture Completer ID (if its a completion)
        cpl_id_capt  = {rx_hdr[39:32], rx_hdr[47:40]};

        // Capture Completion Info
        cpl_tag_capt = rx_hdr[87:80];

        // Capture Message Code from header
        msg_code_capt = rx_hdr[63:56];

        // Capture AT Code
        at_field = rx_hdr[19:18];

        // Capture header information required for generating completions
        dut_req_tag_capt      =  rx_hdr[55:48];
        dut_req_id_capt       = {rx_hdr[39:32], rx_hdr[47:40]};
        dut_req_attr_capt     = {rx_hdr[10], rx_hdr[21:20]};
        dut_cpl_req_tag_capt  =  rx_hdr[87:80];
        dut_cpl_req_id_capt   = {rx_hdr[71:64], rx_hdr[79:72]};
        dut_cpl_status        =  rx_hdr[55:53];
        dut_cpl_bcount_capt   = {rx_hdr[51:48], rx_hdr[63:56]};
        dut_cpl_low_addr_capt =  rx_hdr[94:88];
        dut_req_tc_capt       =  rx_hdr[14:12];
        dut_req_length_msb    =  ({rx_hdr[17:16], rx_hdr[31:24]} == 10'b0) ? 1'b1 : 1'b0;
        dut_req_length_capt   = {dut_req_length_msb,rx_hdr[17:16], rx_hdr[31:24]};
        dut_req_fdw_be_capt   =  rx_hdr[8] ? 4'hf : rx_hdr[59:56];                                              // if TH set, first_be = 4'hf
        dut_req_ldw_be_capt   =  rx_hdr[8] ? ((dut_req_length_capt == 11'h1) ? 4'h0 : 4'hf) : rx_hdr[63:60];    // if TH set, last_be = 4'h0 for 1-word req, 4'hf for >1 word req

        case (fmt_and_type)
            7'b00_00000 : begin valid = 1; ctype = TYPE_N; has_payload = 0; req_c = 1; mem = 1; io = 0; cfg = 0; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 0; end  // Memory Read Request (32-bit address)
            7'b01_00000 : begin valid = 1; ctype = TYPE_N; has_payload = 0; req_c = 1; mem = 1; io = 0; cfg = 0; cpl = 0; msg = 0; addr64 = 1; wr_rd_n = 0; end  // Memory Read Request (64-bit address)

            7'b00_00001 : begin valid = 1; ctype = TYPE_N; has_payload = 0; req_c = 1; mem = 1; io = 0; cfg = 0; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 0; end  // Memory Read Request-Locked (32-bit address); Endpoints do not support locked transactions
            7'b01_00001 : begin valid = 1; ctype = TYPE_N; has_payload = 0; req_c = 1; mem = 1; io = 0; cfg = 0; cpl = 0; msg = 0; addr64 = 1; wr_rd_n = 0; end  // Memory Read Request-Locked (64-bit address); Endpoints do not support locked transactions

            7'b10_00000 : begin valid = 1; ctype = TYPE_P; has_payload = 1; req_c = 0; mem = 1; io = 0; cfg = 0; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 1; end  // Memory Write Request (32-bit address)
            7'b11_00000 : begin valid = 1; ctype = TYPE_P; has_payload = 1; req_c = 0; mem = 1; io = 0; cfg = 0; cpl = 0; msg = 0; addr64 = 1; wr_rd_n = 1; end  // Memory Write Request (64-bit address)

            7'b00_00010 : begin valid = 1; ctype = TYPE_N; has_payload = 0; req_c = 1; mem = 0; io = 1; cfg = 0; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 0; end  // I/O Read Request (32-bit address)
            7'b10_00010 : begin valid = 1; ctype = TYPE_N; has_payload = 1; req_c = 1; mem = 0; io = 1; cfg = 0; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 1; end  // I/O Write Request (32-bit address)

            7'b00_00100 : begin valid = 1; ctype = TYPE_N; has_payload = 0; req_c = 1; mem = 0; io = 0; cfg = 1; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 0; end  // Configuration Read  (Type 0)
            7'b10_00100 : begin valid = 1; ctype = TYPE_N; has_payload = 1; req_c = 1; mem = 0; io = 0; cfg = 1; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 1; end  // Configuration Write (Type 0)

            7'b00_00101 : begin valid = 1; ctype = TYPE_N; has_payload = 0; req_c = 1; mem = 0; io = 0; cfg = 1; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 0; end  // Configuration Read  (Type 1)
            7'b10_00101 : begin valid = 1; ctype = TYPE_N; has_payload = 1; req_c = 1; mem = 0; io = 0; cfg = 1; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 1; end  // Configuration Write (Type 1)

            7'b01_10000 : begin valid = 1; ctype = TYPE_P; has_payload = 0; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 1; addr64 = 1; wr_rd_n = 1; end  // Message Request without Data Payload; Routed to Root Complex
            7'b01_10001 : begin valid = 1; ctype = TYPE_P; has_payload = 0; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 1; addr64 = 1; wr_rd_n = 1; end  // Message Request without Data Payload; Routed by Address
            7'b01_10010 : begin valid = 1; ctype = TYPE_P; has_payload = 0; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 1; addr64 = 1; wr_rd_n = 1; end  // Message Request without Data Payload; Routed by ID
            7'b01_10011 : begin valid = 1; ctype = TYPE_P; has_payload = 0; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 1; addr64 = 1; wr_rd_n = 1; end  // Message Request without Data Payload; Broadcast from Root
            7'b01_10100 : begin valid = 1; ctype = TYPE_P; has_payload = 0; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 1; addr64 = 1; wr_rd_n = 1; end  // Message Request without Data Payload; Local - Terminate at Receiver
            7'b01_10101 : begin valid = 1; ctype = TYPE_P; has_payload = 0; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 1; addr64 = 1; wr_rd_n = 1; end  // Message Request without Data Payload; Gathered and Routed to Root Complex
            7'b01_10110 : begin valid = 1; ctype = TYPE_P; has_payload = 0; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 1; addr64 = 1; wr_rd_n = 1; end  // Message Request without Data Payload; Reserved - Terminate at Receiver
            7'b01_10111 : begin valid = 1; ctype = TYPE_P; has_payload = 0; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 1; addr64 = 1; wr_rd_n = 1; end  // Message Request without Data Payload; Reserved - Terminate at Receiver

            7'b11_10000 : begin valid = 1; ctype = TYPE_P; has_payload = 1; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 1; addr64 = 1; wr_rd_n = 1; end  // Message Request with Data Payload; Routed to Root Complex
            7'b11_10001 : begin valid = 1; ctype = TYPE_P; has_payload = 1; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 1; addr64 = 1; wr_rd_n = 1; end  // Message Request with Data Payload; Routed by Address
            7'b11_10010 : begin valid = 1; ctype = TYPE_P; has_payload = 1; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 1; addr64 = 1; wr_rd_n = 1; end  // Message Request with Data Payload; Routed by ID
            7'b11_10011 : begin valid = 1; ctype = TYPE_P; has_payload = 1; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 1; addr64 = 1; wr_rd_n = 1; end  // Message Request with Data Payload; Broadcast from Root
            7'b11_10100 : begin valid = 1; ctype = TYPE_P; has_payload = 1; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 1; addr64 = 1; wr_rd_n = 1; end  // Message Request with Data Payload; Local - Terminate at Receiver
            7'b11_10101 : begin valid = 1; ctype = TYPE_P; has_payload = 1; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 1; addr64 = 1; wr_rd_n = 1; end  // Message Request with Data Payload; Gathered and Routed to Root Complex
            7'b11_10110 : begin valid = 1; ctype = TYPE_P; has_payload = 1; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 1; addr64 = 1; wr_rd_n = 1; end  // Message Request with Data Payload; Reserved - Terminate at Receiver
            7'b11_10111 : begin valid = 1; ctype = TYPE_P; has_payload = 1; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 1; addr64 = 1; wr_rd_n = 1; end  // Message Request with Data Payload; Reserved - Terminate at Receiver

            7'b00_01010 : begin valid = 1; ctype = TYPE_C; has_payload = 0; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 1; msg = 0; addr64 = 0; wr_rd_n = 1; end  // Completion without Data
            7'b10_01010 : begin valid = 1; ctype = TYPE_C; has_payload = 1; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 1; msg = 0; addr64 = 0; wr_rd_n = 1; end  // Completion with Data

            default     : begin valid = 0; ctype = TYPE_P; has_payload = 0; req_c = 0; mem = 0; io = 0; cfg = 0; cpl = 0; msg = 0; addr64 = 0; wr_rd_n = 0; end  // Unsupported Fmt_and_Type
        endcase

        if (addr64)
            dut_req_addr0_capt = {rx_hdr[103:96],rx_hdr[111:104],rx_hdr[119:112],rx_hdr[127:120]};
        else
            dut_req_addr0_capt = {rx_hdr[71:64],rx_hdr[79:72],rx_hdr[87:80],rx_hdr[95:88]};

        dut_req_addr1_capt = {rx_hdr[71:64],rx_hdr[79:72],rx_hdr[87:80],rx_hdr[95:88]};

        if (!$test$plusargs("pcie_traffic_msgs_off"))
        begin
            ctime = $time/1000; // nS

            // Parse address for configuration information
            bs = dut_req_addr0_capt[31:24];
            dv = dut_req_addr0_capt[23:19];
            f  = dut_req_addr0_capt[18:16];
            r  = {dut_req_addr0_capt[11:2], 2'b00};

            casex (fmt_and_type)
                //                                  Cmd                                                                      Tg                    RqID                 CpID             Length               Addr64              Addr32              L                    F                    CplStat,        CplLowAddr             CplBcount            Bs  Dv  F  R  MsgCode                                     Time
                7'b00_00000 : $display ("U : MRd32 %h %h ---- %h -------- %h %h %h - -- --- -- -- - --- -- %d",       dut_req_tag_capt,     dut_req_id_capt,                      dut_req_length_capt,                     dut_req_addr0_capt, dut_req_ldw_be_capt, dut_req_fdw_be_capt,                                                                                                                       ctime);
                7'b01_00000 : $display ("U : MRd64 %h %h ---- %h %h %h %h %h - -- --- -- -- - --- -- %d",             dut_req_tag_capt,     dut_req_id_capt,                      dut_req_length_capt, dut_req_addr1_capt, dut_req_addr0_capt, dut_req_ldw_be_capt, dut_req_fdw_be_capt,                                                                                                                       ctime);

                7'b00_00001 : $display ("U : MRL32 %h %h ---- %h -------- %h %h %h - -- --- -- -- - --- -- %d",       dut_req_tag_capt,     dut_req_id_capt,                      dut_req_length_capt,                     dut_req_addr0_capt, dut_req_ldw_be_capt, dut_req_fdw_be_capt,                                                                                                                       ctime);
                7'b01_00001 : $display ("U : MRL64 %h %h ---- %h %h %h %h %h - -- --- -- -- - --- -- %d",             dut_req_tag_capt,     dut_req_id_capt,                      dut_req_length_capt, dut_req_addr1_capt, dut_req_addr0_capt, dut_req_ldw_be_capt, dut_req_fdw_be_capt,                                                                                                                       ctime);

                7'b10_00000 : $display ("U : MWr32 -- %h ---- %h -------- %h %h %h - -- --- -- -- - --- -- %d",                             dut_req_id_capt,                      dut_req_length_capt,                     dut_req_addr0_capt, dut_req_ldw_be_capt, dut_req_fdw_be_capt,                                                                                                                       ctime);
                7'b11_00000 : $display ("U : MWr64 -- %h ---- %h %h %h %h %h - -- --- -- -- - --- -- %d",                                   dut_req_id_capt,                      dut_req_length_capt, dut_req_addr1_capt, dut_req_addr0_capt, dut_req_ldw_be_capt, dut_req_fdw_be_capt,                                                                                                                       ctime);

                7'b00_00010 : $display ("U : IoR32 %h %h ---- %h -------- %h %h %h - -- --- -- -- - --- -- %d",       dut_req_tag_capt,     dut_req_id_capt,                      dut_req_length_capt,                     dut_req_addr0_capt, dut_req_ldw_be_capt, dut_req_fdw_be_capt,                                                                                                                       ctime);
                7'b10_00010 : $display ("U : IoW32 %h %h ---- %h -------- %h %h %h - -- --- -- -- - --- -- %d",       dut_req_tag_capt,     dut_req_id_capt,                      dut_req_length_capt,                     dut_req_addr0_capt, dut_req_ldw_be_capt, dut_req_fdw_be_capt,                                                                                                                       ctime);

                7'b00_00100 : $display ("U : Cf0Rd %h %h ---- %h -------- -------- - - - -- --- %h %h %h %h -- %d",   dut_req_tag_capt,     dut_req_id_capt,                      dut_req_length_capt,                                         dut_req_ldw_be_capt, dut_req_fdw_be_capt,                                                             bs, dv, f, r,                                             ctime);
                7'b10_00100 : $display ("U : Cf0Wr %h %h ---- %h -------- -------- - - - -- --- %h %h %h %h -- %d",   dut_req_tag_capt,     dut_req_id_capt,                      dut_req_length_capt,                                         dut_req_ldw_be_capt, dut_req_fdw_be_capt,                                                             bs, dv, f, r,                                             ctime);

                7'b00_00101 : $display ("U : Cf1Rd %h %h ---- %h -------- -------- - - - -- --- %h %h %h %h -- %d",   dut_req_tag_capt,     dut_req_id_capt,                      dut_req_length_capt,                                         dut_req_ldw_be_capt, dut_req_fdw_be_capt,                                                             bs, dv, f, r,                                             ctime);
                7'b10_00101 : $display ("U : Cf1Wr %h %h ---- %h -------- -------- - - - -- --- %h %h %h %h -- %d",   dut_req_tag_capt,     dut_req_id_capt,                      dut_req_length_capt,                                         dut_req_ldw_be_capt, dut_req_fdw_be_capt,                                                             bs, dv, f, r,                                             ctime);

                7'b01_10xxx : $display ("U : Msg   -- %h ---- %h -------- -------- - - - -- --- -- -- - --- %h %d",                         dut_req_id_capt,                      dut_req_length_capt,                                                                                                                                                             {dut_req_ldw_be_capt, dut_req_fdw_be_capt}, ctime);
                7'b11_10xxx : $display ("U : MsgD  -- %h ---- %h -------- -------- - - - -- --- -- -- - --- %h %d",                         dut_req_id_capt,                      dut_req_length_capt,                                                                                                                                                             {dut_req_ldw_be_capt, dut_req_fdw_be_capt}, ctime);

                7'b00_01010 : $display ("U : Cpl   %h %h %h %h -------- -------- - - %h %h %h -- -- - --- -- %d",      dut_cpl_req_tag_capt, dut_cpl_req_id_capt, dut_req_id_capt, dut_req_length_capt,                                                                                  dut_cpl_status, dut_cpl_low_addr_capt, dut_cpl_bcount_capt,                                                           ctime);
                7'b10_01010 : $display ("U : CplD  %h %h %h %h -------- -------- - - %h %h %h -- -- - --- -- %d",      dut_cpl_req_tag_capt, dut_cpl_req_id_capt, dut_req_id_capt, dut_req_length_capt,                                                                                  dut_cpl_status, dut_cpl_low_addr_capt, dut_cpl_bcount_capt,                                                           ctime);

                default     : $display ("U : ERROR -- ---- ---- %h -------- -------- - - - -- --- -- -- - --- -- %d",                                                             dut_req_length_capt,                                                                                                                                                                                                         ctime);
            endcase
        end

        // If its a completion to a Cfg Wr/Rd, then verify that the completer ID is from the expected function
        if (bfm_init_tag_is_cfg_en & (cpl == 1'b1) & bfm_init_tag_is_cfg[cpl_tag_capt[4:0]] & (cpl_id_capt != bfm_init_tag_is_cfg_id[cpl_tag_capt[4:0]]) & bfm_init_tag_status[cpl_tag_capt[4:0]])
        begin
            if (bfm_init_tag_is_cfg_en_ur_0 & (cpl_id_capt == (bfm_init_tag_is_cfg_id[cpl_tag_capt[4:0]] & 16'hff00)) & (dut_cpl_status == 3'b001))
                $display ("%m : NOTE : BFM Configuration Request was completed with UR from function 0 (time %t)", $time);
            else begin
                $display ("%m : ERROR : BFM Configuration Request was completed with an incorrect Completer ID; CompleterID==0x%x, Expected==0x%x (time %t)", cpl_id_capt, bfm_init_tag_is_cfg_id[cpl_tag_capt[4:0]], $time);
                inc_errors;
            end
        end

        // If TD Check is set, confirm that the TD bit is set/clr as appropraiate for the selected sending function
        if (bfm_check_td_en == 1'b1)
        begin
            if (dut_req_id_capt == bfm_check_td_id) begin
                if (td_present != bfm_check_td_present) begin
                    $display ("%m : ERROR : TLP received with incorrect TD bit (time %t)", $time);
                    inc_errors;
                end
            end
        end

        // Subtract out header & TLP Digest (if present) to leave just data payload size in bcnt
        bcnt = addr64 ? (r_rx_bcnt - (td_present ? 13'd20 : 13'd16)) :
                        (r_rx_bcnt - (td_present ? 13'd16 : 13'd12));
        st_bcnt = bcnt;   // Save off the starting Byte Count

        // Clear completion payload
        cpl_payload = 0;

        // Save memory base address that is being targeted
        if (addr64)
            mem_addr = {dut_req_addr1_capt[31:0], dut_req_addr0_capt[31:2], 2'b00};
        else
            mem_addr = {32'h0,                    dut_req_addr0_capt[31:2], 2'b00};

        // Check received Memory Read/Write against ATC table
        if (mem)
            ats_memory_access_check(mem_addr, wr_rd_n, dut_req_id_capt[7:0], at_field);  // Address, Write_Read_n, Function, AT_Field

        // MSI-X/MSI Interrupt Controller - MSI-X/MSI Memory Write Address Hit Detection
        hit_msix            = 1'b0;
        hit_msi             = 1'b0;
        hit_msix_vector     = 12'h0;
        hit_msix_valid      = 1'b0;
        hit_msi_vector      = 5'h0;
        check_msi_vector    = 16'h0;
        expected_msi_vector = 16'h0;
        hit_msi_valid       = 1'b0;

        hit_bfm_mem32       = 1'b0;
        hit_bfm_mem64       = 1'b0;
        hit_mem_user        = 1'b0;
        hit_io_user         = 1'b0;
        hit_user            = 1'b0;

        // decode address to determine if MSI-X/MSI Interrupt Controller Hit
        if ((fmt_and_type == 7'b10_00000) | (fmt_and_type == 7'b11_00000)) // if (MemWr32 | MemWr64)
        begin
            // Check for MSI-X Table Hits - Vector Hit size depends upon the number of vectors present;
            //   each vector present is given a DWORD address incrementing from int_msix_addr[63:0]
            if ( (mem_addr[63:2] >= (int_msix_addr[63:2]                       )) &
                 (mem_addr[63:2] <  (int_msix_addr[63:2] + int_msix_num_vectors)) )
            begin
                if ((bcnt >= 13'h1) & (bcnt <= 13'h4)) // Byte count is 1-4 bytes
                begin
                    hit_msix        = 1;
                    hit_msix_vector = (mem_addr[63:2] - {int_msix_addr[63:3], 1'b0}); // Get MSI-X Vector which was hit
                end
                else
                begin
                    $display ("%m : ERROR : MSI-X Interrupt Hit Detected but Payload was not 1 DWORD : Payload DWORDS==%d (time %t)", (bcnt/4), $time);
                    inc_errors;
                    report_status;
                end
            end

            // Check for MSI Hits - Only one MSI Address is defined and must be hit exactly
            if (mem_addr[63:2] == int_msi_addr[63:2])
            begin
                if ((bcnt >= 13'h1) & (bcnt <= 13'h4)) // Byte count is 1-4 bytes
                    hit_msi = 1;
                else
                begin
                    $display ("%m : ERROR : MSI Interrupt Hit Detected but Payload was not 1 DWORD : Payload DWORDS==%d (time %t)", (bcnt/4), $time);
                    inc_errors;
                    report_status;
                end
            end
        end

        // decode address to determine if BFM Mem Hit or User Mem Hit
        if ( (fmt_and_type == 7'b10_00000) |  // MemWr32
             (fmt_and_type == 7'b11_00000) |  // MemWr64
             (fmt_and_type == 7'b00_00000) |  // MemRd32
             (fmt_and_type == 7'b01_00000) )  // MemRd64
        begin
            // Check for BFM Memory hit
            if ( (mem_addr[63:32] == 32'h0                                  ) &
                 (mem_addr[31: 2] >=  bfm_base_addr32[31:2]                 ) &
                 (mem_addr[31: 2] <  (bfm_base_addr32[31:2] + BFM_MEM_DEPTH)) ) // BFM_MEM_DEPTH indicates number of DWORDs supported.
            begin
                hit_bfm_mem32 = 1'b1;
            end
            else if ( (mem_addr[63:32] != 32'h0                            ) &
                      (mem_addr[63: 2] >= ({bfm_base_addr64 [31:0], 30'h0})) &
                      (mem_addr[63: 2] <  ({bfm_limit_addr64[31:0], 30'h0})) )
            begin
                hit_bfm_mem64 = 1'b1;
            end
            else if ((cmd_data[1:0] != 2'b00) && (fmt_and_type[5] == 1'b0)) // 32-bit BAR Decode
            begin
                hit_bfm_mem32 = 1'b1;
            end
            else if ((cmd_data[1:0] != 2'b00) && (fmt_and_type[5] == 1'b1)) // 64-bit BAR Decode
            begin
                hit_bfm_mem64 = 1'b1;
            end
            else
            begin
                hit_mem_user = 1'b1;
            end
        end

        // decode address to determine if BFM IO Hit or User IO Hit
        if ( (fmt_and_type == 7'b10_00010) |  // IOWr
             (fmt_and_type == 7'b00_00010) )  // IORd
        begin
            // Check for BFM IO hit
            if ( (mem_addr[31:2] >=  bfm_base_io_addr32[31:2]                 ) &
                 (mem_addr[31:2] <  (bfm_base_io_addr32[31:2] + BFM_MEM_DEPTH)) ) // BFM_MEM_DEPTH indicates number of DWORDs supported.
            begin
                hit_bfm_mem32 = 1'b1;
            end
            else
            begin
                hit_io_user = 1'b1;
            end
        end

        hit_user = hit_mem_user | hit_io_user;

        dw_sel = addr64 ? 1'b0 : 1'b1;

        // store data if there is a payload
        if (has_payload)
        begin
            // Entire header already consumed
            if (addr64)
                @(posedge clk);

            first_dw_be = rx_hdr[59:56];
            last_dw_be  = (bcnt <= 13'h4) ? first_dw_be : rx_hdr[63:60]; // if only one dword, use first_dw_be
            cpl_mem_index = bfm_init_tag_index[cpl_tag_capt[4:0]];

            bfm_addr = bfm_addr_from_mem_addr(mem_addr);

            // Get expected payload for completions
            exp_cpl_payload = bfm_init_tag_expected_payload[cpl_tag_capt[4:0]];

            while (bcnt > 13'h0)
            begin
                case (dw_sel)
                    1'b0 : data_dword = rx_data[ 31: 0];
                    1'b1 : data_dword = rx_data[ 63:32];
                endcase

                data_be     = (bcnt <= 13'h4) ? last_dw_be : first_dw_be;
                first_dw_be = 4'b1111; // Once first DWORD BE is applied, set to all bytes enabled for subsequent data

                if (!$test$plusargs("pcie_traffic_msgs_off"))
                begin
                    if (cpl)
                        $display ("U :   Data (Tag 0x%x) == 0x%x", dut_cpl_req_tag_capt, data_dword[31:0]);
                    else
                        $display ("U :   Data == 0x%x", data_dword[31:0]);
                end

                // If the transaction is an MSI-X Interrupt Memory Write then check the vector number and generate the interrupt
                if (hit_msix)
                begin
                    if (data_be == 4'hf)
                    begin
                        if (data_dword[31:0] == {20'h0, hit_msix_vector[11:0]}) // Check for data value == vector # as expected
                        begin
                            // Valid MSI-X Vector Write occurred; assert the interrupt
                            if (!$test$plusargs("pcie_traffic_msgs_off"))
                            begin
                                $display ("%m : MSI-X Interrupt Received : Vector=0x%x, Addr=0x%x, Data=0x%x", hit_msix_vector[11:0], mem_addr[63:0], data_dword);
                            end
                            hit_msix_valid                                 <= 1'b1;
                            int_int_msix_vector_hit[hit_msix_vector[11:0]] <= 1'b1;
                        end
                        else
                        begin
                            $display ("%m : ERROR : Invalid MSI-X Vector Data Value : Expected=0x%x : Received=0x%x (time %t)", {20'h0, hit_msix_vector[11:0]}, data_dword[31:0], $time);
                            inc_errors;
                            report_status;
                        end
                    end
                    else
                    begin
                        $display ("%m : ERROR : MSI-X Interrupt writes must have all 4 byte enables asserted : BE==%d (time %t)", data_be, $time);
                        inc_errors;
                        report_status;
                    end
                end
                // If the transaction is an MSI Interrupt Memory Write then check the vector number and generate the interrupt
                else if (hit_msi)
                begin
                    if (data_be == 4'hf)
                    begin
                        // Check for Hitting Data Value and a vector number <= max vectors
                        if ((data_dword[31:5] == {16'h0, int_msi_data[15:5]}) & ({1'b0, data_dword[4:0]} < int_msi_num_vectors))
                        begin
                            hit_msi_vector = data_dword[4:0];

                            // Valid MSI Vector Write occurred; assert the interrupt
                            if (!$test$plusargs("pcie_traffic_msgs_off"))
                            begin
                                $display ("%m : MSI Interrupt Received : Vector=0x%x, Addr=0x%x, Data=0x%x", hit_msi_vector, mem_addr[63:0], data_dword);
                            end
                            hit_msi_valid                               <= 1'b1;
                            int_int_msi_vector_hit[hit_msi_vector[4:0]] <= 1'b1; // Assert the interrupt
                        end
                        else
                        begin
                            $display ("%m : ERROR : Invalid MSI Vector Data Value : Expected=0x%x : Received=0x%x (time %t)", {16'h0, int_msi_data[15:5], 5'bxxxxx}, data_dword[31:0], $time);
                            inc_errors;
                            report_status;
                        end
                    end
                    else
                    begin
                        $display ("%m : ERROR : MSI Interrupt writes must have all 4 byte enables asserted : BE==%d (time %t)", data_be, $time);
                        inc_errors;
                        report_status;
                    end
                end
                else if (hit_bfm_mem32 | hit_bfm_mem64)
                begin
                    if (data_be[0])
                        case (dw_sel)
                            1'b0:  bfm_mem[bfm_addr][7:0] = rx_data[ 7: 0];
                            1'b1:  bfm_mem[bfm_addr][7:0] = rx_data[39:32];
                        endcase

                    if (data_be[1])
                        case (dw_sel)
                            1'b0:  bfm_mem[bfm_addr][15:8] = rx_data[15: 8];
                            1'b1:  bfm_mem[bfm_addr][15:8] = rx_data[47:40];
                        endcase

                    if (data_be[2])
                        case (dw_sel)
                            1'b0:  bfm_mem[bfm_addr][23:16] = rx_data[23:16];
                            1'b1:  bfm_mem[bfm_addr][23:16] = rx_data[55:48];
                        endcase

                    if (data_be[3])
                        case (dw_sel)
                            1'b0:  bfm_mem[bfm_addr][31:24] = rx_data[31:24];
                            1'b1:  bfm_mem[bfm_addr][31:24] = rx_data[63:56];
                        endcase
                    bfm_addr = bfm_addr + {{(BFM_MEM_SIZE_BITS-1){1'b0}}, 1'b1};

                end
                else if (hit_user)
                begin
                    // *******************************************************
                    // To Do: Custom Logic for non-BFM memory writes goes here
                    // *******************************************************
                    $display ("Error: A memory write to user address location found. Data is ignored.");
                    inc_errors;
                    if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                    // *******************************************************
                    // To Do: Custom Logic for non-BFM memory writes goes here
                    // *******************************************************
                end
                else if (cpl)
                begin
                    if (bfm_init_tag_expect_timeout[cpl_tag_capt[4:0]] == 1'b1)
                    begin
                        $display ("%m : ERROR : Received completion for a request that was expected to result in a completion timeout : Tag == 0x%x, Time == %t", cpl_tag_capt[4:0], $time);
                        inc_errors;
                    end

                    // Check completion data against expected completion data
                    if (bfm_init_tag_check_payload[cpl_tag_capt[4:0]] == 1'b1)
                    begin
                        for (j=0; j<32; j=j+1)
                            exp_data[j] = exp_cpl_payload[(cpl_mem_index*32)+j];

                        data_error = 1'b0; // Default to no data error

//                        $display ("%m : READ_COMPARE_LENGTH_INFO: TAG(0x%0X), bcnt(0x%0X), st_bcnt(0x%0X), st_length(0x%0X), curr_length(0x%0X)",
//                                                                     cpl_tag_capt[4:0],  bcnt,     st_bcnt,
//                                                                                                       bfm_init_tag_st_length[cpl_tag_capt[4:0]],
//                                                                                                                                bfm_init_tag_length[cpl_tag_capt[4:0]]);
                        // Determine the Byte Enables to be used based on "length's"!
            if( bfm_init_tag_st_length[cpl_tag_capt[4:0]] == bfm_init_tag_length[cpl_tag_capt[4:0]] &&  // Check if the LENGTH matches
                           bcnt == st_bcnt          //  And the starting byte count matches.
                          ) // Check if there is only one DWORD or the first DWORD is to be compared
                           mask_dw_be = bfm_init_tag_fdw_be[cpl_tag_capt[4:0]];
                        else if(bcnt <= 4 && (bfm_init_tag_length[cpl_tag_capt[4:0]] * 4 - st_bcnt) < 4)
                           mask_dw_be = bfm_init_tag_ldw_be[cpl_tag_capt[4:0]];
                        else
                           mask_dw_be = 4'b1111;    // Otherwise compare ALL bytes

            mask_dw_data = {{8{mask_dw_be[3]}},{8{mask_dw_be[2]}},{8{mask_dw_be[1]}},{8{mask_dw_be[0]}}}; // Generate comparison masks based on the current BEs.

//                        $display ("%m : READ_COMPARE_INFO: TAG(0x%0X), ByteCount(%0d) out of %0d, with %0d actually remaining, DW_SEL(%0d) DW_MASK('h%0X)",
//                                                             cpl_tag_capt[4:0],        bcnt,
//                                                                           bfm_init_tag_st_length[cpl_tag_capt[4:0]]*4,
//                                                                                           bfm_init_tag_length[cpl_tag_capt[4:0]]*4,
//                                                                                                                                           dw_sel,  mask_dw_data);
                         case (dw_sel)
                            2'h0:  if ((mask_dw_data & exp_data) != (mask_dw_data & rx_data[ 31: 0])) begin data_error = 1'b1; rcv_data = rx_data[ 31: 0]; end
                            2'h1:  if ((mask_dw_data & exp_data) != (mask_dw_data & rx_data[ 63:32])) begin data_error = 1'b1; rcv_data = rx_data[ 63:32]; end
                        endcase

                        if (data_error == 1'b1)
                        begin
                            $display ("%m : ERROR : Received completion data does not match expected : Tag == 0x%x, DWORD_Offset == 0x%x, Expected Data == 0x%x, Read Data == 0x%x, Time == %t", cpl_tag_capt[4:0], cpl_mem_index, exp_data, rcv_data, $time);
                            inc_errors;
                        end
                    end

                    // Record received completion data into cpl_payload
                    case (dw_sel)
                        1'b0 : for (j=0; j<32; j=j+1) cpl_payload[(cpl_mem_index*32)+j] = rx_data[j+ 0];
                        1'b1 : for (j=0; j<32; j=j+1) cpl_payload[(cpl_mem_index*32)+j] = rx_data[j+32];
                    endcase

                    cpl_mem_index = cpl_mem_index + 1;
                    bfm_init_tag_index[cpl_tag_capt[4:0]] = cpl_mem_index;
                end

                bcnt   = bcnt - 13'h4;
                dw_sel = dw_sel + 1'b1;

                // Move to next data after processing the last DWORD of the current data phase
                if ((dw_sel == 1'b0) & (bcnt != 13'h0))
                    @(posedge clk);
            end // while

            // OR data received from this completion into the master completion data array for all completions
            bfm_init_tag_payload[cpl_tag_capt[4:0]] = bfm_init_tag_payload[cpl_tag_capt[4:0]] | cpl_payload;
        end // if payload

        // If we received a message; decode and display the message type
        if (!$test$plusargs("pcie_traffic_msgs_off"))
            begin
                if ( (fmt_and_type[6:3] == 4'b01_10) |
                     (fmt_and_type[6:3] == 4'b11_10) )
                begin
                    case ({dut_req_ldw_be_capt, dut_req_fdw_be_capt})

                        8'h00   : $display ("BFM Received Message : Unlock");

                        8'h14   : $display ("BFM Received Message : PM_Active_State_Nak");
                        8'h18   : $display ("BFM Received Message : PM_PME");
                        8'h19   : $display ("BFM Received Message : PME_Turn_Off");
                        8'h1b   : $display ("BFM Received Message : PME_TO_Ack");

                        8'h20   : $display ("BFM Received Message : Assert_INTA");
                        8'h21   : $display ("BFM Received Message : Assert_INTB");
                        8'h22   : $display ("BFM Received Message : Assert_INTC");
                        8'h23   : $display ("BFM Received Message : Assert_INTD");

                        8'h24   : $display ("BFM Received Message : Deassert_INTA");
                        8'h25   : $display ("BFM Received Message : Deassert_INTB");
                        8'h26   : $display ("BFM Received Message : Deassert_INTC");
                        8'h27   : $display ("BFM Received Message : Deassert_INTD");

                        8'h30   : $display ("BFM Received Message : ERR_COR");
                        8'h31   : $display ("BFM Received Message : ERR_NONFATAL");
                        8'h33   : $display ("BFM Received Message : ERR_FATAL");

                        8'h40   : $display ("BFM Received Message : Ignored_Message - Only supported in PCIe 1.0a Specification");
                        8'h41   : $display ("BFM Received Message : Ignored_Message - Only supported in PCIe 1.0a Specification");
                        8'h43   : $display ("BFM Received Message : Ignored_Message - Only supported in PCIe 1.0a Specification");
                        8'h44   : $display ("BFM Received Message : Ignored_Message - Only supported in PCIe 1.0a Specification");
                        8'h45   : $display ("BFM Received Message : Ignored_Message - Only supported in PCIe 1.0a Specification");
                        8'h47   : $display ("BFM Received Message : Ignored_Message - Only supported in PCIe 1.0a Specification");
                        8'h48   : $display ("BFM Received Message : Ignored_Message - Only supported in PCIe 1.0a Specification");

                        8'h50   : $display ("BFM Received Message : Set_Slot_Power_Limit");

                        8'h7e   : $display ("BFM Received Message : Vendor_Defined_Type_0");
                        8'h7f   : $display ("BFM Received Message : Vendor_Defined_Type_1");

                        default : $display ("BFM Received Message : Unknown message code");

                    endcase
                end
            end

        // log requests for packets that require completion
        if (req_c)
        begin
            if (hit_bfm_mem32 | hit_bfm_mem64)
            begin
                if ((at_field == 2'b01) & ((fmt_and_type == 7'b00_00000) | (fmt_and_type == 7'b01_00000))) begin    // ATS Request
                    ats_request_cpl(dut_req_tc_capt, dut_req_tag_capt, dut_req_id_capt, mem_addr);
                end
                else begin
                    dut_req_tag                       = dut_req_tag_capt;
                    dut_req_fmt_and_type[dut_req_tag] = fmt_and_type;
                    dut_req_id[dut_req_tag]           = dut_req_id_capt;
                    dut_req_tc[dut_req_tag]           = dut_req_tc_capt;
                    dut_req_attr[dut_req_tag]         = dut_req_attr_capt;
                    dut_req_length[dut_req_tag]       = dut_req_length_capt;
                    dut_req_fdw_be[dut_req_tag]       = dut_req_fdw_be_capt;
                    dut_req_ldw_be[dut_req_tag]       = dut_req_ldw_be_capt;
                    dut_req_addr0[dut_req_tag]        = dut_req_addr0_capt;
                    dut_req_addr1[dut_req_tag]        = dut_req_addr1_capt;
                    dut_req_time[dut_req_tag]         = $time;
                    dut_req_tag_status[dut_req_tag]   = 1'b1;
                end
            end
            else if (hit_user)
            begin
                // *******************************************************
                // To Do: Custom Logic for non-BFM memory reads goes here
                // *******************************************************
                $display ("%m: Error: A memory read to user address location found. Completion will not be issued.");
                inc_errors;
                if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                // *******************************************************
                // To Do: Custom Logic for non-BFM memory reads goes here
                // *******************************************************
            end
        end


        if (cpl)
        begin
            // Save completion status
            bfm_init_tag_cpl_status[cpl_tag_capt] = dut_cpl_status;

            // close tag after the expected length is fullfilled or a completion is received with status != succesful
            if ((bfm_init_tag_length[cpl_tag_capt] <= dut_req_length_capt) | (dut_cpl_status != 3'h0))
                bfm_init_tag_status[cpl_tag_capt] = 1'b0;
            else
                bfm_init_tag_length[cpl_tag_capt] = bfm_init_tag_length[cpl_tag_capt] - dut_req_length_capt;
        end

        @(posedge clk);

        // MSI-X Interrupts are a single clock pulse; de-activate
        if (hit_msix_valid)
        begin
            int_int_msix_vector_hit[hit_msix_vector[11:0]] <= 1'b0;
            hit_msix_valid                                 <= 1'b0;
        end

        // MSI Interrupts are a single clock pulse; de-activate
        if (hit_msi_valid)
        begin
            int_int_msi_vector_hit[hit_msi_vector[4:0]] <= 1'b0;
            hit_msi_valid                               <= 1'b0;
        end
    end
end

assign #100 int_msix_vector_hit = int_int_msix_vector_hit;

assign #100 int_msi_vector_hit  = int_int_msi_vector_hit;

assign #100 int_legi_vector_hit[0] = inta;
assign #100 int_legi_vector_hit[1] = intb;
assign #100 int_legi_vector_hit[2] = intc;
assign #100 int_legi_vector_hit[3] = intd;



// -------------------------------------------------------------------
// Issue completions in response to a read or non-posted write request

reg     [31:0]      min_cpl_dly;

initial min_cpl_dly = MIN_CPL_DELAY;

initial
begin : rx_completor

    reg     [32767:0]   payload;

    integer             i;
    integer             j;
    integer             k [0:31];
    reg     [31:0]      rnd;
    integer             count;
    integer             loop_count;

    @(posedge clk);
    while (1) begin : rx_completer_while

    // Unused for completions since data is taken from bfm_mem
    payload = 0;

    if (random_completion_order_mode == 1'b1) begin
        i = 0;
        loop_count = 0;
        count = 0;
        for (j=0; j<32; j=j+1)                          // count active requests
            if ((dut_req_tag_status[j] == 1'b1) & (($time - dut_req_time[j]) > min_cpl_dly)) begin
                k[count] = j;                           // Keep a list of them
                count = count + 1;
            end
        if (count != 0) begin
            rnd = $random;
            i = k[rnd % count];                             // select one at random
            loop_count = 1;                                 // service just this one
        end
    end
    else begin
        // find tag that was received first
        i = 0;
        loop_count = dut_req_tag_status[0] & (($time - dut_req_time[0]) > min_cpl_dly);
        for (j=1; j<32; j=j+1)
        begin
            if (dut_req_tag_status[j] == 1'b1)
            begin
                if (loop_count == 0 || dut_req_time[j] < dut_req_time[i])
                begin
                    i = j;
                    loop_count = (($time - dut_req_time[j]) > min_cpl_dly);
                end
            end
        end

    end
    // Continuously walk through tags and look for open tags to complete
    if (loop_count == 1)
    begin
        if (dut_req_tag_status[i] == 1'b1)
        begin // Issue completion packet(s) to complete request
            case (dut_req_fmt_and_type[i])

                7'b10_00010,                // I/O Write
                7'b10_00100 :               // Config Write Type 0
                    begin
                        // Non-posted requests requiring completion w/o data
                        // xfer (tc,            attr, fmt_and_type, length,            first_dw_be,       last_dw_be,        addr,             req_id,        tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
                        xfer    (dut_req_tc[i], 3'h0, 7'b00_01010,  dut_req_length[i], dut_req_fdw_be[i], dut_req_ldw_be[i], dut_req_addr0[i], dut_req_id[i], i,   payload, 1'b0,       1'b0,            1'b0,           1'b0);
                    end

                7'b00_00000, 7'b01_00000,   // Memory Reads
                7'b00_00001, 7'b01_00001,   // Memory Reads - locked
                7'b00_00010,                // I/O Read
                7'b00_00100 :               // Config Read
                    begin
                        // Non-Posted requests requiring completion with data
                        // xfer (tc,            attr,            fmt_and_type, length,            first_dw_be,       last_dw_be,        addr,             req_id,        tag, payload, check_data, no_wait_for_cpl, expect_timeout, payload_for_cpl
                        xfer    (dut_req_tc[i], dut_req_attr[i], 7'b10_01010,  dut_req_length[i], dut_req_fdw_be[i], dut_req_ldw_be[i], dut_req_addr0[i], dut_req_id[i], i,   payload, 1'b0,       1'b0,            1'b0,           1'b0);
                    end

                default:
                    begin
                        $display ("%m : ERROR : Unrecognized request; cannot generate completions for this request");
                        inc_errors;
                        if (STOP_ON_ERR) if (FINISH_STOP_N) $finish; else $stop;
                    end

            endcase



            if (payload[9:0] == 10'b0) begin        // Check for no more data to send (payload[9:0] = remaining length)
                dut_req_tag_status[i] = 1'b0;
            end
            else begin
                if (dut_req_length[i] == 10'b0)
                    dut_req_addr0[i] = dut_req_addr0[i] + ((11'h400 - {1'b0,payload[9:0]}) << 2);       // length==0 means 1024 words
                else
                    dut_req_addr0[i] = dut_req_addr0[i] + ((dut_req_length[i] - payload[9:0]) << 2);    // Advance address
                dut_req_length[i] = payload[9:0];                                                       // Update word count
                if (payload[9:0] == 1) begin                                                            // Update byte-enables
                    dut_req_fdw_be[i] = dut_req_ldw_be[i];                                              // special case: 1 word left
                    dut_req_ldw_be[i] = 4'h0;
                end
                else
                    dut_req_fdw_be[i] = 4'hf;                                                           // general case
            end
        end
    end
    else
        @(posedge clk);

    end // while
end



// ------------------------------
// Information Display Statements

assign link_speed     = mgmt_pcie_status[25:24];
assign neg_link_width = mgmt_pcie_status[31:26];
assign lts_state      = mgmt_pcie_status[ 7: 2];

always @(posedge clk or negedge pl_link_up)
begin
    if (pl_link_up == 1'b0)
    begin
        r_link_speed     <= 2'b00;
        r_neg_link_width <= 6'b000000;
    end
    else
    begin
        r_link_speed     <= link_speed;
        r_neg_link_width <= neg_link_width;
    end
end

always @(posedge clk)
begin
    if ((pl_link_up === 1'b1) & (link_speed != r_link_speed))
    begin
        case (link_speed)
            2'b00   : $display ("%m : INFO : Link speed change : New link speed is 2.5 Gb/s");
            2'b01   : $display ("%m : INFO : Link speed change : New link speed is 5.0 Gb/s");
            2'b10   : $display ("%m : INFO : Link speed change : New link speed is 8.0 Gb/s");
            default : $display ("%m : INFO : Link speed change : New link speed is Reserved Value");
        endcase
    end
end

always @(posedge clk or negedge pl_link_up)
begin
    if ((pl_link_up === 1'b1) & (neg_link_width != r_neg_link_width))
    begin
        case (neg_link_width)
            6'b000000 : $display ("%m : INFO : Link width change : New link width is undefined");
            6'b000001 : $display ("%m : INFO : Link width change : New link width is x1");
            6'b000010 : $display ("%m : INFO : Link width change : New link width is x2");
            6'b000100 : $display ("%m : INFO : Link width change : New link width is x4");
            6'b001000 : $display ("%m : INFO : Link width change : New link width is x8");
            6'b010000 : $display ("%m : INFO : Link width change : New link width is x16");
            default   : $display ("%m : INFO : Link width change : New link width is Reserved Value");
        endcase
    end

end



// ------------------------------------------------------------
// Assertions to check protocol at user side of PCIe core model

always @(posedge clk)
begin : protocol_check_tx
    if (tx_en == 1'b1 && (^tx_data) === 1'bx)
        $display ("%m : WARNING : x's detected at the transmit interface of PCIe model.  This will likely disrupt the serial link., %t", $time);
end

always @(posedge clk)
begin : protocol_check_rx
    if (rx_en == 1'b1 && (^rx_data) === 1'bx)
        $display ("%m : WARNING : x's detected at the receive interface of PCIe model.  This will likely disrupt the serial link., %t", $time);
end



endmodule
// -------------------------------------------------------------------------
//
//  PROJECT: PCI Express Core
//  COMPANY: Northwest Logic, Inc.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//             (c) Copyright 2009 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or
//  transmitted in any form or by any means, electronic or mechanical,
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module pcie_model (

    rst_n,                  // Asynchronous, active low reset

    core_rst_n,             // Core asynchronous active low reset output; synchrous deassert with core_clk
    core_clk,               // Core clock output; all core I/O synchronous to this clock

    tx_p,                   // PCIe differential transmit output
    tx_n,                   // PCIe differential transmit output

    rx_p,                   // PCIe differential receive output
    rx_n,                   // PCIe differential receive output

    vc0_tx_sop,             // User packet transmit bus
    vc0_tx_eop,             //
    vc0_tx_eop_n,           //
    vc0_tx_en,              //
    vc0_tx_data,            //

    vc0_rx_sel,             // PCI Express Core Local Receive Interface
    vc0_rx_cmd_data,        //
    vc0_rx_err_ecrc,        //
    vc0_rx_sop,             //
    vc0_rx_eop,             //
    vc0_rx_en,              //
    vc0_rx_data,            //

    pl_link_up,             // Set if Physical Link Trained
    dl_link_up,             // Set if Data Link Layer negotiated Credit Values

    mgmt_cfg_id,            //
    mgmt_cfg_constants,     // Constants to control misc behavior
    mgmt_rp_status,         // Status from Root Port functionality
    mgmt_rp_control,
    mgmt_pcie_status        // Status from misc functionality

);



// ----------------
// -- Parameters --
// ----------------

// The following parameter controls the lanes that will be modelled as
//   detecting/not detecting receivers; to cause a lane not to detect
//   a receiver, set (1) the corresponding bit in PIPE_PHY_LANE_MASK;
//     For example: PHY_LANE_MASK == 16'h800C : Lanes[15, 3:2] will
//       emulate being disconnected
//   Note: The serial transmit lines of lanes that are modelled as
//   disconnected will output High-Z
parameter   BFM_PHY_LANE_MASK       = 16'h0000;

parameter   BFM_HALF_PERIOD_5G      = 100;  // 5 GHz
parameter   BFM_HALF_PERIOD_2G5     = 200;  // 2.5 GHz

parameter   SIM_EL_IDLE_TYPE        = 2'b10; // Electrical Idle Emulation: 11 == 1'b1 : Common Mode 1
                                             //                            10 == 1'b0 : Common Mode 0
                                             //                            01 == 1'bx : Undefined
                                             //                            00 == Reserved

parameter   RX_IDLE_ACTIVE_8G_ONLY_EIE = 0; // Set to 1 to cause rx_elec_idle to be 0 at 8G speed only when EIEOS == {8{8'h00, 8'ff}} is received;                                            0 == Use emulated analog comparator
parameter   RX_IDLE_ACTIVE_5G_ONLY_EIE = 0; // Set to 1 to cause rx_elec_idle to be 0 at 5G speed only when EIE == symbols are received back-back; alternating pattern of 5 zeros and 5 ones; 0 == Use emulated analog comparator

// ---------------------------------------------------------------------------
// NOTE: None of the following parameters are available for user's to modify;
//       They are provided for Northwest Logic core maintanence purposes only.
//       Northwest Logic will only support the shipped parameter configuration.
//       The mgmt_cfg_constants amd mgmt_cfg_control ports are provided for
//       users to personalize the core for their application

parameter   BFM_ECRC_GEN_ON         = 1;    // ECRC Generation Enable - Default On when present
parameter   BFM_ECRC_CHECK_ON       = 1;    // ECRC Checking Enable   - Default On when present

localparam  NUM_LANES               = 8;
localparam  NUM_VF                  = 32;

localparam  CORE_DATA_WIDTH         = 64;  // Width of input and output data
localparam  CORE_REMAIN_WIDTH       = 3;   // Width of input and output remainder
localparam  CORE_K_WIDTH            = 8;   // Width of input and output K
localparam  CORE_STS_WIDTH          = 24;  // Width of input and output sts

localparam  LANE_WIDTH              = 1;

// The PCI Express core decodes packets as they are received to determine critical information that is useful for
//   routing and processing the packet; this information is stored in the Ordered Buffer Control Memory so that it
//   is available before the packet must be read; the width of cmd_data fields are set by CMD_DATA_WIDTH
localparam  CMD_DATA_WIDTH          = 13;                 // Local Interface cmd_data width

// Set CLK_PERIOD_IN_NS to the clock period (in nanoseconds) of clk;
//   used to generate counts that must last a fixed time duration
localparam  CLK_PERIOD_IN_NS        = (4 * LANE_WIDTH);
localparam  CLK_PERIOD_IN_NS_5G     = (2 * LANE_WIDTH);
parameter   PCIE_REPLAY_BADDR_WIDTH = 14;   // Use max size Replay buffer in BFM

localparam  FDBK_BITS               = 8;   // Number of bits provided by PHY when reporting equalization quality

// 32 interrupt vectors are implemented by the DMA Back-End
//   and reference design logic; an MSI-X interrupt expansion
//   port is also provided for users to support additional vectors;
//   If more than 256 additional vectors are supported, then
//   the default location of the MSI-X Table and MSI-X PBA cannot
//   be used because the MSI-X Table is too large; in in this case
//   BAR0 size is doubled and different MSI-X Table and MSI-X PBA
//   locations are assigned than default.  See mgmt_msix_table_offset
//   and mgmt_msix_pba_offset below for details.
parameter   INTERRUPT_VECTOR_BITS       = 5;  // Valid values are 5 (32 vectors) to 11 (2048 vectors)
localparam  NUM_INTERRUPT_VECTORS       = (1 << INTERRUPT_VECTOR_BITS);



// ----------------------
// -- Port Definitions --
// ----------------------

input                               rst_n;

output                              core_rst_n;
output                              core_clk;

output  [NUM_LANES-1:0]             tx_p;
output  [NUM_LANES-1:0]             tx_n;

input   [NUM_LANES-1:0]             rx_p;
input   [NUM_LANES-1:0]             rx_n;

input                               vc0_tx_sop;
input                               vc0_tx_eop;
input                               vc0_tx_eop_n;
output                              vc0_tx_en;
input   [CORE_DATA_WIDTH-1:0]       vc0_tx_data;

output  [1:0]                       vc0_rx_sel;
output  [CMD_DATA_WIDTH-1:0]        vc0_rx_cmd_data;
output                              vc0_rx_err_ecrc;
output                              vc0_rx_sop;
output                              vc0_rx_eop;
input                               vc0_rx_en;
output  [CORE_DATA_WIDTH-1:0]       vc0_rx_data;

output                              pl_link_up;
output                              dl_link_up;

output  [15:0]                      mgmt_cfg_id;
input   [1023:0]                    mgmt_cfg_constants;
output  [511:0]                     mgmt_rp_status;
input   [31:0]                      mgmt_rp_control;
output  [1535:0]                    mgmt_pcie_status;



// ----------------
// -- Port Types --
// ----------------

wire                                rst_n;

wire                                core_rst_n;
wire                                core_clk;

wire    [NUM_LANES-1:0]             tx_p;
wire    [NUM_LANES-1:0]             tx_n;

wire    [NUM_LANES-1:0]             rx_p;
wire    [NUM_LANES-1:0]             rx_n;

// PCI Express Core
wire                                vc0_tx_sop;
wire                                vc0_tx_eop;
wire                                vc0_tx_eop_n;
wire                                vc0_tx_en;
wire    [CORE_DATA_WIDTH-1:0]       vc0_tx_data;

wire    [1:0]                       vc0_rx_sel;
wire    [CMD_DATA_WIDTH-1:0]        vc0_rx_cmd_data;
wire                                vc0_rx_err_ecrc;
wire                                vc0_rx_sop;
wire                                vc0_rx_eop;
wire                                vc0_rx_en;
wire    [CORE_DATA_WIDTH-1:0]       vc0_rx_data;

wire                                pl_link_up;
wire                                dl_link_up;

wire    [1023:0]                    mgmt_cfg_constants;
wire    [511:0]                     mgmt_rp_status;
wire    [31:0]                      mgmt_rp_control;
wire    [1535:0]                    mgmt_pcie_status;



// -------------------
// -- Local Signals --
// -------------------

// PCIe Core Link Up Buffering
wire                                mgmt_pl_link_up_o;
wire                                mgmt_dl_link_up_o;
wire                                mgmt_tl_link_up_o;

reg                                 d_mgmt_pl_link_up_i;
reg                                 mgmt_pl_link_up_i;

reg                                 d_mgmt_dl_link_up_i;
reg                                 mgmt_dl_link_up_i;

reg                                 d_mgmt_tl_link_up_i;
reg                                 mgmt_tl_link_up_i;

// PCI Express PIPE PHY w/ PCIe Core Interface
wire    [5:0]                       core_clk_period_in_ns;

wire                                phy_tx_swing;
wire    [2:0]                       phy_tx_margin;
wire    [NUM_LANES-1:0]             phy_tx_deemph;

wire    [NUM_LANES-1:0]             phy_tx_detect_rx_loopback;
wire    [(NUM_LANES*2)-1:0]         phy_power_down;
wire    [NUM_LANES-1:0]             phy_rate;
wire    [NUM_LANES-1:0]             phy_phy_status;
wire    [NUM_LANES-1:0]             phy_rx_polarity;

wire    [CORE_DATA_WIDTH-1:0]       phy_tx_data;
wire    [CORE_K_WIDTH-1:0]          phy_tx_is_k;
wire    [CORE_K_WIDTH-1:0]          phy_tx_fneg;
wire    [NUM_LANES-1:0]             phy_tx_idle;

wire    [CORE_DATA_WIDTH-1:0]       phy_rx_data;
wire    [CORE_K_WIDTH-1:0]          phy_rx_is_k;
wire    [NUM_LANES-1:0]             phy_rx_data_valid;
wire    [NUM_LANES-1:0]             phy_rx_valid;
wire    [CORE_STS_WIDTH-1:0]        phy_rx_status;
wire    [NUM_LANES-1:0]             phy_rx_idle;

// PCIe Engine
wire                                msg_en;
wire    [159:0]                     msg_data;


genvar                              p;
reg                                 vc0_tx_perr;
wire    [(CORE_DATA_WIDTH/8)-1:0]   vc0_tx_datap;
wire    [(CORE_DATA_WIDTH/8)-1:0]   vc0_rx_datap;

wire    [863:0]                     mgmt_cfg_control;
wire    [2047:0]                    mgmt_cfg_status;
wire                                mgmt_interrupt_o;
wire                                mgmt_interrupt;
wire    [31:0]                      mgmt_core_version;
wire    [15:0]                      mgmt_cfg_id;

reg                                 pm_d3cold_n_pme_assert;

// Personalize MSI-X & MSI Capabilities
wire    [2:0]                       msi_multiple_message_capable;
wire    [11:0]                      msix_table_size;
wire    [2:0]                       msix_table_bir;
wire    [31:0]                      mgmt_msix_table_offset;
wire    [2:0]                       msix_pba_bir;
wire    [31:0]                      mgmt_msix_pba_offset;

// PCIe Core : Personalize PCI Express Capability Item
wire    [7:0]                       port_number;
wire    [2:0]                       ep_l1_acceptable_latency;
wire    [2:0]                       l1_exit_latency;

wire    [2:0]                       ep_l0s_acceptable_latency;
wire    [2:0]                       l0s_exit_latency;
wire    [1:0]                       aspm_support;
wire                                aspm_optionality_compliance;

wire    [2:0]                       max_payload_size_supported;
wire                                transactions_pending;

wire    [13:0]                      ctrl_pcie_capabilities;
wire    [31:0]                      ctrl_pcie_device_capabilities;
wire    [15:0]                      ctrl_pcie_device_status;
wire    [31:0]                      ctrl_pcie_link_capabilities;
wire    [15:0]                      ctrl_pcie_link_status;
wire    [31:0]                      ctrl_pcie_slot_capabilities;
wire    [15:0]                      ctrl_pcie_slot_status;
wire    [31:0]                      ctrl_pcie_device_capabilities2;

wire                                aux_power_good;
wire                                aux_power_required;

// PCIe Core : Personalize Power Management Capabilities
wire    [15:0]                      ctrl_pm_capabilities;
wire    [15:0]                      ctrl_pm_control_status;
wire    [7:0]                       ctrl_pm_pmcsr;
wire    [7:0]                       ctrl_pm_data;

// Define PCIe Status Port
reg                                 link_speed_5g_2_5g_n;
reg     [7:0]                       clk_period;
reg     [7:0]                       mgmt_clk_period_in_ns;



// ---------------
// -- Equations --
// ---------------

// ---------------------------
// PCIe Core Link Up Buffering

// mgmt_pl_link_up_i & mgmt_dl_link_up_i are active low asynchronous resets
//   to large portions of the PCIe Core and DMA BE Core.
// mgmt_dl_link_up_i should be the reset to all user logic interacting with the
//   DMA BE or PCIe Core interfaces.
// mgmt_pl_link_up_i & mgmt_dl_link_up_i must be buffered versions of the the
//   PCIe core's mgmt_pl_link_up_o & mgmt_dl_link_up_o outputs.
// For ASIC test, test reset logic should be added to mgmt_pl_link_up_i &
//   mgmt_dl_link_up_i generation logic.
always @(posedge core_clk or negedge core_rst_n)
begin
    if (core_rst_n == 1'b0)
    begin
        d_mgmt_pl_link_up_i <= 1'b0;
        mgmt_pl_link_up_i   <= 1'b0;
    end
    else
    begin
        d_mgmt_pl_link_up_i <=   mgmt_pl_link_up_o;
        mgmt_pl_link_up_i   <= d_mgmt_pl_link_up_i;
    end
end

always @(posedge core_clk or negedge core_rst_n)
begin
    if (core_rst_n == 1'b0)
    begin
        d_mgmt_dl_link_up_i <= 1'b0;
        mgmt_dl_link_up_i   <= 1'b0;
    end
    else
    begin
        d_mgmt_dl_link_up_i <=   mgmt_dl_link_up_o;
        mgmt_dl_link_up_i   <= d_mgmt_dl_link_up_i;
    end
end

always @(posedge core_clk or negedge core_rst_n)
begin
    if (core_rst_n == 1'b0)
    begin
        d_mgmt_tl_link_up_i <= 1'b0;
        mgmt_tl_link_up_i   <= 1'b0;
    end
    else
    begin
        d_mgmt_tl_link_up_i <=   mgmt_tl_link_up_o;
        mgmt_tl_link_up_i   <= d_mgmt_tl_link_up_i;
    end
end

assign pl_link_up = mgmt_pl_link_up_i; // Set when the Physical Layer is up (link is trained)
assign dl_link_up = mgmt_dl_link_up_i; // Set when the Data Link Layer is up (link is trained and inital flow control values are determined)



// -------------------------------------------
// PCI Express PIPE PHY w/ PCIe Core Interface

bfmp_pipe_if #(

    .PIPE_PHY_LANE_MASK             (BFM_PHY_LANE_MASK          ),

    .HALF_PERIOD_5G                 (BFM_HALF_PERIOD_5G         ),
    .HALF_PERIOD_2G5                (BFM_HALF_PERIOD_2G5        ),

    .SIM_EL_IDLE_TYPE               (SIM_EL_IDLE_TYPE           ),
    .RX_IDLE_ACTIVE_8G_ONLY_EIE     (RX_IDLE_ACTIVE_8G_ONLY_EIE ),
    .RX_IDLE_ACTIVE_5G_ONLY_EIE     (RX_IDLE_ACTIVE_5G_ONLY_EIE )


) pipe_if (

    .rst_n                      (rst_n                      ),

    .tx_p                       (tx_p                       ),
    .tx_n                       (tx_n                       ),

    .rx_p                       (rx_p                       ),
    .rx_n                       (rx_n                       ),

    .phy_rst_n                  (core_rst_n                 ),
    .phy_clk                    (core_clk                   ),
    .phy_clk_period_in_ns       (core_clk_period_in_ns      ),

    .phy_tx_swing               (phy_tx_swing               ),
    .phy_tx_margin              (phy_tx_margin              ),
    .phy_tx_deemph              (phy_tx_deemph              ),
    .phy_tx_detect_rx_loopback  (phy_tx_detect_rx_loopback  ),
    .phy_power_down             (phy_power_down             ),
    .phy_rate                   (phy_rate                   ),
    .phy_phy_status             (phy_phy_status             ),
    .phy_rx_polarity            (phy_rx_polarity            ),
    .phy_tx_data                (phy_tx_data                ),
    .phy_tx_data_k              (phy_tx_is_k                ),
    .phy_tx_compliance          (phy_tx_fneg                ),
    .phy_tx_elec_idle           (phy_tx_idle                ),
    .phy_rx_data                (phy_rx_data                ),
    .phy_rx_data_k              (phy_rx_is_k                ),
    .phy_rx_data_valid          (phy_rx_data_valid          ),
    .phy_rx_valid               (phy_rx_valid               ),
    .phy_rx_status              (phy_rx_status              ),
    .phy_rx_elec_idle           (phy_rx_idle                )

);

assign phy_rx_data_valid = {NUM_LANES{1'b1}};

`ifdef SIMULATION
// To accellerate sims, cause BFM rx_idle to go low at start
// of simulation until receiver detection sequence begins
// This will kick us out of the DETECT state quickly on both
// sides of the link

reg fast_rx_idle_init;
always @(posedge core_clk or negedge core_rst_n)
begin
    if (core_rst_n === 1'b0)
        fast_rx_idle_init <= 1'b1;
    else if (|phy_tx_detect_rx_loopback)
        fast_rx_idle_init <= 1'b0;
end

always @(core_rst_n or fast_rx_idle_init)
    if (fast_rx_idle_init == 1'b1)
        force phy_rx_idle = {NUM_LANES{1'b0}};
    else if (fast_rx_idle_init == 1'b0)
        release phy_rx_idle;
`endif

// ---------
// PCIe Core


initial
    vc0_tx_perr = 1'b0;

// Generate data parity for core-generated Tx TLPs
generate for (p=0; p<(CORE_DATA_WIDTH/8); p=p+1)
    begin : gen_vc0_tx_datap
        assign vc0_tx_datap[p] = ^vc0_tx_data[((p+1)*8)-1:(p*8)] ^ vc0_tx_perr;
    end
endgenerate

 `ifdef SIMULATION
// Note checking all bytes of vc0_rx_data even those beyond the TLP end
generate for (p=0; p<(CORE_DATA_WIDTH/8); p=p+1)
    begin : gen_vc0_rx_datap_check
        always @(posedge core_clk)
        begin
            if ((core_rst_n === 1'b1) & vc0_rx_en & (vc0_rx_datap[p] != ^vc0_rx_data[((p+1)*8)-1:(p*8)]))
                $display ("%m : ERROR : Parity Error : vc0_rx_data[%d:%d] (time %t)", ((p+1)*8)-1, (p*8), $time);
        end
    end
endgenerate
 `endif

bfm_pcie_core_vc1 #(
    .PCIE_REPLAY_BADDR_WIDTH    (PCIE_REPLAY_BADDR_WIDTH        )
) pcie_core_vc1 (
    .rst_n                      (core_rst_n                     ),
    .clk                        (core_clk                       ),
    .clk_period_in_ns           (core_clk_period_in_ns          ),
    .flr                        (                               ),
    .flr_ack                    ({NUM_VF{1'b1}}                 ),

    .vc0_tx_sop                 (vc0_tx_sop                     ),
    .vc0_tx_eop                 (vc0_tx_eop                     ),
    .vc0_tx_eop_n               (vc0_tx_eop_n                   ),
    .vc0_tx_en                  (vc0_tx_en                      ),
    .vc0_tx_data                (vc0_tx_data                    ),
    .vc0_tx_datap               (vc0_tx_datap                   ),
    .vc0_tx_np_ok               (                               ),

    .vc0_rx_sel                 (vc0_rx_sel                     ),
    .vc0_rx_cmd_data            (vc0_rx_cmd_data                ),
    .vc0_rx_err_ecrc            (vc0_rx_err_ecrc                ),
    .vc0_rx_sop                 (vc0_rx_sop                     ),
    .vc0_rx_eop                 (vc0_rx_eop                     ),
    .vc0_rx_valid               (                               ),
    .vc0_rx_ready               (vc0_rx_en                      ),
    .vc0_rx_data                (vc0_rx_data                    ),
    .vc0_rx_datap               (vc0_rx_datap                   ),
    .vc0_rx_f                   (                               ),

    .mgmt_pl_link_up_o          (mgmt_pl_link_up_o              ),
    .mgmt_pl_link_up_i          (mgmt_pl_link_up_i              ),
    .mgmt_dl_link_up_o          (mgmt_dl_link_up_o              ),
    .mgmt_dl_link_up_i          (mgmt_dl_link_up_i              ),
    .mgmt_tl_link_up_o          (mgmt_tl_link_up_o              ),
    .mgmt_tl_link_up_i          (mgmt_tl_link_up_i              ),

    .mgmt_cfg_constants         (mgmt_cfg_constants             ),
    .mgmt_cfg_control           (mgmt_cfg_control               ),
    .mgmt_cfg_status            (mgmt_cfg_status                ),
    .mgmt_cfg_estatus           (                               ),
    .mgmt_cfg_status_func_num   (8'h00                          ),
    .mgmt_pcie_status           (mgmt_pcie_status               ),
    .mgmt_core_version          (mgmt_core_version              ),
    .mgmt_cfg_id                (mgmt_cfg_id                    ),
    .mgmt_cfg_slot_control_wr_en(                               ),
    .mgmt_cfg_slot_control      (                               ),
    .mgmt_function_enable       (                               ),

    .mgmt_power_state           (                               ),
    .mgmt_bus_master_en         (                               ),
    .mgmt_mem_en                (                               ),
    .mgmt_io_en                 (                               ),
    .mgmt_int_disable           (                               ),
    .mgmt_max_payload_size      (                               ),
    .mgmt_max_rd_req_size       (                               ),
    .mgmt_en_no_snoop           (                               ),
    .mgmt_en_relaxed_ordering   (                               ),
    .mgmt_crs_sw_vis_en         (                               ),
    .mgmt_ido_req_en            (                               ),
    .mgmt_ido_cpl_en            (                               ),
    .mgmt_en_ecrc_gen           (                               ),
    .mgmt_en_ecrc_check         (                               ),
    .mgmt_msix_function_mask    (                               ),
    .mgmt_msix_en               (                               ),
    .mgmt_msi_en                (                               ),
    .mgmt_msi_mult_msg_en       (                               ),
    .mgmt_msi_addr              (                               ),
    .mgmt_msi_data              (                               ),
    .mgmt_cpl_timeout_disable   (                               ),
    .mgmt_cpl_timeout_value     (                               ),
    .mgmt_interrupt_message_num (                               ),
    .mgmt_msix_table_offset     (                               ),
    .mgmt_msix_pba_offset       (                               ),
    .mgmt_msix_page_size        (                               ),

    .mgmt_interrupt_o           (mgmt_interrupt_o               ),
    .mgmt_interrupt_leg         (mgmt_interrupt                 ),
    .mgmt_interrupt_msix_req    (1'b0                           ),
    .mgmt_interrupt_msix_ack    (                               ),
    .mgmt_interrupt_msix_ack_status (                           ),
    .mgmt_interrupt_msix_vector (128'h0                         ),
    .mgmt_interrupt_msix_function   (8'h0                       ),

    .mgmt_rp_status             (mgmt_rp_status                 ),
    .mgmt_rp_control            (mgmt_rp_control                ),

    .port_rx_active_i           (1'b0                           ), // Switch only
    .port_rx_l1_i               (1'b0                           ), //   ..
    .port_rx_l2_i               (1'b0                           ), //   ..
    .port_rx_l1_exit_i          (1'b0                           ), //   ..
    .port_rx_turn_off_i         (1'b0                           ), //   ..
    .port_rx_to_ack_i           (1'b0                           ), //   ..
    .port_rx_mask_i             (1'b0                           ), //   ..
    .port_rx_active_o           (                               ), //   ..
    .port_rx_l1_o               (                               ), //   ..
    .port_rx_l2_o               (                               ), //   ..
    .port_rx_l1_exit_o          (                               ), //   ..
    .port_rx_turn_off_o         (                               ), //   ..
    .port_rx_to_ack_o           (                               ), //   ..
    .pm_user_tx_pending         (1'b0                           ), // Switch only
    .pm_power_state             (                               ), // Unused: port is for Endpoint applications
    .pm_l1_enter                (                               ),
    .pm_l1_exit                 (                               ),
    .pm_l2_enter                (                               ),
    .pm_l2_enter_ack            (1'b0                           ),
    .pm_l2_exit                 (                               ),
    .pm_l2_store                (                               ),
    .pm_d3cold_exit             (1'b0                           ),
    .pm_d3cold_exit_ack         (                               ),
    .pm_d3cold_restore          (3'h0                           ),
    .pm_d3cold_pme_asserted     (1'b0                           ),
    .pm_d3cold_n_pme_assert     (pm_d3cold_n_pme_assert         ),

    .msg_en                     (msg_en                         ),
    .msg_data                   (msg_data                       ),

    .f_tlps_pending                 ({(NUM_VF){1'b0}}           ),
    .f_err_poisoned_tlp_received    ({(NUM_VF){1'b0}}           ),
    .f_err_completion_timeout       ({(NUM_VF){1'b0}}           ),
    .f_err_completer_abort          ({(NUM_VF){1'b0}}           ),
    .f_err_unexpected_completion    ({(NUM_VF){1'b0}}           ),
    .f_err_unsupported_request      ({(NUM_VF){1'b0}}           ),
    .f_err_ucorr_internal_error     ({(NUM_VF){1'b0}}           ),
    .f_adv_poisoned_tlp_received    ({(NUM_VF){1'b0}}           ),
    .f_adv_completion_timeout       ({(NUM_VF){1'b0}}           ),
    .f_adv_completer_abort          ({(NUM_VF){1'b0}}           ),
    .f_adv_unexpected_completion    ({(NUM_VF){1'b0}}           ),
    .f_adv_unsupported_request      ({(NUM_VF){1'b0}}           ),
    .f_adv_ucorr_internal_error     ({(NUM_VF){1'b0}}           ),
    .f_error_header                 (128'h0                     ),
    .f_error_header_f               ({(NUM_VF){1'b0}}           ),
    .cfg_exp_wr_addr            (                               ),
    .cfg_exp_wr_en              (                               ),
    .cfg_exp_wr_data            (                               ),
    .cfg_exp_wr_be              (                               ),
    .cfg_exp_rd_en              (                               ),
    .cfg_exp_rd_addr            (                               ),
    .cfg_exp_rd_data            (32'h00000000                   ),
    .cfg_exp_f                  (                               ),

    .phy_tx_swing               (phy_tx_swing                   ),
    .phy_tx_margin              (phy_tx_margin                  ),
    .phy_tx_deemph              (phy_tx_deemph                  ),
    .phy_tx_detect_rx_loopback  (phy_tx_detect_rx_loopback      ),
    .phy_power_down             (phy_power_down                 ),
    .phy_rate                   (phy_rate                       ),
    .phy_clk_en                 (1'b1                           ),
    .phy_phy_status             (phy_phy_status                 ),
    .phy_rx_polarity            (phy_rx_polarity                ),
    .phy_tx_data                (phy_tx_data                    ),
    .phy_tx_data_k              (phy_tx_is_k                    ),
    .phy_tx_compliance          (phy_tx_fneg                    ),
    .phy_tx_elec_idle           (phy_tx_idle                    ),
    .phy_rx_data                (phy_rx_data                    ),
    .phy_rx_data_k              (phy_rx_is_k                    ),
    .phy_rx_data_skip           (phy_rx_data_valid              ),
    .phy_rx_valid               (phy_rx_valid                   ),
    .phy_rx_status              (phy_rx_status                  ),
    .phy_rx_elec_idle           (phy_rx_idle                    )

);

// Mo locally generated interrupts
assign mgmt_interrupt     = mgmt_interrupt_o;

// This signal is asserted for one clock to wake up the Root Port from D3hot(L2)
initial pm_d3cold_n_pme_assert = 1'b0;



// ------------------------------------
// Personalize MSI-X & MSI Capabilities

// Multiple Message MSI and MSI-X mgmt_cfg_control ports
assign msi_multiple_message_capable[2:0] = (NUM_INTERRUPT_VECTORS <= 1 ) ? 3'b000 : (
                                           (NUM_INTERRUPT_VECTORS <= 2 ) ? 3'b001 : (
                                           (NUM_INTERRUPT_VECTORS <= 4 ) ? 3'b010 : (
                                           (NUM_INTERRUPT_VECTORS <= 8 ) ? 3'b011 : (
                                           (NUM_INTERRUPT_VECTORS <= 16) ? 3'b100 :
                                                                           3'b101   ))));

assign msix_table_size[11:0]             = (NUM_INTERRUPT_VECTORS - 1); // One less than requested number of vectors

// The PCI Express Complete Core implements an MSI-X arbiter, Table, and PBA
//   in BAR0 register space using the Register Interface; msix_table_bir and
//   msix_pba_bir must be 000 to indicate BAR0.  The MSI-X Table and PBA may
//   be moved by changing the default values below (mgmt_msix_table_offset &
//   mgmt_msix_pba_offset are outputs of this module to the MSI-X arbiter).  If
//   selecting non-default values, care must be taken to place the Table and
//   PBA in their own 4 KByte (8 KB prefferred) page of memory.  The Tables
//   must also be placed on addresses that are aligned to the Table Size.
// Each MSI-X vector requires 16 bytes of MSI-X Table Space.  The default Table
//   and PBA locations below allow for a maximum of 256 vectors.  The default
//   table size in this module is 32 vectors.  A MSI-X table supporting all
//   2048 possible MSI-X vectors requires 32 KBytes.  There is no unoccupied
//   space in the default BAR0 register layout to supports a MSI-X Table larger
//   than 256 vectors.  When more than 256 vectors are selected, the MSI-X Tables
//   are moved to offset +64KB and +96 KB respectively which can support the
//   maximum of 2048 vectors.  Since BAR0 is normally only 64 KBytes, BAR0 size
//   is expanded to 128 Bytes in this case.

assign msix_table_bir[2:0]          = 3'b000;                                        // MSI-X Table located in BAR0
assign mgmt_msix_table_offset[31:0] = (NUM_INTERRUPT_VECTORS > 256) ? 32'h00010000 : //   at offset 0x00010000 (+64 KBytes)
                                                                      32'h0000e000;  //   at offset 0x0000E000 (+56 KBytes)

assign msix_pba_bir[2:0]            = msix_table_bir;                                // MSI-X PBA located in same BAR as MSI-X Table
assign mgmt_msix_pba_offset[31:0]   = (NUM_INTERRUPT_VECTORS > 256) ? 32'h00018000 : //   at offset 0x00018000 (+96 KBytes)
                                                                      32'h0000f000;  //   at offset 0x0000F000 (+60 Kbytes)



// ---------------------------------------
// Personalize PCI Express Capability Item

// Port Number to report in pcie_link_capabilities
assign port_number = 8'h00;

// Endpoint L1 Acceptable Latency - From PCI Express Base Specification, Rev 2.1 section 7.8.3:
//   "This field indicates the acceptable latency that an Endpoint can withstand due to the transition
//   from L1 state to the L0 state. It is essentially an indirect measure of the Endpoint's internal
//   buffering.  Power management software uses the reported L1 Acceptable Latency number to compare
//   against the L1 Exit Latencies reported (see below) by all components comprising the data path
//   from this Endpoint to the Root Complex Root Port to determine whether ASPM L1 entry can be used
//   with no loss of performance."  Note that the amount of buffering does not refer to Expresso 2.0
//   Core buffering, but rather to user application buffering.  Users should set this field in accordance
//   with how long a delay is acceptable for their application.
//     000 - Maximum of 1 us
//     001 - Maximum of 2 us
//     010 - Maximum of 4 us
//     011 - Maximum of 8 us
//     100 - Maximum of 16 us
//     101 - Maximum of 32 us
//     110 - Maximum of 64 us
//     111 - No limit
//   Non-Endpoints must hard wire this field to 000.
assign ep_l1_acceptable_latency = 3'b000;

// L1 Exit Latency - Length of time required to complete transition from L1 to L0:
//     000 - Less than 1us
//     001 - 1 us to less than 2 us
//     010 - 2 us to less than 4 us
//     011 - 4 us to less than 8 us
//     100 - 8 us to less than 16 us
//     101 - 16 us to less than 32 us
//     110 - 32 us-64 us
//     111 - More than 64 us
//  Exit latencies may be significantly increased if the PCI Express reference clocks used by the two devices in the link are common or separate.
assign l1_exit_latency = 3'b111;

// Endpoint L0s Acceptable Latency - From PCI Express Base Specification, Rev 2.1 section 7.8.3:
//   "Acceptable total latency that an Endpoint can withstand due to the transition from L0s state to
//   the L0 state. It is essentially an indirect measure of the Endpoint's internal buffering.
//   Power management software uses the reported L0s Acceptable Latency number to compare against the
//   L0s exit latencies reported by all components comprising the data path from this Endpoint to
//   the Root Complex Root Port to determine whether ASPM L0s entry can be used with no loss of performance."
//   Note that the amount of buffering does not refer to Expresso 2.0 Core buffering, but rather to user
//   application buffering.  Users should set this field in accordance with how long a delay is acceptable
//   for their application.
//     000 - Maximum of 64 ns
//     001 - Maximum of 128 ns
//     010 - Maximum of 256 ns
//     011 - Maximum of 512 ns
//     100 - Maximum of 1 us
//     101 - Maximum of 2 us
//     110 - Maximum of 4 us
//     111 - No limit
//   Non-Endpoints must hard wire this field to 000.
assign ep_l0s_acceptable_latency = 3'b000;

// L0s Exit Latency - Length of time required to complete transition from L0s to L0; varies by PHY and core NFTS setting
//     000 - Less than 64 ns
//     001 - 64 ns to less than 128 ns
//     010 - 128 ns to less than 256 ns
//     011 - 256 ns to less than 512 ns
//     100 - 512 ns to less than 1 us
//     101 - 1 us to less than 2 us
//     110 - 2 us-4 us
//     111 - More than 4 us
// Exit latencies may be significantly increased if the PCI Express reference clocks used by the two devices in the link are common or separate.
assign l0s_exit_latency = 3'b111;

// Active State Power Management (ASPM) Support
//     00 - None
//     01 - Support ASPM L0s only
//     10 - Support ASPM L1  only
//     11 - Support ASPM L0s & L1
assign aspm_support = 2'b11;

// ASPM Optionality Compliance; should be set for all functions (rev 0.9 of spec)
//   This is a new bit defined by ASPM Optionality ECN
assign aspm_optionality_compliance = 1'b1;

// Maximum Payload Size Supported - All PCI Express device's advertise the Maximum Payload Size that they can support.
//   The value advertised is a function of how much buffering has been included in the PCI Express core and how large
//   payloads are supported by the back-end design.  The system configures all devices in the PCI Express hierarchy to
//   the lowest common supported Max Payload Size.  Chipsets typically support 128 or 256 byte Max Payload Sizes.
//   The DMA Back-End supports up to 512 byte payload sizes to support current and future generation chipsets.
//   The Expreso 2.0 Core's default Replay and Receive Buffer credit allocations are designed for 512 byte Maximum
//   Payload Sizes; to support larger Max Payload Sizes, the core's PD and CD buffers must be increased to approximately 4
//   times the advertised supported Max Payload Size.
//     000 -  128 bytes
//     001 -  256 bytes
//     010 -  512 bytes
//     011 - 1024 bytes
//     100 - 2048 bytes
//     101 - 4096 bytes
//     110 - Reserved
//     111 - Reserved
assign max_payload_size_supported = 3'b010; // 512 bytes

// Transactions Pending - Set whenever there are outstanding master requests that have not yet completed;
//   The PCI Express Complete Core hierarchy (Expresso 2.0 Core + DMA Back-End) sets this bit for designs
//   containing a full version of the DMA Back-End.  If the DMA Back-End is used, then user logic must
//   integrate any required requests into the DMA Back-End request logic since requests need to share a
//   single pool of tags and CH/CD receive buffer resources.
assign transactions_pending = 1'b0;

// PCI Express Capabilities Register
assign ctrl_pcie_capabilities[       13: 9] = 5'h0; // Interrupt Message Number used for any interrupts generated by this capability
assign ctrl_pcie_capabilities[           8] = 1'b1; // 1 == Slot Implemented; 0 == Slot not implemented
assign ctrl_pcie_capabilities[        7: 4] = 4'h4; // Device/Port Type == Root Port of PCI Express Root Complex
assign ctrl_pcie_capabilities[        3: 0] = 4'h2; // Capability Version; 4'h2 == PCIe 2.0/3.0

// PCI Express Device Capabilities Register
assign ctrl_pcie_device_capabilities[31:28] = 4'h0;                         // Core: Reserved; tied to 0
assign ctrl_pcie_device_capabilities[27:26] = 2'b00;                        // Core: Captured Slot Power Limit Scale; core implements register
assign ctrl_pcie_device_capabilities[25:18] = 8'b00000000;                  // Core: Captured Slot Power Limit Value; core implements register
assign ctrl_pcie_device_capabilities[17:16] = 2'b00;                        // Core: Reserved; tied to 0
assign ctrl_pcie_device_capabilities[   15] = 1'b0;                         // Core: Role-Based Error Reporting; tied to 1
assign ctrl_pcie_device_capabilities[   14] = 1'b0;                         // User: Was Power Indicator Present; for PCIe 1.1 and greater must be 0
assign ctrl_pcie_device_capabilities[   13] = 1'b0;                         // User: Was Attention Indicator Present; for PCIe 1.1 and greater must be 0
assign ctrl_pcie_device_capabilities[   12] = 1'b0;                         // User: Was Attention Buttton Present; for PCIe 1.1 and greater must be 0
assign ctrl_pcie_device_capabilities[11: 9] = ep_l1_acceptable_latency;     // User: Endpoint L1 Acceptable Latency
assign ctrl_pcie_device_capabilities[ 8: 6] = ep_l0s_acceptable_latency;    // user: Endpoint L0s Acceptable Latency
assign ctrl_pcie_device_capabilities[    5] = 1'b0;                         // Core: Implementing only 5-bit Tags; tied to 0
assign ctrl_pcie_device_capabilities[ 4: 3] = 2'b00;                        // Core: No function # bits used for Phantom Functions (tied to 0)
assign ctrl_pcie_device_capabilities[ 2: 0] = max_payload_size_supported;   // User: Max_Payload_Size_Supported

// PCI Express Device Status Register
assign ctrl_pcie_device_status[      15: 6] = 10'b0000000000;               // Core: Reserved; tied to 0
assign ctrl_pcie_device_status[          5] = transactions_pending;         // User: Transactions Pending
assign ctrl_pcie_device_status[          4] = aux_power_good;               // User: AUX Power Detected
assign ctrl_pcie_device_status[          3] = 1'b0;                         // Core: Unsupported Request Detected; core implements register
assign ctrl_pcie_device_status[          2] = 1'b0;                         // Core: Fatal Error Detected; core implements register
assign ctrl_pcie_device_status[          1] = 1'b0;                         // Core: Non-Fatal Error Detected; core implements register
assign ctrl_pcie_device_status[          0] = 1'b0;                         // Core: Correctable Error Detected; core implements register

// PCI Express Link Capabilities Register
assign ctrl_pcie_link_capabilities[  31:24] = port_number;                  // User: Port Number
assign ctrl_pcie_link_capabilities[     23] = 1'b0;                         // Core: Reserved; tied to 0
assign ctrl_pcie_link_capabilities[     22] = 1'b0;                         // Core: ASPM Optionality Compliance; tied to 1
assign ctrl_pcie_link_capabilities[  21:18] = 4'b0000;                      // Core: Reserved; tied to 0
assign ctrl_pcie_link_capabilities[  17:15] = l1_exit_latency;              // User: L1 Exit Latency
assign ctrl_pcie_link_capabilities[  14:12] = l0s_exit_latency;             // User: L0s Exit Latency
assign ctrl_pcie_link_capabilities[  11:10] = aspm_support;                 // User: ASPM Support
assign ctrl_pcie_link_capabilities[   9: 4] = 6'b000000;                    // Core: Maximum Link Width; tied to core Maximum Link Width
assign ctrl_pcie_link_capabilities[   3: 0] = 4'b0000;                      // Core: Maximum Link Speed; tied to core Maximum Link Speed

// PCI Express Link Status Register
assign ctrl_pcie_link_status[        15:13] = 3'b000;                       // Core: Reserved; tied to 0
assign ctrl_pcie_link_status[           12] = 1'b1;                         // User: Slot Clock == 1; using slot-provided PCI Express Reference clock
assign ctrl_pcie_link_status[        11:10] = 2'b00;                        // Core: Reserved; tied to 0
assign ctrl_pcie_link_status[         9: 4] = 5'b00000;                     // Core: Negotiated Link Width; core implements register
assign ctrl_pcie_link_status[         3 :0] = 4'b0000;                      // Core: Link Speed; core implements register

// PCI Express Slot Capabilities
assign ctrl_pcie_slot_capabilities[  31:19] = 13'h0;                        // physical_slot_number
assign ctrl_pcie_slot_capabilities[     18] = 1'b0;                         // no_command_completed_support
assign ctrl_pcie_slot_capabilities[     17] = 1'b0;                         // electromechanical_interlock_present
assign ctrl_pcie_slot_capabilities[  16:15] = 2'b00;                        // ctrl_slot_power_limit_scale; 00=1.0x, 01=0.1x, 10=0.01x, 11=0.001x
assign ctrl_pcie_slot_capabilities[  14: 7] = 8'd15;                        // ctrl_slot_power_limit_value; SlotPowerLimit=(slot_power_limit_value * ctrl_slot_power_limit_scale multiplier) Watts
assign ctrl_pcie_slot_capabilities[      6] = 1'b0;                         // hot_plug_capable
assign ctrl_pcie_slot_capabilities[      5] = 1'b0;                         // hot_plug_surprise
assign ctrl_pcie_slot_capabilities[      4] = 1'b0;                         // power_indicator_present
assign ctrl_pcie_slot_capabilities[      3] = 1'b0;                         // attention_indicator_present
assign ctrl_pcie_slot_capabilities[      2] = 1'b0;                         // mrl_sensor_present
assign ctrl_pcie_slot_capabilities[      1] = 1'b0;                         // power_controller_present
assign ctrl_pcie_slot_capabilities[      0] = 1'b0;                         // attention_button_present

// PCI Express Slot Status
assign ctrl_pcie_slot_status[        15: 9] = 1'b0;                         // Core: Reserved
assign ctrl_pcie_slot_status[            8] = 1'b0;                         // Core: Data Link Layer State Changed
assign ctrl_pcie_slot_status[            7] = 1'b0;                         // User: Current state of Electromechanical Interlock Sensor; 1 == Engaged; 0 == Disengaged
assign ctrl_pcie_slot_status[            6] = 1'b0;                         // User: Current state of Presence Detect Sensor; 1 == Card Present; 0 == Slot Empty
assign ctrl_pcie_slot_status[            5] = 1'b0;                         // User: Current state of MRL Sensor; 1 == Open; 0 == Closed
assign ctrl_pcie_slot_status[            4] = 1'b0;                         // User: Set for one clock to cause Command Complete to be set
assign ctrl_pcie_slot_status[            3] = 1'b0;                         // Core: Reserved
assign ctrl_pcie_slot_status[            2] = 1'b0;                         // Core: Reserved
assign ctrl_pcie_slot_status[            1] = 1'b0;                         // User: Set for one clock to cause Power Fault Detected to be set
assign ctrl_pcie_slot_status[            0] = 1'b0;                         // User: Set for one clock to cause Attention Button Pressed to be set

// Gen 2 Cores should indicate support for "Completion Timeout Disable"
//   by setting "Completion Timeout Disable Supported"; Note: designs can optionaly
//   set ctrl_pcie_device_capabilities2[3:0] non-zero to indicate that a
//   range of completion timeout values are supported (see PCIe Spec for details);
//   Leaving ctrl_pcie_device_capabilities2[3:0] == 4'h0, indicates that the user design
//   only supports the default Completion Timeout range of 50uS to 50 mS (but strongly
//   encouraged to be 10 mS to 50 mS)
// Note that Gen 2 Cores should disable their completion timeouts when
//   "Completion Timeout Disable" == mgmt_cfg_status[836] == 1'b1.
// Note that Gen 2 Cores should vary their timeout value when
//   "Completion Timeout Value" == mgmt_cfg_status[835:832] != 4'h0 and
//   the design is advertising support for variable timeout
//   ranges via ctrl_pcie_device_capabilities2[3:0] != 4'h0.
assign ctrl_pcie_device_capabilities2 = 32'h00000010;



// -----------------------------------------
// Personalize Power Management Capabilities

assign ctrl_pm_capabilities             = 16'h0603;     // PCI Power Management Interface Specification 1.3; D1 support; D2 support; no PME support; no Aux current
assign ctrl_pm_control_status           = 16'h0008;     // No_Soft_Reset == 1
assign ctrl_pm_pmcsr                    = 8'h00;        //
assign ctrl_pm_data                     = 8'h00;        //

// Aux power not used so tie to 0
assign aux_power_good     = 1'b0;
assign aux_power_required = 1'b0;



// -----------------------------------------------------
// Personalize PCI Express Core through mgmt_cfg_control

assign mgmt_cfg_control[ 31:  0] = 32'h0;
assign mgmt_cfg_control[ 63: 32] = ctrl_pcie_device_capabilities;
assign mgmt_cfg_control[ 79: 64] = ctrl_pcie_device_status;
assign mgmt_cfg_control[111: 80] = ctrl_pcie_link_capabilities;
assign mgmt_cfg_control[127:112] = ctrl_pcie_link_status;
assign mgmt_cfg_control[143:128] = ctrl_pm_capabilities;
assign mgmt_cfg_control[159:144] = ctrl_pm_control_status;
assign mgmt_cfg_control[167:160] = ctrl_pm_pmcsr;
assign mgmt_cfg_control[175:168] = ctrl_pm_data;
assign mgmt_cfg_control[    176] = aux_power_required;
assign mgmt_cfg_control[    177] = 1'b0;    // Reserved
assign mgmt_cfg_control[191:178] = ctrl_pcie_capabilities[13:0];
assign mgmt_cfg_control[223:192] = ctrl_pcie_device_capabilities2;
assign mgmt_cfg_control[245:224] = 22'h0; // Reserved
assign mgmt_cfg_control[    246] = 1'b0;
assign mgmt_cfg_control[255:247] = 9'h0; // Reserved
assign mgmt_cfg_control[383:256] = 128'h0;
assign mgmt_cfg_control[415:384] = 32'h0; // Reserved
assign mgmt_cfg_control[    416] = 1'b0;  // ctrl_aer_version_0x2_enable: 1 == Implement AER to version 0x2 (PCIe 2.1 and later   Specification revisions)
                                          //                              0 == Implement AER to version 0x1 (PCIe 2.0 and earlier Specification revisions)
assign mgmt_cfg_control[430:417] = 14'h0; // Reserved
assign mgmt_cfg_control[    431] = 1'b1;  // Vendor-Specific Extended Capability Enable: 1 == Enable; 0 == Disable
assign mgmt_cfg_control[    432] = 1'b1;  // MSI   Capability Disable: 1 == Disable; 0 == Enable
assign mgmt_cfg_control[    433] = 1'b1;  // MSI-X Capability Disable: 1 == Disable; 0 == Enable
assign mgmt_cfg_control[436:434] = msi_multiple_message_capable[2:0];
assign mgmt_cfg_control[447:437] = msix_table_size[10:0];
assign mgmt_cfg_control[450:448] = msix_table_bir[2:0];
assign mgmt_cfg_control[479:451] = mgmt_msix_table_offset[31:3];
assign mgmt_cfg_control[482:480] = msix_pba_bir[2:0];
assign mgmt_cfg_control[511:483] = mgmt_msix_pba_offset[31:3];
assign mgmt_cfg_control[543:512] = ctrl_pcie_slot_capabilities;
assign mgmt_cfg_control[559:544] = ctrl_pcie_slot_status;
assign mgmt_cfg_control[575:560] = 16'h0001; // Vendor-Specific Extended Capability VSEC_ID
assign mgmt_cfg_control[583:576] = 8'h0;
assign mgmt_cfg_control[589:584] = 6'h0;    // VSEC input bits
assign mgmt_cfg_control[591:590] = 2'h0;    // reserved
assign mgmt_cfg_control[607:592] = 16'h0;   // VSEC ID
assign mgmt_cfg_control[671:608] = 64'h0;   // DSN Serial Number
assign mgmt_cfg_control[863:672] = 192'h0;  // Resize BAR configuration


endmodule
// -------------------------------------------------------------------------
//
//  PROJECT: PCI Express Verification Suite
//  COMPANY: Northwest Logic, Inc.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//             (c) Copyright 2009 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or 
//  transmitted in any form or by any means, electronic or mechanical, 
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//  
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module bfmp_tx_8b_to_10b
    (
     
     rst_n,                                  // Asynchronous reset
     
     serial_clk,
     symbol_clk,
     
     tx_data,                                // 8-bit Data value
     tx_data_k,                              // Set for K characters, clear for others
     tx_compliance,                          // If set, encode this byte assuming disparity is currently negative
     tx_elec_idle,
     
     tx_serial_sdp_start,                    // Indication that current transmit serial bit is start of K_SDP
     tx_serial_stp_start,                    // Indication that current transmit serial bit is start of K_STP

     tx_p,                                   // Transmit serial stream; positive
     tx_n                                    // Transmit serial stream; negative

);  



// ----------------
// -- Parameters --
// ----------------

parameter   SIM_EL_IDLE_TYPE                = 2'b10; // Electrical Idle Emulation: 11 == 1'b1 : Common Mode 1
                                                     //                            10 == 1'b0 : Common Mode 0
                                                     //                            01 == 1'bx : Undefined
                                                     //                            00 == Reserved

parameter   PHY_K_WIDTH                     = 4;

localparam  FIFO_ADDR_WIDTH                 = 6;     // Storing only 50-bits
localparam  FIFO_DEPTH                      = 50;    // Depth must be a multiple of 10

localparam  K_SDP_10N = 10'b00_1111_0101;
localparam  K_SDP_10P = 10'b11_0000_1010;
localparam  K_STP_10N = 10'b11_0110_1000;
localparam  K_STP_10P = 10'b00_1001_0111;

// ----------------------
// -- Port Definitions --
// ----------------------

input                                       rst_n;

input                                       serial_clk;
input                                       symbol_clk;

input   [7:0]                               tx_data;
input                                       tx_data_k;
input                                       tx_compliance;
input                                       tx_elec_idle;

output                                      tx_serial_sdp_start;
output                                      tx_serial_stp_start;

output                                      tx_p;
output                                      tx_n;



// ----------------
// -- Port Types --
// ----------------

wire                                        rst_n;

wire                                        serial_clk;
wire                                        symbol_clk;

wire    [7:0]                               tx_data;
wire                                        tx_data_k;
wire                                        tx_compliance;
wire                                        tx_elec_idle;

wire                                        tx_serial_sdp_start;
wire                                        tx_serial_stp_start;

wire                                        tx_p;
wire                                        tx_n;



// -------------------
// -- Local Signals --
// -------------------

// Encode data 8b -> 10b
reg     [9:0]                               curr_10b_data;
reg     [3:0]                               curr_10b_data_disp;
reg                                         parallel_idle;
reg     [9:0]                               parallel_data10;
reg                                         running_disp;
wire                                        curr_disp;

// Input Clock Domain
reg                                         in_en;
reg     [FIFO_ADDR_WIDTH-1:0]               in_addr;
reg                                         in_half;

genvar                                      i;
wire    [FIFO_ADDR_WIDTH-1:0]               in_a               [9:0];

reg                                         fifo_parallel_stp  [FIFO_DEPTH-1:0];
reg                                         fifo_parallel_sdp  [FIFO_DEPTH-1:0];
reg                                         fifo_parallel_idle [FIFO_DEPTH-1:0];
reg                                         fifo_parallel_data [FIFO_DEPTH-1:0];

// Output Clock Domain
reg     [FIFO_ADDR_WIDTH-1:0]               out_addr;
reg                                         out_half;

reg                                         serial_idle;
reg                                         serial_data;
reg                                         serial_sdp;
reg                                         serial_stp;

reg     [FIFO_ADDR_WIDTH-1:0]               out_in_addr;
reg                                         out_in_half;

reg     [FIFO_ADDR_WIDTH:0]                 out_level;
reg                                         out_en;

wire                                        diff_half;
wire    [FIFO_ADDR_WIDTH:0]                 c_out_level;

wire                                        serial_idle_dly;
wire                                        el_idle;



// ---------------
// -- Equations --
// ---------------

// ---------------------
// Encode data 8b -> 10b

always @*
begin
    case ({curr_disp, tx_data_k, tx_data[7:0]})

        // Current Running Disparity == Negative (curr_disp == 0); Data
        10'h000 : begin curr_10b_data = 10'b100111_0100; end // Curr Disp Neg: D0.0    
        10'h001 : begin curr_10b_data = 10'b011101_0100; end // Curr Disp Neg: D1.0    
        10'h002 : begin curr_10b_data = 10'b101101_0100; end // Curr Disp Neg: D2.0    
        10'h003 : begin curr_10b_data = 10'b110001_1011; end // Curr Disp Neg: D3.0    
        10'h004 : begin curr_10b_data = 10'b110101_0100; end // Curr Disp Neg: D4.0    
        10'h005 : begin curr_10b_data = 10'b101001_1011; end // Curr Disp Neg: D5.0    
        10'h006 : begin curr_10b_data = 10'b011001_1011; end // Curr Disp Neg: D6.0    
        10'h007 : begin curr_10b_data = 10'b111000_1011; end // Curr Disp Neg: D7.0    
        10'h008 : begin curr_10b_data = 10'b111001_0100; end // Curr Disp Neg: D8.0    
        10'h009 : begin curr_10b_data = 10'b100101_1011; end // Curr Disp Neg: D9.0    
        10'h00A : begin curr_10b_data = 10'b010101_1011; end // Curr Disp Neg: D10.0   
        10'h00B : begin curr_10b_data = 10'b110100_1011; end // Curr Disp Neg: D11.0   
        10'h00C : begin curr_10b_data = 10'b001101_1011; end // Curr Disp Neg: D12.0   
        10'h00D : begin curr_10b_data = 10'b101100_1011; end // Curr Disp Neg: D13.0   
        10'h00E : begin curr_10b_data = 10'b011100_1011; end // Curr Disp Neg: D14.0   
        10'h00F : begin curr_10b_data = 10'b010111_0100; end // Curr Disp Neg: D15.0   
        10'h010 : begin curr_10b_data = 10'b011011_0100; end // Curr Disp Neg: D16.0   
        10'h011 : begin curr_10b_data = 10'b100011_1011; end // Curr Disp Neg: D17.0   
        10'h012 : begin curr_10b_data = 10'b010011_1011; end // Curr Disp Neg: D18.0   
        10'h013 : begin curr_10b_data = 10'b110010_1011; end // Curr Disp Neg: D19.0   
        10'h014 : begin curr_10b_data = 10'b001011_1011; end // Curr Disp Neg: D20.0   
        10'h015 : begin curr_10b_data = 10'b101010_1011; end // Curr Disp Neg: D21.0   
        10'h016 : begin curr_10b_data = 10'b011010_1011; end // Curr Disp Neg: D22.0   
        10'h017 : begin curr_10b_data = 10'b111010_0100; end // Curr Disp Neg: D23.0   
        10'h018 : begin curr_10b_data = 10'b110011_0100; end // Curr Disp Neg: D24.0   
        10'h019 : begin curr_10b_data = 10'b100110_1011; end // Curr Disp Neg: D25.0   
        10'h01A : begin curr_10b_data = 10'b010110_1011; end // Curr Disp Neg: D26.0   
        10'h01B : begin curr_10b_data = 10'b110110_0100; end // Curr Disp Neg: D27.0   
        10'h01C : begin curr_10b_data = 10'b001110_1011; end // Curr Disp Neg: D28.0   
        10'h01D : begin curr_10b_data = 10'b101110_0100; end // Curr Disp Neg: D29.0   
        10'h01E : begin curr_10b_data = 10'b011110_0100; end // Curr Disp Neg: D30.0   
        10'h01F : begin curr_10b_data = 10'b101011_0100; end // Curr Disp Neg: D31.0   
        10'h020 : begin curr_10b_data = 10'b100111_1001; end // Curr Disp Neg: D0.1    
        10'h021 : begin curr_10b_data = 10'b011101_1001; end // Curr Disp Neg: D1.1    
        10'h022 : begin curr_10b_data = 10'b101101_1001; end // Curr Disp Neg: D2.1    
        10'h023 : begin curr_10b_data = 10'b110001_1001; end // Curr Disp Neg: D3.1    
        10'h024 : begin curr_10b_data = 10'b110101_1001; end // Curr Disp Neg: D4.1    
        10'h025 : begin curr_10b_data = 10'b101001_1001; end // Curr Disp Neg: D5.1    
        10'h026 : begin curr_10b_data = 10'b011001_1001; end // Curr Disp Neg: D6.1    
        10'h027 : begin curr_10b_data = 10'b111000_1001; end // Curr Disp Neg: D7.1    
        10'h028 : begin curr_10b_data = 10'b111001_1001; end // Curr Disp Neg: D8.1    
        10'h029 : begin curr_10b_data = 10'b100101_1001; end // Curr Disp Neg: D9.1    
        10'h02A : begin curr_10b_data = 10'b010101_1001; end // Curr Disp Neg: D10.1   
        10'h02B : begin curr_10b_data = 10'b110100_1001; end // Curr Disp Neg: D11.1   
        10'h02C : begin curr_10b_data = 10'b001101_1001; end // Curr Disp Neg: D12.1   
        10'h02D : begin curr_10b_data = 10'b101100_1001; end // Curr Disp Neg: D13.1   
        10'h02E : begin curr_10b_data = 10'b011100_1001; end // Curr Disp Neg: D14.1   
        10'h02F : begin curr_10b_data = 10'b010111_1001; end // Curr Disp Neg: D15.1   
        10'h030 : begin curr_10b_data = 10'b011011_1001; end // Curr Disp Neg: D16.1   
        10'h031 : begin curr_10b_data = 10'b100011_1001; end // Curr Disp Neg: D17.1   
        10'h032 : begin curr_10b_data = 10'b010011_1001; end // Curr Disp Neg: D18.1   
        10'h033 : begin curr_10b_data = 10'b110010_1001; end // Curr Disp Neg: D19.1   
        10'h034 : begin curr_10b_data = 10'b001011_1001; end // Curr Disp Neg: D20.1   
        10'h035 : begin curr_10b_data = 10'b101010_1001; end // Curr Disp Neg: D21.1   
        10'h036 : begin curr_10b_data = 10'b011010_1001; end // Curr Disp Neg: D22.1   
        10'h037 : begin curr_10b_data = 10'b111010_1001; end // Curr Disp Neg: D23.1   
        10'h038 : begin curr_10b_data = 10'b110011_1001; end // Curr Disp Neg: D24.1   
        10'h039 : begin curr_10b_data = 10'b100110_1001; end // Curr Disp Neg: D25.1   
        10'h03A : begin curr_10b_data = 10'b010110_1001; end // Curr Disp Neg: D26.1   
        10'h03B : begin curr_10b_data = 10'b110110_1001; end // Curr Disp Neg: D27.1   
        10'h03C : begin curr_10b_data = 10'b001110_1001; end // Curr Disp Neg: D28.1   
        10'h03D : begin curr_10b_data = 10'b101110_1001; end // Curr Disp Neg: D29.1   
        10'h03E : begin curr_10b_data = 10'b011110_1001; end // Curr Disp Neg: D30.1   
        10'h03F : begin curr_10b_data = 10'b101011_1001; end // Curr Disp Neg: D31.1   
        10'h040 : begin curr_10b_data = 10'b100111_0101; end // Curr Disp Neg: D0.2    
        10'h041 : begin curr_10b_data = 10'b011101_0101; end // Curr Disp Neg: D1.2    
        10'h042 : begin curr_10b_data = 10'b101101_0101; end // Curr Disp Neg: D2.2    
        10'h043 : begin curr_10b_data = 10'b110001_0101; end // Curr Disp Neg: D3.2    
        10'h044 : begin curr_10b_data = 10'b110101_0101; end // Curr Disp Neg: D4.2    
        10'h045 : begin curr_10b_data = 10'b101001_0101; end // Curr Disp Neg: D5.2    
        10'h046 : begin curr_10b_data = 10'b011001_0101; end // Curr Disp Neg: D6.2    
        10'h047 : begin curr_10b_data = 10'b111000_0101; end // Curr Disp Neg: D7.2    
        10'h048 : begin curr_10b_data = 10'b111001_0101; end // Curr Disp Neg: D8.2    
        10'h049 : begin curr_10b_data = 10'b100101_0101; end // Curr Disp Neg: D9.2    
        10'h04A : begin curr_10b_data = 10'b010101_0101; end // Curr Disp Neg: D10.2   
        10'h04B : begin curr_10b_data = 10'b110100_0101; end // Curr Disp Neg: D11.2   
        10'h04C : begin curr_10b_data = 10'b001101_0101; end // Curr Disp Neg: D12.2   
        10'h04D : begin curr_10b_data = 10'b101100_0101; end // Curr Disp Neg: D13.2   
        10'h04E : begin curr_10b_data = 10'b011100_0101; end // Curr Disp Neg: D14.2   
        10'h04F : begin curr_10b_data = 10'b010111_0101; end // Curr Disp Neg: D15.2   
        10'h050 : begin curr_10b_data = 10'b011011_0101; end // Curr Disp Neg: D16.2   
        10'h051 : begin curr_10b_data = 10'b100011_0101; end // Curr Disp Neg: D17.2   
        10'h052 : begin curr_10b_data = 10'b010011_0101; end // Curr Disp Neg: D18.2   
        10'h053 : begin curr_10b_data = 10'b110010_0101; end // Curr Disp Neg: D19.2   
        10'h054 : begin curr_10b_data = 10'b001011_0101; end // Curr Disp Neg: D20.2   
        10'h055 : begin curr_10b_data = 10'b101010_0101; end // Curr Disp Neg: D21.2   
        10'h056 : begin curr_10b_data = 10'b011010_0101; end // Curr Disp Neg: D22.2   
        10'h057 : begin curr_10b_data = 10'b111010_0101; end // Curr Disp Neg: D23.2   
        10'h058 : begin curr_10b_data = 10'b110011_0101; end // Curr Disp Neg: D24.2   
        10'h059 : begin curr_10b_data = 10'b100110_0101; end // Curr Disp Neg: D25.2   
        10'h05A : begin curr_10b_data = 10'b010110_0101; end // Curr Disp Neg: D26.2   
        10'h05B : begin curr_10b_data = 10'b110110_0101; end // Curr Disp Neg: D27.2   
        10'h05C : begin curr_10b_data = 10'b001110_0101; end // Curr Disp Neg: D28.2   
        10'h05D : begin curr_10b_data = 10'b101110_0101; end // Curr Disp Neg: D29.2   
        10'h05E : begin curr_10b_data = 10'b011110_0101; end // Curr Disp Neg: D30.2   
        10'h05F : begin curr_10b_data = 10'b101011_0101; end // Curr Disp Neg: D31.2   
        10'h060 : begin curr_10b_data = 10'b100111_0011; end // Curr Disp Neg: D0.3    
        10'h061 : begin curr_10b_data = 10'b011101_0011; end // Curr Disp Neg: D1.3    
        10'h062 : begin curr_10b_data = 10'b101101_0011; end // Curr Disp Neg: D2.3    
        10'h063 : begin curr_10b_data = 10'b110001_1100; end // Curr Disp Neg: D3.3    
        10'h064 : begin curr_10b_data = 10'b110101_0011; end // Curr Disp Neg: D4.3    
        10'h065 : begin curr_10b_data = 10'b101001_1100; end // Curr Disp Neg: D5.3    
        10'h066 : begin curr_10b_data = 10'b011001_1100; end // Curr Disp Neg: D6.3    
        10'h067 : begin curr_10b_data = 10'b111000_1100; end // Curr Disp Neg: D7.3    
        10'h068 : begin curr_10b_data = 10'b111001_0011; end // Curr Disp Neg: D8.3    
        10'h069 : begin curr_10b_data = 10'b100101_1100; end // Curr Disp Neg: D9.3    
        10'h06A : begin curr_10b_data = 10'b010101_1100; end // Curr Disp Neg: D10.3   
        10'h06B : begin curr_10b_data = 10'b110100_1100; end // Curr Disp Neg: D11.3   
        10'h06C : begin curr_10b_data = 10'b001101_1100; end // Curr Disp Neg: D12.3   
        10'h06D : begin curr_10b_data = 10'b101100_1100; end // Curr Disp Neg: D13.3   
        10'h06E : begin curr_10b_data = 10'b011100_1100; end // Curr Disp Neg: D14.3   
        10'h06F : begin curr_10b_data = 10'b010111_0011; end // Curr Disp Neg: D15.3   
        10'h070 : begin curr_10b_data = 10'b011011_0011; end // Curr Disp Neg: D16.3   
        10'h071 : begin curr_10b_data = 10'b100011_1100; end // Curr Disp Neg: D17.3   
        10'h072 : begin curr_10b_data = 10'b010011_1100; end // Curr Disp Neg: D18.3   
        10'h073 : begin curr_10b_data = 10'b110010_1100; end // Curr Disp Neg: D19.3   
        10'h074 : begin curr_10b_data = 10'b001011_1100; end // Curr Disp Neg: D20.3   
        10'h075 : begin curr_10b_data = 10'b101010_1100; end // Curr Disp Neg: D21.3   
        10'h076 : begin curr_10b_data = 10'b011010_1100; end // Curr Disp Neg: D22.3   
        10'h077 : begin curr_10b_data = 10'b111010_0011; end // Curr Disp Neg: D23.3   
        10'h078 : begin curr_10b_data = 10'b110011_0011; end // Curr Disp Neg: D24.3   
        10'h079 : begin curr_10b_data = 10'b100110_1100; end // Curr Disp Neg: D25.3   
        10'h07A : begin curr_10b_data = 10'b010110_1100; end // Curr Disp Neg: D26.3   
        10'h07B : begin curr_10b_data = 10'b110110_0011; end // Curr Disp Neg: D27.3   
        10'h07C : begin curr_10b_data = 10'b001110_1100; end // Curr Disp Neg: D28.3   
        10'h07D : begin curr_10b_data = 10'b101110_0011; end // Curr Disp Neg: D29.3   
        10'h07E : begin curr_10b_data = 10'b011110_0011; end // Curr Disp Neg: D30.3   
        10'h07F : begin curr_10b_data = 10'b101011_0011; end // Curr Disp Neg: D31.3   
        10'h080 : begin curr_10b_data = 10'b100111_0010; end // Curr Disp Neg: D0.4    
        10'h081 : begin curr_10b_data = 10'b011101_0010; end // Curr Disp Neg: D1.4    
        10'h082 : begin curr_10b_data = 10'b101101_0010; end // Curr Disp Neg: D2.4    
        10'h083 : begin curr_10b_data = 10'b110001_1101; end // Curr Disp Neg: D3.4    
        10'h084 : begin curr_10b_data = 10'b110101_0010; end // Curr Disp Neg: D4.4    
        10'h085 : begin curr_10b_data = 10'b101001_1101; end // Curr Disp Neg: D5.4    
        10'h086 : begin curr_10b_data = 10'b011001_1101; end // Curr Disp Neg: D6.4    
        10'h087 : begin curr_10b_data = 10'b111000_1101; end // Curr Disp Neg: D7.4    
        10'h088 : begin curr_10b_data = 10'b111001_0010; end // Curr Disp Neg: D8.4    
        10'h089 : begin curr_10b_data = 10'b100101_1101; end // Curr Disp Neg: D9.4    
        10'h08A : begin curr_10b_data = 10'b010101_1101; end // Curr Disp Neg: D10.4   
        10'h08B : begin curr_10b_data = 10'b110100_1101; end // Curr Disp Neg: D11.4   
        10'h08C : begin curr_10b_data = 10'b001101_1101; end // Curr Disp Neg: D12.4   
        10'h08D : begin curr_10b_data = 10'b101100_1101; end // Curr Disp Neg: D13.4   
        10'h08E : begin curr_10b_data = 10'b011100_1101; end // Curr Disp Neg: D14.4   
        10'h08F : begin curr_10b_data = 10'b010111_0010; end // Curr Disp Neg: D15.4   
        10'h090 : begin curr_10b_data = 10'b011011_0010; end // Curr Disp Neg: D16.4   
        10'h091 : begin curr_10b_data = 10'b100011_1101; end // Curr Disp Neg: D17.4   
        10'h092 : begin curr_10b_data = 10'b010011_1101; end // Curr Disp Neg: D18.4   
        10'h093 : begin curr_10b_data = 10'b110010_1101; end // Curr Disp Neg: D19.4   
        10'h094 : begin curr_10b_data = 10'b001011_1101; end // Curr Disp Neg: D20.4   
        10'h095 : begin curr_10b_data = 10'b101010_1101; end // Curr Disp Neg: D21.4   
        10'h096 : begin curr_10b_data = 10'b011010_1101; end // Curr Disp Neg: D22.4   
        10'h097 : begin curr_10b_data = 10'b111010_0010; end // Curr Disp Neg: D23.4   
        10'h098 : begin curr_10b_data = 10'b110011_0010; end // Curr Disp Neg: D24.4   
        10'h099 : begin curr_10b_data = 10'b100110_1101; end // Curr Disp Neg: D25.4   
        10'h09A : begin curr_10b_data = 10'b010110_1101; end // Curr Disp Neg: D26.4   
        10'h09B : begin curr_10b_data = 10'b110110_0010; end // Curr Disp Neg: D27.4   
        10'h09C : begin curr_10b_data = 10'b001110_1101; end // Curr Disp Neg: D28.4   
        10'h09D : begin curr_10b_data = 10'b101110_0010; end // Curr Disp Neg: D29.4   
        10'h09E : begin curr_10b_data = 10'b011110_0010; end // Curr Disp Neg: D30.4   
        10'h09F : begin curr_10b_data = 10'b101011_0010; end // Curr Disp Neg: D31.4   
        10'h0A0 : begin curr_10b_data = 10'b100111_1010; end // Curr Disp Neg: D0.5    
        10'h0A1 : begin curr_10b_data = 10'b011101_1010; end // Curr Disp Neg: D1.5    
        10'h0A2 : begin curr_10b_data = 10'b101101_1010; end // Curr Disp Neg: D2.5    
        10'h0A3 : begin curr_10b_data = 10'b110001_1010; end // Curr Disp Neg: D3.5    
        10'h0A4 : begin curr_10b_data = 10'b110101_1010; end // Curr Disp Neg: D4.5    
        10'h0A5 : begin curr_10b_data = 10'b101001_1010; end // Curr Disp Neg: D5.5    
        10'h0A6 : begin curr_10b_data = 10'b011001_1010; end // Curr Disp Neg: D6.5    
        10'h0A7 : begin curr_10b_data = 10'b111000_1010; end // Curr Disp Neg: D7.5    
        10'h0A8 : begin curr_10b_data = 10'b111001_1010; end // Curr Disp Neg: D8.5    
        10'h0A9 : begin curr_10b_data = 10'b100101_1010; end // Curr Disp Neg: D9.5    
        10'h0AA : begin curr_10b_data = 10'b010101_1010; end // Curr Disp Neg: D10.5   
        10'h0AB : begin curr_10b_data = 10'b110100_1010; end // Curr Disp Neg: D11.5   
        10'h0AC : begin curr_10b_data = 10'b001101_1010; end // Curr Disp Neg: D12.5   
        10'h0AD : begin curr_10b_data = 10'b101100_1010; end // Curr Disp Neg: D13.5   
        10'h0AE : begin curr_10b_data = 10'b011100_1010; end // Curr Disp Neg: D14.5   
        10'h0AF : begin curr_10b_data = 10'b010111_1010; end // Curr Disp Neg: D15.5   
        10'h0B0 : begin curr_10b_data = 10'b011011_1010; end // Curr Disp Neg: D16.5   
        10'h0B1 : begin curr_10b_data = 10'b100011_1010; end // Curr Disp Neg: D17.5   
        10'h0B2 : begin curr_10b_data = 10'b010011_1010; end // Curr Disp Neg: D18.5   
        10'h0B3 : begin curr_10b_data = 10'b110010_1010; end // Curr Disp Neg: D19.5   
        10'h0B4 : begin curr_10b_data = 10'b001011_1010; end // Curr Disp Neg: D20.5   
        10'h0B5 : begin curr_10b_data = 10'b101010_1010; end // Curr Disp Neg: D21.5   
        10'h0B6 : begin curr_10b_data = 10'b011010_1010; end // Curr Disp Neg: D22.5   
        10'h0B7 : begin curr_10b_data = 10'b111010_1010; end // Curr Disp Neg: D23.5   
        10'h0B8 : begin curr_10b_data = 10'b110011_1010; end // Curr Disp Neg: D24.5   
        10'h0B9 : begin curr_10b_data = 10'b100110_1010; end // Curr Disp Neg: D25.5   
        10'h0BA : begin curr_10b_data = 10'b010110_1010; end // Curr Disp Neg: D26.5   
        10'h0BB : begin curr_10b_data = 10'b110110_1010; end // Curr Disp Neg: D27.5   
        10'h0BC : begin curr_10b_data = 10'b001110_1010; end // Curr Disp Neg: D28.5   
        10'h0BD : begin curr_10b_data = 10'b101110_1010; end // Curr Disp Neg: D29.5   
        10'h0BE : begin curr_10b_data = 10'b011110_1010; end // Curr Disp Neg: D30.5   
        10'h0BF : begin curr_10b_data = 10'b101011_1010; end // Curr Disp Neg: D31.5   
        10'h0C0 : begin curr_10b_data = 10'b100111_0110; end // Curr Disp Neg: D0.6    
        10'h0C1 : begin curr_10b_data = 10'b011101_0110; end // Curr Disp Neg: D1.6    
        10'h0C2 : begin curr_10b_data = 10'b101101_0110; end // Curr Disp Neg: D2.6    
        10'h0C3 : begin curr_10b_data = 10'b110001_0110; end // Curr Disp Neg: D3.6    
        10'h0C4 : begin curr_10b_data = 10'b110101_0110; end // Curr Disp Neg: D4.6    
        10'h0C5 : begin curr_10b_data = 10'b101001_0110; end // Curr Disp Neg: D5.6    
        10'h0C6 : begin curr_10b_data = 10'b011001_0110; end // Curr Disp Neg: D6.6    
        10'h0C7 : begin curr_10b_data = 10'b111000_0110; end // Curr Disp Neg: D7.6    
        10'h0C8 : begin curr_10b_data = 10'b111001_0110; end // Curr Disp Neg: D8.6    
        10'h0C9 : begin curr_10b_data = 10'b100101_0110; end // Curr Disp Neg: D9.6    
        10'h0CA : begin curr_10b_data = 10'b010101_0110; end // Curr Disp Neg: D10.6   
        10'h0CB : begin curr_10b_data = 10'b110100_0110; end // Curr Disp Neg: D11.6   
        10'h0CC : begin curr_10b_data = 10'b001101_0110; end // Curr Disp Neg: D12.6   
        10'h0CD : begin curr_10b_data = 10'b101100_0110; end // Curr Disp Neg: D13.6   
        10'h0CE : begin curr_10b_data = 10'b011100_0110; end // Curr Disp Neg: D14.6   
        10'h0CF : begin curr_10b_data = 10'b010111_0110; end // Curr Disp Neg: D15.6   
        10'h0D0 : begin curr_10b_data = 10'b011011_0110; end // Curr Disp Neg: D16.6   
        10'h0D1 : begin curr_10b_data = 10'b100011_0110; end // Curr Disp Neg: D17.6   
        10'h0D2 : begin curr_10b_data = 10'b010011_0110; end // Curr Disp Neg: D18.6   
        10'h0D3 : begin curr_10b_data = 10'b110010_0110; end // Curr Disp Neg: D19.6   
        10'h0D4 : begin curr_10b_data = 10'b001011_0110; end // Curr Disp Neg: D20.6   
        10'h0D5 : begin curr_10b_data = 10'b101010_0110; end // Curr Disp Neg: D21.6   
        10'h0D6 : begin curr_10b_data = 10'b011010_0110; end // Curr Disp Neg: D22.6   
        10'h0D7 : begin curr_10b_data = 10'b111010_0110; end // Curr Disp Neg: D23.6   
        10'h0D8 : begin curr_10b_data = 10'b110011_0110; end // Curr Disp Neg: D24.6   
        10'h0D9 : begin curr_10b_data = 10'b100110_0110; end // Curr Disp Neg: D25.6   
        10'h0DA : begin curr_10b_data = 10'b010110_0110; end // Curr Disp Neg: D26.6   
        10'h0DB : begin curr_10b_data = 10'b110110_0110; end // Curr Disp Neg: D27.6   
        10'h0DC : begin curr_10b_data = 10'b001110_0110; end // Curr Disp Neg: D28.6   
        10'h0DD : begin curr_10b_data = 10'b101110_0110; end // Curr Disp Neg: D29.6   
        10'h0DE : begin curr_10b_data = 10'b011110_0110; end // Curr Disp Neg: D30.6   
        10'h0DF : begin curr_10b_data = 10'b101011_0110; end // Curr Disp Neg: D31.6   
        10'h0E0 : begin curr_10b_data = 10'b100111_0001; end // Curr Disp Neg: D0.7    
        10'h0E1 : begin curr_10b_data = 10'b011101_0001; end // Curr Disp Neg: D1.7    
        10'h0E2 : begin curr_10b_data = 10'b101101_0001; end // Curr Disp Neg: D2.7    
        10'h0E3 : begin curr_10b_data = 10'b110001_1110; end // Curr Disp Neg: D3.7    
        10'h0E4 : begin curr_10b_data = 10'b110101_0001; end // Curr Disp Neg: D4.7    
        10'h0E5 : begin curr_10b_data = 10'b101001_1110; end // Curr Disp Neg: D5.7    
        10'h0E6 : begin curr_10b_data = 10'b011001_1110; end // Curr Disp Neg: D6.7    
        10'h0E7 : begin curr_10b_data = 10'b111000_1110; end // Curr Disp Neg: D7.7    
        10'h0E8 : begin curr_10b_data = 10'b111001_0001; end // Curr Disp Neg: D8.7    
        10'h0E9 : begin curr_10b_data = 10'b100101_1110; end // Curr Disp Neg: D9.7    
        10'h0EA : begin curr_10b_data = 10'b010101_1110; end // Curr Disp Neg: D10.7   
        10'h0EB : begin curr_10b_data = 10'b110100_1110; end // Curr Disp Neg: D11.7   
        10'h0EC : begin curr_10b_data = 10'b001101_1110; end // Curr Disp Neg: D12.7   
        10'h0ED : begin curr_10b_data = 10'b101100_1110; end // Curr Disp Neg: D13.7   
        10'h0EE : begin curr_10b_data = 10'b011100_1110; end // Curr Disp Neg: D14.7   
        10'h0EF : begin curr_10b_data = 10'b010111_0001; end // Curr Disp Neg: D15.7   
        10'h0F0 : begin curr_10b_data = 10'b011011_0001; end // Curr Disp Neg: D16.7   
        10'h0F1 : begin curr_10b_data = 10'b100011_0111; end // Curr Disp Neg: D17.7   
        10'h0F2 : begin curr_10b_data = 10'b010011_0111; end // Curr Disp Neg: D18.7   
        10'h0F3 : begin curr_10b_data = 10'b110010_1110; end // Curr Disp Neg: D19.7   
        10'h0F4 : begin curr_10b_data = 10'b001011_0111; end // Curr Disp Neg: D20.7   
        10'h0F5 : begin curr_10b_data = 10'b101010_1110; end // Curr Disp Neg: D21.7   
        10'h0F6 : begin curr_10b_data = 10'b011010_1110; end // Curr Disp Neg: D22.7   
        10'h0F7 : begin curr_10b_data = 10'b111010_0001; end // Curr Disp Neg: D23.7   
        10'h0F8 : begin curr_10b_data = 10'b110011_0001; end // Curr Disp Neg: D24.7   
        10'h0F9 : begin curr_10b_data = 10'b100110_1110; end // Curr Disp Neg: D25.7   
        10'h0FA : begin curr_10b_data = 10'b010110_1110; end // Curr Disp Neg: D26.7   
        10'h0FB : begin curr_10b_data = 10'b110110_0001; end // Curr Disp Neg: D27.7   
        10'h0FC : begin curr_10b_data = 10'b001110_1110; end // Curr Disp Neg: D28.7   
        10'h0FD : begin curr_10b_data = 10'b101110_0001; end // Curr Disp Neg: D29.7   
        10'h0FE : begin curr_10b_data = 10'b011110_0001; end // Curr Disp Neg: D30.7   
        10'h0FF : begin curr_10b_data = 10'b101011_0001; end // Curr Disp Neg: D31.7   

        // Current Running Disparity == Negative (curr_disp == 0); K Characters
        10'h11C : begin curr_10b_data = 10'b001111_0100; end // Curr Disp Neg: K28.0   
        10'h13C : begin curr_10b_data = 10'b001111_1001; end // Curr Disp Neg: K28.1   
        10'h15C : begin curr_10b_data = 10'b001111_0101; end // Curr Disp Neg: K28.2   
        10'h17C : begin curr_10b_data = 10'b001111_0011; end // Curr Disp Neg: K28.3   
        10'h19C : begin curr_10b_data = 10'b001111_0010; end // Curr Disp Neg: K28.4   
        10'h1BC : begin curr_10b_data = 10'b001111_1010; end // Curr Disp Neg: K28.5 (COM)
        10'h1DC : begin curr_10b_data = 10'b001111_0110; end // Curr Disp Neg: K28.6   
        10'h1FC : begin curr_10b_data = 10'b001111_1000; end // Curr Disp Neg: K28.7   
        10'h1F7 : begin curr_10b_data = 10'b111010_1000; end // Curr Disp Neg: K23.7   
        10'h1FB : begin curr_10b_data = 10'b110110_1000; end // Curr Disp Neg: K27.7   
        10'h1FD : begin curr_10b_data = 10'b101110_1000; end // Curr Disp Neg: K29.7   
        10'h1FE : begin curr_10b_data = 10'b011110_1000; end // Curr Disp Neg: K30.7   

        // Current Running Disparity == Positive (curr_disp == 1); Data
        10'h200 : begin curr_10b_data = 10'b011000_1011; end // Curr Disp Pos: D0.0    
        10'h201 : begin curr_10b_data = 10'b100010_1011; end // Curr Disp Pos: D1.0    
        10'h202 : begin curr_10b_data = 10'b010010_1011; end // Curr Disp Pos: D2.0    
        10'h203 : begin curr_10b_data = 10'b110001_0100; end // Curr Disp Pos: D3.0    
        10'h204 : begin curr_10b_data = 10'b001010_1011; end // Curr Disp Pos: D4.0    
        10'h205 : begin curr_10b_data = 10'b101001_0100; end // Curr Disp Pos: D5.0    
        10'h206 : begin curr_10b_data = 10'b011001_0100; end // Curr Disp Pos: D6.0    
        10'h207 : begin curr_10b_data = 10'b000111_0100; end // Curr Disp Pos: D7.0    
        10'h208 : begin curr_10b_data = 10'b000110_1011; end // Curr Disp Pos: D8.0    
        10'h209 : begin curr_10b_data = 10'b100101_0100; end // Curr Disp Pos: D9.0    
        10'h20A : begin curr_10b_data = 10'b010101_0100; end // Curr Disp Pos: D10.0   
        10'h20B : begin curr_10b_data = 10'b110100_0100; end // Curr Disp Pos: D11.0   
        10'h20C : begin curr_10b_data = 10'b001101_0100; end // Curr Disp Pos: D12.0   
        10'h20D : begin curr_10b_data = 10'b101100_0100; end // Curr Disp Pos: D13.0   
        10'h20E : begin curr_10b_data = 10'b011100_0100; end // Curr Disp Pos: D14.0   
        10'h20F : begin curr_10b_data = 10'b101000_1011; end // Curr Disp Pos: D15.0   
        10'h210 : begin curr_10b_data = 10'b100100_1011; end // Curr Disp Pos: D16.0   
        10'h211 : begin curr_10b_data = 10'b100011_0100; end // Curr Disp Pos: D17.0   
        10'h212 : begin curr_10b_data = 10'b010011_0100; end // Curr Disp Pos: D18.0   
        10'h213 : begin curr_10b_data = 10'b110010_0100; end // Curr Disp Pos: D19.0   
        10'h214 : begin curr_10b_data = 10'b001011_0100; end // Curr Disp Pos: D20.0   
        10'h215 : begin curr_10b_data = 10'b101010_0100; end // Curr Disp Pos: D21.0   
        10'h216 : begin curr_10b_data = 10'b011010_0100; end // Curr Disp Pos: D22.0   
        10'h217 : begin curr_10b_data = 10'b000101_1011; end // Curr Disp Pos: D23.0   
        10'h218 : begin curr_10b_data = 10'b001100_1011; end // Curr Disp Pos: D24.0   
        10'h219 : begin curr_10b_data = 10'b100110_0100; end // Curr Disp Pos: D25.0   
        10'h21A : begin curr_10b_data = 10'b010110_0100; end // Curr Disp Pos: D26.0   
        10'h21B : begin curr_10b_data = 10'b001001_1011; end // Curr Disp Pos: D27.0   
        10'h21C : begin curr_10b_data = 10'b001110_0100; end // Curr Disp Pos: D28.0   
        10'h21D : begin curr_10b_data = 10'b010001_1011; end // Curr Disp Pos: D29.0   
        10'h21E : begin curr_10b_data = 10'b100001_1011; end // Curr Disp Pos: D30.0   
        10'h21F : begin curr_10b_data = 10'b010100_1011; end // Curr Disp Pos: D31.0   
        10'h220 : begin curr_10b_data = 10'b011000_1001; end // Curr Disp Pos: D0.1    
        10'h221 : begin curr_10b_data = 10'b100010_1001; end // Curr Disp Pos: D1.1    
        10'h222 : begin curr_10b_data = 10'b010010_1001; end // Curr Disp Pos: D2.1    
        10'h223 : begin curr_10b_data = 10'b110001_1001; end // Curr Disp Pos: D3.1    
        10'h224 : begin curr_10b_data = 10'b001010_1001; end // Curr Disp Pos: D4.1    
        10'h225 : begin curr_10b_data = 10'b101001_1001; end // Curr Disp Pos: D5.1    
        10'h226 : begin curr_10b_data = 10'b011001_1001; end // Curr Disp Pos: D6.1    
        10'h227 : begin curr_10b_data = 10'b000111_1001; end // Curr Disp Pos: D7.1    
        10'h228 : begin curr_10b_data = 10'b000110_1001; end // Curr Disp Pos: D8.1    
        10'h229 : begin curr_10b_data = 10'b100101_1001; end // Curr Disp Pos: D9.1    
        10'h22A : begin curr_10b_data = 10'b010101_1001; end // Curr Disp Pos: D10.1   
        10'h22B : begin curr_10b_data = 10'b110100_1001; end // Curr Disp Pos: D11.1   
        10'h22C : begin curr_10b_data = 10'b001101_1001; end // Curr Disp Pos: D12.1   
        10'h22D : begin curr_10b_data = 10'b101100_1001; end // Curr Disp Pos: D13.1   
        10'h22E : begin curr_10b_data = 10'b011100_1001; end // Curr Disp Pos: D14.1   
        10'h22F : begin curr_10b_data = 10'b101000_1001; end // Curr Disp Pos: D15.1   
        10'h230 : begin curr_10b_data = 10'b100100_1001; end // Curr Disp Pos: D16.1   
        10'h231 : begin curr_10b_data = 10'b100011_1001; end // Curr Disp Pos: D17.1   
        10'h232 : begin curr_10b_data = 10'b010011_1001; end // Curr Disp Pos: D18.1   
        10'h233 : begin curr_10b_data = 10'b110010_1001; end // Curr Disp Pos: D19.1   
        10'h234 : begin curr_10b_data = 10'b001011_1001; end // Curr Disp Pos: D20.1   
        10'h235 : begin curr_10b_data = 10'b101010_1001; end // Curr Disp Pos: D21.1   
        10'h236 : begin curr_10b_data = 10'b011010_1001; end // Curr Disp Pos: D22.1   
        10'h237 : begin curr_10b_data = 10'b000101_1001; end // Curr Disp Pos: D23.1   
        10'h238 : begin curr_10b_data = 10'b001100_1001; end // Curr Disp Pos: D24.1   
        10'h239 : begin curr_10b_data = 10'b100110_1001; end // Curr Disp Pos: D25.1   
        10'h23A : begin curr_10b_data = 10'b010110_1001; end // Curr Disp Pos: D26.1   
        10'h23B : begin curr_10b_data = 10'b001001_1001; end // Curr Disp Pos: D27.1   
        10'h23C : begin curr_10b_data = 10'b001110_1001; end // Curr Disp Pos: D28.1   
        10'h23D : begin curr_10b_data = 10'b010001_1001; end // Curr Disp Pos: D29.1   
        10'h23E : begin curr_10b_data = 10'b100001_1001; end // Curr Disp Pos: D30.1   
        10'h23F : begin curr_10b_data = 10'b010100_1001; end // Curr Disp Pos: D31.1   
        10'h240 : begin curr_10b_data = 10'b011000_0101; end // Curr Disp Pos: D0.2    
        10'h241 : begin curr_10b_data = 10'b100010_0101; end // Curr Disp Pos: D1.2    
        10'h242 : begin curr_10b_data = 10'b010010_0101; end // Curr Disp Pos: D2.2    
        10'h243 : begin curr_10b_data = 10'b110001_0101; end // Curr Disp Pos: D3.2    
        10'h244 : begin curr_10b_data = 10'b001010_0101; end // Curr Disp Pos: D4.2    
        10'h245 : begin curr_10b_data = 10'b101001_0101; end // Curr Disp Pos: D5.2    
        10'h246 : begin curr_10b_data = 10'b011001_0101; end // Curr Disp Pos: D6.2    
        10'h247 : begin curr_10b_data = 10'b000111_0101; end // Curr Disp Pos: D7.2    
        10'h248 : begin curr_10b_data = 10'b000110_0101; end // Curr Disp Pos: D8.2    
        10'h249 : begin curr_10b_data = 10'b100101_0101; end // Curr Disp Pos: D9.2    
        10'h24A : begin curr_10b_data = 10'b010101_0101; end // Curr Disp Pos: D10.2   
        10'h24B : begin curr_10b_data = 10'b110100_0101; end // Curr Disp Pos: D11.2   
        10'h24C : begin curr_10b_data = 10'b001101_0101; end // Curr Disp Pos: D12.2   
        10'h24D : begin curr_10b_data = 10'b101100_0101; end // Curr Disp Pos: D13.2   
        10'h24E : begin curr_10b_data = 10'b011100_0101; end // Curr Disp Pos: D14.2   
        10'h24F : begin curr_10b_data = 10'b101000_0101; end // Curr Disp Pos: D15.2   
        10'h250 : begin curr_10b_data = 10'b100100_0101; end // Curr Disp Pos: D16.2   
        10'h251 : begin curr_10b_data = 10'b100011_0101; end // Curr Disp Pos: D17.2   
        10'h252 : begin curr_10b_data = 10'b010011_0101; end // Curr Disp Pos: D18.2   
        10'h253 : begin curr_10b_data = 10'b110010_0101; end // Curr Disp Pos: D19.2   
        10'h254 : begin curr_10b_data = 10'b001011_0101; end // Curr Disp Pos: D20.2   
        10'h255 : begin curr_10b_data = 10'b101010_0101; end // Curr Disp Pos: D21.2   
        10'h256 : begin curr_10b_data = 10'b011010_0101; end // Curr Disp Pos: D22.2   
        10'h257 : begin curr_10b_data = 10'b000101_0101; end // Curr Disp Pos: D23.2   
        10'h258 : begin curr_10b_data = 10'b001100_0101; end // Curr Disp Pos: D24.2   
        10'h259 : begin curr_10b_data = 10'b100110_0101; end // Curr Disp Pos: D25.2   
        10'h25A : begin curr_10b_data = 10'b010110_0101; end // Curr Disp Pos: D26.2   
        10'h25B : begin curr_10b_data = 10'b001001_0101; end // Curr Disp Pos: D27.2   
        10'h25C : begin curr_10b_data = 10'b001110_0101; end // Curr Disp Pos: D28.2   
        10'h25D : begin curr_10b_data = 10'b010001_0101; end // Curr Disp Pos: D29.2   
        10'h25E : begin curr_10b_data = 10'b100001_0101; end // Curr Disp Pos: D30.2   
        10'h25F : begin curr_10b_data = 10'b010100_0101; end // Curr Disp Pos: D31.2   
        10'h260 : begin curr_10b_data = 10'b011000_1100; end // Curr Disp Pos: D0.3    
        10'h261 : begin curr_10b_data = 10'b100010_1100; end // Curr Disp Pos: D1.3    
        10'h262 : begin curr_10b_data = 10'b010010_1100; end // Curr Disp Pos: D2.3    
        10'h263 : begin curr_10b_data = 10'b110001_0011; end // Curr Disp Pos: D3.3    
        10'h264 : begin curr_10b_data = 10'b001010_1100; end // Curr Disp Pos: D4.3    
        10'h265 : begin curr_10b_data = 10'b101001_0011; end // Curr Disp Pos: D5.3    
        10'h266 : begin curr_10b_data = 10'b011001_0011; end // Curr Disp Pos: D6.3    
        10'h267 : begin curr_10b_data = 10'b000111_0011; end // Curr Disp Pos: D7.3    
        10'h268 : begin curr_10b_data = 10'b000110_1100; end // Curr Disp Pos: D8.3    
        10'h269 : begin curr_10b_data = 10'b100101_0011; end // Curr Disp Pos: D9.3    
        10'h26A : begin curr_10b_data = 10'b010101_0011; end // Curr Disp Pos: D10.3   
        10'h26B : begin curr_10b_data = 10'b110100_0011; end // Curr Disp Pos: D11.3   
        10'h26C : begin curr_10b_data = 10'b001101_0011; end // Curr Disp Pos: D12.3   
        10'h26D : begin curr_10b_data = 10'b101100_0011; end // Curr Disp Pos: D13.3   
        10'h26E : begin curr_10b_data = 10'b011100_0011; end // Curr Disp Pos: D14.3   
        10'h26F : begin curr_10b_data = 10'b101000_1100; end // Curr Disp Pos: D15.3   
        10'h270 : begin curr_10b_data = 10'b100100_1100; end // Curr Disp Pos: D16.3   
        10'h271 : begin curr_10b_data = 10'b100011_0011; end // Curr Disp Pos: D17.3   
        10'h272 : begin curr_10b_data = 10'b010011_0011; end // Curr Disp Pos: D18.3   
        10'h273 : begin curr_10b_data = 10'b110010_0011; end // Curr Disp Pos: D19.3   
        10'h274 : begin curr_10b_data = 10'b001011_0011; end // Curr Disp Pos: D20.3   
        10'h275 : begin curr_10b_data = 10'b101010_0011; end // Curr Disp Pos: D21.3   
        10'h276 : begin curr_10b_data = 10'b011010_0011; end // Curr Disp Pos: D22.3   
        10'h277 : begin curr_10b_data = 10'b000101_1100; end // Curr Disp Pos: D23.3   
        10'h278 : begin curr_10b_data = 10'b001100_1100; end // Curr Disp Pos: D24.3   
        10'h279 : begin curr_10b_data = 10'b100110_0011; end // Curr Disp Pos: D25.3   
        10'h27A : begin curr_10b_data = 10'b010110_0011; end // Curr Disp Pos: D26.3   
        10'h27B : begin curr_10b_data = 10'b001001_1100; end // Curr Disp Pos: D27.3   
        10'h27C : begin curr_10b_data = 10'b001110_0011; end // Curr Disp Pos: D28.3   
        10'h27D : begin curr_10b_data = 10'b010001_1100; end // Curr Disp Pos: D29.3   
        10'h27E : begin curr_10b_data = 10'b100001_1100; end // Curr Disp Pos: D30.3   
        10'h27F : begin curr_10b_data = 10'b010100_1100; end // Curr Disp Pos: D31.3   
        10'h280 : begin curr_10b_data = 10'b011000_1101; end // Curr Disp Pos: D0.4    
        10'h281 : begin curr_10b_data = 10'b100010_1101; end // Curr Disp Pos: D1.4    
        10'h282 : begin curr_10b_data = 10'b010010_1101; end // Curr Disp Pos: D2.4    
        10'h283 : begin curr_10b_data = 10'b110001_0010; end // Curr Disp Pos: D3.4    
        10'h284 : begin curr_10b_data = 10'b001010_1101; end // Curr Disp Pos: D4.4    
        10'h285 : begin curr_10b_data = 10'b101001_0010; end // Curr Disp Pos: D5.4    
        10'h286 : begin curr_10b_data = 10'b011001_0010; end // Curr Disp Pos: D6.4    
        10'h287 : begin curr_10b_data = 10'b000111_0010; end // Curr Disp Pos: D7.4    
        10'h288 : begin curr_10b_data = 10'b000110_1101; end // Curr Disp Pos: D8.4    
        10'h289 : begin curr_10b_data = 10'b100101_0010; end // Curr Disp Pos: D9.4    
        10'h28A : begin curr_10b_data = 10'b010101_0010; end // Curr Disp Pos: D10.4   
        10'h28B : begin curr_10b_data = 10'b110100_0010; end // Curr Disp Pos: D11.4   
        10'h28C : begin curr_10b_data = 10'b001101_0010; end // Curr Disp Pos: D12.4   
        10'h28D : begin curr_10b_data = 10'b101100_0010; end // Curr Disp Pos: D13.4   
        10'h28E : begin curr_10b_data = 10'b011100_0010; end // Curr Disp Pos: D14.4   
        10'h28F : begin curr_10b_data = 10'b101000_1101; end // Curr Disp Pos: D15.4   
        10'h290 : begin curr_10b_data = 10'b100100_1101; end // Curr Disp Pos: D16.4   
        10'h291 : begin curr_10b_data = 10'b100011_0010; end // Curr Disp Pos: D17.4   
        10'h292 : begin curr_10b_data = 10'b010011_0010; end // Curr Disp Pos: D18.4   
        10'h293 : begin curr_10b_data = 10'b110010_0010; end // Curr Disp Pos: D19.4   
        10'h294 : begin curr_10b_data = 10'b001011_0010; end // Curr Disp Pos: D20.4   
        10'h295 : begin curr_10b_data = 10'b101010_0010; end // Curr Disp Pos: D21.4   
        10'h296 : begin curr_10b_data = 10'b011010_0010; end // Curr Disp Pos: D22.4   
        10'h297 : begin curr_10b_data = 10'b000101_1101; end // Curr Disp Pos: D23.4   
        10'h298 : begin curr_10b_data = 10'b001100_1101; end // Curr Disp Pos: D24.4   
        10'h299 : begin curr_10b_data = 10'b100110_0010; end // Curr Disp Pos: D25.4   
        10'h29A : begin curr_10b_data = 10'b010110_0010; end // Curr Disp Pos: D26.4   
        10'h29B : begin curr_10b_data = 10'b001001_1101; end // Curr Disp Pos: D27.4   
        10'h29C : begin curr_10b_data = 10'b001110_0010; end // Curr Disp Pos: D28.4   
        10'h29D : begin curr_10b_data = 10'b010001_1101; end // Curr Disp Pos: D29.4   
        10'h29E : begin curr_10b_data = 10'b100001_1101; end // Curr Disp Pos: D30.4   
        10'h29F : begin curr_10b_data = 10'b010100_1101; end // Curr Disp Pos: D31.4   
        10'h2A0 : begin curr_10b_data = 10'b011000_1010; end // Curr Disp Pos: D0.5    
        10'h2A1 : begin curr_10b_data = 10'b100010_1010; end // Curr Disp Pos: D1.5    
        10'h2A2 : begin curr_10b_data = 10'b010010_1010; end // Curr Disp Pos: D2.5    
        10'h2A3 : begin curr_10b_data = 10'b110001_1010; end // Curr Disp Pos: D3.5    
        10'h2A4 : begin curr_10b_data = 10'b001010_1010; end // Curr Disp Pos: D4.5    
        10'h2A5 : begin curr_10b_data = 10'b101001_1010; end // Curr Disp Pos: D5.5    
        10'h2A6 : begin curr_10b_data = 10'b011001_1010; end // Curr Disp Pos: D6.5    
        10'h2A7 : begin curr_10b_data = 10'b000111_1010; end // Curr Disp Pos: D7.5    
        10'h2A8 : begin curr_10b_data = 10'b000110_1010; end // Curr Disp Pos: D8.5    
        10'h2A9 : begin curr_10b_data = 10'b100101_1010; end // Curr Disp Pos: D9.5    
        10'h2AA : begin curr_10b_data = 10'b010101_1010; end // Curr Disp Pos: D10.5   
        10'h2AB : begin curr_10b_data = 10'b110100_1010; end // Curr Disp Pos: D11.5   
        10'h2AC : begin curr_10b_data = 10'b001101_1010; end // Curr Disp Pos: D12.5   
        10'h2AD : begin curr_10b_data = 10'b101100_1010; end // Curr Disp Pos: D13.5   
        10'h2AE : begin curr_10b_data = 10'b011100_1010; end // Curr Disp Pos: D14.5   
        10'h2AF : begin curr_10b_data = 10'b101000_1010; end // Curr Disp Pos: D15.5   
        10'h2B0 : begin curr_10b_data = 10'b100100_1010; end // Curr Disp Pos: D16.5   
        10'h2B1 : begin curr_10b_data = 10'b100011_1010; end // Curr Disp Pos: D17.5   
        10'h2B2 : begin curr_10b_data = 10'b010011_1010; end // Curr Disp Pos: D18.5   
        10'h2B3 : begin curr_10b_data = 10'b110010_1010; end // Curr Disp Pos: D19.5   
        10'h2B4 : begin curr_10b_data = 10'b001011_1010; end // Curr Disp Pos: D20.5   
        10'h2B5 : begin curr_10b_data = 10'b101010_1010; end // Curr Disp Pos: D21.5   
        10'h2B6 : begin curr_10b_data = 10'b011010_1010; end // Curr Disp Pos: D22.5   
        10'h2B7 : begin curr_10b_data = 10'b000101_1010; end // Curr Disp Pos: D23.5   
        10'h2B8 : begin curr_10b_data = 10'b001100_1010; end // Curr Disp Pos: D24.5   
        10'h2B9 : begin curr_10b_data = 10'b100110_1010; end // Curr Disp Pos: D25.5   
        10'h2BA : begin curr_10b_data = 10'b010110_1010; end // Curr Disp Pos: D26.5   
        10'h2BB : begin curr_10b_data = 10'b001001_1010; end // Curr Disp Pos: D27.5   
        10'h2BC : begin curr_10b_data = 10'b001110_1010; end // Curr Disp Pos: D28.5   
        10'h2BD : begin curr_10b_data = 10'b010001_1010; end // Curr Disp Pos: D29.5   
        10'h2BE : begin curr_10b_data = 10'b100001_1010; end // Curr Disp Pos: D30.5   
        10'h2BF : begin curr_10b_data = 10'b010100_1010; end // Curr Disp Pos: D31.5   
        10'h2C0 : begin curr_10b_data = 10'b011000_0110; end // Curr Disp Pos: D0.6    
        10'h2C1 : begin curr_10b_data = 10'b100010_0110; end // Curr Disp Pos: D1.6    
        10'h2C2 : begin curr_10b_data = 10'b010010_0110; end // Curr Disp Pos: D2.6    
        10'h2C3 : begin curr_10b_data = 10'b110001_0110; end // Curr Disp Pos: D3.6    
        10'h2C4 : begin curr_10b_data = 10'b001010_0110; end // Curr Disp Pos: D4.6    
        10'h2C5 : begin curr_10b_data = 10'b101001_0110; end // Curr Disp Pos: D5.6    
        10'h2C6 : begin curr_10b_data = 10'b011001_0110; end // Curr Disp Pos: D6.6    
        10'h2C7 : begin curr_10b_data = 10'b000111_0110; end // Curr Disp Pos: D7.6    
        10'h2C8 : begin curr_10b_data = 10'b000110_0110; end // Curr Disp Pos: D8.6    
        10'h2C9 : begin curr_10b_data = 10'b100101_0110; end // Curr Disp Pos: D9.6    
        10'h2CA : begin curr_10b_data = 10'b010101_0110; end // Curr Disp Pos: D10.6   
        10'h2CB : begin curr_10b_data = 10'b110100_0110; end // Curr Disp Pos: D11.6   
        10'h2CC : begin curr_10b_data = 10'b001101_0110; end // Curr Disp Pos: D12.6   
        10'h2CD : begin curr_10b_data = 10'b101100_0110; end // Curr Disp Pos: D13.6   
        10'h2CE : begin curr_10b_data = 10'b011100_0110; end // Curr Disp Pos: D14.6   
        10'h2CF : begin curr_10b_data = 10'b101000_0110; end // Curr Disp Pos: D15.6   
        10'h2D0 : begin curr_10b_data = 10'b100100_0110; end // Curr Disp Pos: D16.6   
        10'h2D1 : begin curr_10b_data = 10'b100011_0110; end // Curr Disp Pos: D17.6   
        10'h2D2 : begin curr_10b_data = 10'b010011_0110; end // Curr Disp Pos: D18.6   
        10'h2D3 : begin curr_10b_data = 10'b110010_0110; end // Curr Disp Pos: D19.6   
        10'h2D4 : begin curr_10b_data = 10'b001011_0110; end // Curr Disp Pos: D20.6   
        10'h2D5 : begin curr_10b_data = 10'b101010_0110; end // Curr Disp Pos: D21.6   
        10'h2D6 : begin curr_10b_data = 10'b011010_0110; end // Curr Disp Pos: D22.6   
        10'h2D7 : begin curr_10b_data = 10'b000101_0110; end // Curr Disp Pos: D23.6   
        10'h2D8 : begin curr_10b_data = 10'b001100_0110; end // Curr Disp Pos: D24.6   
        10'h2D9 : begin curr_10b_data = 10'b100110_0110; end // Curr Disp Pos: D25.6   
        10'h2DA : begin curr_10b_data = 10'b010110_0110; end // Curr Disp Pos: D26.6   
        10'h2DB : begin curr_10b_data = 10'b001001_0110; end // Curr Disp Pos: D27.6   
        10'h2DC : begin curr_10b_data = 10'b001110_0110; end // Curr Disp Pos: D28.6   
        10'h2DD : begin curr_10b_data = 10'b010001_0110; end // Curr Disp Pos: D29.6   
        10'h2DE : begin curr_10b_data = 10'b100001_0110; end // Curr Disp Pos: D30.6   
        10'h2DF : begin curr_10b_data = 10'b010100_0110; end // Curr Disp Pos: D31.6   
        10'h2E0 : begin curr_10b_data = 10'b011000_1110; end // Curr Disp Pos: D0.7    
        10'h2E1 : begin curr_10b_data = 10'b100010_1110; end // Curr Disp Pos: D1.7    
        10'h2E2 : begin curr_10b_data = 10'b010010_1110; end // Curr Disp Pos: D2.7    
        10'h2E3 : begin curr_10b_data = 10'b110001_0001; end // Curr Disp Pos: D3.7    
        10'h2E4 : begin curr_10b_data = 10'b001010_1110; end // Curr Disp Pos: D4.7    
        10'h2E5 : begin curr_10b_data = 10'b101001_0001; end // Curr Disp Pos: D5.7    
        10'h2E6 : begin curr_10b_data = 10'b011001_0001; end // Curr Disp Pos: D6.7    
        10'h2E7 : begin curr_10b_data = 10'b000111_0001; end // Curr Disp Pos: D7.7    
        10'h2E8 : begin curr_10b_data = 10'b000110_1110; end // Curr Disp Pos: D8.7    
        10'h2E9 : begin curr_10b_data = 10'b100101_0001; end // Curr Disp Pos: D9.7    
        10'h2EA : begin curr_10b_data = 10'b010101_0001; end // Curr Disp Pos: D10.7   
        10'h2EB : begin curr_10b_data = 10'b110100_1000; end // Curr Disp Pos: D11.7   
        10'h2EC : begin curr_10b_data = 10'b001101_0001; end // Curr Disp Pos: D12.7   
        10'h2ED : begin curr_10b_data = 10'b101100_1000; end // Curr Disp Pos: D13.7   
        10'h2EE : begin curr_10b_data = 10'b011100_1000; end // Curr Disp Pos: D14.7   
        10'h2EF : begin curr_10b_data = 10'b101000_1110; end // Curr Disp Pos: D15.7   
        10'h2F0 : begin curr_10b_data = 10'b100100_1110; end // Curr Disp Pos: D16.7   
        10'h2F1 : begin curr_10b_data = 10'b100011_0001; end // Curr Disp Pos: D17.7   
        10'h2F2 : begin curr_10b_data = 10'b010011_0001; end // Curr Disp Pos: D18.7   
        10'h2F3 : begin curr_10b_data = 10'b110010_0001; end // Curr Disp Pos: D19.7   
        10'h2F4 : begin curr_10b_data = 10'b001011_0001; end // Curr Disp Pos: D20.7   
        10'h2F5 : begin curr_10b_data = 10'b101010_0001; end // Curr Disp Pos: D21.7   
        10'h2F6 : begin curr_10b_data = 10'b011010_0001; end // Curr Disp Pos: D22.7   
        10'h2F7 : begin curr_10b_data = 10'b000101_1110; end // Curr Disp Pos: D23.7   
        10'h2F8 : begin curr_10b_data = 10'b001100_1110; end // Curr Disp Pos: D24.7   
        10'h2F9 : begin curr_10b_data = 10'b100110_0001; end // Curr Disp Pos: D25.7   
        10'h2FA : begin curr_10b_data = 10'b010110_0001; end // Curr Disp Pos: D26.7   
        10'h2FB : begin curr_10b_data = 10'b001001_1110; end // Curr Disp Pos: D27.7   
        10'h2FC : begin curr_10b_data = 10'b001110_0001; end // Curr Disp Pos: D28.7   
        10'h2FD : begin curr_10b_data = 10'b010001_1110; end // Curr Disp Pos: D29.7   
        10'h2FE : begin curr_10b_data = 10'b100001_1110; end // Curr Disp Pos: D30.7   
        10'h2FF : begin curr_10b_data = 10'b010100_1110; end // Curr Disp Pos: D31.7   

        // Current Running Disparity == Positive (curr_disp == 1); K Characters
        10'h31C : begin curr_10b_data = 10'b110000_1011; end // Curr Disp Pos: K28.0
        10'h33C : begin curr_10b_data = 10'b110000_0110; end // Curr Disp Pos: K28.1
        10'h35C : begin curr_10b_data = 10'b110000_1010; end // Curr Disp Pos: K28.2
        10'h37C : begin curr_10b_data = 10'b110000_1100; end // Curr Disp Pos: K28.3
        10'h39C : begin curr_10b_data = 10'b110000_1101; end // Curr Disp Pos: K28.4
        10'h3BC : begin curr_10b_data = 10'b110000_0101; end // Curr Disp Pos: K28.5 (COM)
        10'h3DC : begin curr_10b_data = 10'b110000_1001; end // Curr Disp Pos: K28.6
        10'h3FC : begin curr_10b_data = 10'b110000_0111; end // Curr Disp Pos: K28.7
        10'h3F7 : begin curr_10b_data = 10'b000101_0111; end // Curr Disp Pos: K23.7
        10'h3FB : begin curr_10b_data = 10'b001001_0111; end // Curr Disp Pos: K27.7
        10'h3FD : begin curr_10b_data = 10'b010001_0111; end // Curr Disp Pos: K29.7
        10'h3FE : begin curr_10b_data = 10'b100001_0111; end // Curr Disp Pos: K30.7

        // Not in either Current Running Disparity Table
        default : begin curr_10b_data = 10'b011000_1011; end // Not in table; output D0.0 (Positive)

    endcase
end

// Get disparity of the current word by summing the number of 1's in the vector
always @*
begin
    curr_10b_data_disp = curr_10b_data[0] +
                         curr_10b_data[1] + 
                         curr_10b_data[2] +
                         curr_10b_data[3] +
                         curr_10b_data[4] + 
                         curr_10b_data[5] +
                         curr_10b_data[6] +
                         curr_10b_data[7] +
                         curr_10b_data[8] +
                         curr_10b_data[9];
end    

always @(posedge symbol_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        parallel_idle    <= 1'b1;
        parallel_data10  <= 10'b0;
        running_disp     <= 1'b0; // Initialize to negative
    end
    else
    begin
        parallel_idle    <= tx_elec_idle;
        parallel_data10  <= curr_10b_data;

        if (curr_10b_data_disp < 4'd5)
            running_disp <= 1'b0;
        else if (curr_10b_data_disp > 4'd5)
            running_disp <= 1'b1;
    end
end

// If tx_compliance is set, then force negative current disparity
assign curr_disp = tx_compliance ? 1'b0 : running_disp;



// ------------------
// Input Clock Domain

always @(posedge symbol_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        in_en   <= 1'b0;
        in_addr <= {FIFO_ADDR_WIDTH{1'b0}};
        in_half <= 1'b0;
    end
    else
    begin
        in_en <= 1'b1;

        // Keeping one extra address bit to tell full from empty when doing an address compare
        if (in_en)
        begin
            if (in_addr >= (FIFO_DEPTH - 10))
                in_addr <= {FIFO_ADDR_WIDTH{1'b0}};
            else
                in_addr <= in_addr + {{FIFO_ADDR_WIDTH-4{1'b0}}, 4'd10};

            if (in_addr >= (FIFO_DEPTH - 10))
                in_half <= ~in_half;
        end
    end
end

generate
    for (i=0; i<10; i=i+1)
    begin : fifo
        // Drop the extra address bit when accessing the FIFO
        assign in_a[i] = in_addr[FIFO_ADDR_WIDTH-1:0] + i;

        // Write input data into the FIFO
        always @(posedge symbol_clk)
        begin
            if (in_en)
            begin
                fifo_parallel_stp[in_a[i]]  <= (i == 0) & ((parallel_data10 == K_STP_10N) | (parallel_data10 == K_STP_10P));
                fifo_parallel_sdp[in_a[i]]  <= (i == 0) & ((parallel_data10 == K_SDP_10N) | (parallel_data10 == K_SDP_10P));
                fifo_parallel_idle[in_a[i]] <= parallel_idle;
                fifo_parallel_data[in_a[i]] <= parallel_data10[9-i];
            end
        end
    end
endgenerate



// -------------------
// Output Clock Domain

// Manage the output address pointer
always @(posedge serial_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        out_addr         <= {FIFO_ADDR_WIDTH{1'b0}};
        out_half         <= 1'b0;

        serial_idle      <= 1'b1;
        serial_data      <= 1'b0;

        out_in_addr      <= {FIFO_ADDR_WIDTH{1'b0}};
        out_in_half      <= 1'b0;

        out_level        <= {(FIFO_ADDR_WIDTH+1){1'b0}};
        out_en           <= 1'b0;
    end
    else             
    begin
        // Keeping one extra address bit to tell full from empty when doing an address compare
        if (out_en)
        begin
            if (out_addr == (FIFO_DEPTH-1))
                out_addr <= {FIFO_ADDR_WIDTH{1'b0}};
            else
                out_addr <= out_addr + {{(FIFO_ADDR_WIDTH-1){1'b0}}, 1'b1};

            if (out_addr == (FIFO_DEPTH-1))
                out_half <= ~out_half;
        end
        else
        begin
            out_addr <= {FIFO_ADDR_WIDTH{1'b0}};
            out_half <= 1'b0;
        end

        if (out_en)
        begin
            serial_stp  <= fifo_parallel_stp[out_addr] & ~fifo_parallel_idle[out_addr];
            serial_sdp  <= fifo_parallel_sdp[out_addr] & ~fifo_parallel_idle[out_addr];
            serial_idle <= fifo_parallel_idle[out_addr];
            serial_data <= fifo_parallel_data[out_addr];
        end
        else
        begin
            serial_idle <= 1'b1;
            serial_data <= 1'b0;
        end

        // Note: Behavioral, so not synchronizing as must be done
        //   in a hardware implementation
        out_in_addr <= in_addr;
        out_in_half <= in_half;

        out_level   <= c_out_level;

        // Enable output once the FIFO has 20 bits of data
        if (out_level >= (FIFO_DEPTH + 20))
            out_en <= 1'b1;
    end
end

// If the extra address bit (in_half, out_half) being carried in each of the FIFO addresses have
//   different values, then the write address has wrapped relative to the read
//   address and FIFO_DEPTH must be added to the write address in order to determining the true level
assign diff_half   = (out_in_half != out_half);
assign c_out_level = (FIFO_DEPTH + out_in_addr[FIFO_ADDR_WIDTH-1:0]) - {1'b0, out_addr[FIFO_ADDR_WIDTH-1:0]};

// Delay onset of electrical idle by 50 pS to avoid 0 time
//   issues with PHY receiving the last bit transmitted
assign #50 serial_idle_dly = serial_idle;

assign el_idle =  (SIM_EL_IDLE_TYPE == 2'b11) ? 1'b1 :
                 ((SIM_EL_IDLE_TYPE == 2'b10) ? 1'b0 :
                 ((SIM_EL_IDLE_TYPE == 2'b01) ? 1'bx : 1'bz));

assign tx_p = serial_idle_dly ? el_idle :  serial_data;
assign tx_n = serial_idle_dly ? el_idle : ~serial_data;

assign tx_serial_sdp_start = serial_sdp;
assign tx_serial_stp_start = serial_stp;

endmodule
// -------------------------------------------------------------------------
//
//  PROJECT: PCI Express Verification Suite
//  COMPANY: Northwest Logic, Inc.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//             (c) Copyright 2009 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or 
//  transmitted in any form or by any means, electronic or mechanical, 
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//  
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module bfmp_tx_128b_to_130b (

    rst_n,         

    serial_clk,    
    symbol_clk,

    tx_data,       
    tx_data_valid,
    tx_start_block,
    tx_sync_header,
    tx_elec_idle,       

    tx_p,          
    tx_n

);  



// ----------------
// -- Parameters --
// ----------------

parameter   SIM_EL_IDLE_TYPE                = 2'b10; // Electrical Idle Emulation: 11 == 1'b1 : Common Mode 1
                                                     //                            10 == 1'b0 : Common Mode 0
                                                     //                            01 == 1'bx : Undefined
                                                     //                            00 == Reserved

parameter   PHY_K_WIDTH                     = 4;

localparam  FIFO_ADDR_WIDTH                 =  (PHY_K_WIDTH == 4) ? 7 :     // Needs enough depth to tolerate up to 
                                              ((PHY_K_WIDTH == 2) ? 6 : 5); //   PHY_K_WIDTH tx_data_valid in a row (8 * PHY_K_WIDTH)

localparam  FIFO_DEPTH                      = 1 << FIFO_ADDR_WIDTH; 



// ----------------------
// -- Port Definitions --
// ----------------------

input                                       rst_n;

input                                       serial_clk;
input                                       symbol_clk;

input   [7:0]                               tx_data;
input                                       tx_data_valid;
input                                       tx_start_block;
input   [1:0]                               tx_sync_header;
input                                       tx_elec_idle;

output                                      tx_p;
output                                      tx_n;



// ----------------
// -- Port Types --
// ----------------

wire                                        rst_n;

wire                                        serial_clk;
wire                                        symbol_clk;

wire    [7:0]                               tx_data;
wire                                        tx_data_valid;
wire                                        tx_start_block;
wire    [1:0]                               tx_sync_header;
wire                                        tx_elec_idle;

wire                                        tx_p;
wire                                        tx_n;



// -------------------
// -- Local Signals --
// -------------------

// Input Clock Domain
reg     [FIFO_ADDR_WIDTH:0]                 in_addr;

wire                                        in_tx_data_valid;

genvar                                      i;
wire    [FIFO_ADDR_WIDTH-1:0]               in_a                [9:0];

wire    [9:0]                               tx_data_sync;
wire    [9:0]                               tx_data_pad;

reg                                         fifo_tx_data        [FIFO_DEPTH-1:0];

// Output Clock Domain
reg     [FIFO_ADDR_WIDTH:0]                 out_addr;

reg                                         out_tx_elec_idle;
reg                                         out_tx_data;

reg                                         serial_idle;
reg                                         serial_data;

reg     [FIFO_ADDR_WIDTH:0]                 out_in_addr;
reg     [FIFO_ADDR_WIDTH:0]                 out_level;
reg                                         out_en;

wire                                        diff_half;
wire    [FIFO_ADDR_WIDTH:0]                 c_out_level;

wire                                        serial_idle_dly;
wire                                        el_idle;

reg                                         in_block;
reg     [3:0]                               in_ctr;
reg     [3:0]                               idle_ctr;


// ---------------
// -- Equations --
// ---------------

// ------------------
// Input Clock Domain

assign in_tx_data_valid = tx_data_valid & ~tx_elec_idle;

always @(posedge symbol_clk or negedge rst_n)
begin
    if (rst_n == 1'b0) begin
        in_addr     <= {(FIFO_ADDR_WIDTH+1){1'b0}};
        in_block    <= 1'b0;
        in_ctr      <= 4'h0;
        idle_ctr    <= 4'h0;
    end    
    else begin
        if (tx_start_block & in_tx_data_valid) begin
            in_block <= 1'b1;
            in_ctr   <= 4'h1;
        end
        else if (in_block & in_tx_data_valid) begin
            if (in_ctr != 4'h0)
                in_ctr <= in_ctr + 1;
            if (in_ctr == 4'hf)
                in_block <= 1'b0;
        end
        else if (~tx_start_block & ~in_block & in_tx_data_valid) begin
            idle_ctr <= idle_ctr + 1;
        end

        // Keeping one extra address bit to tell full from empty when doing an address compare
        if (in_tx_data_valid & tx_start_block)
            in_addr <= in_addr + {{FIFO_ADDR_WIDTH-3{1'b0}}, 4'd10};
        else if (in_tx_data_valid & in_block)
            in_addr <= in_addr + {{FIFO_ADDR_WIDTH-3{1'b0}}, 4'd8};
        else if (in_tx_data_valid & ~tx_start_block & ~in_block & idle_ctr == 4'h0)
            in_addr <= in_addr + {{FIFO_ADDR_WIDTH-3{1'b0}}, 4'd10};
        else if (in_tx_data_valid & ~tx_start_block & ~in_block)
            in_addr <= in_addr + {{FIFO_ADDR_WIDTH-3{1'b0}}, 4'd8};
    end
end

// Intermediate wires used to avoid compile warnings with below generate block
assign tx_data_sync = {tx_data,tx_sync_header};
assign tx_data_pad  = {2'b0,tx_data};

generate
for (i=0; i<10; i=i+1) begin : fifo_1

assign  in_a[i] = in_addr[FIFO_ADDR_WIDTH-1:0] + i;

always @(posedge symbol_clk) begin
    if (tx_start_block & in_tx_data_valid) begin
        fifo_tx_data       [in_a[i]] <= tx_data_sync[i];
    end
    else if (in_block & in_tx_data_valid) begin
        if (i<8) begin
            fifo_tx_data       [in_a[i]] <= tx_data_pad[i];
        end
    end
    else if (~tx_start_block & ~in_block & in_tx_data_valid & (idle_ctr == 4'h0)) begin
        fifo_tx_data       [in_a[i]] <= 1'b0;
    end
    else if (~tx_start_block & ~in_block & in_tx_data_valid) begin
        if (i<8) begin
            fifo_tx_data       [in_a[i]] <= 1'b0;
        end
    end
end

end
endgenerate

// -------------------
// Output Clock Domain

// Manage the output address pointer
always @(posedge serial_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        out_addr         <= {(FIFO_ADDR_WIDTH+1){1'b0}};

        out_tx_elec_idle <= 1'b1;
        out_tx_data      <= 1'b0;

        serial_idle      <= 1'b1;
        serial_data      <= 1'b0;

        out_in_addr      <= {(FIFO_ADDR_WIDTH+1){1'b0}};
        out_level        <= {(FIFO_ADDR_WIDTH+1){1'b0}};
        out_en           <= 1'b0;
    end
    else
    begin
        // Keeping one extra address bit to tell full from empty when doing an address compare
        if (out_en & (c_out_level != 0))
            out_addr <= out_addr + {{FIFO_ADDR_WIDTH{1'b0}}, 1'b1};

        if (out_en)
        begin
            out_tx_elec_idle <= 1'b0;
            out_tx_data      <= fifo_tx_data     [out_addr[FIFO_ADDR_WIDTH-1:0]];
        end
        else
            out_tx_elec_idle <= 1'b1;

        serial_idle <= out_tx_elec_idle;
        serial_data <= out_tx_data;

        // Note: Behavioral, so not synchronizing as must be done
        //   in a hardware implementation
        out_in_addr <= in_addr;
        out_level   <= c_out_level;

        // Enable output once the FIFO has filled quarter-way
        if (out_level[FIFO_ADDR_WIDTH-2] == 1'b1)
            out_en <= 1'b1;
        else if (c_out_level == 0)
            out_en <= 1'b0;
    end
end

// If the extra address bit being carried in each of the FIFO addresses have
//   different values, then the write address has wrapped relative to the read
//   address and 2^ADDR_WIDTH must be added to the write address in order to
//   determining the true level
assign diff_half   = (out_in_addr[FIFO_ADDR_WIDTH] != out_addr[FIFO_ADDR_WIDTH]);
assign c_out_level = {diff_half, out_in_addr[FIFO_ADDR_WIDTH-1:0]} - {1'b0, out_addr[FIFO_ADDR_WIDTH-1:0]};

// Delay onset of electrical idle by 50 pS to avoid 0 time
//   issues with PHY receiving the last bit transmitted
assign #50 serial_idle_dly = serial_idle;

assign el_idle =  (SIM_EL_IDLE_TYPE == 2'b11) ? 1'b1 :
                 ((SIM_EL_IDLE_TYPE == 2'b10) ? 1'b0 :
                 ((SIM_EL_IDLE_TYPE == 2'b01) ? 1'bx : 1'bz));

//assign tx_p = serial_idle_dly ? el_idle :  serial_data;
//assign tx_n = serial_idle_dly ? el_idle : ~serial_data;

assign tx_p = serial_idle ? el_idle :  serial_data;
assign tx_n = serial_idle ? el_idle : ~serial_data;



endmodule

// -------------------------------------------------------------------------
//
//  PROJECT: PCI Express Verification Suite
//  COMPANY: Northwest Logic, Inc.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//             (c) Copyright 2009 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or 
//  transmitted in any form or by any means, electronic or mechanical, 
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//  
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module bfmp_phy_tx (

    rst_n,             
    rate,

    in_tx_clk,        
    in_tx_clk_en,
    in_tx_data,       
    in_tx_data_k,     
    in_tx_data_valid,  
    in_tx_start_block,
    in_tx_sync_header,
    in_tx_compliance, 
    in_tx_elec_idle,  

    out_tx_clk,       
    out_tx_data,      
    out_tx_data_k,     
    out_tx_data_valid,
    out_tx_start_block,
    out_tx_sync_header,
    out_tx_compliance,
    out_tx_elec_idle  
            
);      



// ----------------
// -- Parameters --
// ----------------

parameter   PHY_K_WIDTH                     = 4;                            // Byte width of in_* port

localparam  FIFO_ADDR_WIDTH                 =  (PHY_K_WIDTH == 4) ? 5 :     // FIFO address size (1 word == 1 symbol)
                                              ((PHY_K_WIDTH == 2) ? 4 : 3); // Want FIFO to hold 8 * PHY_K_WIDTH symbols

localparam  FIFO_DEPTH                      = (1 << FIFO_ADDR_WIDTH);       // Depth of FIFO



// ----------------------
// -- Port Definitions --
// ----------------------

input                                       rst_n;
input   [1:0]                               rate;

input                                       in_tx_clk;
input                                       in_tx_clk_en;
input   [(PHY_K_WIDTH*8)-1:0]               in_tx_data;
input   [PHY_K_WIDTH-1:0]                   in_tx_data_k;
input                                       in_tx_data_valid;
input                                       in_tx_start_block;
input   [1:0]                               in_tx_sync_header;
input   [PHY_K_WIDTH-1:0]                   in_tx_compliance;
input                                       in_tx_elec_idle;

input                                       out_tx_clk;
output  [7:0]                               out_tx_data;
output                                      out_tx_data_k;
output                                      out_tx_data_valid;
output                                      out_tx_start_block;
output  [1:0]                               out_tx_sync_header;
output                                      out_tx_compliance;
output                                      out_tx_elec_idle;



// ----------------
// -- Port Types --
// ----------------

wire                                        rst_n;
wire    [1:0]                               rate;
  
wire                                        in_tx_clk;
wire                                        in_tx_clk_en;
wire    [(PHY_K_WIDTH*8)-1:0]               in_tx_data;
wire    [PHY_K_WIDTH-1:0]                   in_tx_data_k;
wire                                        in_tx_data_valid;
wire                                        in_tx_start_block;
wire    [1:0]                               in_tx_sync_header;
wire    [PHY_K_WIDTH-1:0]                   in_tx_compliance;
wire                                        in_tx_elec_idle;

wire                                        out_tx_clk;
reg     [7:0]                               out_tx_data;
reg                                         out_tx_data_k;
reg                                         out_tx_data_valid;
reg                                         out_tx_start_block;
reg     [1:0]                               out_tx_sync_header;
reg                                         out_tx_compliance;
reg                                         out_tx_elec_idle;



// -------------------
// -- Local Signals --
// -------------------

reg     [1:0]                               r_rate;
reg                                         rate_change;
reg                                         out_rate_change;

// Input Clock Domain
reg                                         in_en;
reg     [FIFO_ADDR_WIDTH:0]                 in_addr;

genvar                                      i;
wire    [FIFO_ADDR_WIDTH-1:0]               in_a                [PHY_K_WIDTH-1:0];

reg     [7:0]                               fifo_tx_data        [FIFO_DEPTH-1:0];
reg                                         fifo_tx_data_k      [FIFO_DEPTH-1:0];
reg                                         fifo_tx_data_valid  [FIFO_DEPTH-1:0];
reg                                         fifo_tx_start_block [FIFO_DEPTH-1:0];
reg     [1:0]                               fifo_tx_sync_header [FIFO_DEPTH-1:0];
reg                                         fifo_tx_compliance  [FIFO_DEPTH-1:0];
reg                                         fifo_tx_elec_idle   [FIFO_DEPTH-1:0];
// Output Clock Domain
reg     [FIFO_ADDR_WIDTH:0]                 out_addr;
reg     [FIFO_ADDR_WIDTH:0]                 out_in_addr;
reg     [FIFO_ADDR_WIDTH:0]                 out_level;
reg                                         out_en;

wire                                        diff_half;
wire    [FIFO_ADDR_WIDTH:0]                 c_out_level;

wire [31:0]                                 in_bytes;


// ---------------
// -- Equations --
// ---------------

assign in_bytes = PHY_K_WIDTH;

always @(posedge in_tx_clk or negedge rst_n) begin
    if (rst_n == 1'b0) begin
        r_rate <= 2'b00;
        rate_change <= 1'b0;
    end
    else if (in_tx_clk_en) begin
        r_rate <= rate;
        if (r_rate != rate)
            rate_change <= 1'b1;
        else if (out_rate_change)
            rate_change <= 1'b0;
    end
end

// ------------------
// Input Clock Domain

always @(posedge in_tx_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        in_en   <= 1'b0;
        in_addr <= {(FIFO_ADDR_WIDTH+1){1'b0}};
    end
    else if (in_tx_clk_en)
    begin
        in_en <= 1'b1;

        // Keeping one extra address bit to tell full from empty when doing an address compare
        if (rate_change)
            in_addr <= {(FIFO_ADDR_WIDTH+1){1'b0}};
        else if (in_en)
            in_addr <= in_addr + in_bytes;
    end
end

generate
    for (i=0; i<PHY_K_WIDTH; i=i+1)
    begin : fifo
        // Drop the extra address bit when accessing the FIFO
        assign in_a[i] = in_addr[FIFO_ADDR_WIDTH-1:0] + i;

        // Write input data into the FIFO
        always @(posedge in_tx_clk)
        begin
            if (in_tx_clk_en & (i < in_bytes))
            begin
                fifo_tx_data        [in_a[i]] <= in_tx_data[((i+1)*8)-1:(i*8)];
                fifo_tx_data_k      [in_a[i]] <= in_tx_data_k[i];
                fifo_tx_data_valid  [in_a[i]] <= in_tx_data_valid;
                fifo_tx_start_block [in_a[i]] <= (i == 0) ? in_tx_start_block : 1'b0;
                fifo_tx_sync_header [in_a[i]] <= (i == 0) ? in_tx_sync_header : 2'b00;
                fifo_tx_compliance  [in_a[i]] <= in_tx_compliance[i];
                fifo_tx_elec_idle   [in_a[i]] <= in_tx_elec_idle;
            end
        end
    end
endgenerate



// -------------------
// Output Clock Domain

// Manage the output address pointer
always @(posedge out_tx_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        out_addr    <= {(FIFO_ADDR_WIDTH+1){1'b0}};
        out_in_addr <= {(FIFO_ADDR_WIDTH+1){1'b0}};
        out_level   <= {(FIFO_ADDR_WIDTH+1){1'b0}};
        out_en      <= 1'b0;
    end
    else
    begin
        // Keeping one extra address bit to tell full from empty when doing an address compare
        if (out_rate_change)
            out_addr <= {(FIFO_ADDR_WIDTH+1){1'b0}};
        else if (out_en)
            out_addr <= out_addr + {{FIFO_ADDR_WIDTH{1'b0}}, 1'b1};
        else
            out_addr <= {(FIFO_ADDR_WIDTH+1){1'b0}};

        // Note: Behavioral, so not synchronizing as must be done
        //   in a hardware implementation
        if (out_rate_change)
            out_in_addr <= {(FIFO_ADDR_WIDTH+1){1'b0}};
        else
            out_in_addr <= in_addr;
        if (out_rate_change)
            out_level <= {(FIFO_ADDR_WIDTH+1){1'b0}};
        else
            out_level   <= c_out_level;
        out_rate_change <= rate_change;

        // Enable output once the FIFO has filled half-way
        if (out_rate_change)
            out_en <= 1'b0;
        else if (out_level[FIFO_ADDR_WIDTH-1] == 1'b1)
            out_en <= 1'b1;
    end
end

// If the extra address bit being carried in each of the FIFO addresses have
//   different values, then the write address has wrapped relative to the read
//   address and 2^ADDR_WIDTH must be added to the write address in order to
//   determining the true level
assign diff_half   = (out_in_addr[FIFO_ADDR_WIDTH] != out_addr[FIFO_ADDR_WIDTH]);
assign c_out_level = {diff_half, out_in_addr[FIFO_ADDR_WIDTH-1:0]} - {1'b0, out_addr[FIFO_ADDR_WIDTH-1:0]};

always @(posedge out_tx_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        out_tx_data        <= 8'h0;
        out_tx_data_k      <= 1'b0;
        out_tx_data_valid  <= 1'b0;
        out_tx_start_block <= 1'b0;
        out_tx_sync_header <= 2'b00;
        out_tx_compliance  <= 1'b0;
        out_tx_elec_idle   <= 1'b1;
    end
    else
    begin
        if (out_en == 1'b0)
        begin
            out_tx_data        <= 8'h0;
            out_tx_data_k      <= 1'b0;
            out_tx_data_valid  <= 1'b0;
            out_tx_start_block <= 1'b0;
            out_tx_sync_header <= 2'b00;
            out_tx_compliance  <= 1'b0;
            out_tx_elec_idle   <= 1'b1;
        end
        else
        begin
            out_tx_data        <= fifo_tx_data        [out_addr[FIFO_ADDR_WIDTH-1:0]];
            out_tx_data_k      <= fifo_tx_data_k      [out_addr[FIFO_ADDR_WIDTH-1:0]];
            out_tx_data_valid  <= fifo_tx_data_valid  [out_addr[FIFO_ADDR_WIDTH-1:0]];
            out_tx_start_block <= fifo_tx_start_block [out_addr[FIFO_ADDR_WIDTH-1:0]];
            out_tx_sync_header <= fifo_tx_sync_header [out_addr[FIFO_ADDR_WIDTH-1:0]];
            out_tx_compliance  <= fifo_tx_compliance  [out_addr[FIFO_ADDR_WIDTH-1:0]];
            out_tx_elec_idle   <= fifo_tx_elec_idle   [out_addr[FIFO_ADDR_WIDTH-1:0]];
        end
    end
end



endmodule

// -------------------------------------------------------------------------
//
//  PROJECT: PCI Express Verification Suite
//  COMPANY: Northwest Logic, Inc.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//             (c) Copyright 2009 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or 
//  transmitted in any form or by any means, electronic or mechanical, 
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//  
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module bfmp_rx_10b_to_8b (

    rst_n,                                  // Asynchronous reset

    rx_serial_clk,                          // Serial clock
    rx_symbol_clk,                          // Symbol Clock

    rx_p,                                   // Received serial stream
    rx_n,                                   //   ..

    rx_valid,                               // Set if syncronization was lost; asynchronous signal
    rx_data_valid,                           // Clock enable for RX
    rx_data,                                // 8-bit Data value
    rx_data_k,                              // Set for K characters, clear for others
    rx_err_disp,                            // Set if a disparity error was received; clear otherwise
    rx_err_8b10                             // Set if the received value is not in the 10b->8b table; clear otherwise

);  



// ----------------
// -- Parameters --
// ----------------

localparam  COM_NEG                         = 10'b001111_1010;  // COM character with current running disparity == negative
localparam  COM_POS                         = 10'b110000_0101;  // COM character with current running disparity == positive



// ----------------------
// -- Port Definitions --
// ----------------------

input                                       rst_n;

input                                       rx_serial_clk;
input                                       rx_symbol_clk;

input                                       rx_p;
input                                       rx_n;

output                                      rx_valid;
output                                      rx_data_valid;
output  [7:0]                               rx_data;
output                                      rx_data_k;
output                                      rx_err_disp;
output                                      rx_err_8b10;



// ----------------
// -- Port Types --
// ----------------

wire                                        rst_n;

wire                                        rx_serial_clk;
wire                                        rx_symbol_clk;

wire                                        rx_p;
wire                                        rx_n;

reg                                         rx_valid;
wire                                        rx_data_valid;
reg     [7:0]                               rx_data;
reg                                         rx_data_k;
wire                                        rx_err_disp;
reg                                         rx_err_8b10;



// -------------------
// -- Local Signals --
// -------------------

// Convert the serial stream to parallel 10-bit data
reg                                         a, b, c, d, e, f, g, h, i, j;
wire    [9:0]                               data10;

// Look for COM to determine code sync
wire                                        data10_com;
wire                                        c_code_sync;
reg                                         code_sync;
reg     [3:0]                               ctr;
reg     [9:0]                               parallel_data10;

reg     [9:0]                               sync_pos;
reg                                         data10_com_good;
reg                                         data10_com_error;
reg     [1:0]                               com_good_ctr;
reg                                         resync;

// Decode data 10b -> 8b
reg                                         out_disp;

reg                                         disp6b;
reg                                         disp4b;

reg     [3:0]                               parallel_data10_disp;
reg                                         next_disp;

reg                                         curr_disp;
reg                                         r_curr_disp;



// ---------------
// -- Equations --
// ---------------

// -------------------------------------------------
// Convert the serial stream to parallel 10-bit data

always @(posedge rx_serial_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        a <= 1'b0;
        b <= 1'b0;
        c <= 1'b0;
        d <= 1'b0;
        e <= 1'b0;
        f <= 1'b0;
        g <= 1'b0;
        h <= 1'b0;
        i <= 1'b0;
        j <= 1'b0;
    end
    else
    begin
        a <= rx_p;
        b <= a;
        c <= b;
        d <= c;
        e <= d;
        f <= e;
        g <= f;
        h <= g;
        i <= h;
        j <= i;
    end
end

assign data10 = {j, i, h, g, f, e, d, c, b, a};



// -----------------------------------
// Look for COM to determine code sync

// Detect new code sync 
assign data10_com  = (data10 == COM_POS) |
                     (data10 == COM_NEG);

assign c_code_sync = data10_com & ~code_sync;

always @(posedge rx_serial_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        code_sync        <= 1'b0;
        ctr              <= 4'h0;
        parallel_data10  <= 10'b0;
    end
    else
    begin    
        if (c_code_sync)
            code_sync <= 1;
        else if (resync)
            code_sync <= 0;

        if (c_code_sync)
            ctr <= 0;
        else if (code_sync)
            ctr <= (ctr == 9) ? 0 : (ctr + 1);

        // Transfer data and disparity into parallel clock domain
        if (c_code_sync | (code_sync & (ctr == 9)))
            parallel_data10 <= data10;

    end
end

// Implement logic to relock when lock position is detected to be bad
always @(posedge rx_serial_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        sync_pos         <= 10'b0;
        data10_com_good  <= 1'b0;
        data10_com_error <= 1'b0;
        com_good_ctr     <= 2'b00;
        resync           <= 1'b0;
        rx_valid         <= 1'b0;
    end
    else
    begin    
        // Mark COM position used to obtain sync; future COM
        //   should be detected at the same position if the
        //   sync operation locked to a valid COM
        if (c_code_sync)
            sync_pos <= 10'b0000000001;
        else
            sync_pos <= {sync_pos[8:0], sync_pos[9]};

        data10_com_good  <= code_sync & data10_com &  sync_pos[9];
        data10_com_error <= code_sync & data10_com & ~sync_pos[9];

        if (code_sync)
        begin
            if (data10_com_good & (com_good_ctr != 2'b11))
                com_good_ctr <= com_good_ctr + 2'b01;
            else if (data10_com_error)
                com_good_ctr <= 2'b00;
        end
        else
        begin
            com_good_ctr <= 2'b00;
        end

        resync <= code_sync & data10_com_error;

        if (code_sync & data10_com_error)
            rx_valid <= 1'b0;
        else if (code_sync & data10_com_good & (com_good_ctr == 2'b11))
            rx_valid <= 1'b1;
    end
end


// ---------------------
// Decode data 10b -> 8b

assign rx_data_valid = 1'b1; // Not clock enabled

always @(posedge rx_symbol_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        // Output 1FE while reset is asserted
        rx_data_k   <= 1'b1;
        rx_data     <= 8'b111_11110;
        rx_err_8b10 <= 1'b0;
        out_disp    <= 1'b0;
    end
    else
    begin
        // Note: Commented lines are duplicates (same 10b symbol for + and -)
        case (parallel_data10)

            // Data Characters; CurrRD-
            10'b100111_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_00000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011101_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_00001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101101_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_00010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110001_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_00011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110101_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_00100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101001_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_00101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011001_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_00110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b111000_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_00111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b111001_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_01000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100101_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_01001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b010101_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_01010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110100_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_01011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001101_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_01100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101100_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_01101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011100_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_01110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b010111_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_01111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011011_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_10000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100011_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_10001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b010011_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_10010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110010_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_10011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001011_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_10100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101010_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_10101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011010_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_10110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b111010_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_10111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110011_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_11000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100110_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_11001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b010110_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_11010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110110_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_11011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001110_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101110_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_11101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011110_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_11110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101011_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_11111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100111_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_00000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011101_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_00001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101101_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_00010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110001_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_00011; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b110101_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_00100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101001_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_00101; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b011001_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_00110; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b111000_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_00111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b111001_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_01000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100101_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_01001; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b010101_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_01010; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b110100_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_01011; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b001101_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_01100; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b101100_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_01101; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b011100_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_01110; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b010111_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_01111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011011_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_10000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100011_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_10001; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b010011_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_10010; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b110010_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_10011; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b001011_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_10100; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b101010_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_10101; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b011010_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_10110; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b111010_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_10111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110011_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_11000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100110_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_11001; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b010110_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_11010; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b110110_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_11011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001110_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_11100; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b101110_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_11101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011110_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_11110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101011_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_11111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100111_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_00000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011101_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_00001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101101_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_00010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110001_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_00011; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b110101_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_00100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101001_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_00101; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b011001_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_00110; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b111000_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_00111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b111001_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_01000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100101_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_01001; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b010101_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_01010; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b110100_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_01011; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b001101_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_01100; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b101100_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_01101; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b011100_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_01110; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b010111_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_01111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011011_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_10000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100011_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_10001; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b010011_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_10010; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b110010_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_10011; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b001011_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_10100; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b101010_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_10101; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b011010_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_10110; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b111010_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_10111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110011_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_11000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100110_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_11001; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b010110_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_11010; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b110110_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_11011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001110_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_11100; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b101110_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_11101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011110_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_11110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101011_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_11111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100111_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_00000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011101_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_00001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101101_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_00010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110001_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_00011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110101_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_00100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101001_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_00101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011001_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_00110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b111000_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_00111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b111001_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_01000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100101_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_01001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b010101_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_01010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110100_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_01011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001101_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_01100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101100_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_01101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011100_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_01110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b010111_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_01111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011011_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_10000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100011_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_10001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b010011_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_10010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110010_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_10011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001011_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_10100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101010_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_10101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011010_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_10110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b111010_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_10111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110011_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_11000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100110_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_11001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b010110_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_11010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110110_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_11011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001110_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101110_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_11101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011110_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_11110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101011_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_11111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100111_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_00000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011101_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_00001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101101_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_00010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110001_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_00011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110101_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_00100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101001_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_00101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011001_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_00110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b111000_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_00111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b111001_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_01000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100101_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_01001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b010101_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_01010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110100_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_01011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001101_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_01100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101100_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_01101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011100_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_01110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b010111_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_01111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011011_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_10000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100011_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_10001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b010011_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_10010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110010_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_10011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001011_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_10100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101010_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_10101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011010_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_10110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b111010_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_10111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110011_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_11000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100110_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_11001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b010110_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_11010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110110_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_11011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001110_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101110_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_11101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011110_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_11110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101011_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_11111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100111_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_00000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011101_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_00001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101101_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_00010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110001_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_00011; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b110101_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_00100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101001_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_00101; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b011001_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_00110; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b111000_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_00111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b111001_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_01000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100101_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_01001; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b010101_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_01010; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b110100_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_01011; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b001101_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_01100; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b101100_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_01101; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b011100_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_01110; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b010111_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_01111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011011_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_10000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100011_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_10001; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b010011_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_10010; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b110010_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_10011; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b001011_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_10100; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b101010_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_10101; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b011010_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_10110; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b111010_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_10111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110011_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_11000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100110_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_11001; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b010110_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_11010; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b110110_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_11011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001110_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_11100; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b101110_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_11101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011110_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_11110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101011_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_11111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100111_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_00000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011101_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_00001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101101_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_00010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110001_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_00011; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b110101_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_00100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101001_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_00101; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b011001_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_00110; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b111000_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_00111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b111001_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_01000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100101_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_01001; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b010101_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_01010; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b110100_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_01011; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b001101_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_01100; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b101100_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_01101; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b011100_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_01110; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b010111_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_01111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011011_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_10000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100011_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_10001; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b010011_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_10010; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b110010_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_10011; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b001011_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_10100; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b101010_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_10101; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b011010_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_10110; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b111010_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_10111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110011_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_11000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100110_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_11001; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b010110_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_11010; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b110110_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_11011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001110_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_11100; rx_err_8b10 <= 1'b0; out_disp <= next_disp; end
            10'b101110_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_11101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011110_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_11110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101011_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_11111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100111_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_00000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011101_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_00001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101101_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_00010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110001_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_00011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110101_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_00100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101001_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_00101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011001_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_00110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b111000_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_00111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b111001_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_01000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100101_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_01001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b010101_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_01010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110100_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_01011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001101_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_01100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101100_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_01101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011100_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_01110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b010111_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_01111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011011_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_10000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100011_0111 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_10001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b010011_0111 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_10010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110010_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_10011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001011_0111 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_10100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101010_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_10101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011010_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_10110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b111010_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_10111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110011_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_11000; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b100110_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_11001; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b010110_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_11010; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110110_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_11011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001110_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101110_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_11101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011110_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_11110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101011_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_11111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end

            // K Characters; CurrRD-
            10'b001111_0100 : begin rx_data_k <= 1'b1; rx_data <= 8'b000_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001111_1001 : begin rx_data_k <= 1'b1; rx_data <= 8'b001_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001111_0101 : begin rx_data_k <= 1'b1; rx_data <= 8'b010_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001111_0011 : begin rx_data_k <= 1'b1; rx_data <= 8'b011_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001111_0010 : begin rx_data_k <= 1'b1; rx_data <= 8'b100_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001111_1010 : begin rx_data_k <= 1'b1; rx_data <= 8'b101_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001111_0110 : begin rx_data_k <= 1'b1; rx_data <= 8'b110_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b001111_1000 : begin rx_data_k <= 1'b1; rx_data <= 8'b111_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b111010_1000 : begin rx_data_k <= 1'b1; rx_data <= 8'b111_10111; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b110110_1000 : begin rx_data_k <= 1'b1; rx_data <= 8'b111_11011; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b101110_1000 : begin rx_data_k <= 1'b1; rx_data <= 8'b111_11101; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end
            10'b011110_1000 : begin rx_data_k <= 1'b1; rx_data <= 8'b111_11110; rx_err_8b10 <= 1'b0; out_disp <= 1'b0;      end

            // Data Characters; CurrRD+
            10'b011000_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_00000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100010_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_00001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010010_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_00010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b110001_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_00011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001010_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_00100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b101001_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_00101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b011001_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_00110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b000111_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_00111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b000110_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_01000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100101_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_01001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010101_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_01010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b110100_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_01011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001101_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_01100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b101100_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_01101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b011100_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_01110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b101000_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_01111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100100_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_10000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100011_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_10001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010011_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_10010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b110010_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_10011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001011_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_10100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b101010_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_10101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b011010_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_10110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b000101_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_10111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001100_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_11000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100110_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_11001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010110_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_11010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001001_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_11011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001110_0100 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010001_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_11101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100001_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_11110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010100_1011 : begin rx_data_k <= 1'b0; rx_data <= 8'b000_11111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b011000_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_00000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100010_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_00001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010010_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_00010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b110001_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_00011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b001010_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_00100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b101001_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_00101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b011001_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_00110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b000111_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_00111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b000110_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_01000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b100101_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_01001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b010101_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_01010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b110100_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_01011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b001101_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_01100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b101100_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_01101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b011100_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_01110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b101000_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_01111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100100_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_10000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b100011_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_10001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b010011_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_10010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b110010_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_10011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b001011_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_10100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b101010_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_10101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b011010_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_10110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b000101_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_10111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001100_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_11000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b100110_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_11001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b010110_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_11010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b001001_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_11011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b001110_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b010001_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_11101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100001_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_11110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010100_1001 : begin rx_data_k <= 1'b0; rx_data <= 8'b001_11111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b011000_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_00000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100010_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_00001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010010_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_00010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b110001_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_00011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b001010_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_00100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b101001_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_00101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b011001_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_00110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b000111_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_00111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b000110_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_01000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b100101_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_01001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b010101_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_01010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b110100_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_01011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b001101_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_01100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b101100_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_01101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b011100_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_01110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b101000_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_01111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100100_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_10000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b100011_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_10001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b010011_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_10010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b110010_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_10011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b001011_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_10100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b101010_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_10101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b011010_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_10110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b000101_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_10111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001100_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_11000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b100110_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_11001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b010110_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_11010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b001001_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_11011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b001110_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b010001_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_11101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100001_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_11110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010100_0101 : begin rx_data_k <= 1'b0; rx_data <= 8'b010_11111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b011000_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_00000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100010_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_00001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010010_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_00010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b110001_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_00011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001010_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_00100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b101001_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_00101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b011001_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_00110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b000111_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_00111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b000110_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_01000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100101_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_01001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010101_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_01010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b110100_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_01011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001101_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_01100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b101100_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_01101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b011100_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_01110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b101000_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_01111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100100_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_10000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100011_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_10001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010011_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_10010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b110010_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_10011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001011_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_10100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b101010_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_10101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b011010_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_10110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b000101_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_10111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001100_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_11000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100110_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_11001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010110_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_11010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001001_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_11011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001110_0011 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010001_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_11101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100001_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_11110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010100_1100 : begin rx_data_k <= 1'b0; rx_data <= 8'b011_11111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b011000_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_00000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100010_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_00001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010010_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_00010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b110001_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_00011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001010_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_00100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b101001_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_00101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b011001_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_00110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b000111_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_00111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b000110_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_01000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100101_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_01001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010101_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_01010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b110100_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_01011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001101_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_01100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b101100_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_01101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b011100_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_01110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b101000_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_01111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100100_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_10000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100011_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_10001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010011_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_10010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b110010_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_10011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001011_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_10100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b101010_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_10101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b011010_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_10110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b000101_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_10111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001100_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_11000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100110_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_11001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010110_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_11010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001001_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_11011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001110_0010 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010001_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_11101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100001_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_11110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010100_1101 : begin rx_data_k <= 1'b0; rx_data <= 8'b100_11111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b011000_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_00000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100010_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_00001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010010_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_00010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b110001_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_00011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b001010_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_00100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b101001_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_00101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b011001_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_00110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b000111_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_00111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b000110_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_01000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b100101_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_01001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b010101_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_01010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b110100_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_01011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b001101_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_01100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b101100_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_01101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b011100_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_01110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b101000_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_01111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100100_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_10000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b100011_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_10001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b010011_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_10010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b110010_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_10011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b001011_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_10100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b101010_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_10101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b011010_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_10110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b000101_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_10111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001100_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_11000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b100110_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_11001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b010110_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_11010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b001001_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_11011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b001110_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b010001_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_11101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100001_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_11110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010100_1010 : begin rx_data_k <= 1'b0; rx_data <= 8'b101_11111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b011000_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_00000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100010_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_00001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010010_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_00010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b110001_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_00011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b001010_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_00100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b101001_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_00101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b011001_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_00110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b000111_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_00111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b000110_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_01000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b100101_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_01001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b010101_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_01010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b110100_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_01011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b001101_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_01100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b101100_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_01101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b011100_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_01110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b101000_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_01111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100100_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_10000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b100011_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_10001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b010011_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_10010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b110010_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_10011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b001011_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_10100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b101010_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_10101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b011010_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_10110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b000101_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_10111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001100_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_11000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b100110_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_11001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
//            10'b010110_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_11010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b001001_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_11011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
//            10'b001110_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1; end
            10'b010001_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_11101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100001_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_11110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010100_0110 : begin rx_data_k <= 1'b0; rx_data <= 8'b110_11111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b011000_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_00000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100010_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_00001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010010_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_00010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b110001_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_00011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001010_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_00100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b101001_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_00101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b011001_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_00110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b000111_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_00111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b000110_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_01000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100101_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_01001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010101_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_01010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b110100_1000 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_01011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001101_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_01100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b101100_1000 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_01101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b011100_1000 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_01110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b101000_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_01111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100100_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_10000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100011_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_10001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010011_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_10010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b110010_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_10011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001011_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_10100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b101010_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_10101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b011010_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_10110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b000101_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_10111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001100_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_11000; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100110_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_11001; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010110_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_11010; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001001_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_11011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001110_0001 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010001_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_11101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100001_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_11110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010100_1110 : begin rx_data_k <= 1'b0; rx_data <= 8'b111_11111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end

            // K Characters; CurrRD+
            10'b110000_1011 : begin rx_data_k <= 1'b1; rx_data <= 8'b000_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b110000_0110 : begin rx_data_k <= 1'b1; rx_data <= 8'b001_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b110000_1010 : begin rx_data_k <= 1'b1; rx_data <= 8'b010_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b110000_1100 : begin rx_data_k <= 1'b1; rx_data <= 8'b011_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b110000_1101 : begin rx_data_k <= 1'b1; rx_data <= 8'b100_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b110000_0101 : begin rx_data_k <= 1'b1; rx_data <= 8'b101_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b110000_1001 : begin rx_data_k <= 1'b1; rx_data <= 8'b110_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b110000_0111 : begin rx_data_k <= 1'b1; rx_data <= 8'b111_11100; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b000101_0111 : begin rx_data_k <= 1'b1; rx_data <= 8'b111_10111; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b001001_0111 : begin rx_data_k <= 1'b1; rx_data <= 8'b111_11011; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b010001_0111 : begin rx_data_k <= 1'b1; rx_data <= 8'b111_11101; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end
            10'b100001_0111 : begin rx_data_k <= 1'b1; rx_data <= 8'b111_11110; rx_err_8b10 <= 1'b0; out_disp <= 1'b1;      end

            // 10b input symbol does not appear in the 10b8b table; encode as 1FE
            default         : begin rx_data_k <= 1'b1; rx_data <= 8'b111_11110; rx_err_8b10 <= 1'b1; out_disp <= 1'b0;      end

        endcase
    end
end

// Compute Disparity of 1st 6-bit sub-block
always @*
begin
    case (parallel_data10[9:4])

        6'b00_0000 : disp6b = 1'b0;
        6'b00_0001 : disp6b = 1'b0;
        6'b00_0010 : disp6b = 1'b0;
        6'b00_0011 : disp6b = 1'b0;
        6'b00_0100 : disp6b = 1'b0;
        6'b00_0101 : disp6b = 1'b0;
        6'b00_0110 : disp6b = 1'b0;
        6'b00_0111 : disp6b = 1'b1; // Special case
        6'b00_1000 : disp6b = 1'b0;
        6'b00_1001 : disp6b = 1'b0;
        6'b00_1010 : disp6b = 1'b0;
        6'b00_1011 : disp6b = curr_disp;
        6'b00_1100 : disp6b = 1'b0;
        6'b00_1101 : disp6b = curr_disp;
        6'b00_1110 : disp6b = curr_disp;
        6'b00_1111 : disp6b = 1'b1;

        6'b01_0000 : disp6b = 1'b0;
        6'b01_0001 : disp6b = 1'b0;
        6'b01_0010 : disp6b = 1'b0;
        6'b01_0011 : disp6b = curr_disp;
        6'b01_0100 : disp6b = 1'b0;
        6'b01_0101 : disp6b = curr_disp;
        6'b01_0110 : disp6b = curr_disp;
        6'b01_0111 : disp6b = 1'b1;
        6'b01_1000 : disp6b = 1'b0;
        6'b01_1001 : disp6b = curr_disp;
        6'b01_1010 : disp6b = curr_disp;
        6'b01_1011 : disp6b = 1'b1;
        6'b01_1100 : disp6b = curr_disp;
        6'b01_1101 : disp6b = 1'b1;
        6'b01_1110 : disp6b = 1'b1;
        6'b01_1111 : disp6b = 1'b1;

        6'b10_0000 : disp6b = 1'b0;
        6'b10_0001 : disp6b = 1'b0;
        6'b10_0010 : disp6b = 1'b0;
        6'b10_0011 : disp6b = curr_disp;
        6'b10_0100 : disp6b = 1'b0;
        6'b10_0101 : disp6b = curr_disp;
        6'b10_0110 : disp6b = curr_disp;
        6'b10_0111 : disp6b = 1'b1;
        6'b10_1000 : disp6b = 1'b0;
        6'b10_1001 : disp6b = curr_disp;
        6'b10_1010 : disp6b = curr_disp;
        6'b10_1011 : disp6b = 1'b1;
        6'b10_1100 : disp6b = curr_disp;
        6'b10_1101 : disp6b = 1'b1;
        6'b10_1110 : disp6b = 1'b1;
        6'b10_1111 : disp6b = 1'b1;

        6'b11_0000 : disp6b = 1'b0;
        6'b11_0001 : disp6b = curr_disp;
        6'b11_0010 : disp6b = curr_disp;
        6'b11_0011 : disp6b = 1'b1;
        6'b11_0100 : disp6b = curr_disp;
        6'b11_0101 : disp6b = 1'b1;
        6'b11_0110 : disp6b = 1'b1;
        6'b11_0111 : disp6b = 1'b1;
        6'b11_1000 : disp6b = 1'b0; // Special case
        6'b11_1001 : disp6b = 1'b1;
        6'b11_1010 : disp6b = 1'b1;
        6'b11_1011 : disp6b = 1'b1;
        6'b11_1100 : disp6b = 1'b1;
        6'b11_1101 : disp6b = 1'b1;
        6'b11_1110 : disp6b = 1'b1;
        6'b11_1111 : disp6b = 1'b1;

    endcase
end

// Compute Disparity of 4-bit sub-block
always @*
begin
    case (parallel_data10[3:0])

        4'b0000 : disp4b = 1'b0;
        4'b0001 : disp4b = 1'b0;
        4'b0010 : disp4b = 1'b0;
        4'b0011 : disp4b = 1'b1; // Special case
        4'b0100 : disp4b = 1'b0;
        4'b0101 : disp4b = disp6b;
        4'b0110 : disp4b = disp6b;
        4'b0111 : disp4b = 1'b1;
        4'b1000 : disp4b = 1'b0;
        4'b1001 : disp4b = disp6b;
        4'b1010 : disp4b = disp6b;
        4'b1011 : disp4b = 1'b1;
        4'b1100 : disp4b = 1'b0; // Special case
        4'b1101 : disp4b = 1'b1;
        4'b1110 : disp4b = 1'b1;
        4'b1111 : disp4b = 1'b1;

    endcase
end

// Get disparity of the current word by summing the number of 1's in the vector
always @(posedge rx_symbol_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
        parallel_data10_disp <= 4'h5;
    else
        parallel_data10_disp <= parallel_data10[0] +
                                parallel_data10[1] + 
                                parallel_data10[2] +
                                parallel_data10[3] +
                                parallel_data10[4] + 
                                parallel_data10[5] +
                                parallel_data10[6] +
                                parallel_data10[7] +
                                parallel_data10[8] +
                                parallel_data10[9];
end    

// Calculate disparity expected for next symbol based on current symbol
always @*
begin
    if (parallel_data10_disp == 4'h5)
        next_disp = out_disp;
    else if (parallel_data10_disp < 4'h5)
        next_disp = 1'b0;
    else
        next_disp = 1'b1;
end
    
always @(posedge rx_symbol_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        curr_disp   <= 1'b0; // Assume negative disparity on reset
        r_curr_disp <= 1'b0;
    end
    else
    begin
        curr_disp   <= disp4b;
        r_curr_disp <= curr_disp;
    end
end

// There was a disparity error if the incoming symbol was encoded for a different
//   disparity than the current running disparity; if an 10b8b error occurred, this
//   takes precedence
assign rx_err_disp = (out_disp != r_curr_disp) & ~rx_err_8b10;



endmodule

// -------------------------------------------------------------------------
//
//  PROJECT: PCI Express Verification Suite
//  COMPANY: Northwest Logic, Inc.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//             (c) Copyright 2009 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or
//  transmitted in any form or by any means, electronic or mechanical,
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module bfmp_rx_130b_to_128b (

    rst_n,                      // Asynchronous reset

    rx_serial_clk,              // Serial clock
    rx_symbol_clk,              // Symbol Clock

    rx_p,                       // Receive serial stream
    rx_n,                       //   ..
    analog_rx_elec_idle,        // 1 == Receive serial stream electrically idle

    rx_block_realign,
    rx_valid,                   // Set if synchronization lost
    rx_data_valid,               // Indicates a valid received byte
    rx_data,                    // 8-bit data value
    rx_start_block,             // Indicates start of a block (first byte)
    rx_sync_header,             // Indicates header on block
    rx_frame_error              // Error indication

);



// ----------------
// -- Parameters --
// ----------------

localparam  FIFO_ADDR_WIDTH         = 7;                        // FIFO address size (1 word == 1 bit)
localparam  FIFO_DEPTH              = (1 << FIFO_ADDR_WIDTH);   // Depth of FIFO

localparam                          STATE_UNALIGNED = 3'b001;
localparam                          STATE_ALIGNED   = 3'b010;
localparam                          STATE_LOCKED    = 3'b100;



// ----------------------
// -- Port Definitions --
// ----------------------

input                           rst_n;

input                           rx_serial_clk;
input                           rx_symbol_clk;

input                           rx_p;
input                           rx_n;
input                           analog_rx_elec_idle;

input                           rx_block_realign;
output                          rx_valid;
output                          rx_data_valid;
output  [7:0]                   rx_data;
output                          rx_start_block;
output  [1:0]                   rx_sync_header;
output                          rx_frame_error;



// ----------------
// -- Port Types --
// ----------------

wire                            rst_n;

wire                            rx_serial_clk;
wire                            rx_symbol_clk;

wire                            rx_p;
wire                            rx_n;
wire                            analog_rx_elec_idle;

wire                            rx_block_realign;
reg                             rx_valid;
reg                             rx_data_valid;
reg     [7:0]                   rx_data;
reg                             rx_start_block;
reg     [1:0]                   rx_sync_header;
wire                            rx_frame_error;



// -------------------
// -- Local Signals --
// -------------------

reg     [169:0]                     in_d;
reg                                 in_rx_p;

wire                                eie_os_detect;
wire                                sds_os_detect;

wire                                skp_os_start;
wire                                skp08_os_detect;
wire                                skp12_os_detect;
wire                                skp16_os_detect;
wire                                skp20_os_detect;
wire                                skp24_os_detect;
wire                                skp_os_end_detect;

reg     [2:0]                       state;

reg                                 in_skp;
reg     [1:0]                       hdr_ctr;
reg     [7:0]                       ctr;

wire                                err_eieos_loc;
wire                                err_sync_header;

reg                                 rx_in_en;
reg                                 rx_serial_start_block;
reg     [1:0]                       rx_serial_sync_header;

wire                                rx_serial_en;
wire                                rx_serial_data;

// Input Clock Domain
wire                                fifo_rst_n;

reg                                 in_en;
reg     [FIFO_ADDR_WIDTH:0]         in_addr;
reg                                 in_data;
reg                                 in_start_block;
reg     [1:0]                       in_sync_header;

wire    [FIFO_ADDR_WIDTH-1:0]       fifo_in_addr;

reg                                 fifo_rx_data        [FIFO_DEPTH-1:0];
reg                                 fifo_rx_start_block [FIFO_DEPTH-1:0];
reg     [1:0]                       fifo_rx_sync_header [FIFO_DEPTH-1:0];

// Output Clock Domain
reg     [FIFO_ADDR_WIDTH:0]         out_addr;
reg     [FIFO_ADDR_WIDTH:0]         out_in_addr;
reg                                 out_en;
reg     [6:0]                       out_data_valid_ctr;
wire                                out_data_valid;

wire                                diff_half;
wire    [FIFO_ADDR_WIDTH:0]         c_out_level;
wire                                fifo_empty;

genvar                              x;
wire    [FIFO_ADDR_WIDTH-1:0]       out_a               [7:0];

reg     [7:0]                       u_ctr;
reg     [1:0]                       u_hdr_ctr;
wire    [1:0]                       rx_hdr_ctr;


// ---------------
// -- Equations --
// ---------------

// -------------------------------------------------
// Convert the serial stream to parallel 8-bit data

always @(posedge rx_serial_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        in_d    <= 170'b0;
        in_rx_p <= 1'b0;
    end
    else
    begin
        if (analog_rx_elec_idle) begin
            in_d    <= 170'b0;
            in_rx_p <= 1'b0;
        end
        else begin
            in_rx_p <= rx_p;
            in_d    <= {rx_p, in_d[169:1]};
        end
    end
end

assign eie_os_detect = (in_d[169:40] == { {8{16'hff00}},      2'b01});
assign sds_os_detect = (in_d[169:40] == { {15{8'h55}}, 8'he1, 2'b01});

assign skp_os_start    = (in_d[169:170-(( 4*8)+2)] == {       { 4{8'haa}}, 2'b01}) & (ctr == 8'h60);
assign skp08_os_detect = (in_d[169:170-(( 5*8)+2)] == {8'he1, { 4{8'haa}}, 2'b01}) & (ctr == 8'h98);
assign skp12_os_detect = (in_d[169:170-(( 9*8)+2)] == {8'he1, { 8{8'haa}}, 2'b01}) & (ctr == 8'h78);
assign skp16_os_detect = (in_d[169:170-((13*8)+2)] == {8'he1, {12{8'haa}}, 2'b01}) & (ctr == 8'h58);
assign skp20_os_detect = (in_d[169:170-((17*8)+2)] == {8'he1, {16{8'haa}}, 2'b01}) & (ctr == 8'h38);
assign skp24_os_detect = (in_d[169:170-((21*8)+2)] == {8'he1, {20{8'haa}}, 2'b01}) & (ctr == 8'h18);

assign skp_os_end_detect = skp08_os_detect |
                           skp12_os_detect |
                           skp16_os_detect |
                           skp20_os_detect |
                           skp24_os_detect;

always @(posedge rx_serial_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        state <= STATE_UNALIGNED;
    end
    else
    begin
        case (state)
            
            STATE_UNALIGNED :
                if (eie_os_detect & ~out_en)    // Delay state transition if fifo is not yet empty
                    state <= STATE_ALIGNED;
            
            STATE_ALIGNED   :
                if (err_sync_header)
                    state <= STATE_UNALIGNED;
                else if (sds_os_detect)
                    state <= STATE_LOCKED;
            
            STATE_LOCKED    :
                if (err_sync_header)
                    state <= STATE_UNALIGNED;
            
            default         :
                state <= STATE_UNALIGNED;
            
        endcase
    end
end

always @(posedge rx_serial_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        in_skp       <= 1'b0;
        hdr_ctr      <= 2'd0;
        ctr          <= 8'd0;
    end
    else
    begin
        if (analog_rx_elec_idle) begin
            in_skp       <= 1'b0;
            hdr_ctr      <= 2'd0;
            ctr          <= 8'd0;
        end
        else begin
            // (State Aligned and not leaving state) &
            // (first 4 symbols of SKP OS Detected )
            if (((state == STATE_ALIGNED) | (state == STATE_LOCKED)) & ~err_sync_header & skp_os_start & (ctr == 8'd96))
                in_skp <= 1'b1;
            else if (skp_os_end_detect | (ctr == 8'd0))
                in_skp <= 1'b0;

            // (State Aligned and not leaving state and EIEOS Detected  ) |
            // (State Unaligned                       and EIEOS Detected) |
            // (State Aligned/Locked, reset ctr when the block ends     )
            if ( ((state == STATE_ALIGNED  ) & ~err_sync_header & eie_os_detect        ) |
                 ((state == STATE_UNALIGNED) &                    eie_os_detect        ) |
                 (((state == STATE_ALIGNED) | (state == STATE_LOCKED)) & (ctr == 8'd0) ) )
            begin
                hdr_ctr <= 2'd2;
                ctr     <= 8'd127;
            end
            // SKP OS start (hdr & first 8 symbols) detected in Aligned or Locked state
            else if (((state == STATE_ALIGNED) | (state == STATE_LOCKED)) & skp_os_start)
            begin
                hdr_ctr <= hdr_ctr;
                ctr     <= 8'd159; // Reset ctr to largest SKP set size (20 additional symbols after (skp_os_start == 1))
            end
            else if (in_skp)
            begin
                hdr_ctr <= hdr_ctr;

                // Reset counter to end after 3 remaining symbols of the SKP OS
                if (skp_os_end_detect)
                    ctr <= 8'd23;
                else if (hdr_ctr == 2'd0)
                    ctr <= ctr - 8'd1;
            end
            else if ((state == STATE_ALIGNED) | (state == STATE_LOCKED))
            begin
                if (hdr_ctr != 2'd0)
                    hdr_ctr <= hdr_ctr - 2'd1;

                if (hdr_ctr == 2'd0)
                    ctr <= ctr - 8'd1;
            end
            else
            begin
                hdr_ctr <= 2'd0;
                ctr     <= 8'd0;
            end
        end
    end
end

// EIEOS found at different than expected location
assign err_eieos_loc   = (state == STATE_ALIGNED) & eie_os_detect & ~((hdr_ctr == 2'd0) & (ctr == 8'd0));

// Invalid sync header detected
assign err_sync_header = (((state == STATE_ALIGNED) | (state == STATE_LOCKED)) & (hdr_ctr == 2'd1) & ~((in_d[169:168] == 2'b01) | (in_d[169:168] == 2'b10))) | rx_block_realign;
`ifdef SIMULATION

reg     [2:0]   r_state;
always @(posedge rx_serial_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
        r_state <= STATE_UNALIGNED;
    else begin
        r_state <= state;
    end
end

reg mask_rx_assert = 1'b0;

always @(posedge rx_serial_clk)
begin
    if (rst_n == 1'b1)
    begin
        if (!$test$plusargs("pcie_traffic_msgs_off"))
        begin
            if (state !== r_state)
            begin
                case (state)
                    STATE_UNALIGNED : $display ("%m : INFO : PHY 130b Data Stream Alignment/Lock State Machine Change: New state == UNALIGNED");
                    STATE_ALIGNED   : $display ("%m : INFO : PHY 130b Data Stream Alignment/Lock State Machine Change: New state == ALIGNED");
                    STATE_LOCKED    : $display ("%m : INFO : PHY 130b Data Stream Alignment/Lock State Machine Change: New state == LOCKED");
                    default         : $display ("%m : INFO : PHY 130b Data Stream Alignment/Lock State Machine Change: New state == UNDEFINED");
                endcase
            end
        end

        if (err_eieos_loc)
        begin
            $display ("%m : WARNING : While aligned (not locked), an EIE OS was found at a different location than expected (time %t)", $time);
            $display ("%m :           Adjusting alignment; the data stream will be temporarily disrupted");
        end

        if (err_sync_header & ~analog_rx_elec_idle & ~mask_rx_assert)
        begin
            $display ("%m : WARNING : While aligned or locked an invalid sync header was detected (time %t)", $time);
            $display ("%m :           This is an indication that the data stream has lost lock");
        end
    end
end
`endif

always @(posedge rx_serial_clk or negedge rst_n) 
begin
    if (rst_n == 1'b0) 
    begin
        u_hdr_ctr <= 2'd0;
        u_ctr     <= 8'h0;
    end
    else begin
        if ((state == STATE_ALIGNED) & (hdr_ctr == 2'd1)) begin
            u_hdr_ctr <= 2'd0;
            u_ctr     <= 8'd127;
        end
        else if ((u_hdr_ctr != 2'd0) & rx_in_en) 
            u_hdr_ctr <= u_hdr_ctr - 2'd1;
        else if ((u_hdr_ctr == 2'd0) & (u_ctr != 8'd0) & rx_in_en)
            u_ctr     <= u_ctr - 8'd1;
        else if ((u_hdr_ctr == 2'd0) & (u_ctr == 8'd0) & rx_in_en) begin
            u_hdr_ctr <= 2'd2;
            u_ctr     <= 8'd127;
        end
    end
end

assign  rx_hdr_ctr = (state == STATE_UNALIGNED) ? u_hdr_ctr : hdr_ctr;

always @(posedge rx_serial_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        rx_in_en              <= 1'b0;
        rx_serial_start_block <= 1'b0;
        rx_serial_sync_header <= 2'b00;
    end
    else
    begin
        if (analog_rx_elec_idle | ~fifo_rst_n) begin
            rx_in_en              <= 1'b0;
            rx_serial_start_block <= 1'b0;
            rx_serial_sync_header <= 2'b00;
        end
        else begin
            // Don't enable FIFO until the start of a block is detected
            if (hdr_ctr == 2'd1)
                rx_in_en <= 1'b1;

            rx_serial_start_block <= (rx_hdr_ctr == 2'd1);
            rx_serial_sync_header <= (rx_hdr_ctr == 2'd1) ? in_d[169:168] : 2'b00;
        end
    end
end

assign rx_serial_en   = (rx_hdr_ctr == 2'd0);
assign rx_serial_data = in_rx_p;



// ------------------
// Input Clock Domain

assign fifo_rst_n = ((state == STATE_ALIGNED) | (state == STATE_LOCKED));
//assign fifo_rst_n = 1'b1;

always @(posedge rx_serial_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        in_en          <= 1'b0;
        in_addr        <= {(FIFO_ADDR_WIDTH+1){1'b0}};
        in_data        <= 1'b0;
        in_start_block <= 1'b0;
        in_sync_header <= 2'b00;
    end
    else
    begin
        if ((fifo_rst_n == 1'b0) && ~out_en) begin
            in_en          <= 1'b0;
            in_addr        <= {(FIFO_ADDR_WIDTH+1){1'b0}};
            in_data        <= 1'b0;
            in_start_block <= 1'b0;
            in_sync_header <= 2'b00;
        end
        else begin
            in_en <= rx_in_en & rx_serial_en;
            // Keeping one extra address bit to tell full from empty when doing an address compare
            if (in_en)
                in_addr <= in_addr + {{FIFO_ADDR_WIDTH{1'b0}}, 1'b1};
            in_data        <= rx_serial_data;
            in_start_block <= rx_serial_start_block;
            in_sync_header <= rx_serial_sync_header;
        end
    end
end

// Drop the extra address bit when accessing the FIFO
assign fifo_in_addr = in_addr[FIFO_ADDR_WIDTH-1:0];

// Write input data into the FIFO
always @(posedge rx_serial_clk)
begin
    if (in_en)
    begin
        fifo_rx_data        [fifo_in_addr] <= in_data;
        fifo_rx_start_block [fifo_in_addr] <= in_start_block;
        fifo_rx_sync_header [fifo_in_addr] <= in_sync_header;
    end
end



// -------------------
// Output Clock Domain

// Manage the output address pointer
always @(posedge rx_symbol_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        out_addr           <= {(FIFO_ADDR_WIDTH+1){1'b0}};
        out_in_addr        <= {(FIFO_ADDR_WIDTH+1){1'b0}};
        out_en             <= 1'b0;
    end
    else
    begin
        // Keeping one extra address bit to tell full from empty when doing an address compare
        if (out_en & out_data_valid & ~fifo_empty)                       // output data
            out_addr <= out_addr + {{(FIFO_ADDR_WIDTH-3){1'b0}}, 4'd8};
        else if (~out_en)                                                // reset when disabled
            out_addr <= {(FIFO_ADDR_WIDTH+1){1'b0}};

        // Note: Behavioral, so not synchronizing as must be done
        //   in a hardware implementation
        out_in_addr <= in_addr;

        // Enable output once the FIFO has filled half-way
        if (c_out_level >= ((FIFO_DEPTH/2) - 8))
            out_en     <= 1'b1;
        else if (fifo_empty)
            out_en     <= 1'b0;

    end
end

// Use free running counter for data_valid
// (want this to run even if FIFO is reset due to sync header errors)
always @(posedge rx_symbol_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        out_data_valid_ctr <= 7'd0;
    end
    else
    begin
        if (out_data_valid_ctr == 7'd64)
            out_data_valid_ctr <= 7'd0;
        else
            out_data_valid_ctr <= out_data_valid_ctr + 7'd1;
    end
end

assign out_data_valid = (out_data_valid_ctr != 7'd64);

// If the extra address bit being carried in each of the FIFO addresses have
//   different values, then the write address has wrapped relative to the read
//   address and 2^ADDR_WIDTH must be added to the write address in order to
//   determining the true level
assign diff_half   = (out_in_addr[FIFO_ADDR_WIDTH] != out_addr[FIFO_ADDR_WIDTH]);
assign c_out_level = {diff_half, out_in_addr[FIFO_ADDR_WIDTH-1:0]} - {1'b0, out_addr[FIFO_ADDR_WIDTH-1:0]};
assign fifo_empty  = (c_out_level <= 8);

generate
    for (x=0; x<8; x=x+1)
    begin : out_symbols
        assign out_a[x] = out_addr[FIFO_ADDR_WIDTH-1:0] + x;

        always @(posedge rx_symbol_clk or negedge rst_n)
        begin
            if (rst_n == 1'b0)
                rx_data[x] <= 1'b0;
            else
            begin
                if ((out_data_valid == 1'b0) || (out_en == 1'b0))
                    rx_data[x] <= 1'b0;
                else
                    rx_data[x] <= fifo_rx_data[out_a[x]];
            end
        end
    end
endgenerate

always @(posedge rx_symbol_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        rx_valid       <= 1'b0;
        rx_start_block <= 1'b0;
        rx_sync_header <= 2'b00;
        rx_data_valid  <= 1'b0;
    end
    else
    begin
        if ((out_data_valid == 1'b0) || (out_en == 1'b0))
        begin
            rx_start_block <= 1'b0;
            rx_sync_header <= 2'b00;
        end
        else
        begin
            rx_start_block <= fifo_rx_start_block[out_addr[FIFO_ADDR_WIDTH-1:0]];
            rx_sync_header <= fifo_rx_sync_header[out_addr[FIFO_ADDR_WIDTH-1:0]];
        end

        rx_valid       <= out_en;
        rx_data_valid  <= out_data_valid;
    end
end

assign rx_frame_error = 1'b0;                // Error indication

endmodule
// -------------------------------------------------------------------------
//
//  PROJECT: PCI Express Verification Suite
//  COMPANY: Northwest Logic, Inc.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//             (c) Copyright 2009 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or
//  transmitted in any form or by any means, electronic or mechanical,
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module bfmp_phy_rx (

    rst_n,
    rx_lock,
    rate,

    in_rx_clk,        
    in_rx_valid,      
    in_rx_data_valid,  
    in_rx_data,       
    in_rx_data_k,     
    in_rx_err_disp,   
    in_rx_err_8b10,   
    in_rx_start_block,
    in_rx_sync_header,
    in_rx_frame_error,

    out_rx_clk,        
    out_rx_clk_en,
    out_rx_valid,      
    out_rx_data_valid,  
    out_rx_data,       
    out_rx_data_k,    
    out_rx_err_disp,   
    out_rx_err_8b10,   
    out_rx_start_block,
    out_rx_sync_header,
    out_rx_frame_error,
    out_rx_err_uflow,  
    out_rx_err_oflow  

);
    

// ----------------
// -- Parameters --
// ----------------

parameter   LANE_NUMBER                     = 0;
parameter   PHY_K_WIDTH                     = 2;                            // Byte width of out_* port


localparam  FIFO_ADDR_WIDTH                 =  (PHY_K_WIDTH == 4) ? 6 :     // FIFO address size (1 word == 1 symbol)
                                              ((PHY_K_WIDTH == 2) ? 5 : 4); // Want FIFO to hold 16 * PHY_K_WIDTH symbols
localparam  FIFO_DEPTH                      = 1 << FIFO_ADDR_WIDTH;         // Depth of FIFO
localparam  FIFO_THRESH                     = (FIFO_DEPTH/2) - PHY_K_WIDTH; // FIFO threshold to hit before enabling FIFO output
localparam  FIFO_MAX                        = FIFO_DEPTH-1-2*(PHY_K_WIDTH)-8;
localparam  FIFO_MIN                        = PHY_K_WIDTH*2+1+8;

localparam  K_COM = 9'h1bc;
localparam  K_SKP = 9'h11c;


// ----------------------
// -- Port Definitions --
// ----------------------

input                                       rst_n;              
input                                       rx_lock;
input   [1:0]                               rate;

input                                       in_rx_clk;        
input                                       in_rx_valid;      
input                                       in_rx_data_valid;  
input   [7:0]                               in_rx_data;       
input                                       in_rx_data_k;     
input                                       in_rx_err_disp;   
input                                       in_rx_err_8b10;   
input                                       in_rx_start_block;
input   [1:0]                               in_rx_sync_header;
input                                       in_rx_frame_error;

input                                       out_rx_clk;        
input                                       out_rx_clk_en;
output  [PHY_K_WIDTH-1:0]                   out_rx_valid;      
output  [PHY_K_WIDTH-1:0]                   out_rx_data_valid;  
output  [(PHY_K_WIDTH*8)-1:0]               out_rx_data;       
output  [PHY_K_WIDTH-1:0]                   out_rx_data_k;    
output  [PHY_K_WIDTH-1:0]                   out_rx_err_disp;   
output  [PHY_K_WIDTH-1:0]                   out_rx_err_8b10;   
output  [PHY_K_WIDTH-1:0]                   out_rx_start_block;
output  [(PHY_K_WIDTH*2)-1:0]               out_rx_sync_header;
output  [PHY_K_WIDTH-1:0]                   out_rx_frame_error;
output  [PHY_K_WIDTH-1:0]                   out_rx_err_uflow;  
output  [PHY_K_WIDTH-1:0]                   out_rx_err_oflow;  



// ----------------
// -- Port Types --
// ----------------

wire                                        rst_n;              
wire                                        rx_lock;
wire    [1:0]                               rate;

wire                                        in_rx_clk;        
wire                                        in_rx_valid;      
wire                                        in_rx_data_valid;  
wire    [7:0]                               in_rx_data;       
wire                                        in_rx_data_k;     
wire                                        in_rx_err_disp;   
wire                                        in_rx_err_8b10;   
wire                                        in_rx_start_block;
wire    [1:0]                               in_rx_sync_header;
wire                                        in_rx_frame_error;

wire                                        out_rx_clk;        
wire                                        out_rx_clk_en;
reg     [PHY_K_WIDTH-1:0]                   out_rx_valid;      
reg     [PHY_K_WIDTH-1:0]                   out_rx_data_valid;  
reg     [(PHY_K_WIDTH*8)-1:0]               out_rx_data;       
reg     [PHY_K_WIDTH-1:0]                   out_rx_data_k;    
reg     [PHY_K_WIDTH-1:0]                   out_rx_err_disp;   
reg     [PHY_K_WIDTH-1:0]                   out_rx_err_8b10;   
reg     [PHY_K_WIDTH-1:0]                   out_rx_start_block;
reg     [(PHY_K_WIDTH*2)-1:0]               out_rx_sync_header;
reg     [PHY_K_WIDTH-1:0]                   out_rx_frame_error;
wire    [PHY_K_WIDTH-1:0]                   out_rx_err_uflow;
wire    [PHY_K_WIDTH-1:0]                   out_rx_err_oflow;  



// -------------------
// -- Local Signals --
// -------------------

wire                                        rate_8g;
reg     [1:0]                               d1_in_rate;
reg     [1:0]                               d2_in_rate;
reg                                         in_rate_rst_n;
reg     [1:0]                               d1_out_rate;
reg     [1:0]                               d2_out_rate;
reg     [1:0]                               out_rate_rst_ctr;
reg                                         out_rate_rst_n;

reg     [FIFO_ADDR_WIDTH-1:0]               fifo_thresh;
reg     [FIFO_ADDR_WIDTH-1:0]               fifo_max;
reg     [FIFO_ADDR_WIDTH-1:0]               fifo_min;
reg                                         rand_skp_ovrrd;

reg                                         debug;
reg                                         last_was_com;
reg                                         last_was_skp;
reg     [FIFO_ADDR_WIDTH-1:0]               a;
reg     [FIFO_ADDR_WIDTH-1:0]               b;
integer                                     k;
reg     [FIFO_ADDR_WIDTH-1:0]               out_a           [PHY_K_WIDTH-1:0];
reg     [3:0]                               rpt_ctr;
reg     [2:0]                               skp_ctr;
reg     [3:0]                               r_rpt_ctr;
reg     [2:0]                               r_skp_ctr;

// Input Clock Domain
wire                                        flush_rst_n;
reg                                         in_flush;
reg                                         in_flush_done;
reg                                         out_flush_done_s1;
reg                                         out_flush_done_s2;
reg                                         out_flush_done_s3;
reg                                         out_flush_done_s4;
reg                                         in_flush_wait;
reg                                         out_flush_wait_s1;
reg                                         out_flush_wait_s2;

reg     [FIFO_ADDR_WIDTH:0]                 in_addr;
wire    [FIFO_ADDR_WIDTH-1:0]               fifo_in_addr;

reg                                         fifo_rx_valid       [FIFO_DEPTH-1:0];
reg                                         fifo_rx_data_valid  [FIFO_DEPTH-1:0];
reg     [7:0]                               fifo_rx_data        [FIFO_DEPTH-1:0];
reg                                         fifo_rx_data_k      [FIFO_DEPTH-1:0];
reg                                         fifo_rx_err_disp    [FIFO_DEPTH-1:0];
reg                                         fifo_rx_err_8b10    [FIFO_DEPTH-1:0];
reg                                         fifo_rx_start_block [FIFO_DEPTH-1:0];
reg     [1:0]                               fifo_rx_sync_header [FIFO_DEPTH-1:0];
reg                                         fifo_rx_frame_error [FIFO_DEPTH-1:0];
reg                                         fifo_rx_skp_start   [FIFO_DEPTH-1:0];

// Output Clock Domain
reg                                         in_flush_s1;
reg                                         in_flush_s2;
reg                                         in_flush_s3;
reg                                         in_flush_s4;
reg                                         out_in_addr_invalid;

reg     [FIFO_ADDR_WIDTH:0]                 out_addr;
wire    [FIFO_ADDR_WIDTH:0]                 out_in_addr;
reg     [FIFO_ADDR_WIDTH:0]                 out_level;
reg                                         out_en;
reg [3:0]                                   out_clr_timer;
reg                                         out_flush_done;
reg                                         out_flush_wait;

wire                                        diff_half;
wire    [FIFO_ADDR_WIDTH:0]                 c_out_level;

wire    [31:0]                              out_bytes;

genvar                                      i;
genvar                                      j;

// ---------------
// -- Equations --
// ---------------

assign rate_8g = rate[1];

always @(posedge out_rx_clk or negedge rst_n)
begin
    if (rst_n == 1'b0) 
    begin
        d1_out_rate <= 2'b0;
        d2_out_rate <= 2'b0;
        out_rate_rst_ctr <= 2'h0;
        out_rate_rst_n <= 1'b0;
    end
    else begin
        d1_out_rate <= rate;
        d2_out_rate <= d1_out_rate;
        if (d1_out_rate != d2_out_rate) begin
            out_rate_rst_ctr <= 2'h3;
            out_rate_rst_n   <= 1'b0;
        end
        else if (~out_rate_rst_n & (out_rate_rst_ctr != 0))
            out_rate_rst_ctr <= out_rate_rst_ctr - 2'h1;
        else
            out_rate_rst_n <= 1'b1;
    end
end

always @(posedge in_rx_clk or negedge rst_n)
begin
    if (rst_n == 1'b0) 
    begin
        d1_in_rate <= 2'b0;
        d2_in_rate <= 2'b0;
        in_rate_rst_n <= 1'b0;
    end
    else begin
        d1_in_rate <= rate;
        d2_in_rate <= d1_in_rate;
        if (d1_in_rate != d2_in_rate)
            in_rate_rst_n <= 1'b0;
        else
            in_rate_rst_n <= 1'b1;
    end
end


initial
    begin
        debug = 0; // Disable Debug Messages by default
        rand_skp_ovrrd = 1'b0;
    end

always @(posedge out_rx_clk or negedge rst_n) begin
    if (rst_n == 1'b0) begin
        fifo_thresh <= (FIFO_DEPTH/4) - PHY_K_WIDTH;
        fifo_max    <= (FIFO_DEPTH/2)-1 - PHY_K_WIDTH - 2;
        fifo_min    <= PHY_K_WIDTH + 1 + 2;
    end
    else begin
        fifo_thresh <= (rate_8g) ? (FIFO_DEPTH/2) - PHY_K_WIDTH   : (FIFO_DEPTH/4) - PHY_K_WIDTH;
        fifo_max    <= (rate_8g) ? FIFO_DEPTH-1-2*(PHY_K_WIDTH)-8 : (FIFO_DEPTH/2)-1 - PHY_K_WIDTH - 2;
        fifo_min    <= (rate_8g) ? PHY_K_WIDTH*2+1+8              : PHY_K_WIDTH + 1 + 2;
    end
end

assign out_bytes = PHY_K_WIDTH;

// ------------------
// Input Clock Domain

// Keeping one extra address bit to tell full from empty when doing an address compare
always @(posedge in_rx_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
        in_addr <= {(FIFO_ADDR_WIDTH+1){1'b0}};
    else if (in_rate_rst_n == 1'b0)
        in_addr <= {(FIFO_ADDR_WIDTH+1){1'b0}};
    else if (in_flush & in_flush_done)
        // reset when output address counter is cleared
        in_addr <= {(FIFO_ADDR_WIDTH+1){1'b0}};
    else if (in_rx_valid & rx_lock & ~in_flush & ~in_flush_wait)
        // advance input counter if we are locked and valid, and we are not flushing
        in_addr <= in_addr + {{FIFO_ADDR_WIDTH{1'b0}}, 1'b1};
end

always @(posedge in_rx_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        out_flush_done_s1 <= 1'b1;
        out_flush_done_s2 <= 1'b1;
        out_flush_done_s3 <= 1'b1;
        out_flush_done_s4 <= 1'b1;
        in_flush_done     <= 1'b1;
        out_flush_wait_s1 <= 1'b0;
        out_flush_wait_s2 <= 1'b0;
        in_flush_wait     <= 1'b0;
    end
    else
    begin
        out_flush_done_s1 <= out_flush_done;
        out_flush_done_s2 <= out_flush_done_s1;
        out_flush_done_s3 <= out_flush_done_s2;
        out_flush_done_s4 <= out_flush_done_s3;
        in_flush_done     <= out_flush_done_s4;
        out_flush_wait_s1 <= out_flush_wait;
        out_flush_wait_s2 <= out_flush_wait_s1;
        in_flush_wait     <= out_flush_wait_s2;
   end
end

assign flush_rst_n = rst_n & rx_lock;

always @(posedge in_rx_clk or negedge flush_rst_n)
begin
    if (flush_rst_n == 1'b0)
        in_flush <= 1'b1;
    else if (in_flush_done)
        in_flush <= 1'b0;
    else if (~(in_rx_valid & rx_lock))
        in_flush <= 1'b1;
end

// Drop the extra address bit when accessing the FIFO
assign fifo_in_addr = in_addr[FIFO_ADDR_WIDTH-1:0];

// Write input data into the FIFO
always @(posedge in_rx_clk)
begin
    if (in_rx_valid & rst_n & rx_lock & ~in_flush & ~in_flush_wait)
    begin
        fifo_rx_valid       [fifo_in_addr] <= in_rx_valid;      
        fifo_rx_data_valid  [fifo_in_addr] <= in_rx_data_valid;
        fifo_rx_data        [fifo_in_addr] <= in_rx_data;       
        fifo_rx_data_k      [fifo_in_addr] <= in_rx_data_k;     
        fifo_rx_err_disp    [fifo_in_addr] <= in_rx_err_disp;   
        fifo_rx_err_8b10    [fifo_in_addr] <= in_rx_err_8b10;   
        fifo_rx_start_block [fifo_in_addr] <= in_rx_start_block;
        fifo_rx_sync_header [fifo_in_addr] <= in_rx_sync_header;
        fifo_rx_frame_error [fifo_in_addr] <= in_rx_frame_error;
    end
end

// ---------------
// Detect SKP sets
always@(posedge in_rx_clk or negedge rst_n) begin
    if (rst_n == 1'b0)
    begin
        last_was_com <= 1'b0;
        last_was_skp <= 1'b0;
    end
    else if ((rx_lock == 1'b0) || (in_flush == 1'b1) || (in_flush_wait == 1'b1) || (in_rx_valid == 1'b0))
    begin
        last_was_com <= 1'b0;
        last_was_skp <= 1'b0;
    end
    else begin
        // COM detection
        if (({in_rx_data_k, in_rx_data} == K_COM))
            last_was_com <= 1'b1;
        else
            last_was_com <= 1'b0;
        // 8G SKP Start detection
        if ((in_rx_data == 8'haa) & (in_rx_sync_header == 2'h1) & in_rx_start_block)
            last_was_skp <= 1'b1;
        else
            last_was_skp <= 1'b0;

        // 128b/130b SKP detection
        if ((in_rx_data == 8'hAA) & last_was_skp)
            fifo_rx_skp_start[fifo_in_addr] <= 1'b1;
        // 8b/10b SKP detection
        else if (({in_rx_data_k, in_rx_data} == K_SKP) & last_was_com)
            fifo_rx_skp_start[fifo_in_addr] <= 1'b1;
        else
            fifo_rx_skp_start[fifo_in_addr] <= 1'b0;
    end
end  



// -------------------
// Output Clock Domain

// Identify when the input address pointer is valid
// after a flush
always @(posedge out_rx_clk or negedge rst_n)
    if (rst_n == 1'b0)
    begin
        in_flush_s1         <= 1'b1;
        in_flush_s2         <= 1'b1;
        in_flush_s3         <= 1'b1;
        in_flush_s4         <= 1'b1;
        out_in_addr_invalid <= 1'b1;
    end
    else
    begin
        in_flush_s1         <= in_flush;
        in_flush_s2         <= in_flush_s1;
        in_flush_s3         <= in_flush_s2;
        in_flush_s4         <= in_flush_s3;
        out_in_addr_invalid <= in_flush_s4;
    end

// Manage the output address pointer
always @(posedge out_rx_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        out_addr           <= {(FIFO_ADDR_WIDTH+1){1'b0}};
        out_level          <= {(FIFO_ADDR_WIDTH+1){1'b0}};
        out_en             <= 1'b0;
        out_clr_timer      <= 4'h0;
        out_flush_done     <= 1'b1;
        out_flush_wait     <= 1'b0;
    end
    else if (out_rate_rst_n == 1'b0) begin
        out_addr           <= {(FIFO_ADDR_WIDTH+1){1'b0}};
        out_level          <= {(FIFO_ADDR_WIDTH+1){1'b0}};
        out_en             <= 1'b0;
        out_clr_timer      <= 4'h0;
        out_flush_done     <= 1'b1;
        out_flush_wait     <= 1'b0;
    end
    else begin
        if (out_rx_clk_en)
        begin
            // Keeping one extra address bit to tell full from empty when doing an address compare
            if (out_en & c_out_level > PHY_K_WIDTH)
                out_addr <= out_addr + b;
            else if (~out_en)
                out_addr <= {(FIFO_ADDR_WIDTH+1){1'b0}};

            if ((out_en == 1'b1) | (out_clr_timer == 4'b0))
                out_level <= c_out_level;

            if ((out_en == 1'b1) && (c_out_level <= PHY_K_WIDTH))
            begin
                // init clear timer when we are going empty
                // count 15 clocks so output and input addresses can
                // be reset cleanly before processing output level
                // Also assert out_flush_done when going empty
                // so the input address counter can be cleared
                out_clr_timer  <= 4'hf;
                out_flush_done <= 1'b1;
                out_flush_wait <= 1'b1;
            end
            else if (out_en == 1'b1)
            begin
                out_clr_timer  <= 4'h0;
                out_flush_done <= 1'b0;
                out_flush_wait <= 1'b0;
            end
            else if ((out_en == 1'b0) && (out_clr_timer > 4'h0))
                out_clr_timer <= out_clr_timer - 4'h1;
            else if (out_clr_timer == 4'h0)
                out_flush_wait <= 1'b0;

            if ((out_level >= fifo_thresh) && (out_in_addr_invalid == 1'b0))
                // Enable output once the FIFO has filled half-way
                // Don't set out_en if the input address is not yet valid after a flush
                out_en <= 1'b1;
            else if (c_out_level <= PHY_K_WIDTH)
                // Stop when underflowing
                out_en <= 1'b0;
        end
    end
end

bfmp_phy_gray_sync_bus #(
    .WIDTH           (FIFO_ADDR_WIDTH + 1),
    .REGISTER_OUTPUT (0)
) fifo_addr (
    .d_rst_n    (rst_n),
    .d_clk      (in_rx_clk),
    .d_clr      (1'b0),
    .d          (in_addr),

    .q_rst_n    (rst_n),
    .q_clk      (out_rx_clk),
    .q_clr      (1'b0),
    .q          (out_in_addr)
);

// If the extra address bit being carried in each of the FIFO addresses have
//   different values, then the write address has wrapped relative to the read
//   address and 2^ADDR_WIDTH must be added to the write address in order to
//   determining the true level
assign diff_half   = (out_in_addr[FIFO_ADDR_WIDTH] != out_addr[FIFO_ADDR_WIDTH]);
assign c_out_level = {diff_half, out_in_addr[FIFO_ADDR_WIDTH-1:0]} - {1'b0, out_addr[FIFO_ADDR_WIDTH-1:0]};


always @(out_addr or r_rpt_ctr or out_en) begin
    a = out_addr[FIFO_ADDR_WIDTH-1:0];
    b = 0;
    rpt_ctr = r_rpt_ctr;
    skp_ctr = r_skp_ctr;
    for (k=0; k<out_bytes; k=k+1)
        out_a[k] = a;
    k = 0;
    while ((k<out_bytes) & out_en) begin
        if (fifo_rx_data_valid[a]) begin
            // Dropping SKP symbols
            if (skp_ctr != 3'h0) begin
                a = a + 1;
                b = b + 1;
                skp_ctr = skp_ctr - 3'h1;
            end
            // Starting 8b/10b SKP add
            else if (fifo_rx_skp_start[a] & fifo_rx_data_k[a] & (rpt_ctr == 4'h0) & (out_level < fifo_thresh)) begin
                rpt_ctr = 4'h1;
                out_a[k] = a;
                k = k + 1;
            end
                // Starting 128b/130b SKP add
            else if (fifo_rx_skp_start[a] & ~fifo_rx_data_k[a] & (rpt_ctr == 4'h0) & (out_level < fifo_thresh)) begin
                rpt_ctr = 4'h4;
                out_a[k] = a;
                k = k + 1;
            end
                // End of SKP add
            else if (fifo_rx_skp_start[a] & (rpt_ctr != 4'h0)) begin
                out_a[k] = a;
                rpt_ctr = rpt_ctr - 4'h1;
                if (rpt_ctr == 4'h0) begin
                    a = a + 1;
                    b = b + 1;
                end
                k = k + 1;
            end
                // Start of 8b/10b SKP drop
            else if (fifo_rx_skp_start[a] & fifo_rx_data_k[a] & (out_level > fifo_thresh)) begin
                skp_ctr = 3'h0;
//                out_a[k] = a + 1;
//                a = a + 2;
//                b = b + 2;
//                k = k + 1;
                a = a + 1;
                b = b + 1;
            end
                // Start of 128b/130b SKP drop
            else if (fifo_rx_skp_start[a] & ~fifo_rx_data_k[a] & (out_level > fifo_thresh)) begin
                skp_ctr = 3'h3;
                a = a + 1;
                b = b + 1;
            end
            else begin
                out_a[k] = a;
                a = a + 1;
                b = b + 1;
                k = k + 1;
            end
        end
        else begin
            out_a[k] = a;
            a = a + 1;
            b = b + 1;
            k = k + 1;
        end
    end
end

always @(posedge out_rx_clk or negedge rst_n) begin
    if (rst_n == 1'b0) begin
        r_rpt_ctr <= 4'h0;
        r_skp_ctr <= 3'h0;
    end
    else if (out_rx_clk_en) begin
        if (out_en) begin
            r_rpt_ctr <= rpt_ctr;
            r_skp_ctr <= skp_ctr;
        end else begin
            r_rpt_ctr <= 4'h0;
            r_skp_ctr <= 3'h0;
        end
    end
end

generate
    for (i=0; i<PHY_K_WIDTH; i=i+1)
    begin : out_symbols
        always @(posedge out_rx_clk or negedge rst_n)
        begin
            if (rst_n == 1'b0)
            begin
                out_rx_valid        [  i              ] <= 1'b0;      
                out_rx_data_valid   [  i              ] <= 1'b1; 
                out_rx_data         [((i+1)*8)-1:(i*8)] <= 8'hFE; 
                out_rx_data_k       [  i              ] <= 1'b1; 
                out_rx_err_disp     [  i              ] <= 1'b0; 
                out_rx_err_8b10     [  i              ] <= 1'b0; 
                out_rx_start_block  [  i              ] <= 1'b0; 
                out_rx_sync_header  [((i+1)*2)-1:(i*2)] <= 2'b00; 
                out_rx_frame_error  [  i              ] <= 1'b0; 
            end
            else if (out_rx_clk_en)
            begin
                // Equation Below Reduces to just out_en == 0 when not
                // using HALF_PHY_DATA_WIDTH_SUPPORT
                if (out_en == 1'b0 || ((i >= out_bytes)))
                begin
                    out_rx_valid        [  i              ] <= 1'b0;      
                    out_rx_data_valid   [  i              ] <= ~out_en; 
                    out_rx_data         [((i+1)*8)-1:(i*8)] <= 8'hFE; 
                    out_rx_data_k       [  i              ] <= 1'b1; 
                    out_rx_err_disp     [  i              ] <= 1'b0; 
                    out_rx_err_8b10     [  i              ] <= 1'b0; 
                    out_rx_start_block  [  i              ] <= 1'b0; 
                    out_rx_sync_header  [((i+1)*2)-1:(i*2)] <= 2'b00; 
                    out_rx_frame_error  [  i              ] <= 1'b0; 
                end
                else
                begin
                    out_rx_valid        [  i              ] <= fifo_rx_valid       [out_a[i]];  
                    out_rx_data_valid   [  i              ] <= fifo_rx_data_valid  [out_a[i]];
                    out_rx_data         [((i+1)*8)-1:(i*8)] <= fifo_rx_data        [out_a[i]];
                    out_rx_data_k       [  i              ] <= fifo_rx_data_k      [out_a[i]];
                    out_rx_err_disp     [  i              ] <= fifo_rx_err_disp    [out_a[i]];
                    out_rx_err_8b10     [  i              ] <= fifo_rx_err_8b10    [out_a[i]];
                    out_rx_start_block  [  i              ] <= fifo_rx_start_block [out_a[i]];
                    out_rx_sync_header  [((i+1)*2)-1:(i*2)] <= fifo_rx_sync_header [out_a[i]];
                    out_rx_frame_error  [  i              ] <= fifo_rx_frame_error [out_a[i]];
                end
            end
        end
    end
endgenerate

// This module does not support different frequency clocks, so can't overflow/underflow
assign out_rx_err_uflow = {PHY_K_WIDTH{1'b0}};
assign out_rx_err_oflow = {PHY_K_WIDTH{1'b0}};



endmodule

// --------------------------------------------------------------------------
//
//  PROJECT:             PCI Core
//  COMPANY:             Northwest Logic, Inc.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//                 Copyright 2010 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or
//  transmitted in any form or by any means, electronic or mechanical,
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

// -------------------------------------------------------------------------
//
//  FUNCTIONAL DESCRIPTION
//
//  Synchronizes an entire bus using Gray Codes
//    Latency is 1 d_clk and 3-4 q_clks depending upon whether
//    the current or last state of d was caught by the synchronizing
//    operation
//
//  For proper operation, the input bus, d, must only increment,
//    decrement, or stay the same on any given clock cycle.  This is
//    because Gray coding ensures that exactly one bit will change in
//    any increment or decrement, but makes no guarantee about other
//    add or subtract conditions.
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module bfmp_phy_gray_sync_bus (
    d_rst_n,
    d_clk,
    d_clr,
    d,

    q_rst_n,
    q_clk,
    q_clr,
    q
);



// ----------------
// -- Parameters --
// ----------------

parameter   WIDTH           = 7; // Valid range: 1 to 16
parameter   REGISTER_OUTPUT = 1; // Set for registered output, clear for combinatorial output



// -----------------------
// -- Port Declarations --
// -----------------------

input                       d_rst_n;
input                       d_clk;
input                       d_clr;
input     [WIDTH-1:0]       d;

input                       q_rst_n;
input                       q_clk;
input                       q_clr;
output     [WIDTH-1:0]      q;



// ----------------
// -- Port Types --
// ----------------

wire                        d_rst_n;
wire                        d_clk;
wire                        d_clr;
wire     [WIDTH-1:0]        d;

wire                        q_rst_n;
wire                        q_clk;
wire                        q_clr;
wire    [WIDTH-1:0]         q;



// ---------------------
// -- Local Variables --
// ---------------------

wire    [WIDTH-1:0]         d_c_gray;
reg     [WIDTH-1:0]         d_r_gray;
reg     [WIDTH-1:0]         q_s1_gray;
reg     [WIDTH-1:0]         q_s2_gray;
wire    [WIDTH-1:0]         q_ungray;
reg     [WIDTH-1:0]         r_q_ungray;



// ---------------
// -- Equations --
// ---------------

// Convert d to gray code
bfmp_phy_bin_to_gray #(WIDTH) bin_to_gray_component (
    .d      (d          ),
    .q      (d_c_gray   )
);

// Register gray coded version of d in d_clk domain
always @(posedge d_clk or negedge d_rst_n)
begin
    if (d_rst_n == 1'b0)
        d_r_gray <= {WIDTH{1'b0}};
    else
        d_r_gray <= d_clr ? {WIDTH{1'b0}} : d_c_gray;
end

// Transfer gray coded d into q_clk domain
//   Double register to reduce metastable propogation
always @(posedge q_clk or negedge q_rst_n)
begin
    if (q_rst_n == 1'b0)
    begin
        q_s1_gray <= {WIDTH{1'b0}};
        q_s2_gray <= {WIDTH{1'b0}};
    end
    else
    begin
        q_s1_gray <= q_clr ? {WIDTH{1'b0}} : d_r_gray;
        q_s2_gray <= q_clr ? {WIDTH{1'b0}} : q_s1_gray;
    end
end

// Un-Gray code to generate q
bfmp_phy_gray_to_bin #(WIDTH) gray_to_bin_component (
    .d      (q_s2_gray  ),
    .q      (q_ungray   )
);

always @(posedge q_clk or negedge q_rst_n)
begin
    if (q_rst_n == 1'b0)
        r_q_ungray <= {WIDTH{1'b0}};
    else
        r_q_ungray <= q_clr ? {WIDTH{1'b0}} : q_ungray;
end

assign q = REGISTER_OUTPUT ? r_q_ungray : q_ungray;



endmodule
// ------------------------- CONFIDENTIAL ----------------------------------
//
//                 Copyright 2010 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or
//  transmitted in any form or by any means, electronic or mechanical,
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

// -------------------------------------------------------------------------
//
//  FUNCTIONAL DESCRIPTION
//
//  This module converts the input d represented in gray code into
//     its corresponding binary value which is output as q.
//
//  This function is completely combinatorial.  Applications requiring
//    high route frequencies should consider registering the q output
//    before using it.
//
//  LIMITATIONS:
//
//    This code is limited to a maximum input bit width of 16 bits.
//    The desired width for the function is chosen via the parameter
//    WIDTH which has a valid range of 1-16.
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module bfmp_phy_gray_to_bin (

    d,
    q

);



// ----------------
// -- Parameters --
// ----------------

parameter WIDTH = 7;        // Valid range: 1 to 16



// -----------------------
// -- Port Declarations --
// -----------------------

input     [WIDTH-1:0]            d;
output     [WIDTH-1:0]            q;



// ----------------
// -- Port Types --
// ----------------

wire     [WIDTH-1:0]            d;
wire    [WIDTH-1:0]            q;



// ---------------------
// -- Local Variables --
// ---------------------

wire     [15:0]                g;
wire     [15:0]                temp;



// ---------------
// -- Equations --
// ---------------

assign g = { {(16-WIDTH){1'b0}}, d};

// Compute gray code to binary conversion
//   Do this in a temporary variable since only
//   WIDTH bits should be output
assign  temp  = { g[15],
                      (g[15] ^ g[14]),
                       (g[15] ^ g[14] ^ g[13]),
                       (g[15] ^ g[14] ^ g[13] ^ g[12]),
                       (g[15] ^ g[14] ^ g[13] ^ g[12] ^ g[11]),
                       (g[15] ^ g[14] ^ g[13] ^ g[12] ^ g[11] ^ g[10]),
                       (g[15] ^ g[14] ^ g[13] ^ g[12] ^ g[11] ^ g[10] ^ g[9]),
                       (g[15] ^ g[14] ^ g[13] ^ g[12] ^ g[11] ^ g[10] ^ g[9] ^ g[8]),
                       (g[15] ^ g[14] ^ g[13] ^ g[12] ^ g[11] ^ g[10] ^ g[9] ^ g[8] ^ g[7]),
                    (g[15] ^ g[14] ^ g[13] ^ g[12] ^ g[11] ^ g[10] ^ g[9] ^ g[8] ^ g[7] ^ g[6]),
                    (g[15] ^ g[14] ^ g[13] ^ g[12] ^ g[11] ^ g[10] ^ g[9] ^ g[8] ^ g[7] ^ g[6] ^ g[5]),
                    (g[15] ^ g[14] ^ g[13] ^ g[12] ^ g[11] ^ g[10] ^ g[9] ^ g[8] ^ g[7] ^ g[6] ^ g[5] ^ g[4]),
                    (g[15] ^ g[14] ^ g[13] ^ g[12] ^ g[11] ^ g[10] ^ g[9] ^ g[8] ^ g[7] ^ g[6] ^ g[5] ^ g[4] ^ g[3]),
                    (g[15] ^ g[14] ^ g[13] ^ g[12] ^ g[11] ^ g[10] ^ g[9] ^ g[8] ^ g[7] ^ g[6] ^ g[5] ^ g[4] ^ g[3] ^ g[2]),
                    (g[15] ^ g[14] ^ g[13] ^ g[12] ^ g[11] ^ g[10] ^ g[9] ^ g[8] ^ g[7] ^ g[6] ^ g[5] ^ g[4] ^ g[3] ^ g[2] ^ g[1]),
                    (g[15] ^ g[14] ^ g[13] ^ g[12] ^ g[11] ^ g[10] ^ g[9] ^ g[8] ^ g[7] ^ g[6] ^ g[5] ^ g[4] ^ g[3] ^ g[2] ^ g[1] ^ g[0]) };

// Assign temp result to relevant output bits
assign q = temp[WIDTH-1:0];



endmodule
// --------------------------------------------------------------------------
//
//  PROJECT:             PCI Core
//  COMPANY:             Northwest Logic, Inc.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//                 Copyright 2010 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or
//  transmitted in any form or by any means, electronic or mechanical,
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

// -------------------------------------------------------------------------
//
//  FUNCTIONAL DESCRIPTION
//
//  This module converts the input d represented in binary into
//    its corresponding Gray Code value which is output as q.
//
//  This function is completely combinatorial.  Applications requiring
//    high route frequencies should consider registering the q output
//    before using it.
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module bfmp_phy_bin_to_gray (
    d,
    q
);



// ----------------
// -- Parameters --
// ----------------

parameter   WIDTH       = 7;



// -----------------------
// -- Port Declarations --
// -----------------------

input   [WIDTH-1:0]     d;
output  [WIDTH-1:0]     q;



// ----------------
// -- Port Types --
// ----------------

wire    [WIDTH-1:0]     d;
reg     [WIDTH-1:0]     q;



// ---------------------
// -- Local Variables --
// ---------------------

genvar                  i;



// ---------------
// -- Equations --
// ---------------

// Compute binary to Gray code conversion
generate for (i=0; i<WIDTH-1; i=i+1) 
    begin : gen_q
        always @(d)
        begin
            q[i] = d[i] ^ d[i+1];
        end
    end
endgenerate

always @(d)
begin
    // MSbit is a special case
    q[WIDTH-1] = d[WIDTH-1];
end



endmodule
// -------------------------------------------------------------------------
//
//  PROJECT: PCI Express Verification Suite
//  COMPANY: Northwest Logic, Inc.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//             (c) Copyright 2009 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or
//  transmitted in any form or by any means, electronic or mechanical,
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module bfmp_pl_rx_skip_comp (

    rst_n,              
    clk,        
    clk_en,

    in_rx_valid,
    in_rx_data_valid,  
    in_rx_data,       
    in_rx_data_k,     
    in_rx_status,   
    in_rx_start_block,
    in_rx_sync_header,

    out_rx_valid,      
    out_rx_data_valid,  
    out_rx_data,       
    out_rx_data_k,    
    out_rx_status,   
    out_rx_start_block,
    out_rx_sync_header

);

    

// ----------------
// -- Parameters --
// ----------------

parameter   PHY_K_WIDTH                     = 1;

localparam PHY_K_WIDTH_USED = 1;



// ----------------------
// -- Port Definitions --
// ----------------------

input                                       rst_n;              
input                                       clk;        
input                                       clk_en;

input   [PHY_K_WIDTH-1:0]                   in_rx_valid;      
input   [PHY_K_WIDTH-1:0]                   in_rx_data_valid;  
input   [(PHY_K_WIDTH*8)-1:0]               in_rx_data;       
input   [PHY_K_WIDTH-1:0]                   in_rx_data_k;     
input   [(PHY_K_WIDTH*3)-1:0]               in_rx_status;   
input   [PHY_K_WIDTH-1:0]                   in_rx_start_block;
input   [(PHY_K_WIDTH*2)-1:0]               in_rx_sync_header;

output                                      out_rx_valid;      
output                                      out_rx_data_valid;  
output  [(PHY_K_WIDTH*8)-1:0]               out_rx_data;       
output  [PHY_K_WIDTH-1:0]                   out_rx_data_k;    
output  [(PHY_K_WIDTH*3)-1:0]               out_rx_status;   
output                                      out_rx_start_block;
output  [1:0]                               out_rx_sync_header;



// ----------------
// -- Port Types --
// ----------------

wire                                        rst_n;              
wire                                        clk;        
wire                                        clk_en;

wire    [PHY_K_WIDTH-1:0]                   in_rx_valid;      
wire    [PHY_K_WIDTH-1:0]                   in_rx_data_valid;  
wire    [(PHY_K_WIDTH*8)-1:0]               in_rx_data;       
wire    [PHY_K_WIDTH-1:0]                   in_rx_data_k;     
wire    [(PHY_K_WIDTH*3)-1:0]               in_rx_status;   
wire    [PHY_K_WIDTH-1:0]                   in_rx_start_block;
wire    [(PHY_K_WIDTH*2)-1:0]               in_rx_sync_header;

wire                                        out_rx_valid;      
wire                                        out_rx_data_valid;  
wire    [(PHY_K_WIDTH*8)-1:0]               out_rx_data;       
wire    [PHY_K_WIDTH-1:0]                   out_rx_data_k;    
wire    [(PHY_K_WIDTH*3)-1:0]               out_rx_status;   
wire                                        out_rx_start_block;
wire    [1:0]                               out_rx_sync_header;



// -------------------
// -- Local Signals --
// -------------------



// ---------------
// -- Equations --
// ---------------

// Data input and data output are the same width, so no conversion is necessary
assign out_rx_valid       = in_rx_valid;      
assign out_rx_data_valid  = in_rx_data_valid;  
assign out_rx_data        = in_rx_data;       
assign out_rx_data_k      = in_rx_data_k;     
assign out_rx_status      = in_rx_status;   
assign out_rx_start_block = in_rx_start_block;
assign out_rx_sync_header = in_rx_sync_header;


endmodule

// -------------------------------------------------------------------------
//
//  PROJECT: PCI Express Verification Suite
//  COMPANY: Northwest Logic, Inc.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//             (c) Copyright 2009 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or 
//  transmitted in any form or by any means, electronic or mechanical, 
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//  
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

// -------------------------------------------------------------------------
// Behavioral PLL for Locking to Rx Serial Stream Clock Frequency
// -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module bfmp_pcie_rx_pll (

    rst_n,              // Asynchronous reset
    ref_serial_clk,     // Reference serial clock
    rate,               // Serial data rate: 00 == 2.5G, 01 == 5G, 10 == 8G

    rx_p,               // Receive serial stream
    rx_n,               //   ..

    rx_elec_idle,       // 1 == serial stream is electrical idle (Hi-Z or Common Mode)
    rx_lock,            // 1 == rx_serial_clk & rx_symbol_clk valid
    rx_serial_clk,      // Recovered RX serial clock
    rx_symbol_clk       // Recovered RX symbol clock

);      



// ----------------
// -- Parameters --
// ----------------



// ----------------------
// -- Port Definitions --
// ----------------------

input                               rst_n;
input                               ref_serial_clk;
input   [1:0]                       rate;

input                               rx_p;
input                               rx_n;

input                               rx_elec_idle;
output                              rx_lock;
output                              rx_serial_clk;
output                              rx_symbol_clk;



// ----------------
// -- Port Types --
// ----------------

wire                                rst_n;
wire                                ref_serial_clk;
wire    [1:0]                       rate;

wire                                rx_p;
wire                                rx_n;

wire                                rx_elec_idle;
reg                                 rx_lock;
reg                                 rx_serial_clk;
reg                                 rx_symbol_clk;



// -------------------
// -- Local Signals --
// -------------------

reg     [1:0]                       r_dly;
reg                                 r_en;
time                                r_old;
time                                r_new;
time                                r_period;

time                                max_ppm_diff;
time                                max_period;  
time                                min_period;  

wire                                f_rst_n;
time                                f_last;
time                                f_time;
time                                n_periods;
time                                f_periodt1;
reg                                 f_lock;

time                                f_period1a;
time                                f_period1b;

reg     [8:0]                       rx_elec_idle_ctr;
reg                                 rx_elec_idle_dly;

reg     [2:0]                       rx_serial_clk_ctr;
reg     [4:0]                       rx_lock_ctr;



// ---------------
// -- Equations --
// ---------------

// Get period of reference serial clock
always @(posedge ref_serial_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        r_dly    <= 2'h0;
        r_en     <= 1'b0;
        r_old    <= 64'b0;
        r_new    <= 64'b0;
        r_period <= 64'b0;
    end
    else
    begin
        // Wait 4 clocks before using reference clock
        if (r_dly != 2'b11)
            r_dly <= r_dly + 2'h1;

        r_en <= (r_dly == 2'b11);

        // Capture time at rising edges to compute the period of the clock
        r_old    <= r_new;
        r_new    <= $time;
        r_period <= r_new - r_old;
    end
end

// Max & Min == ref_clk frequency +/- 600 ppm
always @*
begin
    case (rate)
        2'b10 :
            begin
                max_ppm_diff =            64'd1; // 64'd75;  600 ppm == +/- 0.75 pS
                max_period   = r_period + 64'd1; // 64'd75;   
                min_period   = r_period - 64'd1; // 64'd75;   
            end
        2'b01 :
            begin
                max_ppm_diff =            64'd2; // 64'd120; 600 ppm == +/- 1.20 pS
                max_period   = r_period + 64'd2; // 64'd120;
                min_period   = r_period - 64'd2; // 64'd120;
            end
        default :
            begin
                max_ppm_diff =            64'd3; // 64'd240; 600 ppm == +/- 2.40 pS
                max_period   = r_period + 64'd3; // 64'd240;
                min_period   = r_period - 64'd3; // 64'd240;
            end
    endcase 
end    

always @(posedge ref_serial_clk or negedge rst_n) begin
    if (rst_n == 0) begin
        rx_elec_idle_ctr <= 9'h0;
        rx_elec_idle_dly <= 1'b1;
    end
    else begin
        if (~rx_elec_idle) begin
            rx_elec_idle_ctr <= 9'h0;
            rx_elec_idle_dly <= 1'b0;
        end
        else begin
            if (rx_elec_idle_ctr != 9'h1ff)
                rx_elec_idle_ctr <= rx_elec_idle_ctr + 1;
            else
                rx_elec_idle_dly <= 1'b1;
        end
    end
end


// Do not try to recover clock when not locked to reference clock frequency
//   or when the link is electrcially idle
assign f_rst_n = r_en & ~rx_elec_idle_dly;

// Get period of reference serial clock
always @(rx_p or negedge f_rst_n)
begin
    if (f_rst_n == 1'b0)
    begin
        f_last     <= 64'b0;
        f_time     <= 64'h0;
        n_periods  <= 64'h0;
        f_periodt1 <= 64'h0;
        f_period1a <= 64'd200;
        f_period1b <= 64'd200; 
        f_lock     <= 1'b0;
    end
    else
    begin
        // Capture time at rising edges to compute the period of the clock
        //   Period is actually half this amount
        f_last     <= $time;

        f_time = ($time - f_last);

        n_periods = (f_time + r_period/2)/r_period;

        f_periodt1 = (n_periods == 64'h0) ? 64'h0 : f_time / n_periods;

        // Only use edges if the rx_p edge period is within 600 pS of closely match period of reference clock
        //   and the difference between rx_p frequency and ref_clk frequency is <= 600 ppm
        if ( (f_periodt1 <= max_period) &
             (f_periodt1 >= min_period) &
             ( (f_periodt1 >= r_period) ? ((f_periodt1 - r_period) <= max_ppm_diff) :
                                          ((r_period - f_periodt1) <= max_ppm_diff) ) )

        begin
            f_period1a <= f_periodt1/2;
            f_period1b <= f_periodt1 - (f_periodt1/2);
            f_lock     <= 1'b1;
        end
    end
end

initial
begin
    rx_serial_clk = 1'b0;
    while (1)
    begin
        while (f_lock == 1'b1) begin
            fork
                begin : resync_0
                    @(rx_p);
                    disable coast_0;
                    #(f_period1a);
                end
                begin : coast_0
                    #(f_period1a);
                    disable resync_0;
                end
            join
            rx_serial_clk = 1'b1;
            fork 
                begin : resync_1
                    @(rx_p);
                    disable coast_1;
                end
                begin : coast_1
                    #(f_period1b);
                    disable resync_1;
                end
            join
            rx_serial_clk = 1'b0;
        end
        rx_serial_clk = 1'b0;
        @(posedge rx_p);
    end
end


always @(negedge rx_serial_clk or negedge f_lock)
begin
    if (f_lock == 1'b0)
    begin
        rx_serial_clk_ctr <= 3'd0;
        rx_symbol_clk     <= 1'b0;
        rx_lock_ctr       <= 5'h0;
        rx_lock           <= 1'b0;
    end
    else
    begin
        if ( ((rx_serial_clk_ctr == 3'd3) & (rate == 2'b10)) | // Divide by  8 at 8G Data Rate
             ((rx_serial_clk_ctr == 3'd4) & (rate != 2'b10)) ) // Divide by 10 at 5G & 2.5G Data Rates
            rx_serial_clk_ctr <= 3'd0;
        else
            rx_serial_clk_ctr <= rx_serial_clk_ctr + 3'd1;

        if ( ((rx_serial_clk_ctr == 3'd3) & (rate == 2'b10)) | 
             ((rx_serial_clk_ctr == 3'd4) & (rate != 2'b10)) )
            rx_symbol_clk <= ~rx_symbol_clk;

        if (rx_lock_ctr != 5'h1f)
            rx_lock_ctr <= rx_lock_ctr + 5'h1;

        if (rx_lock_ctr == 5'h1f)
            rx_lock <= 1'b1;
    end
end



endmodule
// -------------------------------------------------------------------------
//
//  PROJECT: PCI Express Verification Suite
//  COMPANY: Northwest Logic, Inc.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//             (c) Copyright 2009 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or 
//  transmitted in any form or by any means, electronic or mechanical, 
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//  
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps


// -----------------------
// -- Module Definition --
// -----------------------

module bfmp_pcie_phy_pll (

    rst_n,              // Asynchronous reset
    rate,               // Clock rate: 00 == 2.5G, 01 == 5G, 10 == 8G

    lock,               // 1 == Output clocks valid
    serial_clk,         // Current serial clock
    symbol_clk,         // Symbol clock   at current serial clock rate
    pclk,               // Pipe clock
    pclk_en             // Pipe clock enable, used in fixed clock rate mode
);      



// ----------------
// -- Parameters --
// ----------------

parameter  PHY_K_WIDTH             = 2;
parameter  HALF_PERIOD_8G          = 62.5;
parameter  HALF_PERIOD_5G          = 100;
parameter  HALF_PERIOD_2G5         = 200;

// ----------------------
// -- Port Definitions --
// ----------------------

input                               rst_n;
input   [1:0]                       rate;

output                              lock;
output                              serial_clk;
output                              symbol_clk;
output                              pclk;
output                              pclk_en;


// ----------------
// -- Port Types --
// ----------------

wire                                rst_n;
wire    [1:0]                       rate;

reg                                 lock;
reg                                 serial_clk;
reg                                 symbol_clk;
wire                                pclk;
wire                                pclk_en;

// -------------------
// -- Local Signals --
// -------------------

reg     [7:0]                       low_pulse_serial;
reg     [7:0]                       high_pulse_serial;

integer                             bits_per_symbol;
reg     [11:0]                      low_pulse_symbol;
reg     [11:0]                      high_pulse_symbol;

reg     [1:0]                       symbol_clk_ctr;
wire                                symbol_clk_div2;
wire                                symbol_clk_div4;

reg     [7:0]                       lock_ctr;


// ---------------
// -- Equations --
// ---------------

// Generate serial Clock
always @(rate or rst_n)
begin
    if (rst_n == 1'b1)
        case (rate)
            2'b10:begin
                low_pulse_serial  = HALF_PERIOD_8G;
                high_pulse_serial = (HALF_PERIOD_8G*2.0) - low_pulse_serial;
            end
            2'b01:begin
                low_pulse_serial  = HALF_PERIOD_5G;
                high_pulse_serial = (HALF_PERIOD_5G*2.0) - low_pulse_serial;
            end
            default:begin
                low_pulse_serial  = HALF_PERIOD_2G5;
                high_pulse_serial = (HALF_PERIOD_2G5*2.0) - low_pulse_serial;
            end
        endcase
    else
    begin
        low_pulse_serial  = HALF_PERIOD_2G5;
        high_pulse_serial = (HALF_PERIOD_2G5*2.0) - low_pulse_serial;
    end
end

initial begin
    low_pulse_serial  = HALF_PERIOD_2G5;
    high_pulse_serial = (HALF_PERIOD_2G5*2.0) - low_pulse_serial;
    forever
    begin
        serial_clk <= 1'b0;
        #low_pulse_serial;
        serial_clk <= 1'b1;
        #high_pulse_serial;
    end
end

always @(rate or rst_n)
begin
    if (rst_n == 1'b1)
        case (rate)
            2'b10:begin
                low_pulse_symbol  = HALF_PERIOD_8G*8.0;
                high_pulse_symbol = (HALF_PERIOD_8G*16.0) - low_pulse_symbol;
                bits_per_symbol   = 8;
            end
            2'b01:begin
                low_pulse_symbol  = HALF_PERIOD_5G*10.0;
                high_pulse_symbol = (HALF_PERIOD_5G*20.0) - low_pulse_symbol;
                bits_per_symbol   = 10;
            end
            default:begin
                low_pulse_symbol  = HALF_PERIOD_2G5*10.0;
                high_pulse_symbol = (HALF_PERIOD_2G5*20.0) - low_pulse_symbol;
                bits_per_symbol   = 10;
            end
        endcase
    else
    begin
        low_pulse_symbol  = HALF_PERIOD_2G5*10;
        high_pulse_symbol = (HALF_PERIOD_2G5*20.0) - low_pulse_symbol;
        bits_per_symbol   = 10;
    end
end

initial begin
    low_pulse_symbol  = HALF_PERIOD_2G5*10.0;
    high_pulse_symbol = (HALF_PERIOD_2G5*20.0) - low_pulse_symbol;
    bits_per_symbol   = 10;
    forever begin
        // Align to Serial Clock
        symbol_clk <= 1'b0;
        repeat(bits_per_symbol/2) @(posedge serial_clk);
        symbol_clk <= 1'b1;
        repeat(bits_per_symbol/2) @(posedge serial_clk);
    end
end

always @(posedge symbol_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        lock_ctr        <= 8'h0;
        lock            <= 1'b0;
    end
    else
    begin
        if (lock_ctr != 8'h1f)
        begin
            lock_ctr <= lock_ctr + 8'h1;
            lock     <= 1'b0;
        end
        else
            lock     <= 1'b1;
    end
end

// Using opposite edge that generate serial_clk and adding so all clock outputs
//    will have an offset relative to one another to avoid 0-time simulator issues
always @(negedge symbol_clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
        symbol_clk_ctr <= 2'b00;
    else
        symbol_clk_ctr <= symbol_clk_ctr + 2'b01;
end

assign symbol_clk_div2 = symbol_clk_ctr[0];
assign symbol_clk_div4 = symbol_clk_ctr[1];

assign pclk =  (PHY_K_WIDTH == 4) ?  symbol_clk_div4 :
              ((PHY_K_WIDTH == 2) ?  symbol_clk_div2 :
               symbol_clk);

assign pclk_en      = 1'b1;

endmodule
// -------------------------------------------------------------------------
//
//  PROJECT: PCI Express Verification Suite
//  COMPANY: Northwest Logic, Inc.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//             (c) Copyright 2009 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or 
//  transmitted in any form or by any means, electronic or mechanical, 
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//  
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps


// -----------------------
// -- Module Definition --
// -----------------------

module bfmp_pcie_phy (

    rst_n,                          // Asynchronous reset
    lane_mask,                      // (1) Mask Lane from detecting receiver; (0) normal operation

    serial_clk,                     // Tx Serial Clock
    symbol_clk,                     // Tx Symbol Clock
    pclk,                           // Parallel interface clock
    pclk_en,                        // Parallel interface clock enable

    tx_p,                           // Transmit serial stream; positive
    tx_n,                           // Transmit serial stream; negative

    rx_p,                           // Receive serial stream; positive
    rx_n,                           // Receive serial stream; negative

    tx_detect_rx_loopback,          // Tx Detect & Rx Loopback
    power_down,                     // Power Down
    rate,                           // Selected data rate: 00 == 2.5g, 01 == 5g, or 10 == 8g
    phy_status,                     // Set to communicate completion of Tx Detect, rate chage, & power_down transitions; on reset stays set until PCLK valid 
    rx_polarity,                    // Set to invert serial polarity of receive stream
    tx_deemph,                      // Set to specify transmitter deemphasis value

    tx_data,                        // Data value
    tx_data_k,                      // 8b10b: 1 == K characters, 0 == Data
    tx_data_valid,                  // Data valid indicator in 8g mode: 1 == Use Data; 0 == Don't Use
    tx_start_block,                 // Start of block in 8g mode
    tx_sync_header,                 // Block type in 8g mode
    tx_compliance,                  // If set, encode this byte assuming disparity is currently negative
    tx_elec_idle,                   // 1 == Idle transmitter; 0 == Active transmitter
    
    rx_data,                        // Data value
    rx_data_k,                      // 8b10b: 1 == K characters, 0 == Data
    rx_data_valid,                  // Data valid indicator in 8g mode: 1 == Use Data; 0 == Don't Use
    rx_start_block,                 // Start of block in 8g mode
    rx_sync_header,                 // Type of block in 8g mode
    rx_block_realign,               // For 8G mode, when high specifies that the PHY should re-acquire alignment
    rx_valid,                       // Set if syncronization was lost; asynchronous signal
    rx_status,                      // 3-bit Status per lane
    rx_elec_idle,                   // 1 == Receiver Idle; 0 == Active Receiver

    rx_serial_stp,                  // rising edge on reception of STP start symbol on RX serial interface.  12 UI late 
    rx_serial_sdp,                  // rising edge on reception of SDP start symbol on RX serial interface.  12 UI late 
    tx_serial_stp,                  // rising edge on transmisson of STP start symbol on TX serial interface
    tx_serial_sdp                   // rising edge on transmisson of SDP start symbol on TX serial interface
);      



// ----------------
// -- Parameters --
// ----------------

parameter   LANE_NUMBER             = 0;
parameter   SIM_EL_IDLE_TYPE        = 2'b10; // Electrical Idle Emulation: 11 == 1'b1 : Common Mode 1
                                             //                            10 == 1'b0 : Common Mode 0
                                             //                            01 == 1'bx : Undefined
                                             //                            00 == Reserved

localparam  START_DLY               = 255 - ((LANE_NUMBER%4)*16);   // Add variability to lane phy_status startup time
localparam  DET_DLY                 =  15 -  (LANE_NUMBER%4);       // Add variability to lane phy_status Tx Detect response
localparam  PD_DLY                  =  15 -  (LANE_NUMBER%4);       // Add variability to lane phy_status power down change acceptance
localparam  RATE_DLY                =  15 -  (LANE_NUMBER%4);       // Add variability to lane phy_status rate change acceptance

parameter   RX_IDLE_ACTIVE_8G_ONLY_EIE = 0; // Set to 1 to cause rx_elec_idle to be 0 at 8G speed only when EIEOS == {8{8'h00, 8'ff}} is received;                                            0 == Use emulated analog comparator 
parameter   RX_IDLE_ACTIVE_5G_ONLY_EIE = 0; // Set to 1 to cause rx_elec_idle to be 0 at 5G speed only when EIE == symbols are received back-back; alternating pattern of 5 zeros and 5 ones; 0 == Use emulated analog comparator 
parameter   RX_IDLE_RANDOM_WHEN_ACTIVE = 0; // When receiver is not idle, value of rx_elec_idle is random

parameter   PHY_K_WIDTH             = 1;                // Output data width in bytes

localparam  LINE_PHY_K_WIDTH        = 1;                // PCI Express-side logic is always 1 byte in width
localparam  LINE_PHY_DATA_WIDTH     = 8;



// ----------------------
// -- Port Definitions --
// ----------------------

input                               rst_n;
input                               lane_mask;

input                               serial_clk;
input                               symbol_clk;
input                               pclk;
input                               pclk_en;

output                              tx_p;
output                              tx_n;

input                               rx_p;
input                               rx_n;

input                               tx_detect_rx_loopback;
input   [1:0]                       power_down;
input   [1:0]                       rate;
output                              phy_status;
input                               rx_polarity;
input                               tx_deemph;
   
input   [(PHY_K_WIDTH*8)-1:0]       tx_data;
input   [PHY_K_WIDTH-1:0]           tx_data_k;
input                               tx_data_valid;
input                               tx_start_block;
input   [1:0]                       tx_sync_header;
input   [PHY_K_WIDTH-1:0]           tx_compliance;
input                               tx_elec_idle;
    
output  [(PHY_K_WIDTH*8)-1:0]       rx_data;
output  [PHY_K_WIDTH-1:0]           rx_data_k;
output                              rx_data_valid;
output                              rx_start_block;
output  [1:0]                       rx_sync_header;
input                               rx_block_realign;
output                              rx_valid;
output  [(PHY_K_WIDTH*3)-1:0]       rx_status;
output                              rx_elec_idle;

output                              rx_serial_stp;
output                              rx_serial_sdp;
output                              tx_serial_stp;
output                              tx_serial_sdp;

// ----------------
// -- Port Types --
// ----------------

wire                                rst_n;
wire                                lane_mask;

wire                                serial_clk;
wire                                symbol_clk;
wire                                pclk;
wire                                pclk_en;

wire                                tx_p;
wire                                tx_n;

wire                                rx_p;
wire                                rx_n;

wire                                tx_detect_rx_loopback;
wire    [1:0]                       power_down;
wire    [1:0]                       rate;
wire                                phy_status;
wire                                rx_polarity;
wire                                tx_deemph;

wire    [(PHY_K_WIDTH*8)-1:0]       tx_data;
wire    [PHY_K_WIDTH-1:0]           tx_data_k;
wire                                tx_data_valid;
wire                                tx_start_block;
wire    [1:0]                       tx_sync_header;
wire    [PHY_K_WIDTH-1:0]           tx_compliance;
wire                                tx_elec_idle;
    
wire    [(PHY_K_WIDTH*8)-1:0]       rx_data;
wire    [PHY_K_WIDTH-1:0]           rx_data_k;
wire                                rx_data_valid;
wire                                rx_start_block;
wire    [1:0]                       rx_sync_header;
wire                                rx_block_realign;
wire                                rx_valid;
wire    [(PHY_K_WIDTH*3)-1:0]       rx_status;          
reg                                 rx_elec_idle;

// -------------------
// -- Local Signals --
// -------------------

// Clock Generation
wire                                rx_lock;           
wire                                rx_serial_clk;     
wire                                rx_symbol_clk;     

// RX Logic
wire                                i_rx_p;
wire                                i_rx_n;

reg     [31:0]                      history_8g;
reg     [5:0]                       ctr_8g;

reg     [14:0]                      history_5g;
reg     [3:0]                       ctr_5g;

reg                                 analog_rx_elec_idle;
reg                                 eie_8g_idle;
reg                                 eie_5g_idle;

wire                                rx_elec_idle_active_8g_only_for_eie;
wire                                rx_elec_idle_active_5g_only_for_eie;
wire                                rx_elec_idle_random_when_active;

reg                                 rx_elec_idle_random;

reg                                 in_rx_valid;
reg                                 in_rx_data_valid;
reg     [7:0]                       in_rx_data;
reg                                 in_rx_data_k;
reg                                 in_rx_err_disp;
reg                                 in_rx_err_8b10;
reg                                 in_rx_start_block;
reg     [1:0]                       in_rx_sync_header;
reg                                 in_rx_frame_error;

wire                                in_rx_10b_valid;     
wire                                in_rx_10b_data_valid;
wire    [7:0]                       in_rx_10b_data;      
wire                                in_rx_10b_data_k;    
wire                                in_rx_10b_err_disp;  
wire                                in_rx_10b_err_8b10;  

wire                                in_rx_130b_valid;      
wire                                in_rx_130b_data_valid; 
wire    [7:0]                       in_rx_130b_data;       
wire                                in_rx_130b_start_block;
wire    [1:0]                       in_rx_130b_sync_header;
wire                                in_rx_130b_frame_error;

wire    [PHY_K_WIDTH-1:0]           out_rx_valid;
wire    [PHY_K_WIDTH-1:0]           out_rx_data_valid;
wire    [(PHY_K_WIDTH*8)-1:0]       out_rx_data;
wire    [PHY_K_WIDTH-1:0]           out_rx_data_k;
wire    [PHY_K_WIDTH-1:0]           out_rx_err_disp;
wire    [PHY_K_WIDTH-1:0]           out_rx_err_8b10;
wire    [PHY_K_WIDTH-1:0]           out_rx_err_uflow;
wire    [PHY_K_WIDTH-1:0]           out_rx_err_oflow; 
wire    [PHY_K_WIDTH-1:0]           out_rx_start_block;
wire    [(PHY_K_WIDTH*2)-1:0]       out_rx_sync_header;
wire    [PHY_K_WIDTH-1:0]           out_rx_frame_error;

wire                                sc_rx_valid;
wire                                sc_rx_data_valid;
wire    [(PHY_K_WIDTH*8)-1:0]       sc_rx_data;
wire    [PHY_K_WIDTH-1:0]           sc_rx_data_k;
wire    [(PHY_K_WIDTH*3)-1:0]       sc_rx_status;
wire                                sc_rx_start_block;
wire    [1:0]                       sc_rx_sync_header;

genvar                              g;
wire    [(PHY_K_WIDTH*3)-1:0]       out_rx_status;

wire    [(PHY_K_WIDTH*3)-1:0]       norm_rx_status;

// TX Logic
wire                                tx_8b_p;
wire                                tx_8b_n;

wire                                tx_128b_p;
wire                                tx_128b_n;

reg                                 c_tx_p;
reg                                 c_tx_n;

wire    [7:0]                       out_tx_data;
wire                                out_tx_data_k;
wire                                out_tx_data_valid;
wire                                out_tx_start_block;
wire    [1:0]                       out_tx_sync_header;
wire                                out_tx_compliance;
wire                                out_tx_elec_idle;
wire                                en_loopback;

reg                                 en_loopback_data;
reg                                 en_data;
reg     [7:0]                       en_loopback_ctr;
reg                                 d_en_loopback;

// Model Reset phy_status
reg     [7:0]                       start_phy_status_ctr;
reg                                 start_phy_status;

// Model Receiver Detect
wire                                rx_detect;

reg                                 r1_rx_detect;  
reg                                 r2_rx_detect;  

reg     [3:0]                       det_ctr;       
reg                                 det_phy_status;

wire    [2:0]                       det_rx_status_yes;
wire    [2:0]                       det_rx_status_no;

wire    [(PHY_K_WIDTH*3)-1:0]       det_rx_status;

// Model Power State Change
reg     [1:0]                       r1_power_down;
reg     [1:0]                       r2_power_down;

reg     [3:0]                       pd_phy_status_ctr;
reg                                 pd_phy_status;

// Model Rate Change
reg     [1:0]                       r1_rate;
reg     [1:0]                       r2_rate;
reg     [3:0]                       rate_phy_status_ctr;
reg                                 rate_phy_status;

integer                             seed;

localparam K_STP = 9'h1FB;
localparam K_SDP = 9'h15C;


// ---------------
// -- Functions --
// ---------------


// ----------------
// Clock Generation

// Instantiate Behavioral Rx PLL
bfmp_pcie_rx_pll rx_pll (

    .rst_n                  (rst_n                  ),
    .ref_serial_clk         (serial_clk             ),
    .rate                   (rate                   ),

    .rx_p                   (rx_p                   ),
    .rx_n                   (rx_n                   ),

    .rx_elec_idle           (analog_rx_elec_idle    ),
    .rx_lock                (rx_lock                ),
    .rx_serial_clk          (rx_serial_clk          ),
    .rx_symbol_clk          (rx_symbol_clk          )

);

// --------------------
// Instantiate RX Logic

// Invert polarity if directed
assign i_rx_p = rx_polarity ? rx_n : rx_p;
assign i_rx_n = rx_polarity ? rx_p : rx_n;

// Detect electrical idle; Hi-Z or common mode
always @*
    analog_rx_elec_idle = (rx_p === 1'bz) |
                          (rx_n === 1'bz) |
                          (rx_p === rx_n);

// Assert eie_8g_idle == 0 only during reception of the low-frequency EIEOS ordered set
always @(posedge rx_serial_clk or negedge rx_lock)
begin
    if (rx_lock == 1'b0)
    begin
        // Choose reset pattern that is most different from the EIE pattern
        history_8g  <= 32'haaaaaaaa;
        ctr_8g      <= 6'h0;
        eie_8g_idle <= 1'b1;
    end
    else
    begin
        // Reset pattern when link is idle
        if ((rx_p === 1'bz) | (rx_n === 1'bz))
            history_8g <= 32'haaaaaaaa;
        else 
            history_8g <= {history_8g[30:0], rx_p};

        if ( (history_8g == 32'hff00ff00) | 
             (history_8g == 32'h00ff00ff) )
            ctr_8g <= 6'h20;
        else if (ctr_8g != 6'h0)
            ctr_8g <= ctr_8g - 6'h1;

        eie_8g_idle <= (ctr_8g == 6'h0);
    end
end

// Assert eie_5g_idle == 0 only during reception of the low-frequency EIE symbols
always @(posedge rx_serial_clk or negedge rx_lock)
begin
    if (rx_lock == 1'b0)
    begin
        // Choose reset pattern that is most different from the EIE pattern
        history_5g  <= 15'b101010101010101;
        ctr_5g      <= 4'h0;
        eie_5g_idle <= 1'b1;
    end
    else
    begin
        // Reset pattern when link is idle
        if ((rx_p === 1'bz) | (rx_n === 1'bz))
            history_5g <= 15'b101010101010101;
        else 
            history_5g <= {history_5g[13:0], rx_p};

        if ( (history_5g == 15'b111110000011111) | 
             (history_5g == 15'b000001111100000) )
            ctr_5g <= 4'hf;
        else if (ctr_5g != 4'h0)
            ctr_5g <= ctr_5g - 4'h1;

        eie_5g_idle <= (ctr_5g == 4'h0);
    end
end

// 10b to 8b Decoder; serial to parallel conversion
bfmp_rx_10b_to_8b rx_10b_to_8b (

    .rst_n                  (rx_lock & ~rate[1]     ),

    .rx_serial_clk          (rx_serial_clk          ),
    .rx_symbol_clk          (rx_symbol_clk          ),

    .rx_p                   (i_rx_p                 ),
    .rx_n                   (i_rx_n                 ),

    .rx_valid               (in_rx_10b_valid        ),
    .rx_data_valid          (in_rx_10b_data_valid   ),
    .rx_data                (in_rx_10b_data         ),
    .rx_data_k              (in_rx_10b_data_k       ),
    .rx_err_disp            (in_rx_10b_err_disp     ),
    .rx_err_8b10            (in_rx_10b_err_8b10     )

);      

// 130b to 128b Decoder; serial to parallel conversion
bfmp_rx_130b_to_128b rx_130b_to_128b (

    .rst_n                  (rx_lock & rate[1]      ),

    .rx_serial_clk          (rx_serial_clk          ),
    .rx_symbol_clk          (rx_symbol_clk          ),

    .rx_p                   (i_rx_p                 ),
    .rx_n                   (i_rx_n                 ),
    .analog_rx_elec_idle    (analog_rx_elec_idle    ),

    .rx_block_realign       (rx_block_realign       ),
    .rx_valid               (in_rx_130b_valid       ),
    .rx_data_valid          (in_rx_130b_data_valid  ),
    .rx_data                (in_rx_130b_data        ),
    .rx_start_block         (in_rx_130b_start_block ),
    .rx_sync_header         (in_rx_130b_sync_header ),
    .rx_frame_error         (in_rx_130b_frame_error )

);

// Select which data stream to use
always @*
begin
    if (rate == 2'b10) // 8g
    begin 
        in_rx_valid       = in_rx_130b_valid;
        in_rx_data_valid  = in_rx_130b_data_valid;
        in_rx_data        = in_rx_130b_data;
        // 5/2.5G only signals
        in_rx_data_k      = 1'b0;
        in_rx_err_disp    = 1'b0;
        in_rx_err_8b10    = 1'b0;
        // 8G only signals
        in_rx_start_block = in_rx_130b_start_block;
        in_rx_sync_header = in_rx_130b_sync_header;
        in_rx_frame_error = in_rx_130b_frame_error;
    end
    else // 2.5g or 5g
    begin
        in_rx_valid       = in_rx_10b_valid;
        in_rx_data_valid  = in_rx_10b_data_valid;
        in_rx_data        = in_rx_10b_data;
        // 5/2.5G only signals
        in_rx_data_k      = in_rx_10b_data_k;
        in_rx_err_disp    = in_rx_10b_err_disp;
        in_rx_err_8b10    = in_rx_10b_err_8b10;
        // 8G only signals
        in_rx_start_block = 1'b0;
        in_rx_sync_header = 2'b0;
        in_rx_frame_error = 1'b0;
    end
end
// -----------------------------
// RX Electrical Idle Generation

assign rx_elec_idle_active_8g_only_for_eie = RX_IDLE_ACTIVE_8G_ONLY_EIE;
assign rx_elec_idle_active_5g_only_for_eie = RX_IDLE_ACTIVE_5G_ONLY_EIE;
assign rx_elec_idle_random_when_active     = RX_IDLE_RANDOM_WHEN_ACTIVE;

initial
begin
    rx_elec_idle_random = 1'b0;
    forever
    begin
        #1000;
        rx_elec_idle_random = $random(seed);
    end
end

always @*
begin
    if (rx_lock)
        case (rate)
            // At 8.0G use analog idle or worst case EIE-only idle if directed
            2'h2    : rx_elec_idle = rx_elec_idle_active_8g_only_for_eie ? eie_8g_idle : (rx_elec_idle_random_when_active ? rx_elec_idle_random | analog_rx_elec_idle : analog_rx_elec_idle);
            // At 5.0G use analog idle or worst case EIE-only idle if directed
            2'h1    : rx_elec_idle = rx_elec_idle_active_5g_only_for_eie ? eie_5g_idle : (rx_elec_idle_random_when_active ? rx_elec_idle_random | analog_rx_elec_idle : analog_rx_elec_idle);
            // At 2.5G use analog idle
            default : rx_elec_idle = analog_rx_elec_idle; 
        endcase
    else
        if (rate == 2'h0)
            rx_elec_idle = analog_rx_elec_idle;
        else
            rx_elec_idle = rx_elec_idle_random_when_active ? rx_elec_idle_random | analog_rx_elec_idle : analog_rx_elec_idle;
end

// STP/SDP RX detection for latency checking, currently only supported at 2.5/5G speed
assign rx_serial_stp = in_rx_valid & (rate !== 2'b10) & ({in_rx_data_k,in_rx_data} == K_STP); // Note - this signal is 2 serial clocks after rx_p/n ports
assign rx_serial_sdp = in_rx_valid & (rate !== 2'b10) & ({in_rx_data_k,in_rx_data} == K_SDP); // Note - this signal is 2 serial clocks after rx_p/n ports 

// link_speed/width matching FIFO
bfmp_phy_rx #(

    .LANE_NUMBER            (LANE_NUMBER            ),
    .PHY_K_WIDTH            (PHY_K_WIDTH            )
    
) phy_rx (

    .rst_n                  (rst_n                  ),
    .rx_lock                (rx_lock                ),
    .rate                   (rate                   ),
            
    .in_rx_clk              (rx_symbol_clk          ),
    .in_rx_valid            (in_rx_valid            ),
    .in_rx_data_valid       (in_rx_data_valid       ),
    .in_rx_data             (in_rx_data             ),
    .in_rx_data_k           (in_rx_data_k           ),
    .in_rx_err_disp         (in_rx_err_disp         ),
    .in_rx_err_8b10         (in_rx_err_8b10         ),
    .in_rx_start_block      (in_rx_start_block      ),
    .in_rx_sync_header      (in_rx_sync_header      ),
    .in_rx_frame_error      (in_rx_frame_error      ),

    .out_rx_clk             (pclk                   ),
    .out_rx_clk_en          (pclk_en                ),
    .out_rx_valid           (out_rx_valid           ),
    .out_rx_data_valid      (out_rx_data_valid      ),
    .out_rx_data            (out_rx_data            ),
    .out_rx_data_k          (out_rx_data_k          ),
    .out_rx_err_disp        (out_rx_err_disp        ),
    .out_rx_err_8b10        (out_rx_err_8b10        ),
    .out_rx_err_uflow       (out_rx_err_uflow       ),
    .out_rx_err_oflow       (out_rx_err_oflow       ),
    .out_rx_start_block     (out_rx_start_block     ),
    .out_rx_sync_header     (out_rx_sync_header     ),
    .out_rx_frame_error     (out_rx_frame_error     )
    
);      

// Generate rx_status from error signals
generate
    for (g=0; g<PHY_K_WIDTH; g=g+1)
    begin : rx_status_gen
        assign out_rx_status[((g+1)*3)-1:(g*3)] =  out_rx_err_oflow[g]   ? 3'b101 :
                                                  (out_rx_err_uflow[g]   ? 3'b110 :
                                                  (out_rx_err_8b10[g]    ? 3'b100 :
                                                  (out_rx_frame_error[g] ? 3'b100 :
                                                  (out_rx_err_disp[g]    ? 3'b111 : 3'b000))));
    end
endgenerate

// Aggregate data into full-width data or full-width valid;
//   Align rx_start_block to byte 0
bfmp_pl_rx_skip_comp pl_rx_skip_comp (

    .rst_n                  (rst_n                  ),
    .clk                    (pclk                   ),
    .clk_en                 (pclk_en                ),

    .in_rx_valid            (out_rx_valid           ),
    .in_rx_data_valid       (out_rx_data_valid      ),
    .in_rx_data             (out_rx_data            ),
    .in_rx_data_k           (out_rx_data_k          ),
    .in_rx_status           (out_rx_status          ),
    .in_rx_start_block      (out_rx_start_block     ),
    .in_rx_sync_header      (out_rx_sync_header     ),

    .out_rx_valid           (sc_rx_valid            ),
    .out_rx_data_valid      (sc_rx_data_valid       ),
    .out_rx_data            (sc_rx_data             ),
    .out_rx_data_k          (sc_rx_data_k           ),
    .out_rx_status          (sc_rx_status           ),
    .out_rx_start_block     (sc_rx_start_block      ),
    .out_rx_sync_header     (sc_rx_sync_header      )
    
);      

assign rx_valid       = sc_rx_valid;
assign rx_data_valid  = sc_rx_data_valid;
assign rx_data        = sc_rx_data;
assign rx_data_k      = sc_rx_data_k;
assign norm_rx_status = sc_rx_status;
assign rx_start_block = sc_rx_start_block;
assign rx_sync_header = sc_rx_sync_header;

// --------------------
// Instantiate TX Logic

// link_speed/width matching FIFO
bfmp_phy_tx #(

    .PHY_K_WIDTH            (PHY_K_WIDTH            )
    
) phy_tx (

    .rst_n                  (rst_n                  ),
    .rate                   (rate                   ),

    .in_tx_clk              (pclk                   ),
    .in_tx_clk_en           (pclk_en                ),
    .in_tx_data             (tx_data                ),
    .in_tx_data_k           (tx_data_k              ),
    .in_tx_data_valid       (tx_data_valid          ),
    .in_tx_start_block      (tx_start_block         ),
    .in_tx_sync_header      (tx_sync_header         ),
    .in_tx_compliance       (tx_compliance          ),
    .in_tx_elec_idle        (tx_elec_idle           ),

    .out_tx_clk             (symbol_clk             ),
    .out_tx_data            (out_tx_data            ),
    .out_tx_data_k          (out_tx_data_k          ),
    .out_tx_data_valid      (out_tx_data_valid      ),
    .out_tx_start_block     (out_tx_start_block     ),
    .out_tx_sync_header     (out_tx_sync_header     ),
    .out_tx_compliance      (out_tx_compliance      ),
    .out_tx_elec_idle       (out_tx_elec_idle       )
);      

// 8b to 10b Encoder; parallel to serial conversion
bfmp_tx_8b_to_10b #(

    .SIM_EL_IDLE_TYPE       (SIM_EL_IDLE_TYPE       )

) tx_8b_to_10b (

    .rst_n                  (rst_n & ~rate[1]       ),

    .serial_clk             (serial_clk             ),
    .symbol_clk             (symbol_clk             ),

    .tx_data                (out_tx_data            ),
    .tx_data_k              (out_tx_data_k          ),
    .tx_compliance          (out_tx_compliance      ),
    .tx_elec_idle           (out_tx_elec_idle       ),

    .tx_serial_stp_start    (tx_serial_stp          ),
    .tx_serial_sdp_start    (tx_serial_sdp          ),

    .tx_p                   (tx_8b_p                ),
    .tx_n                   (tx_8b_n                )

);      

// 128b to 130b Encoder; parallel to serial conversion
bfmp_tx_128b_to_130b #(

    .SIM_EL_IDLE_TYPE       (SIM_EL_IDLE_TYPE       ),
    .PHY_K_WIDTH            (PHY_K_WIDTH            )

) tx_128b_to_130b (

    .rst_n                  (rst_n & rate[1]        ),

    .serial_clk             (serial_clk             ),
    .symbol_clk             (symbol_clk             ),

    .tx_data                (out_tx_data            ),
    .tx_data_valid          (out_tx_data_valid      ),
    .tx_start_block         (out_tx_start_block     ),
    .tx_sync_header         (out_tx_sync_header     ),
    .tx_elec_idle           (out_tx_elec_idle       ),

    .tx_p                   (tx_128b_p              ),
    .tx_n                   (tx_128b_n              )

);

// Select which serial stream to use
always @*
begin
    if (rate[1])
    begin // 8g : use serial clock with data_next as clock enable
        c_tx_p = tx_128b_p;
        c_tx_n = tx_128b_n;
    end
    else
    begin // 2.5g & 5g : user byte clock, with no clock enable
        c_tx_p = tx_8b_p;
        c_tx_n = tx_8b_n;
    end
end

// If en_loopback == 1, loop serial RX back to serial TX
assign tx_p = en_loopback_data ? rx_p : (en_data ? c_tx_p : 1'b0);
assign tx_n = en_loopback_data ? rx_n : (en_data ? c_tx_n : 1'b0);

// --------------------
// Model Slave Loopback

assign en_loopback = (power_down == 2'b00) & // P0
                     tx_detect_rx_loopback &
                     ~tx_elec_idle;

always @(posedge pclk or negedge rst_n) begin
    if (rst_n == 1'b0) begin
        en_loopback_ctr  <= 8'h0;
        en_loopback_data <= 1'b0;
        en_data          <= 1'b1;
        d_en_loopback    <= 1'b0;
    end
    else if (pclk_en) begin
        d_en_loopback <= en_loopback;

        if (en_loopback != d_en_loopback)
            en_loopback_ctr <= 8'h7f;
        else if (en_loopback_ctr != 8'h0)
            en_loopback_ctr <= en_loopback_ctr - 8'h1;

        en_loopback_data <=  d_en_loopback & (en_loopback_ctr == 8'h0);
        en_data          <= ~d_en_loopback & (en_loopback_ctr == 8'h0);
    end
end

// ----------------------
// Model Reset phy_status

always @(posedge pclk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        start_phy_status_ctr <= START_DLY;
        start_phy_status     <= 1'b1;
    end
    else if (pclk_en)
    begin
        if (start_phy_status_ctr != 8'h00)
            start_phy_status_ctr <= start_phy_status_ctr - 8'h1;

        if (start_phy_status_ctr == 8'h1)
            start_phy_status <= 1'b0;
    end
end



// ---------------------
// Model Receiver Detect

assign rx_detect = (power_down == 2'b10) & // P1
                   tx_detect_rx_loopback &
                   tx_elec_idle;

// Look for Receiver Detect
always @(posedge pclk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        r1_rx_detect   <= 1'b0;
        r2_rx_detect   <= 1'b0;

        det_ctr        <= 4'h0;
        det_phy_status <= 1'b0;
    end
    else if (pclk_en)
    begin
        r1_rx_detect <=    rx_detect;
        r2_rx_detect <= r1_rx_detect;
        
        if (r1_rx_detect & ~r2_rx_detect)
            det_ctr <= DET_DLY;
        else if (det_ctr != 4'h0)
            det_ctr <= det_ctr - 4'h1;

        det_phy_status <= (det_ctr == 4'h1);
    end
end

// Encode possible receiver detect status
assign det_rx_status_yes = 3'b011; // Receiver detected
assign det_rx_status_no  = 3'b000; // Receiver not detected

// Use port to decide whether Receivers are detected or not
assign det_rx_status = lane_mask ? det_rx_status_no :
                                   det_rx_status_yes;



// ------------------------
// Model Power State Change

always @(posedge pclk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        r1_power_down     <= 2'b10;
        r2_power_down     <= 2'b10;

        pd_phy_status_ctr <= 4'h0;
        pd_phy_status     <= 1'b0;
    end
    else if (pclk_en)
    begin
        // Emulate phy_physts resulting from succesful power change
        r1_power_down <=    power_down;
        r2_power_down <= r1_power_down;

        if (r1_power_down != r2_power_down)
            pd_phy_status_ctr <= PD_DLY;
        else if (pd_phy_status_ctr != 4'h0)
            pd_phy_status_ctr <= pd_phy_status_ctr - 4'h1;

        pd_phy_status <= (pd_phy_status_ctr == 4'h1);
    end
end



// -----------------
// Model Rate Change

always @(posedge pclk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        r1_rate             <= 2'b0;
        r2_rate             <= 2'b0;

        rate_phy_status_ctr <= 4'h0;
        rate_phy_status     <= 1'b0;
    end
    else if (pclk_en)
    begin
        // Emulate phy_physts resulting from succesful power change
        r1_rate <= rate;
        r2_rate <= r1_rate;
 
        if (r1_rate != r2_rate)
            rate_phy_status_ctr <= RATE_DLY;
        else if (rate_phy_status_ctr != 4'h0)
            rate_phy_status_ctr <= rate_phy_status_ctr - 4'h1;

        rate_phy_status <= (rate_phy_status_ctr == 4'h1);
    end
end



// ----------------------
// Combine Status Sources

// Combine sources of phy_status
assign phy_status = start_phy_status | det_phy_status | pd_phy_status | rate_phy_status;

// Receiver Detect gets priority
assign rx_status = det_phy_status ? det_rx_status : norm_rx_status;

always @(posedge pclk)
begin
    if (rst_n == 1)
    begin
        if ((power_down >= 2'b10) & (rate != r1_rate) & (rate <= 2'h2)) // Rate change when PHY Power Down == P1 or P2
            $display ("%m : ERROR : PHY speed change was requested while PHY was in the P%h Power State; old_rate == %hG%h, new_rate == %hG%h",
                ((power_down == 2'h3) ? 4'h2 : 4'h1),
                ((r1_rate == 2'h2) ? 4'h8 : ((r1_rate == 2'h1) ? 4'h5 : 4'h2)),
                ((r1_rate == 2'h2) ? 4'h0 : ((r1_rate == 2'h1) ? 4'h0 : 4'h5)),
                ((   rate == 2'h2) ? 4'h8 : ((   rate == 2'h1) ? 4'h5 : 4'h2)),
                ((   rate == 2'h2) ? 4'h0 : ((   rate == 2'h1) ? 4'h0 : 4'h5)) );

        if (rate > 2'h2) // Illegal rate value
            $display ("%m : ERROR : PHY rate == %h is set to an illegal value", rate);
    end
end



endmodule
// -------------------------------------------------------------------------
//
//  PROJECT: PCI Express Verification Suite
//  COMPANY: Northwest Logic, Inc.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//             (c) Copyright 2009 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or 
//  transmitted in any form or by any means, electronic or mechanical, 
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//  
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

// -------------------------------------------------------------------------
// 
// pcie_pipe_phy
// 
// This is a PIPE PHY Model implementing the main features of the
//   Intel "PHY Interface for the PCI Express(TM) Architecture Version 2.00"
//
// This module is behavioral and is not designed to be synthesizable or
//   targeted to hardware
//
// This module is included as part of the Northwest Logic PCI Express
//   Verification Suite and is intended to be replaced by a third party
//   synthesizable PHY for hardware designs.
// 
// For simulation, this module implements the main modes of operation
//   of a standard PHY; this module is not intended to be a real or 
//   accurate PHY but rather to quickly and simply model a PCIe PHY
//   to enable simulation when a synthesizable PHY is not available.
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps


// -----------------------
// -- Module Definition --
// -----------------------

module bfmp_pcie_pipe_phy (

    rst_n,                  // Asynchronous reset; resets transmitter and receiver when == 0

    pclk,                   // Parallel interface clock; clock used for all ports except serial TX/RX

    tx_p,                   // Transmit serial stream; positive
    tx_n,                   // Transmit serial stream; negative

    rx_p,                   // Received serial stream; positive
    rx_n,                   // Received serial stream; negative

    tx_swing,               // Analog transmitter control; not modeled; 1 == Low Swing; 0 == Full Swing
    tx_margin,              // Analog transmitter control; not modeled; 000=Normal Operation; others for test
    tx_deemph,              // Analog transmitter control; not modeled

    tx_detect_rx_loopback,  // Set for initiating Tx Detect or Rx Loopback sequences
    power_down,             // 00 == P0; 01 == P0s; 10 == P1; 11 == P2
    rate,                   // PHY Rate; affects PCLK output; 0 == 2.5G; 1 == 5G
    phy_status,             // Set to communicate completion of Tx Detect, rate chage, & power_down transitions; on reset stays set until PCLK valid 
    rx_polarity,            // 1-bit/lane; 1 == invert serial receive polarity; 0 == do not invert

    tx_data,                // Transmit Data Bytes
    tx_data_k,              // For each bit, 1 == the corresponding data byte is a "K" character; 0 == "D" character
    tx_compliance,          // For each bit, 1 == encode the corresponding data byte, K assuming disparity is currently negative; 0 == normal operation
    tx_elec_idle,           // Set to idle transmitter (emulated with 'z')
  
    rx_data,                // Received Data Bytes
    rx_data_k,              // For each bit, 1 == the corresponding data byte is a "K" character; 0 == "D" character
    rx_valid,               // For each bit, 1 == symbol lock and valid data on rx_data, rx_data_k; 0 == no lock
    rx_status,              // 3-bits/lane; 000==DataOK; 001==SKPAdd; 010=SKPDel; 011==RxDetected; 100=Rx8b10b Error; 101==OverFlow; 110==Underflow; 111=RxDisparityError
    rx_elec_idle            // 1 == Receiver is in electrical idle; 0 == receiver is active

);      



// ----------------
// -- Parameters --
// ----------------

// The following parameter controls the lanes that will be modelled as 
//   detecting/not detecting receivers; to cause a lane not to detect 
//   a receiver, set (1) the corresponding bit in PIPE_PHY_LANE_MASK;
//     For example: PHY_LANE_MASK == 16'h800C : Lanes[15, 3:2] will 
//       emulate being disconnected
//   Note: The serial transmit lines of lanes that are modelled as
//   disconnected will output High-Z
parameter   PIPE_PHY_LANE_MASK      = 16'h0000;

parameter   HALF_PERIOD_8G          = 62.5; // 8 GHz
parameter   HALF_PERIOD_5G          = 100;  // 5 GHz
parameter   HALF_PERIOD_2G5         = 200;  // 2.5 GHz

parameter   SIM_EL_IDLE_TYPE        = 2'b10; // Electrical Idle Emulation: 11 == 1'b1 : Common Mode 1
                                             //                            10 == 1'b0 : Common Mode 0
                                             //                            01 == 1'bx : Undefined
                                             //                            00 == Reserved

parameter   RX_IDLE_ACTIVE_8G_ONLY_EIE = 0; // Set to 1 to cause rx_elec_idle to be 0 at 8G speed only when EIEOS == {8{8'h00, 8'ff}} is received;                                            0 == Use emulated analog comparator 
parameter   RX_IDLE_ACTIVE_5G_ONLY_EIE = 0; // Set to 1 to cause rx_elec_idle to be 0 at 5G speed only when EIE == symbols are received back-back; alternating pattern of 5 zeros and 5 ones; 0 == Use emulated analog comparator 
parameter   RX_IDLE_RANDOM_WHEN_ACTIVE = 0; // When receiver is not idle, value of rx_elec_idle is random


localparam  NUM_LANES               = 8;    

localparam  CORE_DATA_WIDTH         = 64;
localparam  CORE_K_WIDTH            = 8;
localparam  CORE_STS_WIDTH          = 24;

parameter   PHY_K_WIDTH             = 1;
localparam  PHY_DATA_WIDTH          = PHY_K_WIDTH * 8;
localparam  PHY_STS_WIDTH           = PHY_K_WIDTH * 3;



// ----------------------
// -- Port Definitions --
// ----------------------

input                               rst_n;

output                              pclk;

output  [NUM_LANES-1:0]             tx_p;
output  [NUM_LANES-1:0]             tx_n;

input   [NUM_LANES-1:0]             rx_p;
input   [NUM_LANES-1:0]             rx_n;

input                               tx_swing;
input   [2:0]                       tx_margin;
input   [NUM_LANES-1:0]             tx_deemph;

input   [NUM_LANES-1:0]             tx_detect_rx_loopback;
input   [(NUM_LANES*2)-1:0]         power_down;
input   [NUM_LANES-1:0]             rate;
output  [NUM_LANES-1:0]             phy_status;
input   [NUM_LANES-1:0]             rx_polarity;

input   [CORE_DATA_WIDTH-1:0]       tx_data;
input   [CORE_K_WIDTH-1:0]          tx_data_k;
input   [CORE_K_WIDTH-1:0]          tx_compliance;
input   [NUM_LANES-1:0]             tx_elec_idle;
  
output  [CORE_DATA_WIDTH-1:0]       rx_data;
output  [CORE_K_WIDTH-1:0]          rx_data_k;
output  [NUM_LANES-1:0]             rx_valid;
output  [CORE_STS_WIDTH-1:0]        rx_status;
output  [NUM_LANES-1:0]             rx_elec_idle;



// ----------------
// -- Port Types --
// ----------------

wire                                rst_n;

wire                                pclk;
wire                                pclk_en;

wire    [NUM_LANES-1:0]             tx_p;
wire    [NUM_LANES-1:0]             tx_n;

wire    [NUM_LANES-1:0]             rx_p;
wire    [NUM_LANES-1:0]             rx_n;

wire                                tx_swing;
wire    [2:0]                       tx_margin;
wire    [NUM_LANES-1:0]             tx_deemph;

wire    [NUM_LANES-1:0]             tx_detect_rx_loopback;
wire    [(NUM_LANES*2)-1:0]         power_down;
wire    [NUM_LANES-1:0]             rate;
wire    [NUM_LANES-1:0]             phy_status;
wire    [NUM_LANES-1:0]             rx_polarity;

wire    [CORE_DATA_WIDTH-1:0]       tx_data;
wire    [CORE_K_WIDTH-1:0]          tx_data_k;
wire    [CORE_K_WIDTH-1:0]          tx_compliance;
wire    [NUM_LANES-1:0]             tx_elec_idle;
  
wire    [CORE_DATA_WIDTH-1:0]       rx_data;
wire    [CORE_K_WIDTH-1:0]          rx_data_k;
wire    [NUM_LANES-1:0]             rx_valid;
wire    [CORE_STS_WIDTH-1:0]        rx_status;
wire    [NUM_LANES-1:0]             rx_elec_idle;



// -------------------
// -- Local Signals --
// -------------------

genvar                              i;

wire    [(NUM_LANES*2)-1:0]         link_speed;

// Instantiate Behavioral PLL
wire                                common_pll_lock;
wire                                symbol_clk;     
wire                                serial_clk;     

// Instantiate PHY Lane Model   
wire    [NUM_LANES-1:0]             c_tx_p;
wire    [NUM_LANES-1:0]             c_tx_n;

// Model Lanes Not Connected
wire    [NUM_LANES-1:0]             size_pipe_phy_lane_mask;

wire    [NUM_LANES-1:0]             rx_serial_stp;
wire    [NUM_LANES-1:0]             rx_serial_sdp;
wire    [NUM_LANES-1:0]             tx_serial_stp;
wire    [NUM_LANES-1:0]             tx_serial_sdp;

wire                                any_rx_serial_stp;
wire                                any_rx_serial_sdp;
wire                                any_tx_serial_stp;
wire                                any_tx_serial_sdp;

// ---------------
// -- Equations --
// ---------------

// Current serial link speed of operation
generate for (i=0; i<NUM_LANES; i=i+1)
    begin : gen_link_speed
        assign link_speed[((i+1)*2)-1:(i*2)] = {1'b0, rate[i]};
    end
endgenerate



// --------------------------
// Instantiate Behavioral PLL

bfmp_pcie_phy_pll #(

    .PHY_K_WIDTH        (PHY_K_WIDTH        ),
    .HALF_PERIOD_8G     (HALF_PERIOD_8G     ), 
    .HALF_PERIOD_5G     (HALF_PERIOD_5G     ),
    .HALF_PERIOD_2G5    (HALF_PERIOD_2G5    )

) pcie_phy_pll (

    .rst_n              (rst_n              ),
    .rate               (link_speed[1:0]    ),

    .lock               (common_pll_lock    ),
    .serial_clk         (serial_clk         ),
    .symbol_clk         (symbol_clk         ),
    .pclk               (pclk               ),
    .pclk_en            (pclk_en            )
);


// --------------------------
// Instantiate PHY Lane Model

// Generate the individual lanes
generate
    for (i=0; i<NUM_LANES; i=i+1)
    begin : gen_phy
        bfmp_pcie_phy #(

            .LANE_NUMBER                (i                                                              ),
            .SIM_EL_IDLE_TYPE           (SIM_EL_IDLE_TYPE                                               ),
            .PHY_K_WIDTH                (PHY_K_WIDTH                                                    ),
            .RX_IDLE_ACTIVE_8G_ONLY_EIE (RX_IDLE_ACTIVE_8G_ONLY_EIE                                     ),
            .RX_IDLE_ACTIVE_5G_ONLY_EIE (RX_IDLE_ACTIVE_5G_ONLY_EIE                                     ),
            .RX_IDLE_RANDOM_WHEN_ACTIVE (RX_IDLE_RANDOM_WHEN_ACTIVE                                     )

        ) pcie_phy (

            .rst_n                  (common_pll_lock                                                    ),
            .lane_mask              (size_pipe_phy_lane_mask[i]                                         ),

            .serial_clk             (serial_clk                                                         ),
            .symbol_clk             (symbol_clk                                                         ),
            .pclk                   (pclk                                                               ),
            .pclk_en                (pclk_en                                                            ),

            .tx_p                   (c_tx_p                [  i                                        ]),
            .tx_n                   (c_tx_n                [  i                                        ]),

            .rx_p                   (rx_p                  [  i                                        ]),
            .rx_n                   (rx_n                  [  i                                        ]),

            .tx_detect_rx_loopback  (tx_detect_rx_loopback [  i                                        ]),
            .power_down             (power_down            [((i+1)*2             )-1:(i*2             )]),
            .rate                   (link_speed            [((i+1)*2             )-1:(i*2             )]),
            .phy_status             (phy_status            [  i                                        ]),
            .rx_polarity            (rx_polarity           [  i                                        ]),

            .tx_data                (tx_data               [((i+1)*PHY_DATA_WIDTH)-1:(i*PHY_DATA_WIDTH)]),
            .tx_data_k              (tx_data_k             [((i+1)*PHY_K_WIDTH   )-1:(i*PHY_K_WIDTH   )]),
            .tx_deemph              (tx_deemph             [  i                                        ]),
            .tx_data_valid          (1'b1                                                               ),
            .tx_start_block         (1'b0                                                               ),
            .tx_sync_header         (2'b00                                                              ),
            .tx_compliance          (tx_compliance         [((i+1)*PHY_K_WIDTH   )-1:(i*PHY_K_WIDTH   )]),
            .tx_elec_idle           (tx_elec_idle          [  i                                        ]),
                                                                
            .rx_data                (rx_data               [((i+1)*PHY_DATA_WIDTH)-1:(i*PHY_DATA_WIDTH)]),
            .rx_data_k              (rx_data_k             [((i+1)*PHY_K_WIDTH   )-1:(i*PHY_K_WIDTH   )]),
            .rx_data_valid          (                                                                   ),
            .rx_start_block         (                                                                   ),
            .rx_sync_header         (                                                                   ),
            .rx_block_realign       (1'b0                                                               ),
            .rx_valid               (rx_valid              [  i                                        ]),
            .rx_status              (rx_status             [((i+1)*PHY_STS_WIDTH )-1:(i*PHY_STS_WIDTH )]),
            .rx_elec_idle           (rx_elec_idle          [  i                                        ]),

            .rx_serial_stp          (rx_serial_stp         [  i                                        ]),
            .rx_serial_sdp          (rx_serial_sdp         [  i                                        ]),
            .tx_serial_stp          (tx_serial_stp         [  i                                        ]),
            .tx_serial_sdp          (tx_serial_sdp         [  i                                        ])
        );
    end
endgenerate

assign   any_rx_serial_stp = |rx_serial_stp;
assign   any_rx_serial_sdp = |rx_serial_sdp;
assign   any_tx_serial_stp = |tx_serial_stp;
assign   any_tx_serial_sdp = |tx_serial_sdp;



// -------------------------
// Model Lanes Not Connected
  
// Size parameter
assign size_pipe_phy_lane_mask = PIPE_PHY_LANE_MASK;

// Emulate lanes not connected with High-Z
generate
    for (i=0; i<NUM_LANES; i=i+1)
    begin : gen_tx_p_n
        // No receiver present is expected to be modeled by High-Z on rx_p and rx_n
        assign tx_p[i] = (size_pipe_phy_lane_mask[i] == 1'b1) ? 1'bz : c_tx_p[i];
        assign tx_n[i] = (size_pipe_phy_lane_mask[i] == 1'b1) ? 1'bz : c_tx_n[i];
    end
endgenerate



endmodule
// -------------------------------------------------------------------------
//
//  PROJECT: PCI Express Core
//  COMPANY: Northwest Logic, Inc.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//             (c) Copyright 2009 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or
//  transmitted in any form or by any means, electronic or mechanical,
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module bfmp_pipe_if (

    rst_n,                      // Asynchronous PHY reset

    tx_p,                       // Serial transmit differential pair(s)
    tx_n,                       //

    rx_p,                       // Serial receive differential pair(s)
    rx_n,                       //

    phy_rst_n,                  // Synchronized rst_n to phy_clk
    phy_clk,                    // Parallel interface clock (clock used for all phy_* ports)
    phy_clk_period_in_ns,       // Period of phy_clk in nanoseconds

    phy_tx_swing,
    phy_tx_margin,
    phy_tx_deemph,

    phy_tx_detect_rx_loopback,
    phy_power_down,
    phy_rate,
    phy_phy_status,
    phy_rx_polarity,

    phy_tx_data,
    phy_tx_data_k,
    phy_tx_compliance,
    phy_tx_elec_idle,

    phy_rx_data,
    phy_rx_data_k,
    phy_rx_data_valid,
    phy_rx_valid,
    phy_rx_status,
    phy_rx_elec_idle

);



// ----------------
// -- Parameters --
// ----------------

// The following parameter controls the lanes that will be modelled as
//   detecting/not detecting receivers; to cause a lane not to detect
//   a receiver, set (1) the corresponding bit in PIPE_PHY_LANE_MASK;
//     For example: PHY_LANE_MASK == 16'h800C : Lanes[15, 3:2] will
//       emulate being disconnected
//   Note: The serial transmit lines of lanes that are modelled as
//   disconnected will output High-Z
parameter   PIPE_PHY_LANE_MASK      = 16'h0000;

parameter   HALF_PERIOD_5G          = 100;  // 5 GHz
parameter   HALF_PERIOD_2G5         = 200;  // 2.5 GHz

parameter   SIM_EL_IDLE_TYPE        = 2'b10; // Electrical Idle Emulation: 11 == 1'b1 : Common Mode 1
                                             //                            10 == 1'b0 : Common Mode 0
                                             //                            01 == 1'bx : Undefined
                                             //                            00 == Reserved

parameter   RX_IDLE_ACTIVE_8G_ONLY_EIE = 0; // Set to 1 to cause rx_elec_idle to be 0 at 8G speed only when EIEOS == {8{8'h00, 8'ff}} is received;                                            0 == Use emulated analog comparator
parameter   RX_IDLE_ACTIVE_5G_ONLY_EIE = 0; // Set to 1 to cause rx_elec_idle to be 0 at 5G speed only when EIE == symbols are received back-back; alternating pattern of 5 zeros and 5 ones; 0 == Use emulated analog comparator
parameter   RX_IDLE_RANDOM_WHEN_ACTIVE = 0; // When receiver is not idle, value of rx_elec_idle is random

// ---------------------------------------------------------------------------
// NOTE: None of the following parameters are available for user's to modify;
//       They are provided for Northwest Logic core maintanence purposes only.
//       Northwest Logic will only support the shipped parameter configuration.
//       The mgmt_cfg_constants amd mgmt_cfg_control ports are provided for
//       users to personalize the core for their application.

localparam  NUM_LANES               = 8;

localparam  CORE_DATA_WIDTH         = 64;
localparam  CORE_K_WIDTH            = 8;
localparam  CORE_STS_WIDTH          = 24;

localparam  FDBK_BITS               = 8;   // Number of bits provided by PHY when reporting equalization quality



// ----------------------
// -- Port Definitions --
// ----------------------

input                               rst_n;

output  [NUM_LANES-1:0]             tx_p;
output  [NUM_LANES-1:0]             tx_n;

input   [NUM_LANES-1:0]             rx_p;
input   [NUM_LANES-1:0]             rx_n;

output                              phy_rst_n;
output                              phy_clk;
output  [5:0]                       phy_clk_period_in_ns;

input                               phy_tx_swing;
input   [2:0]                       phy_tx_margin;
input   [NUM_LANES-1:0]             phy_tx_deemph;

input   [NUM_LANES-1:0]             phy_tx_detect_rx_loopback;
input   [(NUM_LANES*2)-1:0]         phy_power_down;
input   [NUM_LANES-1:0]             phy_rate;
output  [NUM_LANES-1:0]             phy_phy_status;
input   [NUM_LANES-1:0]             phy_rx_polarity;

input   [CORE_DATA_WIDTH-1:0]       phy_tx_data;
input   [CORE_K_WIDTH-1:0]          phy_tx_data_k;
input   [CORE_K_WIDTH-1:0]          phy_tx_compliance;
input   [NUM_LANES-1:0]             phy_tx_elec_idle;

output  [CORE_DATA_WIDTH-1:0]       phy_rx_data;
output  [CORE_K_WIDTH-1:0]          phy_rx_data_k;
output  [NUM_LANES-1:0]             phy_rx_data_valid;
output  [NUM_LANES-1:0]             phy_rx_valid;
output  [(CORE_K_WIDTH*3)-1:0]      phy_rx_status;
output  [NUM_LANES-1:0]             phy_rx_elec_idle;



// ----------------
// -- Port Types --
// ----------------

wire                                rst_n;

wire    [NUM_LANES-1:0]             tx_p;
wire    [NUM_LANES-1:0]             tx_n;

wire    [NUM_LANES-1:0]             rx_p;
wire    [NUM_LANES-1:0]             rx_n;

wire                                phy_rst_n;
wire                                phy_clk;
reg     [5:0]                       phy_clk_period_in_ns;

wire                                phy_tx_swing;
wire    [2:0]                       phy_tx_margin;
wire    [NUM_LANES-1:0]             phy_tx_deemph;

wire    [NUM_LANES-1:0]             phy_tx_detect_rx_loopback;
wire    [(NUM_LANES*2)-1:0]         phy_power_down;
wire    [NUM_LANES-1:0]             phy_rate;
wire    [NUM_LANES-1:0]             phy_phy_status;
wire    [NUM_LANES-1:0]             phy_rx_polarity;

wire    [CORE_DATA_WIDTH-1:0]       phy_tx_data;
wire    [CORE_K_WIDTH-1:0]          phy_tx_data_k;
wire    [CORE_K_WIDTH-1:0]          phy_tx_compliance;
wire    [NUM_LANES-1:0]             phy_tx_elec_idle;

wire    [CORE_DATA_WIDTH-1:0]       phy_rx_data;
wire    [CORE_K_WIDTH-1:0]          phy_rx_data_k;
wire    [NUM_LANES-1:0]             phy_rx_data_valid;
wire    [NUM_LANES-1:0]             phy_rx_valid;
wire    [(CORE_K_WIDTH*3)-1:0]      phy_rx_status;
wire    [NUM_LANES-1:0]             phy_rx_elec_idle;



// -------------------
// -- Local Signals --
// -------------------

// Clock and Reset
reg                                             phy_ready;
reg                                             d2_pclk_rst_n;
reg                                             d1_pclk_rst_n;
reg                                             pclk_rst_n;

// Instantiate Behavioral PIPE PHY
wire                                            pclk;


wire    [CORE_DATA_WIDTH-1:0]                   i_phy_tx_data;
wire    [CORE_K_WIDTH-1:0]                      i_phy_tx_data_k;
wire    [NUM_LANES-1:0]                         i_phy_tx_elec_idle;
wire    [CORE_K_WIDTH-1:0]                      i_phy_tx_compliance;

wire    [CORE_DATA_WIDTH-1:0]                   i_phy_rx_data;
wire    [CORE_K_WIDTH-1:0]                      i_phy_rx_data_k;
wire    [NUM_LANES-1:0]                         i_phy_rx_valid;
wire    [(CORE_K_WIDTH*3)-1:0]                  i_phy_rx_status;
wire    [NUM_LANES-1:0]                         i_phy_rx_elec_idle;
wire    [NUM_LANES-1:0]                         i_phy_phy_status_c;

wire                                            rx_fifo_rd_valid;

wire    [NUM_LANES-1:0]                         phy_phy_status_c;



// ---------------
// -- Equations --
// ---------------

// ---------------
// Clock and Reset

// Output PHY Clock Period
always @*
begin
    case (phy_rate[0])
        1'b1    : phy_clk_period_in_ns = 6'd2;
        default : phy_clk_period_in_ns = 6'd4;
    endcase
end

always @(posedge pclk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        phy_ready     <= 1'b0;
        d2_pclk_rst_n <= 1'b0;
        d1_pclk_rst_n <= 1'b0;
        pclk_rst_n    <= 1'b0;
    end
    else
    begin
        // Hold core in reset until the PHY is ready for operation
        //   as indicated by phy_phy_status going low after reset
        if (phy_phy_status_c == 1'b0)
            phy_ready <= 1'b1;

        d2_pclk_rst_n <= phy_ready;
        d1_pclk_rst_n <= d2_pclk_rst_n;
        pclk_rst_n    <= d1_pclk_rst_n;
    end
end

assign phy_clk   = pclk;
assign phy_rst_n = pclk_rst_n;



// -------------------------------
// Instantiate Behavioral PIPE PHY

bfmp_pcie_pipe_phy #(

    .PIPE_PHY_LANE_MASK         (PIPE_PHY_LANE_MASK         ),

    .HALF_PERIOD_5G             (HALF_PERIOD_5G             ),
    .HALF_PERIOD_2G5            (HALF_PERIOD_2G5            ),

    .SIM_EL_IDLE_TYPE           (SIM_EL_IDLE_TYPE           ),
    .RX_IDLE_ACTIVE_8G_ONLY_EIE (RX_IDLE_ACTIVE_8G_ONLY_EIE ),
    .RX_IDLE_ACTIVE_5G_ONLY_EIE (RX_IDLE_ACTIVE_5G_ONLY_EIE ),
    .RX_IDLE_RANDOM_WHEN_ACTIVE (RX_IDLE_RANDOM_WHEN_ACTIVE )

) phy (

    .rst_n                      (rst_n                      ),

    .pclk                       (pclk                       ),

    .tx_p                       (tx_p                       ),
    .tx_n                       (tx_n                       ),

    .rx_p                       (rx_p                       ),
    .rx_n                       (rx_n                       ),

    .tx_swing                   (phy_tx_swing               ),
    .tx_margin                  (phy_tx_margin              ),
    .tx_deemph                  (phy_tx_deemph              ),

    .tx_detect_rx_loopback      (phy_tx_detect_rx_loopback  ),
    .power_down                 (phy_power_down             ),
    .rate                       (phy_rate                   ),
    .phy_status                 (i_phy_phy_status_c         ),
    .rx_polarity                (phy_rx_polarity            ),

    .tx_data                    (i_phy_tx_data              ),
    .tx_data_k                  (i_phy_tx_data_k            ),
    .tx_compliance              (i_phy_tx_compliance        ),
    .tx_elec_idle               (i_phy_tx_elec_idle         ),

    .rx_data                    (i_phy_rx_data              ),
    .rx_data_k                  (i_phy_rx_data_k            ),
    .rx_valid                   (i_phy_rx_valid             ),
    .rx_status                  (i_phy_rx_status            ),
    .rx_elec_idle               (i_phy_rx_elec_idle         )

);


assign i_phy_tx_data        = phy_tx_data;
assign i_phy_tx_data_k      = phy_tx_data_k;
assign i_phy_tx_compliance  = phy_tx_compliance;
assign i_phy_tx_elec_idle   = phy_tx_elec_idle;


assign phy_rx_data            = i_phy_rx_data;
assign phy_rx_data_k          = i_phy_rx_data_k;
assign phy_rx_valid           = i_phy_rx_valid;
assign phy_rx_status          = i_phy_rx_status;
assign phy_rx_elec_idle       = i_phy_rx_elec_idle;

assign rx_fifo_rd_valid = 1'b1;

assign phy_phy_status_c = i_phy_phy_status_c;


assign phy_phy_status = phy_phy_status_c;
assign phy_rx_data_valid  = {NUM_LANES{1'b1}};

endmodule
// ------------------------- CONFIDENTIAL ----------------------------------
//
//                 Copyright 2011 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or
//  transmitted in any form or by any means, electronic or mechanical,
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

// -------------------------------------------------------------------------
//
//  FUNCTIONAL DESCRIPTION
//
//  Dual (DUAL_CLOCK == 1) or Single clock (DUAL_CLOCK == 0) domain FIFO
//
//  When configured for Dual Clock operation, this module implements
//    a dual clock FIFO that uses Gray Code conversion to handle FIFO
//    level synchronization between the two clock domains.
//         
//  When configured for Single Clock operation, this module implements
//    lower-latency synchronous level computations.
//
//  Includes an output latency reduction circuit; rd_data is available
//    the same clock as rd_valid.
//
//  Includes optional RAM output register (REGISTER_OUTPUT == 1) for
//    higher frequency operation.
//    
//  When configured for Dual Clock operation, this module includes 
//    functionality to cleanly handle only one of the clock domains 
//    being reset.  The ports wr_dc_eop and rd_dc_eop identify packet
//    boundaries on which single clock domain resets should be handled.
//    wr_dc_err indicates when write data fails to transfer due to read
//    clock domain being in reset; rd_dc_err indicates when a read packet
//    was started, but not finished prior to the write clock domain going
//    into reset; when rd_dc_err asserts, this indicates that the current
//    read packet will not complete normally due to a write domain reset;
//    the user should use rd_dc_err to terminate (EOP) the current read
//    packet with error status.  rd_data is invalid when rd_dc_err == 1.
//    If the user application does not use multi-clock cycle packets,
//    then the user must tie wr_dc_eop = 1 and rd_dc_eop = 1 to indicate
//    that a single clock domain reset is allowed on any FIFO word.
//
//  When configured for Single Clock operation, wr_dc_eop, rd_rd_eop are
//    not used, and must be tied to 0 or 1, and wr_dc_err and rd_dc_err are
//    not used and output constant 0.
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module bfmp_util_fifo (

    wr_rst_n,
    wr_clk,
    wr_valid,
    wr_ready,
    wr_aready,
    wr_dc_eop,
    wr_dc_err,
    wr_data,

    rd_rst_n,
    rd_clk,
    rd_valid,
    rd_ready,
    rd_dc_eop,
    rd_dc_err,
    rd_data

);



// ----------------
// -- Parameters --
// ----------------

parameter   DATA_WIDTH          = 64;               // Set to desired number of RAM data bits
parameter   ADDR_WIDTH          = 7;                // Set to desired number of RAM address bits
localparam  READY_LEVEL         = 1 << ADDR_WIDTH;  // Level for which wr_ready  de-asserts; intended to be used for full indication
parameter   AREADY_LEVEL        = 1 << ADDR_WIDTH;  // Level for which wr_aready de-asserts; intended to be used for almost full indication
parameter   DUAL_CLOCK          = 1;                // Set to 0 when ((wr_clk == rd_clk) & (wr_rst_n == rd_rst_n)); set to 1 otherwise (include synchronization)
parameter   REGISTER_OUTPUT     = 1;                // Set to 1 to add an output register on RAM read data; 0 == to omit output register; in both cases the external
                                                    //   read latency is 0 (rd_data valid the same clock as rd_valid == 1); set to 1 for highest frequencies of operation



// -----------------------
// -- Port Declarations --
// -----------------------

input                           wr_rst_n;           // Active low asynchronous assert, wr_clk synchronous de-assert reset
input                           wr_clk;             // Positive edge clock for write clock domain
input                           wr_valid;           // Write data is transferred when ((wr_valid == 1) & (wr_ready == 1))
output                          wr_ready;           //   wr_valid == source ready; wr_ready == destination ready
output                          wr_aready;          // Almost full write ready; deasserted when write level >= AREADY_LEVEL
input                           wr_dc_eop;          // wr_dc_eop and wr_dc_err are only used when DUAL_CLOCK==1 to help the FIFO cleanly handle the read clock domain being reset while the write clock domain is not; 
output                          wr_dc_err;          //   on read reset exit, a write that was active while the read clock domain was in reset continues to be ignored until wr_dc_eop transfers;   
                                                    //   wr_dc_eop must be set to 1 on the last wr_data of a packet; tie wr_dc_eop = 1 if all packets are 1 word;
                                                    //   when (wr_valid & wr_ready & wr_dc_err) == 1, wr_data was not stored because the read clock domain was in reset
input   [DATA_WIDTH-1:0]        wr_data;            // FIFO write data

input                           rd_rst_n;           // Active low asynchronous assert, rd_clk synchronous de-assert reset
input                           rd_clk;             // Positive edge clock for read clock domain
output                          rd_valid;           // Read data is transferred when ((rd_valid == 1) & (rd_ready == 1))
input                           rd_ready;           //   rd_valid == source ready; rd_ready == destination ready
input                           rd_dc_eop;          // rd_dc_eop and rd_dc_err are only used when DUAL_CLOCK==1 to help the FIFO cleanly handle the write clock domain being reset while the read clock domain is not;  
output                          rd_dc_err;          //   if a read is active when a write reset occurs, the FIFO drives rd_valid == 1 & rd_error == 1 to indicate that the current packet will not terminate normally;
                                                    //   rd_dc_eop must be set to 1 on the last rd_data of a packet; tie rd_dc_eop = 1 if all packets are 1 word
output  [DATA_WIDTH-1:0]        rd_data;            // FIFO read data



// ----------------
// -- Port Types --
// ----------------

wire                            wr_rst_n;  
wire                            wr_clk;    
wire                            wr_valid;  
reg                             wr_ready;  
reg                             wr_aready; 
wire                            wr_dc_eop;
wire                            wr_dc_err;
wire    [DATA_WIDTH-1:0]        wr_data;   

wire                            rd_rst_n;  
wire                            rd_clk;    
wire                            rd_valid;  
wire                            rd_ready;  
wire                            rd_dc_eop;
wire                            rd_dc_err;
wire    [DATA_WIDTH-1:0]        rd_data;   



// ---------------------
// -- Local Variables --
// ---------------------

// FIFO Control
wire                            wr_en;
wire                            ram_rd_en;

wire                            rd_sync_wr_rst_n;

wire    [ADDR_WIDTH-1:0]        ram_wr_addr;
wire    [ADDR_WIDTH-1:0]        ram_rd_addr;

// Instantiate Output Stage to Reduce Read Latency to 0
reg                             ram_rd_valid;
wire                            ram_rd_ready;
wire    [DATA_WIDTH-1:0]        ram_rd_data;
wire                            rd_valid_out;



// ---------------
// -- Equations --
// ---------------

// ------------
// FIFO Control

assign wr_en     =     wr_valid &     wr_ready & ~wr_dc_err;
assign ram_rd_en = ram_rd_valid & ram_rd_ready;

generate if (DUAL_CLOCK == 0)
    begin : gen_sc_level
        reg     [ADDR_WIDTH-1:0]        wr_addr;
        reg     [ADDR_WIDTH-1:0]        rd_addr;
        reg     [ADDR_WIDTH:0]          c_level;
        reg     [ADDR_WIDTH:0]          level;

        // Not used in single clock mode
        assign wr_dc_err = 1'b0;
        assign rd_dc_err = 1'b0;

        // Synchronous reset is not needed for single clock FIFO
        assign rd_sync_wr_rst_n = 1'b1;

        always @(posedge wr_clk or negedge wr_rst_n)
        begin
            if (wr_rst_n == 1'b0)
            begin
                wr_addr <= {ADDR_WIDTH{1'b0}};
                rd_addr <= {ADDR_WIDTH{1'b0}};
            end
            else
            begin
                if (wr_en)
                    wr_addr <= wr_addr + {{(ADDR_WIDTH-1){1'b0}}, 1'b1};

                if (ram_rd_en)
                    rd_addr <= rd_addr + {{(ADDR_WIDTH-1){1'b0}}, 1'b1};
            end
        end

        always @*
        begin
            case ({wr_en, ram_rd_en})
                2'b01   : c_level = level - {{(ADDR_WIDTH-1){1'b0}}, 1'b1};
                2'b10   : c_level = level + {{(ADDR_WIDTH-1){1'b0}}, 1'b1};
                default : c_level = level;
            endcase
        end

        always @(posedge wr_clk or negedge wr_rst_n)
        begin
            if (wr_rst_n == 1'b0)
            begin
                level        <= {(ADDR_WIDTH+1){1'b0}};
                wr_ready     <= 1'b0;
                wr_aready    <= 1'b0;
                ram_rd_valid <= 1'b0;
            end
            else
            begin
                level        <=  c_level;
                wr_ready     <= (c_level <  READY_LEVEL);
                wr_aready    <= (c_level < AREADY_LEVEL);
                ram_rd_valid <= (c_level != {(ADDR_WIDTH+1){1'b0}});
            end
        end

        assign ram_wr_addr = wr_addr[ADDR_WIDTH-1:0];
        assign ram_rd_addr = rd_addr[ADDR_WIDTH-1:0];
    end
    else
    begin : gen_dc_level
        // Reset
        reg                             logic_rd_reset;
        reg                             logic_wr_reset;
        wire                            wr_sync_rd_reset;
        wire                            rd_sync_wr_reset;

        reg                             wr_active;
        reg                             wr_active_reset;

        reg                             rd_active;
        reg                             r_rd_dc_err;

        // Writes
        wire    [ADDR_WIDTH:0]          c_wr_addr;
        wire    [ADDR_WIDTH:0]          c_wr_addr_gray;
        reg     [ADDR_WIDTH:0]          wr_addr;
        reg     [ADDR_WIDTH:0]          wr_addr_gray;

        wire    [ADDR_WIDTH:0]          sync2_rd_addr_gray;
        wire    [ADDR_WIDTH:0]          wr_side_rd_addr;

        wire                            wr_diff_half;
        wire    [ADDR_WIDTH:0]          c_wr_level;

        // Reads
        wire    [ADDR_WIDTH:0]          c_rd_addr;
        wire    [ADDR_WIDTH:0]          c_rd_addr_gray;
        reg     [ADDR_WIDTH:0]          rd_addr;
        reg     [ADDR_WIDTH:0]          rd_addr_gray;

        wire    [ADDR_WIDTH:0]          sync2_wr_addr_gray;
        wire    [ADDR_WIDTH:0]          rd_side_wr_addr;

        wire                            rd_diff_half;
        wire    [ADDR_WIDTH:0]          c_rd_level;

        // -----
        // Reset

        // Avoid using rd_rst_n/wr_rst_n as logic inputs since they should only be used as resets
        always @(posedge rd_clk or negedge rd_rst_n) if (rd_rst_n == 1'b0) logic_rd_reset <= 1'b1; else logic_rd_reset <= 1'b0;
        always @(posedge wr_clk or negedge wr_rst_n) if (wr_rst_n == 1'b0) logic_wr_reset <= 1'b1; else logic_wr_reset <= 1'b0;
        // Need to synchronize the opposite clock domain reset into the local clock domain
        //   and use to synchronously reset the FIFO when the opposite side is in reset
        util_sync_flops util_sync_flops0 (.clk (wr_clk), .rst_n (wr_rst_n), .d (logic_rd_reset), .q (wr_sync_rd_reset));
        util_sync_flops util_sync_flops1 (.clk (rd_clk), .rst_n (rd_rst_n), .d (logic_wr_reset), .q (rd_sync_wr_reset));

        // -- Write Clock Domain - Handle read clock domain in reset while write clock domain is not

        always @(posedge wr_clk or negedge wr_rst_n)
        begin
            if (wr_rst_n == 1'b0)
                wr_active <= 1'b0;
            else if (wr_valid & wr_ready)
                wr_active <= ~wr_dc_eop;
        end

        // When a write packet starts while read clock domain is in reset, or a packet is active when reset occurs,
        //   hold this signal until the packet completes, so that the packet can be rejected
        always @(posedge wr_clk or negedge wr_rst_n)
        begin
            if (wr_rst_n == 1'b0)
                wr_active_reset <= 1'b0;
            else
            begin
                if      ( (wr_valid & wr_ready) &  wr_dc_eop)                    // Always clear on EOP transfer
                    wr_active_reset <= 1'b0;
                else if ( (wr_valid & wr_ready) & ~wr_dc_eop & wr_sync_rd_reset) // Set on non-EOP transfer while in reset
                    wr_active_reset <= 1'b1;
                else if (~(wr_valid & wr_ready) & wr_active  & wr_sync_rd_reset) // Needed for packets that started before reset and have no transfers during reset
                    wr_active_reset <= 1'b1;
            end
        end
        
        assign wr_dc_err = wr_sync_rd_reset | wr_active_reset;

        // -- Read Clock Domain - Handle write clock domain in reset while read clock domain is not

        always @(posedge rd_clk or negedge rd_rst_n)
        begin
            if (rd_rst_n == 1'b0)
                rd_active <= 1'b0;
            else
            begin
                if (rd_sync_wr_reset)
                    rd_active <= 1'b0;
                else if (rd_valid & rd_ready)
                    rd_active <= ~rd_dc_eop;
            end
        end

        always @(posedge rd_clk or negedge rd_rst_n)
        begin
            if (rd_rst_n == 1'b0)
            begin
                r_rd_dc_err <= 1'b0;
            end
            else
            begin
                // Write domain in reset while read interface packet started but not completed
                if (rd_sync_wr_reset & rd_active & ~(rd_valid & rd_ready & rd_dc_eop))
                    r_rd_dc_err <= 1'b1;
                else if (rd_valid & rd_ready)
                    r_rd_dc_err <= 1'b0;
            end
        end

        assign rd_dc_err = r_rd_dc_err;        

        // Hold read side in reset while write side remains in reset or r_rd_dc_err has not transferred
        assign rd_sync_wr_rst_n = ~(rd_sync_wr_reset | r_rd_dc_err);

        // ------
        // Writes

        // Generate next write address
        assign c_wr_addr = wr_en ? (wr_addr + {{ADDR_WIDTH{1'b0}}, 1'b1}) : wr_addr;

        // Convert next write address to its Gray Code equivalent
        //   Note: Using wr_addr instead of c_wr_addr to avoid RAM wr/rd collisions; if rd_clk is very fast relative
        //     to wr_clk, a RAM location that was just written could otherwise be read before the written data is stable
        util_bin_to_gray #(.WIDTH (ADDR_WIDTH+1)) util_bin_to_gray_c_wr_addr_gray (.d (wr_addr), .q (c_wr_addr_gray));

        always @(posedge wr_clk or negedge wr_rst_n)
        begin
            if (wr_rst_n == 1'b0)
            begin
                wr_addr      <= {(ADDR_WIDTH+1){1'b0}};
                wr_addr_gray <= {(ADDR_WIDTH+1){1'b0}};
            end
            else if (wr_dc_err)
            begin
                wr_addr      <= {(ADDR_WIDTH+1){1'b0}};
                wr_addr_gray <= {(ADDR_WIDTH+1){1'b0}};
            end
            else
            begin
                wr_addr      <= c_wr_addr;
                wr_addr_gray <= c_wr_addr_gray;
            end
        end

        // Transfer rd_addr_gray to wr_clk domain
        util_sync_flops_width #(.WIDTH (ADDR_WIDTH+1)) util_sync_flops_width0 (.clk (wr_clk), .rst_n (wr_rst_n), .d (rd_addr_gray), .q (sync2_rd_addr_gray));

        // Convert sync2_rd_addr_gray back to its binary equivalent
        util_gray_to_bin #(.WIDTH (ADDR_WIDTH+1)) util_gray_to_bin_wr_side_rd_addr (.d (sync2_rd_addr_gray), .q (wr_side_rd_addr));

        // If the extra address bit being carried in each of the FIFO addresses have
        //   different values, then the write address has wrapped relative to the read
        //   address and 2^ADDR_WIDTH must be added to the write address in order to
        //   determining the true level
        assign wr_diff_half = (c_wr_addr[ADDR_WIDTH] != wr_side_rd_addr[ADDR_WIDTH]);
        assign c_wr_level   = {wr_diff_half, c_wr_addr[ADDR_WIDTH-1:0]} - {1'b0, wr_side_rd_addr[ADDR_WIDTH-1:0]};

        always @(posedge wr_clk or negedge wr_rst_n)
        begin
            if (wr_rst_n == 1'b0)
            begin
                wr_ready  <= 1'b0;
                wr_aready <= 1'b0;
            end
            else if (wr_dc_err)
            begin
                wr_ready  <= 1'b1; // Force wr_ready when read side is in reset so we can flush any writes in the pipeline
                wr_aready <= 1'b1; //   ..
            end
            else
            begin
                wr_ready  <= (c_wr_level <  READY_LEVEL);
                wr_aready <= (c_wr_level < AREADY_LEVEL);
            end
        end

        // -----
        // Reads

        // Generate next read address
        assign c_rd_addr = ram_rd_en ? (rd_addr + {{ADDR_WIDTH{1'b0}}, 1'b1}) : rd_addr;

        // Convert next read address to its Gray Code equivalent
        //   Note: Using rd_addr instead of c_rd_addr to account for read address register in util_ram;
        //     if wr_clk is very fast relative to rd_clk, the read data location could be overwritten while read was occurring
        util_bin_to_gray #(.WIDTH (ADDR_WIDTH+1)) util_bin_to_gray_rd_addr_gray (.d (rd_addr), .q (c_rd_addr_gray));

        always @(posedge rd_clk or negedge rd_rst_n)
        begin
            if (rd_rst_n == 1'b0)
            begin
                rd_addr      <= {(ADDR_WIDTH+1){1'b0}};
                rd_addr_gray <= {(ADDR_WIDTH+1){1'b0}};
            end
            else if (rd_sync_wr_rst_n == 1'b0)
            begin
                rd_addr      <= {(ADDR_WIDTH+1){1'b0}};
                rd_addr_gray <= {(ADDR_WIDTH+1){1'b0}};
            end
            else
            begin
                rd_addr      <= c_rd_addr;
                rd_addr_gray <= c_rd_addr_gray;
            end
        end

        // Transfer wr_addr_gray to rd_clk domain
        util_sync_flops_width #(.WIDTH (ADDR_WIDTH+1)) util_sync_flops_width1 (.clk (rd_clk), .rst_n (rd_rst_n), .d (wr_addr_gray), .q (sync2_wr_addr_gray));

        // Convert sync2_wr_addr_gray back to its binary equivalent
        util_gray_to_bin #(.WIDTH (ADDR_WIDTH+1)) util_gray_to_bin_rd_side_wr_addr (.d (sync2_wr_addr_gray), .q (rd_side_wr_addr));

        // If the extra address bit being carried in each of the FIFO addresses have
        //   different values, then the write address has wrapped relative to the read
        //   address and 2^ADDR_WIDTH must be added to the write address in order to
        //   determining the true level
        assign rd_diff_half = (rd_side_wr_addr[ADDR_WIDTH] != c_rd_addr[ADDR_WIDTH]);
        assign c_rd_level   = {rd_diff_half, rd_side_wr_addr[ADDR_WIDTH-1:0]} - {1'b0, c_rd_addr[ADDR_WIDTH-1:0]};

        always @(posedge rd_clk or negedge rd_rst_n)
        begin
            if (rd_rst_n == 1'b0)
                ram_rd_valid <= 1'b0;
            else if (rd_sync_wr_rst_n == 1'b0)
                ram_rd_valid <= 1'b0;
            else
                ram_rd_valid <= (c_rd_level != {(ADDR_WIDTH+1){1'b0}});
        end

        // Drop the extra address bit being carried to determine the FIFO level before accessing the RAM
        assign ram_wr_addr = wr_addr[ADDR_WIDTH-1:0];
        assign ram_rd_addr = rd_addr[ADDR_WIDTH-1:0];
`ifdef SIMULATION

        initial
        begin
            #10
            if ($test$plusargs("gray_code_msgs_on"))
            begin
                $display ("Gray Code FIFO : %m : wr_addr_gray : synchronized gray-coded address; skew between wr_clk wr_addr_gray bits captured by rd_clk must be < 1 fastest clk cycle for gray code to work");
                $display ("Gray Code FIFO : %m : rd_addr_gray : synchronized gray-coded address; skew between rd_clk rd_addr_gray bits captured by wr_clk must be < 1 fastest clk cycle for gray code to work");
            end
        end
`endif
    end
endgenerate



// --------------------
// Instantiate FIFO RAM

util_ram #(
    .ADDR_WIDTH             (ADDR_WIDTH         ),
    .DATA_WIDTH             (DATA_WIDTH         ),
    .REGISTER_OUTPUT        (REGISTER_OUTPUT    )

) util_ram (

    .wr_clk                 (wr_clk             ), 
    .wr_en                  (wr_en              ),
    .wr_addr                (ram_wr_addr        ),
    .wr_data                (wr_data            ),

    .rd_clk                 (rd_clk             ),
    .rd_en                  (ram_rd_en          ),
    .rd_addr                (ram_rd_addr        ),
    .rd_data                (ram_rd_data        )

);
`ifdef SIMULATION

initial
begin
    #10
    if ($test$plusargs("gray_code_msgs_on"))
    begin
        if (DUAL_CLOCK != 0)
            $display ("Gray Code FIFO : %m.util_ram.mem is a RAM implemented in registers that is written and read from different clock domains");
    end
end
`endif



// ----------------------------------------------------
// Instantiate Output Stage to Reduce Read Latency to 0

generate if (REGISTER_OUTPUT == 0)
    begin : gen_lat1_to_lat0
        util_lat1_to_lat0 #(.DATA_WIDTH (DATA_WIDTH)) util_lat1_to_lat0 (

            .rst_n                  (rd_rst_n           ),
            .clk                    (rd_clk             ),
            .s_rst_n                (rd_sync_wr_rst_n   ),

            .wr_valid               (ram_rd_valid       ),
            .wr_ready               (ram_rd_ready       ),
            .wr_data                (ram_rd_data        ),

            .rd_valid               (rd_valid_out       ),
            .rd_ready               (rd_ready           ),
            .rd_data                (rd_data            )

        );
    end
    else
    begin : gen_lat2_to_lat0
        util_lat2_to_lat0 #(.DATA_WIDTH (DATA_WIDTH)) util_lat2_to_lat0 (

            .rst_n                  (rd_rst_n           ),
            .clk                    (rd_clk             ),
            .s_rst_n                (rd_sync_wr_rst_n   ),

            .wr_valid               (ram_rd_valid       ),
            .wr_ready               (ram_rd_ready       ),
            .wr_data                (ram_rd_data        ),

            .rd_valid               (rd_valid_out       ),
            .rd_ready               (rd_ready           ),
            .rd_data                (rd_data            )

        );
    end
endgenerate

assign rd_valid = rd_valid_out | rd_dc_err;



endmodule
// -------------------------------------------------------------------------
//
//  PROJECT: PCI Express Core
//  COMPANY: Northwest Logic, Inc.
//
// ------------------------- CONFIDENTIAL ----------------------------------
//
//                 (c) Copyright 2009 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or 
//  transmitted in any form or by any means, electronic or mechanical, 
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//  
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
// -------------------------------------------------------------------------

// -------------------------------------------------------------------------
//
// Standard FIFO except wr_data and rd_data are provided 1 clock after the 
//   corresponding enable; FIFO level is kept with wr_en and rd_en strobes
//
// -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module bfmp_util_shallow_fifo (

    rst_n,
    clk,

    wr_en,
    wr_data,
    wr_full,
    wr_almost_full,

    rd_en,
    rd_data,
    rd_empty

);



// ----------------
// -- Parameters --
// ----------------

// FIFO Data Width
parameter   DATA_WIDTH          = 32;
parameter   ADDR_WIDTH          = 4;    // Minimum == 3; minimum recommended == 4

parameter   ALMOST_FULL_THRESH  = (1 >> ADDR_WIDTH) - 4;



// ----------------------
// -- Port Definitions --
// ----------------------

input                           rst_n;
input                           clk;

input                           wr_en;
input   [DATA_WIDTH-1:0]        wr_data;
output                          wr_full;
output                          wr_almost_full;

input                           rd_en;
output  [DATA_WIDTH-1:0]        rd_data;
output                          rd_empty;



// ----------------
// -- Port Types --
// ----------------

wire                            rst_n;
wire                            clk;

wire                            wr_en;
wire    [DATA_WIDTH-1:0]        wr_data;
reg                             wr_full;
reg                             wr_almost_full;

wire                            rd_en;
wire    [DATA_WIDTH-1:0]        rd_data;
reg                             rd_empty;



// -------------------
// -- Local Signals --
// -------------------

// FIFO Implementation
reg                             r_wr_en;
reg                             r2_wr_en;
reg     [ADDR_WIDTH-1:0]        wr_addr;

wire    [ADDR_WIDTH-1:0]        next_rd_addr;
wire    [ADDR_WIDTH-1:0]        fifo_rd_addr;
reg     [ADDR_WIDTH-1:0]        rd_addr;

reg     [ADDR_WIDTH:0]          wr_level;

reg     [ADDR_WIDTH:0]          rd_level;



// ---------------
// -- Equations --
// ---------------

// -------------------
// FIFO Implementation

always @(posedge clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        r_wr_en  <= 1'b0;
        r2_wr_en <= 1'b0;
    end
    else
    begin
        r_wr_en  <= wr_en;
        r2_wr_en <= r_wr_en;
    end
end

always @(posedge clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
        wr_addr <= 1'b0;
    else if (r_wr_en) // Write data has latency 1 relative to wr_en
        wr_addr <= wr_addr + {{(ADDR_WIDTH-1){1'b0}}, 1'b1};
end

// Instantiate dual port RAM for FIFO;
//   read enable is always asserted, so the rd_data
//   output depends exclusively on rd_addr;
bfmp_util_inferred_shallow_ram #(

    .ADDR_WIDTH         (ADDR_WIDTH         ),                  
    .DATA_WIDTH         (DATA_WIDTH         )

) fifo_ram (

    .wr_clk             (clk                ),
    .wr_addr            (wr_addr            ),
    .wr_en              (r_wr_en            ),
    .wr_data            (wr_data            ),

    .rd_clk             (clk                ),
    .rd_addr            (fifo_rd_addr       ),
    .rd_data            (rd_data            )

);

assign next_rd_addr = rd_addr + {{(ADDR_WIDTH-1){1'b0}}, 1'b1};

// Enable look ahead reads to lower latency of FIFO to 0
assign fifo_rd_addr = rd_en ? next_rd_addr : rd_addr; 

always @(posedge clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
        rd_addr <= 1'b0;
    else if (rd_en)
        rd_addr <= next_rd_addr;
end

// Note: Level computations use latency 0 strobes even when DATA_LATENCY == 1
always @(posedge clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        wr_level <= {(ADDR_WIDTH+1){1'b0}};
        wr_full  <= 1'b0;

        wr_almost_full <= 1'b0;
    end
    else 
    begin
        case ({wr_en, rd_en})
            2'b01   : begin wr_level <= wr_level - {{ADDR_WIDTH{1'b0}}, 1'b1}; wr_full <= 1'b0;                                                   end
            2'b10   : begin wr_level <= wr_level + {{ADDR_WIDTH{1'b0}}, 1'b1}; wr_full <= (wr_level[ADDR_WIDTH:0] >= {1'b0, {ADDR_WIDTH{1'b1}}}); end
            default : begin wr_level <= wr_level;                              wr_full <= (wr_level[ADDR_WIDTH:0] >= {1'b1, {ADDR_WIDTH{1'b0}}}); end
        endcase

        wr_almost_full <= (wr_level >= ALMOST_FULL_THRESH);
    end
end

// NOTE: Because this FIFO is using look ahead reads and write data has latency 1 relative to wr_en, it is necessary to delay wr_en
//   for rd_level computation by two clocks to ensure that the write data cannot be read before it has been stored in RAM
always @(posedge clk or negedge rst_n)
begin
    if (rst_n == 1'b0)
    begin
        rd_level <= {(ADDR_WIDTH+1){1'b0}};
        rd_empty <= 1'b1;
    end
    else 
    begin
        case ({r2_wr_en, rd_en}) 
            2'b01   : begin rd_level <= rd_level - {{ADDR_WIDTH{1'b0}}, 1'b1}; rd_empty <= (rd_level == {{ADDR_WIDTH{1'b0}}, 1'b1}); end
            2'b10   : begin rd_level <= rd_level + {{ADDR_WIDTH{1'b0}}, 1'b1}; rd_empty <= 1'b0;                                     end
            default : begin rd_level <= rd_level;                              rd_empty <= (rd_level == {(ADDR_WIDTH+1){1'b0}});     end
        endcase
    end
end



endmodule
//  ------------------------- CONFIDENTIAL ----------------------------------
//
//                 (c) Copyright 2010 by Northwest Logic, Inc.
//
//  All rights reserved.  No part of this source code may be reproduced or
//  transmitted in any form or by any means, electronic or mechanical,
//  including photocopying, recording, or any information storage and
//  retrieval system, without permission in writing from Northest Logic, Inc.
//
//  Further, no use of this source code is permitted in any form or means
//  without a valid, written license agreement with Northwest Logic, Inc.
//
// $Date: 2013-11-20 23:19:56 -0800 (Wed, 20 Nov 2013) $
// $Revision: 42115 $
//
//                         Northwest Logic, Inc.
//                  1100 NW Compton Drive, Suite 100
//                      Beaverton, OR 97006, USA
//
//                       Ph.  +1 503 533 5800
//                       Fax. +1 503 533 5900
//                          www.nwlogic.com
//
//  -------------------------------------------------------------------------

`timescale 1ps / 1ps



// -----------------------
// -- Module Definition --
// -----------------------

module bfmp_util_inferred_shallow_ram (

    wr_clk,
    wr_addr,
    wr_en,
    wr_data,

    rd_clk,
    rd_addr,
    rd_data

);



// ----------------
// -- Parameters --
// ----------------

parameter   ADDR_WIDTH          = 9;                // Set to desired number of address bits
parameter   DATA_WIDTH          = 8;                // Set to desired number of data bits
// bluepearl disable 108
parameter   FAST_READ           = 0;                // If 1, allows simultaneous read and write
// bluepearl enable 108
`ifdef SIMULATION
localparam  INITIAL_SIM_RANDOM  = 1'b1;             // 1-Initialize RAM to random value; 0-Use INITIAL_SIM_VALUE to initialize all bits
localparam  INITIAL_SIM_VALUE   = 1'b0;             // Set the initial value for all ram bits (used for simulation only)
`endif  // SIMULATION
localparam  NUM_WORDS           = 1 << ADDR_WIDTH;  // The number of words is 2^ADDR_WIDTH



// -----------------------
// -- Port Declarations --
// -----------------------

input                           wr_clk;
input   [ADDR_WIDTH-1:0]        wr_addr;
input                           wr_en;
input   [DATA_WIDTH-1:0]        wr_data;

input                           rd_clk;
input   [ADDR_WIDTH-1:0]        rd_addr;
output  [DATA_WIDTH-1:0]        rd_data;



// ----------------
// -- Port Types --
// ----------------

wire                            wr_clk;
wire    [ADDR_WIDTH-1:0]        wr_addr;
wire                            wr_en;
wire    [DATA_WIDTH-1:0]        wr_data;

wire                            rd_clk;
wire    [ADDR_WIDTH-1:0]        rd_addr;
wire    [DATA_WIDTH-1:0]        rd_data;



// ---------------------
// -- Local Variables --
// ---------------------

reg     [ADDR_WIDTH-1:0]        r_rd_addr;
(* ram_style = "distributed" *)
reg     [DATA_WIDTH-1:0]        mem [NUM_WORDS-1:0];



// ---------------
// -- Equations --
// ---------------


// Perform RAM write
always @(posedge wr_clk)
begin
    if (wr_en)
        mem[wr_addr] <= wr_data;
end

// Register read inputs
// bluepearl disable 224 395 24
always @(posedge rd_clk)
begin
    r_rd_addr <= rd_addr;
end
// bluepearl enable 224 395 24

 `ifdef SIMULATION

reg [ADDR_WIDTH-1:0] r_wr_addr;
reg                  r_wr_en;
wire                 wr_rd_collision;

always @(posedge wr_clk)
begin
    r_wr_en <= wr_en;
    if (wr_en == 1'b1)
        r_wr_addr <= wr_addr;
end

assign wr_rd_collision = (FAST_READ == 1) ? 1'b0 : (r_wr_en & (r_wr_addr == r_rd_addr));

initial r_rd_addr = {ADDR_WIDTH{1'b0}};

assign rd_data = (wr_rd_collision == 1'b1) ? {DATA_WIDTH{1'bx}} : mem[r_rd_addr];

// Initialize the memory for simualtion
integer i;
integer j;
reg     [31:0]              r;
reg     [DATA_WIDTH-1:0]    val;
initial
begin
    for (i=0; i<NUM_WORDS; i=i+1)
    begin
        if (INITIAL_SIM_RANDOM != 0)
        begin
            for (j=0; j<DATA_WIDTH;j=j+1)
            begin
                r = $random;
                val[j] = r[0];
            end
            mem[i] = val;
        end
        else
        begin
            mem[i] = {DATA_WIDTH{INITIAL_SIM_VALUE}};
        end
    end
end
 `else // SIMULATION

assign rd_data = mem[r_rd_addr];
 `endif // SIMULATION

`ifdef SIMULATION
initial
begin
    #6
    if ($test$plusargs("ram_size_msgs_on"))
        $display("RAM Instance using ADDR_WIDTH=%d, DATA_WIDTH=%d, FAST_READ=%d : %m",ADDR_WIDTH,DATA_WIDTH,FAST_READ);
end
`endif

endmodule
