/************************************************************************
 * Copyright 2006-2010 Silicon Software GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License (version 2) as
 * published by the Free Software Foundation.
 */
#include <linux/err.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/uaccess.h>
#include "uiq.h"
#include "menable.h"
#include "linux_version.h"

static void
men_uiq_pop(struct menable_uiq *uiq)
{
	unsigned int windex;
	uint32_t val;

	if (unlikely(uiq->length == 0)) {
		do {
			val = ioread32(uiq->reg);
			if (!(val & 0x80000000))
				uiq->lost++;
		} while (!(val & 0x10000));
		return;
	}

	windex = (uiq->rindex + uiq->fill) % uiq->length;
	do {
		val = ioread32(uiq->reg);

		if (!(val & 0x80000000)) {
			if (uiq->fill < uiq->length) {
				uiq->data[windex] = val;
				windex = (windex + 1) % uiq->length;
				uiq->fill++;
			} else {
				uiq->lost++;
			}
		}
	} while (!(val & 0x10000));
	if (uiq->cpltodo && (uiq->cpltodo <= uiq->fill))
		complete(&uiq->cpl);
	uiq->irqcnt++;
}

static void
men_uiq_push(struct menable_uiq *uiq)
{
	unsigned int num, i;

	num = min((unsigned int)uiq->burst, uiq->fill);
	for (i = 0; i < num - 1; i++) {
		iowrite32(uiq->data[uiq->rindex] & 0xffff, uiq->reg);
		uiq->rindex = (uiq->rindex + 1) % uiq->length;
	}
	iowrite32(uiq->data[uiq->rindex] | 0x10000, uiq->reg);
	uiq->rindex = (uiq->rindex + 1) % uiq->length;
	uiq->irqcnt++;
	uiq->fill -= num;
	if (uiq->cpltodo && (uiq->cpltodo <= uiq->length - uiq->fill))
		complete(&uiq->cpl);
}

static void
uiq_cpy(struct menable_uiq *uiq, void *buf, const unsigned int num)
{
	if (unlikely(uiq->rindex + num >= uiq->length)) {
		unsigned int wrap = uiq->length - uiq->rindex;

		memcpy(buf, uiq->data + uiq->rindex,
			wrap * sizeof(*uiq->data));
		memcpy(buf + wrap * sizeof(*uiq->data), uiq->data,
			(num - wrap) * sizeof(*uiq->data));
	} else {
		memcpy(buf, uiq->data + uiq->rindex,
			num * sizeof(*uiq->data));
	}
}

static ssize_t
uiq_read(
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 35)
		struct file *filp,
#endif /* LINUX >= 2.6.35 */
		struct kobject *kobj, struct bin_attribute *attr, char *buf,
		loff_t off, size_t count)
{
	unsigned long flags;
	struct device *dev = container_of(kobj, struct device, kobj);
	struct menable_uiq *uiq = container_of(dev, struct menable_uiq, dev);
	unsigned int cnt = count / sizeof(*uiq->data);
	ssize_t num = 0;

	spin_lock_irqsave(&uiq->lock, flags);
	if (unlikely(uiq->cpltodo)) {
		spin_unlock_irqrestore(&uiq->lock, flags);
		return -EBUSY;
	}
	if (uiq->fill < cnt) {
		unsigned long t = uiq->cpltimeout;
		uiq->cpltodo = cnt;
		do {
			spin_unlock_irqrestore(&uiq->lock, flags);
			t = wait_for_completion_interruptible_timeout(&uiq->cpl,
					t);
			spin_lock_irqsave(&uiq->lock, flags);
			if (uiq->fill >= cnt)
				break;
		} while ((t > 0) && !fatal_signal_pending(current));
		uiq->cpltodo = 0;
	}

	if (uiq->fill > 0) {
		unsigned int c = min(uiq->fill, cnt);

		uiq_cpy(uiq, buf, c);
		uiq->fill -= c;
		if (uiq->fill == 0)
			uiq->rindex = 0;
		else
			uiq->rindex = (uiq->rindex + c) % uiq->length;
		num = c * sizeof(*uiq->data);
	} else {
		num = -ETIMEDOUT;
	}
	spin_unlock_irqrestore(&uiq->lock, flags);
	return num;
}

static ssize_t
uiq_write(
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 35)
		struct file *filp,
#endif /* LINUX >= 2.6.35 */
		struct kobject *kobj, struct bin_attribute *attr, char *buf,
		loff_t off, size_t count)
{
	unsigned long flags;
	struct device *dev = container_of(kobj, struct device, kobj);
	struct menable_uiq *uiq = container_of(dev, struct menable_uiq, dev);
	const unsigned int cnt = count / sizeof(*uiq->data);
	unsigned int wrap;
	unsigned int windex;

	if (unlikely(count == 0))
		return 0;

	if (unlikely(count % sizeof(*uiq->data) != 0))
		return -EINVAL;

	spin_lock_irqsave(&uiq->lock, flags);
	if (unlikely((uiq->length == 0) || (uiq->length < cnt))) {
		spin_unlock_irqrestore(&uiq->lock, flags);
		return -ENOSPC;
	}
	if (unlikely(uiq->cpltodo)) {
		spin_unlock_irqrestore(&uiq->lock, flags);
		return -EBUSY;
	}
	if (uiq->length - uiq->fill < cnt) {
		unsigned long t = uiq->cpltimeout;
		uiq->cpltodo = cnt;
		do {
			spin_unlock_irqrestore(&uiq->lock, flags);
			t = wait_for_completion_interruptible_timeout(&uiq->cpl,
					t);
			spin_lock_irqsave(&uiq->lock, flags);
			if (uiq->cpltodo == -1) {
				/* someone send a clear request while we were
				 * waiting. We act as if the data of this request
				 * would already have been queued: we silently
				 * drop it. */
				uiq->cpltodo = 0;
				spin_unlock_irqrestore(&uiq->lock, flags);
				return cnt * sizeof(*uiq->data);
			}
			if (uiq->length - uiq->fill >= cnt)
				break;
		} while ((t > 0) && !fatal_signal_pending(current));
		uiq->cpltodo = 0;
	}
	if (uiq->length - uiq->fill < cnt) {
		spin_unlock_irqrestore(&uiq->lock, flags);
		return -EBUSY;
	}

	windex = (uiq->rindex + uiq->fill) % uiq->length;
	wrap = uiq->length - windex;

	if (wrap > cnt) {
		memcpy(uiq->data + windex, buf,
			cnt * sizeof(*uiq->data));
	} else {
		memcpy(uiq->data + windex, buf,
			wrap * sizeof(*uiq->data));
		memcpy(uiq->data, buf + wrap * sizeof(*uiq->data),
			(cnt - wrap) * sizeof(*uiq->data));
	}

	uiq->fill += cnt;

	if (!uiq->running) {
		men_uiq_push(uiq);
		uiq->running = true;
	}

	spin_unlock_irqrestore(&uiq->lock, flags);

	return cnt * sizeof(*uiq->data);
}

static ssize_t
men_uiq_readsize(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct menable_uiq *uiq = container_of(dev, struct menable_uiq, dev);

	/* This is not synchronized as this may change anyway
	 * until the user can use the information */
	return sprintf(buf, "%zi\n", uiq->dattr.size);
}

static ssize_t
men_uiq_writesize(struct device *dev, struct device_attribute *attr,
				const char *buf, size_t count)
{
	struct menable_uiq *uiq = container_of(dev, struct menable_uiq, dev);
	unsigned int sz;
	int ret;

	ret = buf_get_uint(buf, count, &sz);
	if (ret)
		return ret;

	ret = men_scale_uiq(uiq, sz);

	return ret ? ret : count;
}

static DEVICE_ATTR(size, 0660, men_uiq_readsize, men_uiq_writesize);

static ssize_t
men_uiq_readlost(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct menable_uiq *uiq = container_of(dev, struct menable_uiq, dev);

	/* This is not synchronized as this may change anyway
	 * until the user can use the information */
	return sprintf(buf, "%i\n", uiq->lost);
}

static DEVICE_ATTR(lost, 0440, men_uiq_readlost, NULL);

static ssize_t
men_uiq_readfill(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct menable_uiq *uiq = container_of(dev, struct menable_uiq, dev);

	/* This is not synchronized as this may change anyway
	 * until the user can use the information */
	return sprintf(buf, "%i\n", uiq->fill);
}

static ssize_t
men_uiq_writefill(struct device *dev, struct device_attribute *attr,
				const char *buf, size_t count)
{
	struct menable_uiq *uiq = container_of(dev, struct menable_uiq, dev);
	unsigned long flags;

	if (strcmp(buf, "0"))
		return -EINVAL;

	spin_lock_irqsave(&uiq->lock, flags);
	uiq->fill = 0;
	uiq->rindex = 0;
	uiq->lost = 0;
	if (uiq->cpltodo) {
		uiq->cpltodo = -1;
		complete(&uiq->cpl);
	}
	spin_unlock_irqrestore(&uiq->lock, flags);

	return count;
}

static DEVICE_ATTR(fill, 0660, men_uiq_readfill, men_uiq_writefill);

static ssize_t
men_uiq_readirqcnt(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct menable_uiq *uiq = container_of(dev, struct menable_uiq, dev);

	return sprintf(buf, "%i\n", uiq->irqcnt);
}

static DEVICE_ATTR(irqcnt, 0440, men_uiq_readirqcnt, NULL);

static ssize_t
men_uiq_writetimeout(struct device *dev, struct device_attribute *attr,
				const char *buf, size_t count)
{
	struct menable_uiq *uiq = container_of(dev, struct menable_uiq, dev);
	unsigned long flags;
	unsigned int t;
	int ret;
	struct timespec timeout;
	unsigned long to;

	ret = buf_get_uint(buf, count, &t);
	if (ret)
		return ret;

	timeout.tv_nsec = (t % 1000) * 1000000;
	timeout.tv_sec = t / 1000;
	to = timespec_to_jiffies(&timeout);

	spin_lock_irqsave(&uiq->lock, flags);
	uiq->cpltimeout = to;
	spin_unlock_irqrestore(&uiq->lock, flags);

	return count;
}

static DEVICE_ATTR(timeout, 0220, NULL, men_uiq_writetimeout);

static void menable_uiq_release(struct device *dev)
{
	struct menable_uiq *uiq = container_of(dev, struct menable_uiq, dev);

	kfree(uiq->data);
	kfree(uiq);
}

void
men_uiq_remove(struct menable_uiq *uiq)
{
	device_remove_file(&uiq->dev, &dev_attr_timeout);
	device_remove_file(&uiq->dev, &dev_attr_fill);
	device_remove_file(&uiq->dev, &dev_attr_irqcnt);
	device_remove_file(&uiq->dev, &dev_attr_lost);
	device_remove_file(&uiq->dev, &dev_attr_size);
	device_remove_bin_file(&uiq->dev, &uiq->dattr);
	device_unregister(&uiq->dev);
}

static struct lock_class_key men_uiq_lock;
static struct lock_class_key men_uiqcpl_lock;

/**
 * men_uiq_init - create a UIQ object for the given queue
 * @chan UIQ channel number
 * @addr address of the queue I/O register
 * @parent grabber this UIQ belongs to
 * @write if queue is a read or write queue
 * @burst FIFO depth for write queues
 *
 * This will allocate the memory for the UIQ object and return
 * the allocated pointer or an ERR_PTR() value on failure.
 */
struct menable_uiq *
men_uiq_init(int chan, void __iomem *addr, struct siso_menable *parent,
		bool write, unsigned char burst)
{
	int ret;
	struct menable_uiq *uiq;

	uiq = kzalloc(sizeof(*uiq), GFP_KERNEL);
	if (uiq == NULL)
		return ERR_PTR(-ENOMEM);

	uiq->chan = chan;
	uiq->reg = addr;
	uiq->parent = parent;
	uiq->dev.driver = parent->dev.driver;
	init_completion(&uiq->cpl);
	spin_lock_init(&uiq->lock);
	lockdep_set_class(&uiq->lock, &men_uiq_lock);
	lockdep_set_class(&uiq->cpl.wait.lock, &men_uiqcpl_lock);

	dev_set_name(&uiq->dev, "uiq%i", chan);

	uiq->dev.parent = &parent->dev;
	uiq->dev.release = menable_uiq_release;
	ret = device_register(&uiq->dev);
	if (ret)
		goto err;

	sysfs_bin_attr_init(&uiq->dattr);
	uiq->dattr.attr.name = "data";
	uiq->write_queue = write;
	if (write) {
		uiq->dattr.attr.mode = S_IFREG | S_IWUSR | S_IWGRP;
		uiq->burst = burst;
	} else {
		uiq->dattr.attr.mode = S_IFREG | S_IRUSR | S_IRGRP;
	}
	uiq->dattr.size = 0;
	uiq->dattr.read = uiq_read;
	uiq->dattr.write = uiq_write;

	ret = device_create_bin_file(&uiq->dev, &uiq->dattr);
	if (ret)
		goto err_data;

	ret = device_create_file(&uiq->dev, &dev_attr_size);
	if (ret)
		goto err_scale;

	ret = device_create_file(&uiq->dev, &dev_attr_lost);
	if (ret)
		goto err_lost;

	ret = device_create_file(&uiq->dev, &dev_attr_irqcnt);
	if (ret)
		goto err_irqcnt;

	ret = device_create_file(&uiq->dev, &dev_attr_fill);
	if (ret)
		goto err_irqfill;

	ret = device_create_file(&uiq->dev, &dev_attr_timeout);
	if (ret)
		goto err_timeout;

	return uiq;
err_timeout:
	device_remove_file(&uiq->dev, &dev_attr_fill);
err_irqfill:
	device_remove_file(&uiq->dev, &dev_attr_irqcnt);
err_irqcnt:
	device_remove_file(&uiq->dev, &dev_attr_lost);
err_lost:
	device_remove_file(&uiq->dev, &dev_attr_size);
err_scale:
	device_remove_bin_file(&uiq->dev, &uiq->dattr);
err_data:
	device_unregister(&uiq->dev);
err:
	kfree(uiq);
	return ERR_PTR(ret);
}

int
men_scale_uiq(struct menable_uiq *uiq, const unsigned int len)
{
	void *mem, *old;
	unsigned long flags;
	size_t cp;
	const unsigned int newlen = len / sizeof(*uiq->data);

	if (uiq->length == newlen)
		return 0;

	if (len == 0) {
		mem = NULL;
	} else {
		mem = kmalloc(len, GFP_USER);
		if (unlikely(mem == NULL))
			return -ENOMEM;
	}

	spin_lock_irqsave(&uiq->lock, flags);
	cp = min(uiq->fill, newlen);
	uiq_cpy(uiq, mem, cp);
	uiq->fill = cp;
	uiq->rindex = 0;
	uiq->dattr.size = len;
	old = uiq->data;
	uiq->data = mem;
	uiq->length = newlen;
	spin_unlock_irqrestore(&uiq->lock, flags);
	kfree(old);

	return 0;
}

void
uiq_irq(struct menable_uiq *uiq)
{
	if (!spin_trylock(&uiq->lock))
		return;

	/* unconfigured board */
	if (unlikely(uiq->reg == NULL)) {
		spin_unlock(&uiq->lock);
		return;
	}

	if (uiq->write_queue == 0) {
		men_uiq_pop(uiq);
	} else {
		WARN_ON(!uiq->running);

		if (uiq->fill == 0) {
			uiq->running = false;
			uiq->rindex = 0;
		} else {
			men_uiq_push(uiq);
		}
	}

	iowrite32(1 << uiq->ackbit, uiq->irqack);

	spin_unlock(&uiq->lock);
}
