/************************************************************************
 * Copyright 2006-2010 Silicon Software GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License (version 2) as
 * published by the Free Software Foundation.
 */
#include <linux/io.h>
#include <linux/pci.h>
#include <linux/scatterlist.h>
#include <linux/slab.h>
#include <linux/vmalloc.h>
#include "menable.h"
#include "linux_version.h"

/**
 * men_get_dmaimg - print current picture number to sysfs
 * @dev: device to query
 * @attr: device attribute of the channel file
 * @buf: buffer to print information to
 *
 * The result will be printed in decimal form into the buffer.
 */
static ssize_t
men_get_dmaimg(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct menable_dmachan *d = container_of(dev,
			struct menable_dmachan, dev);

	return sprintf(buf, "%lli\n", d->goodcnt);
}

static DEVICE_ATTR(img, 0440, men_get_dmaimg, NULL);

/**
 * men_get_dmalost - print current number of lost pictures to sysfs
 * @dev: device to query
 * @attr: device attribute of the channel file
 * @buf: buffer to print information to
 *
 * The result will be printed in decimal form into the buffer.
 */
static ssize_t
men_get_dmalost(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct menable_dmachan *d = container_of(dev,
			struct menable_dmachan, dev);

	return sprintf(buf, "%i\n", d->lost);
}

static DEVICE_ATTR(lost, 0440, men_get_dmalost, NULL);

static struct lock_class_key men_dmacpl_lock;

/**
 * men_wait_dmaimg - wait until the given image is grabbed
 * @d: the DMA channel to watch
 * @img: the image number to wait for
 * @timeout: wait limit
 *
 * This function blocks until the current image number is at least the one
 * requested in @buf.
 *
 * Returns: current picture number on success, error code on failure
 */
int
men_wait_dmaimg(struct menable_dmachan *d, const uint64_t img,
			const struct timespec *timeout, uint64_t *foundframe)
{
	uint64_t cur;
	struct device *dv;
	unsigned long flags;
	DECLARE_COMPLETION_ONSTACK(cpl);

	dv = get_device(&d->dev);
	lockdep_set_class(&cpl.wait.lock, &men_dmacpl_lock);

	spin_lock_irqsave(&d->listlock, flags);
	if (d->goodcnt >= img) {
		spin_unlock_irqrestore(&d->listlock, flags);
		put_device(dv);
		*foundframe = d->goodcnt;
		return 0;
	}
	if (unlikely(d->cpl)) {
		spin_unlock_irqrestore(&d->listlock, flags);
		put_device(dv);
		return -EBUSY;
	}
	if (unlikely(d->running != 1)) {
		spin_unlock_irqrestore(&d->listlock, flags);
		put_device(dv);
		return -ETIMEDOUT;
	}
	d->cplimg = img;
	d->cpl = &cpl;
	spin_unlock_irqrestore(&d->listlock, flags);
	wait_for_completion_interruptible_timeout(&cpl,
				timespec_to_jiffies(timeout));

	spin_lock_irqsave(&d->listlock, flags);
	d->cplimg = 0;
	d->cpl = NULL;
	cur = d->goodcnt;
	spin_unlock_irqrestore(&d->listlock, flags);

	put_device(dv);
	if (cur < img)
		return -ETIMEDOUT;

	*foundframe = d->goodcnt;
	return 0;
}

static void
men_dma_timeout(unsigned long arg)
{
	unsigned long flags;
	struct menable_dmachan *dc = (struct menable_dmachan *)arg;

	spin_lock(&dc->parent->headlock);
	spin_lock_irqsave(&dc->chanlock, flags);
	if (!spin_trylock(&dc->timerlock)) {
		/* If this fails someone else tries to modify this timer.
		 * This means the timer will either get restarted or deleted.
		 * In both cases we don't need to do anything here as the
		 * other function will take care of everything. */
		spin_unlock_irqrestore(&dc->chanlock, flags);
		spin_unlock(&dc->parent->headlock);
		return;
	}

	dc->parent->abortdma(dc->parent, dc);
	dc->running = 2;
	spin_lock(&dc->listlock);
	if (dc->cpl)
		complete(dc->cpl);
	spin_unlock(&dc->listlock);

	spin_unlock(&dc->timerlock);
	spin_unlock_irqrestore(&dc->chanlock, flags);
	spin_unlock(&dc->parent->headlock);
}

static void
menable_chan_release(struct device *dev)
{
	struct menable_dmachan *d = container_of(dev, struct menable_dmachan,
							dev);

	device_remove_file(&d->dev, &dev_attr_img);
	device_remove_file(&d->dev, &dev_attr_lost);
	kfree(d);
}

static struct lock_class_key men_dma_lock;
static struct lock_class_key men_dmachan_lock;
static struct lock_class_key men_dmatimer_lock;

/**
 * dma_clean_sync - put DMA channel in stopped state
 * @db: channel to work
 *
 * This resets everything in the DMA channel to stopped state, e.g. clearing
 * association of a memory buffer.
 *
 * context: IRQ (headlock and chanlock must be locked and released from caller)
 *          the channel must be in "stop pending" state (running == 3)
 */
void
dma_clean_sync(struct menable_dmachan *db)
{
	BUG_ON(db->running != 3);
	del_timer(&db->timer);
	db->running = 2;
	db->transfer_todo = 0;
	spin_lock(&db->listlock);
	if (db->cpl)
		complete(db->cpl);
	spin_unlock(&db->listlock);
}

void
dma_done_work(struct work_struct *work)
{
	struct menable_dmachan *db = container_of(work,
			struct menable_dmachan, dwork);
	struct siso_menable *men = db->parent;
	unsigned long flags;

	spin_lock(&men->headlock);
	spin_lock_irqsave(&db->chanlock, flags);
	/* the timer might have cancelled everything in the mean time */
	if (db->running == 3)
		dma_clean_sync(db);
	spin_unlock_irqrestore(&db->chanlock, flags);
	spin_unlock(&men->headlock);
}

static struct menable_dmachan *
men_create_dmachan(struct siso_menable *parent, const int num)
{
	struct menable_dmachan *res = kzalloc(sizeof(*res), GFP_KERNEL);
	int r;

	if (!res)
		return ERR_PTR(ENOMEM);

	/* our device */
	res->dev.parent = &parent->dev;
	res->dev.release = menable_chan_release;
	res->dev.driver = parent->dev.driver;
	res->number = num;
	parent->dmabase(parent, res);
	dev_set_name(&res->dev, "dma%i", num);

	/* misc setup */
	res->parent = parent;
	spin_lock_init(&res->listlock);
	spin_lock_init(&res->chanlock);
	spin_lock_init(&res->timerlock);
	lockdep_set_class(&res->listlock, &men_dma_lock);
	lockdep_set_class(&res->chanlock, &men_dmachan_lock);
	lockdep_set_class(&res->timerlock, &men_dmatimer_lock);
	INIT_LIST_HEAD(&res->free_list);
	INIT_LIST_HEAD(&res->grabbed_list);
	INIT_LIST_HEAD(&res->hot_list);
	init_timer(&res->timer);
	res->timer.function = men_dma_timeout;
	res->timer.data = (unsigned long)res;
	INIT_WORK(&res->dwork, dma_done_work);

	r = device_register(&res->dev);
	if (r != 0) {
		kfree(res);
		return ERR_PTR(r);
	}

	r = device_create_file(&res->dev, &dev_attr_img);
	if (r != 0) {
		device_unregister(&res->dev);
		return ERR_PTR(r);
	}

	r = device_create_file(&res->dev, &dev_attr_lost);
	if (r != 0) {
		device_unregister(&res->dev);
		return ERR_PTR(r);
	}

	return res;
}

static void
men_dma_remove(struct menable_dmachan *d)
{
	del_timer_sync(&d->timer);
	flush_scheduled_work();
	device_unregister(&d->dev);
}

/**
 * men_add_dmas - add more DMA channels
 * @men: board to pimp
 */
int
men_add_dmas(struct siso_menable *men)
{
	unsigned int i;
	struct menable_dmachan **old, **nc;
	unsigned long flags;
	unsigned int count = men->query_dma(men);
	unsigned int oldcount = men->num_dma;

	WARN_ON(count < oldcount);
	if (count <= oldcount)
		return 0;

	nc = kcalloc(count, sizeof(*nc), GFP_KERNEL);

	if (unlikely(nc == NULL))
		return -ENOMEM;

	/* allocate control structs for all new channels */
	for (i = oldcount; i < count; i++) {
		nc[i] = men_create_dmachan(men, i);

		if (unlikely(IS_ERR(nc[i]))) {
			int ret = PTR_ERR(nc[i]);
			unsigned int j;

			for (j = oldcount; j < i; j++)
				men_dma_remove(nc[j]);

			kfree(nc);
			return ret;
		}
	}

	spin_lock(&men->headlock);
	spin_lock_irqsave(&men->boardlock, flags);
	/* copy old array contents to new one */
	old = men->dmas;
	for (i = 0; i < oldcount; i++)
		nc[i] = old[i];

	men->dmas = nc;
	men->num_dma = count;
	spin_unlock_irqrestore(&men->boardlock, flags);
	spin_unlock(&men->headlock);

	kfree(old);

	return 0;
}

/**
 * men_alloc_dma - change number of DMA channels of this board
 * @men: board
 * @count: new channel count
 *
 * count must not be greater than the current DMA count. Use men_add_dmas()
 * instead.
 */
int
men_alloc_dma(struct siso_menable *men, unsigned int count)
{
	struct menable_dmachan *old[MEN_MAX_DMA];
	struct menable_dmachan **delold = NULL;
	int i, oldcnt;
	unsigned long flags;

	spin_lock_irqsave(&men->boardlock, flags);

	if (unlikely(men->releasing))
		count = 0;

	oldcnt = men->num_dma;
	BUG_ON(oldcnt > MEN_MAX_DMA);

	if (count == oldcnt) {
		spin_unlock_irqrestore(&men->boardlock, flags);
		spin_unlock(&men->headlock);
		return 0;
	} else if (count > oldcnt) {
		WARN_ON(count > oldcnt);
		return 0;
	}

	for (i = count; i < oldcnt; i++) {
		spin_lock(&men->dmas[i]->chanlock);
		men_stop_dma_locked(men->dmas[i]);
		spin_unlock(&men->dmas[i]->chanlock);
	}

	/* don't bother shrinking the array here, it's
	 * not worth the effort */
	memset(old, 0, sizeof(old));

	/* copy those we will free */
	for (i = count; i < oldcnt; i++) {
		old[i] = men->dmas[i];
		men->dmas[i] = NULL;
	}

	men->num_dma = count;
	if (count == 0) {
		delold = men->dmas;
		men->dmas = NULL;
	}
	spin_unlock_irqrestore(&men->boardlock, flags);
	spin_unlock(&men->headlock);

	kfree(delold);

	for (i = oldcnt - 1; i >= 0; --i) {
		if (old[i] == NULL)
			continue;

		men_dma_remove(old[i]);
	}

	return 0;
}

static ssize_t
men_get_dmas(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct siso_menable *men = container_of(dev, struct siso_menable, dev);

	return sprintf(buf, "%i\n", men->num_dma);
}

DEVICE_ATTR(dma_channels, 0444, men_get_dmas, NULL);

static void
men_clean_bh(struct menable_dmachan *dc, const unsigned int startbuf)
{
	long i;
	struct menable_dmahead *active = dc->active;

	dc->free = 0;
	dc->grabbed = 0;
	dc->locked = 0;
	dc->lost = 0;
	dc->hot = 0;
	dc->goodcnt = 0;
	INIT_LIST_HEAD(&dc->free_list);
	INIT_LIST_HEAD(&dc->hot_list);
	INIT_LIST_HEAD(&dc->grabbed_list);

	for (i = startbuf; i < active->num_sb; i++) {
		if (active->bufs[i] == NULL)
			continue;
		list_add_tail(&active->bufs[i]->node, &dc->free_list);
		dc->free++;
	}

	if (startbuf > 0) {
		for (i = 0; i < startbuf; i++) {
			if (active->bufs[i] == NULL)
				continue;
			list_add_tail(&active->bufs[i]->node, &dc->free_list);
			dc->free++;
		}
	}

	active->chan = dc;
}

int
men_start_dma(struct menable_dmachan *dc, struct menable_dmahead *newh,
		const unsigned int startbuf) __releases(&dc->parent->headlock)
{
	int ret = 0;

	if ((newh->chan != NULL) &&
			(((dc->running != 2) && (dc->running != 0)) ||
			((newh->chan->running != 2) &&
					(newh->chan->running != 0)))) {
		spin_unlock(&dc->parent->headlock);
		return -EBUSY;
	}

	if ((dc->active != NULL) && (dc->active != newh))
		dc->active->chan = NULL;

	dc->active = newh;
	dc->running = 1;
	spin_unlock(&dc->parent->headlock);

	spin_lock(&dc->listlock);
	men_clean_bh(dc, startbuf);
	if (dc->free == 0)
		ret = -ENODEV;
	spin_unlock(&dc->listlock);
	if (ret) {
		dc->running = 0;
		return ret;
	}

	ret = dc->parent->startdma(dc);

	if (ret == 0) {
		spin_lock(&dc->timerlock);
		add_timer(&dc->timer);
		spin_unlock(&dc->timerlock);
	}
	return ret;
}
