/************************************************************************
 * Copyright 2006-2010 Silicon Software GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License (version 2) as
 * published by the Free Software Foundation.
 */
#include <linux/io.h>
#include <asm/pgtable.h>
#include <asm/page.h>
#include <linux/kmod.h>
#include <linux/init.h>
#include <linux/device.h>
#include <linux/pci.h>
#include <linux/fs.h>
#include <linux/interrupt.h>
#include <linux/list.h>
#include <linux/lockdep.h>
#include <linux/kobject.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/stddef.h>
#include <linux/vmalloc.h>
#include <linux/module.h>
#include "menable.h"
#include "uiq.h"
#include "linux_version.h"

static dev_t devr;
static DEFINE_SPINLOCK(idxlock);

static struct lock_class_key men_idx_lock;
static struct lock_class_key men_board_lock;
static struct lock_class_key men_design_lock;
static struct lock_class_key men_head_lock;
static int maxidx;

static void menable_obj_release(struct device *dev)
{
	struct siso_menable *men = container_of(dev, struct siso_menable, dev);

	spin_lock(&idxlock);
	if (men->idx == maxidx - 1)
		maxidx--;
	spin_unlock(&idxlock);
	kfree(men);
}

static void
men_cleanup(struct siso_menable *men, unsigned long flags)
		__releases(&men->boardlock)
{
	int i, r;
	struct menable_dmahead *dh, *tmp;

	men->stopirq(men);

	for (i = 0; i < men->num_dma; i++) {
		struct menable_dmachan *dc = men->dmas[i];
		if (dc->running == 1)
			men_stop_dma_locked(dc);
		if (dc->active) {
			dc->active->chan = NULL;
			dc->active = NULL;
		}
	}

	for (i = 0; i < men->num_uiq; i++) {
		struct menable_uiq *uiq = men->uiqs[i];

		spin_lock(&uiq->lock);
		uiq->running = false;
		spin_unlock(&uiq->lock);
	}

	spin_unlock_irqrestore(&men->boardlock, flags);

	list_for_each_entry_safe(dh, tmp, &men->heads, node) {
		r = men_free_buf_head(men, dh);
		BUG_ON(r != 0);
	}
}

static int menable_open(struct inode *inode, struct file *file)
{
	struct siso_menable *men = container_of(inode->i_cdev, struct siso_menable, cdev);
	int ret;
	unsigned long flags;

	file->private_data = men;

	if (men->open) {
		ret = men->open(men, file);
		if (ret)
			return ret;
	}

	spin_lock_irqsave(&men->boardlock, flags);
	if (men->use == 0)
		men->startirq(men);
	men->use++;
	spin_unlock_irqrestore(&men->boardlock, flags);
	return 0;
}

static int menable_release(struct inode *inode, struct file *file)
{
	struct siso_menable *men = file->private_data;
	unsigned long flags;

	/* headlock is needed for men_cleanup() to avoid reverse locking
	 * dependency between headlock and boardlock. */
	spin_lock(&men->headlock);
	spin_lock_irqsave(&men->boardlock, flags);

	if (--men->use == 0)
		men_cleanup(men, flags);
	else
		spin_unlock_irqrestore(&men->boardlock, flags);

	spin_unlock(&men->headlock);
	return 0;
}

static const struct file_operations menable_fops = {
	.owner = THIS_MODULE,
	.unlocked_ioctl = menable_ioctl,
	.compat_ioctl = menable_compat_ioctl,
	.open = menable_open,
	.release = menable_release,
};

static struct class *menable_class;

static DEFINE_PCI_DEVICE_TABLE(menable_pci_table) = {
	/* microEnable III/XXL */
	{ .vendor = 0x10b5, .device = 0x9656, .subvendor = 0x10b5,
			.subdevice = 0x1087, .driver_data = 1 },
	/* microEnable III */
	{ .vendor = 0x1ae8, .device = 0x0a30, .driver_data = 1 },
	/* microEnable III-XXL */
	{ .vendor = 0x1ae8, .device = 0x0a31, .driver_data = 2 },
	/* microEnable IV AD1-CL */
	{ .vendor = 0x1ae8, .device = 0x0a40, .driver_data = 3 },
	/* microEnable IV VD1-CL */
	{ .vendor = 0x1ae8, .device = 0x0a41, .driver_data = 4 },
	/* microEnable IV VD4-CL */
	{ .vendor = 0x1ae8, .device = 0x0a42, .driver_data = 5 },
	/* microEnable IV VD4-CL */
	{ .vendor = 0x1ae8, .device = 0x0a44, .driver_data = 6 },
	/* microEnable IV AS1-CL */
	{ .vendor = 0x1ae8, .device = 0x0a45, .driver_data = 7 },
	/* microEnable IV AQ4-GE */
	{ .vendor = 0x1ae8, .device = 0x0e42, .driver_data = 8 },
	/* microEnable IV VQ4-GE */
	{ .vendor = 0x1ae8, .device = 0x0e44, .driver_data = 9 },
	{ 0 },
};

MODULE_DEVICE_TABLE(pci, menable_pci_table);

static const char *
men_boards[] = {
	"unknown",
	"microEnable III",
	"microEnable III-XXL",
	"microEnable IV AD1-CL",
	"microenable IV VD1-CL",
	"microEnable IV AD4-CL",
	"microenable IV VD4-CL",
	"microEnable IV AS1-CL",
	"microenable IV AQ4-GE",
	"microenable IV VQ4-GE"
};

static void menable_pci_remove(struct pci_dev *pdev)
{
	struct siso_menable *men = pci_get_drvdata(pdev);
	int i;
	unsigned long flags;

	get_device(&men->dev);

	spin_lock(&men->headlock);
	spin_lock_irqsave(&men->designlock, flags);
	while (men->design_changing) {
		spin_unlock_irqrestore(&men->designlock, flags);
		schedule();
		spin_lock_irqsave(&men->designlock, flags);
	}
	men->design_changing = true;
	spin_unlock_irqrestore(&men->designlock, flags);

	spin_lock_irqsave(&men->boardlock, flags);
	men->stopirq(men);
	men->releasing = true;
	spin_unlock_irqrestore(&men->boardlock, flags);

	devm_free_irq(&men->pdev->dev, men->pdev->irq, men);
	i = men_alloc_dma(men, 0);
	BUG_ON(i != 0);

	sysfs_remove_link(&men->dev.kobj, "pci_dev");
	device_remove_file(&men->dev, &dev_attr_design_name);
	device_remove_file(&men->dev, &dev_attr_dma_channels);
	for (i = 0; i < men->num_uiq; i++)
		men_uiq_remove(men->uiqs[i]);

	men->exit(men);

	device_unregister(&men->dev);
	cdev_del(&men->cdev);

	pci_pool_destroy(men->pool);
	pci_set_drvdata(pdev, NULL);
	put_device(&men->dev);
}

static int menable_pci_probe(struct pci_dev *,
				const struct pci_device_id *);

static struct pci_driver menable_pci_driver = {
	.name =		DRIVER_NAME,
	.id_table =	menable_pci_table,
	.probe =	menable_pci_probe,
	.remove =	menable_pci_remove,
};

static int menable_pci_probe(struct pci_dev *pdev,
				const struct pci_device_id *id)
{
	struct siso_menable *men;
	int ret;
	int board;
	unsigned long flags;

	ret = pcim_enable_device(pdev);
	if (ret) {
		dev_err(&pdev->dev, "pcim_enable_device() failed\n");
		return ret;
	}

	board = id->driver_data;

	men = kzalloc(sizeof(*men), GFP_KERNEL);
	if (men == NULL)
		return -ENOMEM;

	men->owner = menable_class->owner;
	men->dev.parent = &pdev->dev;
	men->pdev = pdev;
	men->dev.release = menable_obj_release;
	men->dev.class = menable_class;
	men->dev.driver = &menable_pci_driver.driver;
	spin_lock(&idxlock);
	men->idx = maxidx++;
	spin_unlock(&idxlock);
	dev_set_name(&men->dev, "menable%i", men->idx);

	men->board = board;
	men->releasing = false;
	men->design_changing = true;

	pci_set_master(pdev);
	pci_set_drvdata(pdev, men);
	spin_lock_init(&men->boardlock);
	lockdep_set_class(&men->boardlock, &men_board_lock);
	spin_lock_init(&men->designlock);
	lockdep_set_class(&men->designlock, &men_design_lock);
	spin_lock_init(&men->headlock);
	lockdep_set_class(&men->headlock, &men_head_lock);
	INIT_LIST_HEAD(&men->heads);

	cdev_init(&men->cdev, &menable_fops);
	men->cdev.owner = men->dev.class->owner;
	kobject_set_name(&men->cdev.kobj, "menablec%i", men->idx);

	ret = cdev_add(&men->cdev, devr + men->idx, 1);
	if (ret)
		goto err_release;

	men->dev.devt = men->cdev.dev;
	ret = device_register(&men->dev);
	if (ret)
		goto err_cdev;

	dev_info(&men->dev, "device is a %s\n", men_boards[board]);

	ret = pci_request_regions(pdev, DRIVER_NAME);
	if (ret)
		goto err_dev;

	men->runtime_base = pcim_iomap(pdev, 0, 0);
	if (men->runtime_base == NULL)
		goto err_dev;

	ret = device_create_file(&men->dev, &dev_attr_dma_channels);
	if (ret)
		goto err_dev;

	if (is_me3(men))
		ret = me3_probe(men);
	else
		ret = me4_probe(men);

	if (ret)
		goto err_board_dev;

	ret = device_create_file(&men->dev, &dev_attr_design_name);
	if (ret)
		goto err_sysfs_name;
	ret = sysfs_create_link(&men->dev.kobj, &men->pdev->dev.kobj,
			"pci_dev");
	if (ret)
		goto err_sysfs_link;

	ret = men_add_dmas(men);
	if (ret)
		goto err_dma_scale;

	spin_lock_irqsave(&men->boardlock, flags);
	men->design_changing = false;
	spin_unlock_irqrestore(&men->boardlock, flags);

	return 0;
err_dma_scale:
	sysfs_remove_link(&men->dev.kobj, "pci_dev");
err_sysfs_link:
	device_remove_file(&men->dev, &dev_attr_design_name);
err_sysfs_name:
	if (board == 1)
		me3_init_cleanup(men);
	pci_pool_destroy(men->pool);
err_board_dev:
	device_remove_file(&men->dev, &dev_attr_dma_channels);
err_dev:
	device_unregister(&men->dev);
err_cdev:
	cdev_del(&men->cdev);
err_release:
	pci_set_drvdata(pdev, NULL);
	spin_lock(&idxlock);
	if (men->idx == maxidx - 1)
		maxidx--;
	spin_unlock(&idxlock);
	kfree(men);
	return ret;
}

static int __init menable_init(void)
{
	int ret;

	lockdep_set_class(&idxlock, &men_idx_lock);

	ret = alloc_chrdev_region(&devr, 0, MEN_MAX_NUM, DRIVER_NAME);
	if (ret) {
		printk(KERN_ERR "%s: failed to register device numbers\n",
				DRIVER_NAME);
		return ret;
	}

	menable_class = class_create(THIS_MODULE, "menable");
	if (IS_ERR(menable_class)) {
		ret = PTR_ERR(menable_class);
		goto err_class;
	}

	ret = pci_register_driver(&menable_pci_driver);
	if (ret)
		goto err_reg;

	return 0;
err_reg:
	class_destroy(menable_class);
err_class:
	unregister_chrdev_region(devr, MEN_MAX_NUM);
	return ret;
}

static void __exit menable_exit(void)
{
	pci_unregister_driver(&menable_pci_driver);
	class_destroy(menable_class);
	unregister_chrdev_region(devr, MEN_MAX_NUM);
}

module_init(menable_init);
module_exit(menable_exit);

MODULE_DESCRIPTION("microEnable 3/4 driver");
MODULE_AUTHOR("Silicon Software");
MODULE_LICENSE("GPL");
#if BITS_PER_LONG == 32
MODULE_VERSION("3.9.14");
#else /* BITS_PER_LONG == 32 */
MODULE_VERSION("4.0.3");
#endif /* BITS_PER_LONG == 32 */
